<?php
/**
 * @package         Modules Anywhere
 * @version         8.4.1
 * 
 * @author          Peter van Westen <info@regularlabs.com>
 * @link            https://regularlabs.com
 * @copyright       Copyright © 2025 Regular Labs All Rights Reserved
 * @license         GNU General Public License version 2 or later
 */

namespace RegularLabs\Plugin\System\ModulesAnywhere\Extension;

use RegularLabs\Library\Document as RL_Document;
use RegularLabs\Library\Html as RL_Html;
use RegularLabs\Library\Plugin\System as RL_SystemPlugin;
use RegularLabs\Library\Protect as RL_Protect;
use RegularLabs\Plugin\System\ModulesAnywhere\Params;
use RegularLabs\Plugin\System\ModulesAnywhere\Protect;
use RegularLabs\Plugin\System\ModulesAnywhere\Replace;

defined('_JEXEC') or die;

final class ModulesAnywhere extends RL_SystemPlugin
{
    public    $_lang_prefix = 'MA';
    protected $_jversion    = 4;

    public function processArticle(
        string &$string,
        string $area = 'article',
        string $context = '',
        mixed  $article = null,
        int    $page = 0
    ): void
    {
        Replace::processModules($string, $area, $context, $article);
    }

    /**
     * @param string $buffer
     *
     * @return  bool
     */
    protected function changeDocumentBuffer(string &$buffer): bool
    {
        return Replace::replaceTags($buffer, 'component');
    }

    /**
     * @param string $html
     *
     * @return  bool
     */
    protected function changeFinalHtmlOutput(string &$html): bool
    {
        if (RL_Document::isFeed())
        {
            Replace::replaceTags($html);

            return true;
        }

        // only do stuff in body
        [$pre, $body, $post] = RL_Html::getBody($html);
        Replace::replaceTags($body, 'body');
        $html = $pre . $body . $post;

        return true;
    }

    /**
     * @param string $html
     */
    protected function cleanFinalHtmlOutput(string &$html): void
    {
        RL_Protect::removeAreaTags($html, 'MODA');

        $params = Params::get();

        Protect::unprotectTags($html);

        RL_Protect::removeFromHtmlTagContent($html, Params::getTags(true));
        RL_Protect::removeInlineComments($html, 'Modules Anywhere');

        if ( ! $params->place_comments)
        {
            RL_Protect::removeCommentTags($html, 'Modules Anywhere');
        }
    }

    /**
     * @param object $module
     * @param array  $params
     */
    protected function handleOnAfterRenderModule(object &$module, array &$params): void
    {
        if ( ! isset($module->content))
        {
            return;
        }

        Replace::replaceTags($module->content, 'module');
    }
}
