<?php
/**
 * helper.php - mod_ospropertyrandom
 * 
 * @author    Dang Thuc Dam
 * @copyright Copyright (C) 2022 joomdonation.com. All Rights Reserved.
 * @license   http://www.gnu.org/licenses/gpl-2.0.html GNU/GPL
 * @link      http://www.joomdonation.com
 * @support   Forum - http://www.joomdonation.com/forum.html
 */

defined('_JEXEC') or die('Restricted access');

use Joomla\CMS\Object\CMSObject;
use Joomla\CMS\Factory;
use Joomla\Utilities\ArrayHelper;
use Joomla\CMS\Language\Text;

class modOSpropertyramdomHelper extends CMSObject
{
    /**
     * Constructor
     *
     * @param   object  $params  Module parameters
     */
    public function __construct($params)
    {
        $this->params = $params;
    }

    /**
     * Get properties based on module parameters
     *
     * @return  array  List of property objects
     */
    public function getProperty()
    {
        $db = Factory::getContainer()->get('db');
        $user = Factory::getApplication()->getIdentity();
        $langSuffix = (string)OSPHelper::getFieldSuffix();
        
        // Get module parameters
        $params = [
            'latest' => $this->params->get('latest', 0),
            'category' => $this->params->get('category', 0),
            'type' => $this->params->get('type', 0),
            'featured' => $this->params->get('featured', 0),
            'sold' => $this->params->get('sold', 0),
            'openHouse' => $this->params->get('open_house', 0),
            'orderBy' => $this->params->get('orderby', 'created'),
            'orderType' => $this->params->get('ordertype', 'desc'),
            'recentProperties' => $this->params->get('recent_properties', 0),
            'propertyIds' => $this->params->get('propertyids', ''),
            'agentId' => $this->params->get('agent_id', 0),
            'companyId' => $this->params->get('company_id', 0),
            'countryIds' => $this->params->get('country_ids', ''),
            'stateIds' => $this->params->get('state_ids', ''),
            'cityIds' => $this->params->get('city_ids', ''),
            'specificPropertyIds' => $this->params->get('property_ids', ''),
            'excludePropertyIds' => $this->params->get('exclude_property_ids', ''),
            'openFrom' => $this->params->get('open_from', ''),
            'openTo' => $this->params->get('open_to', ''),
            'onlyOpen' => $this->params->get('only_open', 0),
            'maxProperties' => $this->params->get('max_properties', '')
        ];

        // Build WHERE conditions
        $whereConditions = $this->buildWhereConditions($params);
        
        // Handle recent properties
        $recentIds = [];
        if ($params['recentProperties'] == 1 && !empty($_COOKIE['recent_properties_viewed'])) {
            $recentIds = array_reverse(explode(',', $_COOKIE['recent_properties_viewed']));
            if (!empty($recentIds)) {
                $whereConditions[] = 'p.id IN (' . implode(',', array_map('intval', $recentIds)) . ')';
            }
        }

        // Build the main query
        $query = $db->getQuery(true)
            ->select('p.*, s.state_name, c.country_name, ty.id as typeid, ty.type_name' . $langSuffix . ' as type_name')
            ->from('#__osrs_properties AS p')
            ->join('INNER', '#__osrs_types AS ty ON ty.id = p.pro_type')
            ->join('LEFT', '#__osrs_states AS s ON s.id = p.state')
            ->join('INNER', '#__osrs_countries AS c ON c.id = p.country')
            ->join('INNER', '#__osrs_agents AS ag ON p.agent_id = ag.id')
            ->where('p.published = 1')
            ->where('p.approved = 1')
            ->where('p.access IN (' . implode(',', $user->getAuthorisedViewLevels()) . ')');

        // Add additional conditions
        if (!empty($whereConditions)) {
            $query->where($whereConditions);
        }

        // Handle category filter
        if ($params['category'] > 0) {
            $categoryIds = implode(',', HelperOspropertyCommon::getSubCategories($params['category'], []));
            $query->where('p.id IN (SELECT pid FROM #__osrs_property_categories WHERE category_id IN (' . $categoryIds . '))');
        }

        // Set ordering
        if ($params['recentProperties'] == 1 && !empty($recentIds)) {
            $query->order('FIELD(p.id, ' . implode(',', array_map('intval', $recentIds)) . ')');
        } else {
            $query->order($db->escape($params['orderBy']) . ' ' . $db->escape($params['orderType']));
        }

        // Set limit
        if ($params['maxProperties'] > 0) {
            $query->setLimit((int)$params['maxProperties']);
        }

        $db->setQuery($query);
		//echo $db->getQuery();
        $properties = $db->loadObjectList();

        // Process property data
        return $this->processProperties($properties, $langSuffix);
    }

    /**
     * Build WHERE conditions based on parameters
     *
     * @param   array  $params  Module parameters
     * @return  array  Array of WHERE conditions
     */
    private function buildWhereConditions(array $params): array
    {
        $where = [];

        // Handle specific property IDs
        if (!empty($params['propertyIds'])) {
            $ids = ArrayHelper::toInteger(explode(',', $params['propertyIds']));
            $where[] = 'p.id IN (' . implode(',', $ids) . ')';
        }

        // Handle location filters
        foreach (['country' => 'countryIds', 'state' => 'stateIds', 'city' => 'cityIds'] as $field => $param) {
            if (!empty($params[$param])) {
                $ids = ArrayHelper::toInteger(explode(',', $params[$param]));
                $where[] = "p.{$field} IN (" . implode(',', $ids) . ")";
            }
        }

        // Handle other filters
        if ($params['type'] > 0) {
            $where[] = 'p.pro_type = ' . (int)$params['type'];
        }
        if ($params['featured'] == 1) {
            $where[] = 'p.isFeatured = 1';
        }
        if ($params['sold'] > 0) {
            $where[] = 'p.isSold = ' . (int)$params['sold'];
        }
        if ($params['openHouse'] == 1) {
            $where[] = 'p.id IN (SELECT pid FROM #__osrs_property_open WHERE end_to >= NOW())';
        }
        if ($params['onlyOpen'] == 1) {
            $where[] = 'p.id IN (SELECT pid FROM #__osrs_property_open)';
        }
        if (!empty($params['openFrom'])) {
            $where[] = "p.id IN (SELECT pid FROM #__osrs_property_open WHERE start_from >= " . Factory::getContainer()->get('db')->quote($params['openFrom']) . ")";
        }
        if (!empty($params['openTo'])) {
            $where[] = "p.id IN (SELECT pid FROM #__osrs_property_open WHERE end_to <= " . Factory::getContainer()->get('db')->quote($params['openTo']) . ")";
        }
        if (!empty($params['excludePropertyIds'])) {
            $ids = ArrayHelper::toInteger(explode(',', $params['excludePropertyIds']));
            $where[] = 'p.id NOT IN (' . implode(',', $ids) . ')';
        }
        if ($params['agentId'] > 0) {
            $where[] = 'p.agent_id = ' . (int)$params['agentId'];
        }
        if ($params['companyId'] > 0) {
            $where[] = 'p.company_id = ' . (int)$params['companyId'];
        }
        if (!empty($params['specificPropertyIds'])) {
            $ids = ArrayHelper::toInteger(explode(',', $params['specificPropertyIds']));
            $where[] = 'p.id IN (' . implode(',', $ids) . ')';
        }

        return $where;
    }

    /**
     * Process properties to add additional information
     *
     * @param   array   $properties  Array of property objects
     * @param   string  $langSuffix  Language suffix for field names
     * @return  array   Processed properties
     */
    private function processProperties(array $properties, string $langSuffix = ''): array
    {
        $db = Factory::getContainer()->get('db');

        foreach ($properties as $property) {
            // Handle language-specific fields
            $property->pro_name = OSPHelper::getLanguageFieldValueBackend($property, 'pro_name', $langSuffix);
            $property->pro_small_desc = OSPHelper::getLanguageFieldValue($property, 'pro_small_desc', $langSuffix);

            // Get property image
            $query = $db->getQuery(true)
                ->select('image')
                ->from('#__osrs_photos')
                ->where('pro_id = ' . (int)$property->id)
                ->order('ordering');
            $db->setQuery($query, 0, 1);
            $property->photo = $db->loadResult();

            // Generate price information
            $property->price_information = $this->generatePriceInformation($property);
            
            // Get category names
            $property->category_name = OSPHelper::getCategoryNamesOfPropertyWithLinks($property->id);
        }

        return $properties;
    }

    /**
     * Generate price information HTML for a property
     *
     * @param   object  $property  Property object
     * @return  string  HTML price information
     */
    private function generatePriceInformation($property): string
    {
        ob_start();
        ?>
        <span class="property_price">
            <?php
            $priceText = OSPHelper::getLanguageFieldValue($property, 'price_text');
            if (!empty($priceText)) {
                echo OSPHelper::showPriceText($priceText);
            } elseif ($property->price_call == 1) {
                echo Text::_('OSPROPERTY_CALL_FOR_PRICE');
            } else {
                if ($property->price > 0) {
                    echo OSPHelper::generatePrice($property->curr, $property->price);
                }
                if (!empty($property->rent_time)) {
                    echo ' /' . Text::_($property->rent_time);
                }
            }
            ?>
        </span>
        <?php
        $output = ob_get_contents();
        ob_end_clean();
        return $output;
    }

    /**
     * Get Itemid for a property
     *
     * @param   int  $pid  Property ID
     * @return  int  Itemid
     */
    public static function getItemid($pid): int
    {
        return OSPRoute::getItemid(['property_details', $pid]);
    }
}
