(function (window, document) {
    /**
     * get caret status of the selection of the element
     *
     * @param   {Element}   element         target DOM element
     * @return  {Object}    return
     * @return  {String}    return.text     selected text
     * @return  {Number}    return.start    start position of the selection
     * @return  {Number}    return.end      end position of the selection
     */
    var _getCaretInfo = function (element) {
        var res = {
            text: '',
            start: 0,
            end: 0
        };

        if (!element.value) {
            /* no value or empty string */
            return res;
        }

        try {
            if (window.getSelection) {
                /* except IE */
                res.start = element.selectionStart;
                res.end = element.selectionEnd;
                res.text = element.value.slice(res.start, res.end);
            } else if (document.selection) {
                /* for IE */
                element.focus();

                var range = document.selection.createRange(),
                    range2 = document.body.createTextRange();

                res.text = range.text;

                try {
                    range2.moveToElementText(element);
                    range2.setEndPoint('StartToStart', range);
                } catch (e) {
                    range2 = element.createTextRange();
                    range2.setEndPoint('StartToStart', range);
                }

                res.start = element.value.length - range2.text.length;
                res.end = res.start + range.text.length;
            }
        } catch (e) {
            /* give up */
        }

        return res;
    };

    /**
     * caret operation for the element
     * @type {Object}
     */
    var _CaretOperation = {
        /**
         * get caret position
         *
         * @param   {Element}   element         target element
         * @return  {Object}    return
         * @return  {Number}    return.start    start position for the selection
         * @return  {Number}    return.end      end position for the selection
         */
        getPos: function (element) {
            var tmp = _getCaretInfo(element);
            return {start: tmp.start, end: tmp.end};
        },

        /**
         * set caret position
         *
         * @param   {Element}   element         target element
         * @param   {Object}    toRange         caret position
         * @param   {Number}    toRange.start   start position for the selection
         * @param   {Number}    toRange.end     end position for the selection
         * @param   {String}    caret           caret mode: any of the following: "keep" | "start" | "end"
         */
        setPos: function (element, toRange, caret) {
            caret = this._caretMode(caret);

            if (caret === 'start') {
                toRange.end = toRange.start;
            } else if (caret === 'end') {
                toRange.start = toRange.end;
            }

            element.focus();
            try {
                if (element.createTextRange) {
                    var range = element.createTextRange();

                    if (window.navigator.userAgent.toLowerCase().indexOf("msie") >= 0) {
                        toRange.start = element.value.substr(0, toRange.start).replace(/\r/g, '').length;
                        toRange.end = element.value.substr(0, toRange.end).replace(/\r/g, '').length;
                    }

                    range.collapse(true);
                    range.moveStart('character', toRange.start);
                    range.moveEnd('character', toRange.end - toRange.start);

                    range.select();
                } else if (element.setSelectionRange) {
                    element.setSelectionRange(toRange.start, toRange.end);
                }
            } catch (e) {
                /* give up */
            }
        },

        /**
         * get selected text
         *
         * @param   {Element}   element         target element
         * @return  {String}    return          selected text
         */
        getText: function (element) {
            return _getCaretInfo(element).text;
        },

        /**
         * get caret mode
         *
         * @param   {String}    caret           caret mode
         * @return  {String}    return          any of the following: "keep" | "start" | "end"
         */
        _caretMode: function (caret) {
            caret = caret || "keep";
            if (caret === false) {
                caret = 'end';
            }

            switch (caret) {
                case 'keep':
                case 'start':
                case 'end':
                    break;

                default:
                    caret = 'keep';
            }

            return caret;
        },

        /**
         * replace selected text
         *
         * @param   {Element}   element         target element
         * @param   {String}    text            replacement text
         * @param   {String}    caret           caret mode: any of the following: "keep" | "start" | "end"
         */
        replace: function (element, text, caret) {
            var tmp = _getCaretInfo(element),
                orig = element.value,
                pos = element.scrollTop;
            range = {start: tmp.start, end: tmp.start + text.length};

            element.value = orig.substr(0, tmp.start) + text + orig.substr(tmp.end);
            element.scrollTop = pos;
            this.setPos(element, range, caret);
        },

        /**
         * insert before the selected text
         *
         * @param   {Element}   element         target element
         * @param   {String}    text            insertion text
         * @param   {String}    caret           caret mode: any of the following: "keep" | "start" | "end"
         */
        insertBefore: function (element, text, caret) {
            var tmp = _getCaretInfo(element),
                orig = element.value,
                pos = element.scrollTop;
            range = {start: tmp.start + text.length, end: tmp.end + text.length};

            element.value = orig.substr(0, tmp.start) + text + orig.substr(tmp.start);

            element.scrollTop = pos;

            this.setPos(element, range, caret);
        },

        /**
         * insert after the selected text
         *
         * @param   {Element}   element         target element
         * @param   {String}    text            insertion text
         * @param   {String}    caret           caret mode: any of the following: "keep" | "start" | "end"
         */
        insertAfter: function (element, text, caret) {
            var tmp = _getCaretInfo(element),
                orig = element.value,
                pos = element.scrollTop;

            range = {start: tmp.start, end: tmp.end};

            element.value = orig.substr(0, tmp.end) + text + orig.substr(tmp.end);

            element.scrollTop = pos;
            this.setPos(element, range, caret);
        }
    };

    document.addEventListener('DOMContentLoaded', () => {
        const message = document.getElementById('message');
        message.addEventListener('keydown', function (event) {
            if (event.ctrlKey || event.metaKey) {
                switch (event.key) {
                    case 'q':
                        event.preventDefault();
                        _CaretOperation.insertBefore(message, '[quote]', 'keep');
                        _CaretOperation.insertAfter(message, '[/quote]', 'keep');
                        break;
                    case 'p':
                        event.preventDefault();
                        _CaretOperation.insertBefore(message, '[phpcode]', 'keep');
                        _CaretOperation.insertAfter(message, '[/phpcode]', 'keep');
                        break;
                    case 'j':
                        event.preventDefault();
                        _CaretOperation.insertBefore(message, '[jscode]', 'keep');
                        _CaretOperation.insertAfter(message, '[/jscode]', 'keep');
                        break;
                    case 'h':
                        event.preventDefault();
                        _CaretOperation.insertBefore(message, '[htmlcode]', 'keep');
                        _CaretOperation.insertAfter(message, '[/htmlcode]', 'keep');
                        break;
                    case 'b':
                        event.preventDefault();
                        _CaretOperation.insertBefore(message, '[b]', 'keep');
                        _CaretOperation.insertAfter(message, '[/b]', 'keep');
                        break;
                    case 'i':
                        event.preventDefault();
                        _CaretOperation.insertBefore(message, '[i]', 'keep');
                        _CaretOperation.insertAfter(message, '[/i]', 'keep');
                        break;
                    case 'm':
                        event.preventDefault();
                        _CaretOperation.insertBefore(message, '[mail]', 'keep');
                        _CaretOperation.insertAfter(message, '[/mail]', 'keep');
                        break;
                }
            }
        });
    });
}(window, document));