<?php
/**
 * @package        Joomla
 * @subpackage     Helpdesk Pro
 * @author         Tuan Pham Ngoc
 * @copyright      Copyright (C) 2013 - 2026 Ossolution Team
 * @license        GNU/GPL, see LICENSE.php
 */

namespace OSSolution\HelpdeskPro\Site\Helper;

use Exception;
use Joomla\CMS\Factory;
use Joomla\CMS\Form\FormHelper;
use Joomla\CMS\HTML\HTMLHelper;
use Joomla\CMS\Language\Text;
use Joomla\CMS\Table\Table;
use Joomla\Database\DatabaseDriver;
use Joomla\Registry\Registry;
use RuntimeException;
use SimpleXMLElement;

defined('_JEXEC') or die;

class Html
{
	/**
	 * Render ShowOn string
	 *
	 * @param   array  $fields
	 *
	 * @return string
	 */
	public static function renderShowOn($fields)
	{
		$output = [];

		$i = 0;

		foreach ($fields as $name => $values)
		{
			$i++;

			$values = (array) $values;

			$data = [
				'field'  => $name,
				'values' => $values,
				'sign'   => '=',
			];

			$data['op'] = $i > 1 ? 'AND' : '';

			$output[] = json_encode($data);
		}

		return '[' . implode(',', $output) . ']';
	}

	/**
	 * @param   int     $selected
	 * @param   string  $name
	 * @param ?string   $attr
	 * @param   array   $rows
	 * @param   bool    $reload
	 *
	 * @return mixed
	 */
	public static function buildCategoryDropdown(
		$selected,
		$name = 'parent_id',
		$attr = null,
		$rows = [],
		$reload = true
	) {
		if (empty($rows) && $reload)
		{
			/* @var DatabaseDriver $db */
			$db    = Factory::getContainer()->get('db');
			$query = $db->getQuery(true)
				->select('id, parent_id, title')
				->from('#__helpdeskpro_categories')
				->whereIn('category_type', [0, 1])
				->where('published=1');
			$db->setQuery($query);
			$rows = $db->loadObjectList();
		}

		$children = [];

		// first pass - collect children
		foreach ($rows as $v)
		{
			$pt            = (int) $v->parent_id;
			$list          = $children[$pt] ?? [];
			$list[]        = $v;
			$children[$pt] = $list;
		}

		$list      = HTMLHelper::_('menu.treerecurse', 0, '', [], $children, 9999, 0, 0);
		$options   = [];
		$options[] = HTMLHelper::_('select.option', '0', Text::_('HDP_SELECT_CATEGORY'));

		foreach ($list as $item)
		{
			$options[] = HTMLHelper::_('select.option', $item->id, '&nbsp;&nbsp;&nbsp;' . $item->treename);
		}

		return HTMLHelper::_(
			'select.genericlist',
			$options,
			$name,
			[
				'option.text.toHtml' => false,
				'option.text'        => 'text',
				'option.value'       => 'value',
				'list.attr'          => $attr,
				'list.select'        => $selected,
			]
		);
	}

	/**
	 * Function to render a common layout which is used in different views
	 *
	 * @param   string  $layout
	 * @param   array   $data
	 *
	 * @return string
	 * @throws Exception
	 */
	public static function loadCommonLayout($layout, $data = [])
	{
		$app       = Factory::getApplication();
		$themeFile = str_replace('/tmpl', '', $layout);

		if (is_file($layout))
		{
			$path = $layout;
		}
		elseif (is_file(JPATH_THEMES . '/' . $app->getTemplate() . '/html/com_helpdeskpro/' . $themeFile))
		{
			$path = JPATH_THEMES . '/' . $app->getTemplate() . '/html/com_helpdeskpro/' . $themeFile;
		}
		elseif (is_file(JPATH_ROOT . '/components/com_helpdeskpro/View/' . $layout))
		{
			$path = JPATH_ROOT . '/components/com_helpdeskpro/View/' . $layout;
		}
		else
		{
			throw new RuntimeException(Text::_('The given shared template path is not exist'));
		}

		// Start an output buffer.
		ob_start();
		extract($data);

		// Load the layout.
		include $path;

		// Get the layout contents.
		return ob_get_clean();
	}

	/**
	 * Get page params of the given view
	 *
	 * @param $active
	 * @param $views
	 *
	 * @return Registry
	 */
	public static function getViewParams($active, $views)
	{
		if ($active && isset($active->query['view']) && in_array($active->query['view'], $views))
		{
			return $active->getParams();
		}

		return new Registry();
	}

	/**
	 * Helper method to prepare meta data for the document
	 *
	 * @param   Registry  $params
	 *
	 * @param   Table     $item
	 */
	public static function prepareDocument($params, $item = null)
	{
		$app      = Factory::getApplication();
		$document = $app->getDocument();

		$siteNamePosition = $app->get('sitename_pagetitles');
		$pageTitle        = $params->get('page_title');

		if ($pageTitle)
		{
			if ($siteNamePosition == 0)
			{
				$document->setTitle($pageTitle);
			}
			elseif ($siteNamePosition == 1)
			{
				$document->setTitle($app->get('sitename') . ' - ' . $pageTitle);
			}
			else
			{
				$document->setTitle($pageTitle . ' - ' . $app->get('sitename'));
			}
		}

		if (!empty($item->meta_keywords))
		{
			$document->setMetaData('keywords', $item->meta_keywords);
		}
		elseif ($params->get('menu-meta_keywords'))
		{
			$document->setMetaData('keywords', $params->get('menu-meta_keywords'));
		}

		if (!empty($item->meta_description))
		{
			$document->setMetaData('description', $item->meta_description);
		}
		elseif ($params->get('menu-meta_description'))
		{
			$document->setDescription($params->get('menu-meta_description'));
		}

		if ($params->get('robots'))
		{
			$document->setMetaData('robots', $params->get('robots'));
		}
	}

	/**
	 * Add choices JS to dropdown
	 *
	 * @param   string  $html
	 * @param   string  $hint
	 */
	public static function getChoicesJsSelect($html, $hint = '')
	{
		Text::script('JGLOBAL_SELECT_NO_RESULTS_MATCH');
		Text::script('JGLOBAL_SELECT_PRESS_TO_SELECT');

		Factory::getApplication()->getDocument()->getWebAssetManager()
			->usePreset('choicesjs')
			->useScript('webcomponent.field-fancy-select');

		$attributes = [];

		$hint = $hint ?: Text::_('JGLOBAL_TYPE_OR_SELECT_SOME_OPTIONS');

		$attributes[] = 'placeholder="' . $hint . '""';
		$attributes[] = 'search-placeholder="' . $hint . '""';


		return '<joomla-field-fancy-select ' . implode(' ', $attributes) . '>' . $html . '</joomla-field-fancy-select>';
	}

	/**
	 * Method to get input to allow selecting user
	 *
	 * @param   string  $value
	 * @param   string  $fieldName
	 *
	 * @return string
	 */
	public static function getUserInput($value, $fieldName)
	{
		// Prevent notices on PHP 8.1 due to a bug on user form field
		if (!$value)
		{
			$value = '';
		}

		$field = FormHelper::loadFieldType('User');

		$element = new SimpleXMLElement('<field />');
		$element->addAttribute('name', $fieldName);
		$element->addAttribute('class', 'readonly input-medium');

		$field->setup($element, $value);

		return $field->input;
	}
}