<?php
/**
 * @package        Joomla
 * @subpackage     Helpdesk Pro
 * @author         Tuan Pham Ngoc
 * @copyright      Copyright (C) 2013 - 2026 Ossolution Team
 * @license        GNU/GPL, see LICENSE.php
 */

namespace OSSolution\HelpdeskPro\Site\Helper;

use BBCodeParser;
use Exception;
use Joomla\CMS\Component\ComponentHelper;
use Joomla\CMS\Factory;
use Joomla\CMS\HTML\HTMLHelper;
use Joomla\CMS\Language\LanguageHelper;
use Joomla\CMS\Language\Multilanguage;
use Joomla\CMS\Mail\Mail;
use Joomla\CMS\Mail\MailHelper;
use Joomla\CMS\Router\Route;
use Joomla\CMS\Table\Table;
use Joomla\CMS\Uri\Uri;
use Joomla\CMS\User\User;
use Joomla\CMS\User\UserFactoryInterface;
use Joomla\CMS\User\UserHelper;
use Joomla\Database\DatabaseDriver;
use Joomla\Database\ParameterType;
use Joomla\Event\Event;
use Joomla\Filesystem\File;
use OSL\Config\Config;
use OSL\Utils\Database as DatabaseUtils;
use OSSolution\HelpdeskPro\Admin\Table\EmailLog;
use OSSolution\HelpdeskPro\Admin\Table\Message;
use OSSolution\HelpdeskPro\Admin\Table\Ticket;
use OSSolution\HelpdeskPro\Site\Helper\Route as RouteHelper;
use stdClass;

defined('_JEXEC') or die;

class Helper
{
	/**
	 * Field types that support filtering
	 */
	const FILTERABLE_FIELD_TYPES = ['List', 'Checkboxes', 'Radio'];

	/**
	 * Get current installed version of Helpdesk Pro
	 *
	 * @return string
	 */
	public static function getInstalledVersion()
	{
		return '6.3.0';
	}

	/**
	 * Helper method to check if the extension is running on Joomla 4
	 *
	 * @return bool
	 */
	public static function isJoomla4()
	{
		return version_compare(JVERSION, '4.0.0', '>=');
	}

	/**
	 * Helper method to check if the extension is running on Joomla 5
	 *
	 * @return bool
	 */
	public static function isJoomla5()
	{
		return version_compare(JVERSION, '4.4.99', '>');
	}

	/**
	 * Get hased field name to store the time which form started to be rendered
	 *
	 * @return string
	 */
	public static function getHashedFieldName()
	{
		$app = Factory::getApplication();

		$siteName = $app->get('sitename');
		$secret   = $app->get('secret');

		return md5('HDP' . $siteName . $secret);
	}

	/**
	 * Get configuration data and store in config object
	 *
	 * @return Config
	 */
	public static function getConfig()
	{
		static $config;

		if (!$config)
		{
			/* @var DatabaseDriver $db */
			$db    = Factory::getContainer()->get('db');
			$query = $db->getQuery(true)
				->select('*')->from('#__helpdeskpro_configs');
			$db->setQuery($query);
			$rows = $db->loadObjectList();
			$data = [];

			foreach ($rows as $row)
			{
				$data[$row->config_key] = $row->config_value;
			}

			$config = new Config($data);
		}

		return $config;
	}

	/**
	 * Get the email messages used for sending emails
	 *
	 * @return Config
	 */
	public static function getEmailMessages()
	{
		static $message;

		if (!$message)
		{
			/* @var DatabaseDriver $db */
			$db    = Factory::getContainer()->get('db');
			$query = $db->getQuery(true)
				->select('*')
				->from('#__helpdeskpro_emails');
			$db->setQuery($query);
			$rows = $db->loadObjectList();
			$data = [];

			foreach ($rows as $row)
			{
				$data[$row->email_key] = $row->email_message;
			}

			$message = new Config($data);
		}

		return $message;
	}

	/**
	 * Get all custom fields assigned to certain category
	 *
	 * @param   int    $categoryId
	 * @param   array  $filters
	 *
	 * @return mixed
	 */
	public static function getFields($categoryId, $filters = [], $fieldSuffix = '')
	{
		$user = Factory::getApplication()->getIdentity();
		/* @var DatabaseDriver $db */
		$db    = Factory::getContainer()->get('db');
		$query = $db->getQuery(true)
			->select('id, name, fieldtype')
			->select($db->quoteName('title' . $fieldSuffix, 'title'))
			->from('#__helpdeskpro_fields')
			->where(
				'(category_id = -1 OR id IN (SELECT field_id FROM #__helpdeskpro_field_categories WHERE category_id=' . (int) $categoryId . '))'
			)
			->where('published = 1')
			->order('ordering');

		if (!$user->authorise('core.admin', 'com_helpdeskpro'))
		{
			$query->whereIn('access', $user->getAuthorisedViewLevels());
		}

		foreach ($filters as $filter)
		{
			$query->where($filter);
		}

		$db->setQuery($query);

		return $db->loadObjectList();
	}

	/**
	 * Get all custom fields
	 *
	 * @param   string  $fieldSuffix
	 *
	 * @return array
	 */
	public static function getAllFields($fieldSuffix = null)
	{
		$user = Factory::getApplication()->getIdentity();
		/* @var DatabaseDriver $db */
		$db    = Factory::getContainer()->get('db');
		$query = $db->getQuery(true)
			->select('*, extra AS extra_attributes, 0 AS max_length, "" AS place_holder')
			->from('#__helpdeskpro_fields')
			->where('published = 1')
			->order('ordering');

		if (!$user->authorise('core.admin', 'com_helpdeskpro'))
		{
			$query->whereIn('access', $user->getAuthorisedViewLevels());
		}

		if (!$fieldSuffix)
		{
			$fieldSuffix = Helper::getFieldSuffix();
		}

		if ($fieldSuffix)
		{
			DatabaseUtils::getMultilingualFields(
				$query,
				['title', 'description', 'values', 'default_values'],
				$fieldSuffix
			);
		}

		$db->setQuery($query);

		return $db->loadObjectList();
	}

	/**
	 * Get the association array contains the relationship between field and categories
	 *
	 * @return array
	 */
	public static function getFieldCategoryRelation()
	{
		$user = Factory::getApplication()->getIdentity();
		/* @var DatabaseDriver $db */
		$db    = Factory::getContainer()->get('db');
		$query = $db->getQuery(true)
			->select('id')
			->from('#__helpdeskpro_fields')
			->where('published = 1')
			->where('category_id = -1');

		if (!$user->authorise('core.admin', 'com_helpdeskpro'))
		{
			$query->whereIn('access', $user->getAuthorisedViewLevels());
		}

		$db->setQuery($query);

		$relation[0] = $db->loadColumn();

		$query->clear()
			->select('b.*')
			->from('#__helpdeskpro_fields AS a')
			->innerJoin('#__helpdeskpro_field_categories AS b ON a.id = b.field_id')
			->where('a.published = 1');

		if (!$user->authorise('core.admin', 'com_helpdeskpro'))
		{
			$query->whereIn('a.access', $user->getAuthorisedViewLevels());
		}

		$db->setQuery($query);
		$fieldCategories = $db->loadObjectList();

		foreach ($fieldCategories as $fieldCategory)
		{
			$relation[$fieldCategory->category_id][] = $fieldCategory->field_id;
		}

		return $relation;
	}

	/**
	 * Add result back to event trigger
	 *
	 * @param   Event  $event
	 * @param   mixed  $data
	 *
	 * @return void
	 */
	public static function addEventListenerResult(Event $event, $data)
	{
		$result   = $event->getArgument('result', []);
		$result[] = $data;
		$event->setArgument('result', $result);
	}

	/**
	 * Get specify config value
	 *
	 * @param   string  $key
	 *
	 * @return string
	 */
	public static function getConfigValue($key)
	{
		$config = static::getConfig();

		return $config->get($key);
	}

	/**
	 * Get Itemid of Helpdesk Pro componnent
	 *
	 * @return int
	 */
	public static function getItemid()
	{
		return RouteHelper::findView('tickets', 0);
	}

	/**
	 * Get field suffix used in sql query
	 *
	 * @param   string  $activeLanguage
	 *
	 * @return string
	 */
	public static function getFieldSuffix($activeLanguage = null)
	{
		$prefix = '';

		if ($activeLanguage !== '*' && Multilanguage::isEnabled())
		{
			if (!$activeLanguage)
			{
				$activeLanguage = Factory::getLanguage()->getTag();
			}

			if ($activeLanguage != self::getDefaultLanguage())
			{
				$languages = LanguageHelper::getLanguages('lang_code');

				if (isset($languages[$activeLanguage]))
				{
					$prefix = '_' . $languages[$activeLanguage]->sef;
				}
			}
		}

		return $prefix;
	}

	/**
	 * Get all dependencies custom fields of a given field
	 *
	 * @param $id
	 *
	 * @return array
	 */
	public static function getAllDependencyFields($id)
	{
		/* @var DatabaseDriver $db */
		$db    = Factory::getContainer()->get('db');
		$query = $db->getQuery(true);

		$queue  = [$id];
		$fields = [$id];

		while (count($queue))
		{
			$masterFieldId = array_pop($queue);

			//Get list of dependency fields of this master field
			$query->clear()
				->select('id')
				->from('#__helpdeskpro_fields')
				->where('depend_on_field_id = ' . $masterFieldId);
			$db->setQuery($query);
			$rows = $db->loadObjectList();

			foreach ($rows as $row)
			{
				$queue[]  = $row->id;
				$fields[] = $row->id;
			}
		}

		return $fields;
	}

	/**
	 * Load highlighter script so that the code will be highlighted
	 */
	public static function loadHighlighter()
	{
		$assetVersion = self::getInstalledVersion();

		Factory::getApplication()
			->getDocument()
			->getWebAssetManager()
			->registerAndUseScript(
				'com_helpdeskpro.prism',
				'media/com_helpdeskpro/assets/js/prismjs/prism.js',
				['version' => $assetVersion]
			)
			->registerAndUseStyle(
				'com_helpdeskpro.prism',
				'media/com_helpdeskpro/assets/js/prismjs/prism.css',
				['version' => $assetVersion]
			);
	}

	/**
	 *
	 *
	 * @return string
	 */
	public static function validateEngine()
	{
		$dateNow    = HTMLHelper::_('date', Factory::getDate(), 'Y/m/d');
		$validClass = [
			'',
			'validate[custom[integer]]',
			'validate[custom[number]]',
			'validate[custom[email]]',
			'validate[custom[url]]',
			'validate[custom[phone]]',
			"validate[custom[date],past[$dateNow]]",
			'validate[custom[ipv4]]',
			'validate[minSize[6]]',
			'validate[maxSize[12]]',
			'validate[custom[integer],min[-5]]',
			'validate[custom[integer],max[50]]',
		];

		return json_encode($validClass);
	}

	/**
	 *
	 * Get role of the given user
	 *
	 * @param $userId
	 *
	 * @return mixed
	 */
	public static function getUserRole($userId = 0)
	{
		static $roles = [];

		if (!$userId)
		{
			$userId = Factory::getApplication()->getIdentity()->id;
		}

		if (!isset($roles[$userId]))
		{
			$role = 'user';

			if ($userId)
			{
				$user = Factory::getUser($userId);

				if ($user->authorise('core.admin', 'com_helpdeskpro'))
				{
					$role = 'admin';
				}
				else
				{
					$manageCategoryIds = Helper::getTicketCategoryIds($user->username);

					if (count($manageCategoryIds) >= 1 && $manageCategoryIds[0] != 0)
					{
						$role = 'manager';
					}
					else
					{
						$config = self::getConfig();

						if ($config->staff_group_id)
						{
							$staffs = Database::getAllStaffs($config->staff_group_id);

							foreach ($staffs as $staff)
							{
								if ($staff->id == $userId)
								{
									$role = 'staff';
									break;
								}
							}
						}
					}
				}
			}

			$roles[$userId] = $role;
		}

		return $roles[$userId];
	}

	/**
	 * Get list of current online users
	 *
	 * @return array
	 */
	public static function getOnlineUsers()
	{
		/* @var DatabaseDriver $db */
		$db    = Factory::getContainer()->get('db');
		$query = $db->getQuery(true)
			->select('DISTINCT u.id')
			->from('#__session AS s')
			->innerJoin('#__users AS u ON s.userid = u.id')
			->where('s.guest = 0');
		$db->setQuery($query);

		return $db->loadColumn();
	}

	/**
	 * Load language from main component
	 *
	 */
	public static function loadLanguage()
	{
		static $loaded;

		if (!$loaded)
		{
			$lang = Factory::getLanguage();
			$tag  = $lang->getTag();

			if (!$tag)
			{
				$tag = 'en-GB';
			}

			$lang->load('com_helpdeskpro', JPATH_ROOT, $tag);
			$loaded = true;
		}
	}

	/**
	 * Display copy right information
	 *
	 */
	public static function displayCopyRight()
	{
		echo '<div class="copyright clearfix row-fluid" style="text-align:center;margin-top: 5px;"><a href="http://joomdonation.com/components/helpdesk-pro.html" target="_blank"><strong>Helpdesk Pro</strong></a> version ' . self::getInstalledVersion(
			) . ', Copyright (C) 2012 - ' . date(
				'Y'
			) . ' <a href="http://joomdonation.com" target="_blank"><strong>Ossolution Team</strong></a></div>';
	}

	/**
	 * Method to call a static overridable helper method
	 *
	 * @param   string  $helper
	 * @param   string  $method
	 * @param   array   $methodArgs
	 *
	 * @return mixed
	 *
	 * @throws Exception
	 */
	public static function callOverridableHelperMethod($helper, $method, $methodArgs = [])
	{
		$callableMethods   = [];
		$callableMethods[] = '\\OSSolution\\HelpdeskPro\\Site\\Override\\Helper\\' . ucfirst($helper) . '::' . $method;
		$callableMethods[] = '\\OSSolution\\HelpdeskPro\\Site\\Helper\\' . ucfirst($helper) . '::' . $method;

		foreach ($callableMethods as $callable)
		{
			if (is_callable($callable))
			{
				return call_user_func_array($callable, $methodArgs);
			}
		}

		throw new Exception(sprintf('Method %s does not exist in the helper %s', $method, $helper));
	}

	/**
	 * Get ticket categories managed by the given user
	 *
	 * @param   string  $username
	 *
	 * @return array
	 */
	public static function getTicketCategoryIds($username)
	{
		static $categories = [];

		if (!isset($categories[$username]))
		{
			/* @var DatabaseDriver $db */
			$db    = Factory::getContainer()->get('db');
			$query = $db->getQuery(true)
				->select('id')
				->from('#__helpdeskpro_categories')
				->where(
					"managers='$username' OR managers LIKE '$username,%' OR managers LIKE '%,$username,%' OR managers LIKE '%,$username'"
				);
			$db->setQuery($query);

			$categoryIds = $db->loadColumn();

			if (!count($categoryIds))
			{
				$categoryIds = [0];
			}

			$categories[$username] = $categoryIds;
		}

		return $categories[$username];
	}

	/**
	 * Method to check to see whether the user can edit the comment
	 *
	 * @param   User  $user
	 * @param   int   $id
	 *
	 * @return bool
	 */
	public static function canUpdateComment($user, $id)
	{
		$config = self::getConfig();

		// Only allow editing comment if not using HTML editor
		if (!$user->id || $config->use_html_editor)
		{
			return false;
		}

		// User has edit permission in helpdesk pro will be able to edit the comment
		if ($user->authorise('core.edit', 'com_helpdeskpro'))
		{
			return true;
		}

		// Owner of the ticket can edit the comment

		/* @var DatabaseDriver $db */
		$db    = Factory::getContainer()->get('db');
		$query = $db->getQuery(true)
			->select('user_id')
			->from('#__helpdeskpro_messages')
			->where('id = ' . $id);
		$db->setQuery($query);
		$message = $db->loadObject();

		if ($message && $user->id && ($message->user_id == $user->id) && $user->authorise(
				'core.editown',
				'com_helpdeskpro'
			))
		{
			return true;
		}

		// Otherwise, he could not edit the comment
		return false;
	}

	/**
	 * Helper methd to check to see whether the user can edit the comment
	 *
	 * @param   User  $user
	 * @param   int   $id
	 *
	 * @return bool
	 */
	public static function canDeleteComment($user, $id)
	{
		// User has edit permission in helpdesk pro will be able to edit the comment
		if ($user->authorise('core.delete', 'com_helpdeskpro'))
		{
			return true;
		}

		// Owner of the ticket can edit the comment

		/* @var DatabaseDriver $db */
		$db    = Factory::getContainer()->get('db');
		$query = $db->getQuery(true)
			->select('user_id')
			->from('#__helpdeskpro_messages')
			->where('id = ' . $id);
		$db->setQuery($query);
		$message = $db->loadObject();

		if ($message && $user->id && ($message->user_id == $user->id))
		{
			return true;
		}

		// Otherwise, he could not edit the comment

		return false;
	}

	/**
	 * Check ticket access
	 *
	 * @param   Ticket  $item
	 *
	 * @return bool
	 */
	public static function checkTicketAccess($item)
	{
		$user   = Factory::getApplication()->getIdentity();
		$config = Helper::getConfig();

		if (!$item->id)
		{
			return false;
		}

		if ($item->is_ticket_code && $config->allow_public_user_submit_ticket)
		{
			return true;
		}

		if (!$user->id)
		{
			return false;
		}

		if ($user->id == $item->user_id || $user->email == $item->email)
		{
			return true;
		}

		if ($user->id == $item->staff_id)
		{
			return true;
		}

		if ($user->authorise('core.admin', 'com_helpdeskpro'))
		{
			return true;
		}

		$managedCategoryIds = Helper::getTicketCategoryIds($user->username);

		if (in_array($item->category_id, $managedCategoryIds))
		{
			return true;
		}

		return false;
	}

	/**
	 * Get CB avatar of the given user to display on the ticket
	 *
	 * @param   int  $userId
	 *
	 * @return string relative path to the avatar
	 */
	public static function getCBAvatar($userId)
	{
		static $avatars;

		if (!isset($avatars[$userId]))
		{
			/* @var DatabaseDriver $db */
			$db    = Factory::getContainer()->get('db');
			$query = $db->getQuery(true)
				->select('avatar')
				->from('#__comprofiler')
				->where('user_id=' . $userId . ' AND avatarapproved=1');
			$db->setQuery($query);
			$avatar = $db->loadResult();
			if (!$avatar)
			{
				$avatar = '';
			}
			$avatars[$userId] = $avatar;
		}

		return $avatars[$userId];
	}

	/**
	 * This function is used to check to see whether we need to update the database to support multilingual or not
	 *
	 * @return boolean
	 */
	public static function isSyncronized()
	{
		/* @var DatabaseDriver $db */
		$db             = Factory::getContainer()->get('db');
		$fields         = $db->getTableColumns('#__helpdeskpro_categories');
		$fields         = array_keys($fields);
		$extraLanguages = self::getLanguages();

		if (count($extraLanguages))
		{
			foreach ($extraLanguages as $extraLanguage)
			{
				$prefix = $extraLanguage->sef;

				if (!in_array('alias_' . $prefix, $fields))
				{
					return false;
				}
			}
		}

		return true;
	}

	/**
	 * Syncronize Helpdesk Pro database to support multilingual
	 */
	public static function setupMultilingual()
	{
		/* @var DatabaseDriver $db */
		$db        = Factory::getContainer()->get('db');
		$languages = self::getLanguages();

		if (count($languages))
		{
			foreach ($languages as $language)
			{
				#Process for #__helpdeskpro_categories table							
				$prefix = $language->sef;
				$fields = array_keys($db->getTableColumns('#__helpdeskpro_categories'));

				if (!in_array('title_' . $prefix, $fields))
				{
					$fieldName = 'title_' . $prefix;
					$sql       = "ALTER TABLE  `#__helpdeskpro_categories` ADD  `$fieldName` VARCHAR( 255 );";
					$db->setQuery($sql);
					$db->execute();

					$fieldName = 'description_' . $prefix;
					$sql       = "ALTER TABLE  `#__helpdeskpro_categories` ADD  `$fieldName` TEXT NULL;";
					$db->setQuery($sql);
					$db->execute();
				}

				if (!in_array('alias_' . $prefix, $fields))
				{
					$fieldName = 'alias_' . $prefix;
					$sql       = "ALTER TABLE  `#__helpdeskpro_categories` ADD  `$fieldName` VARCHAR( 255 );";
					$db->setQuery($sql);
					$db->execute();
				}

				#Process for #__helpdeskpro_statuses table
				$fields = array_keys($db->getTableColumns('#__helpdeskpro_statuses'));

				if (!in_array('title_' . $prefix, $fields))
				{
					$fieldName = 'title_' . $prefix;
					$sql       = "ALTER TABLE  `#__helpdeskpro_statuses` ADD  `$fieldName` VARCHAR( 255 );";
					$db->setQuery($sql);
					$db->execute();
				}

				#Process for #__helpdeskpro_priorities table
				$fields = array_keys($db->getTableColumns('#__helpdeskpro_priorities'));

				if (!in_array('title_' . $prefix, $fields))
				{
					$fieldName = 'title_' . $prefix;
					$sql       = "ALTER TABLE  `#__helpdeskpro_priorities` ADD  `$fieldName` VARCHAR( 255 );";
					$db->setQuery($sql);
					$db->execute();
				}

				#Process for #__helpdeskpro_fields table
				$fields = array_keys($db->getTableColumns('#__helpdeskpro_fields'));
				if (!in_array('title_' . $prefix, $fields))
				{
					$fieldName = 'title_' . $prefix;
					$sql       = "ALTER TABLE  `#__helpdeskpro_fields` ADD  `$fieldName` VARCHAR( 255 );";
					$db->setQuery($sql);
					$db->execute();

					$fieldName = 'description_' . $prefix;
					$sql       = "ALTER TABLE  `#__helpdeskpro_fields` ADD  `$fieldName` TEXT NULL;";
					$db->setQuery($sql);
					$db->execute();

					$fieldName = 'values_' . $prefix;
					$sql       = "ALTER TABLE  `#__helpdeskpro_fields` ADD  `$fieldName` TEXT NULL;";
					$db->setQuery($sql);
					$db->execute();

					$fieldName = 'default_values_' . $prefix;
					$sql       = "ALTER TABLE  `#__helpdeskpro_fields` ADD  `$fieldName` TEXT NULL;";
					$db->setQuery($sql);
					$db->execute();
				}

				// Process for #__helpdeskpro_articles tabl
				$fields = array_keys($db->getTableColumns('#__helpdeskpro_articles'));

				if (!in_array('title_' . $prefix, $fields))
				{
					$fieldName = 'title_' . $prefix;
					$sql       = "ALTER TABLE  `#__helpdeskpro_articles` ADD  `$fieldName` VARCHAR( 255 );";
					$db->setQuery($sql);
					$db->execute();

					$fieldName = 'text_' . $prefix;
					$sql       = "ALTER TABLE  `#__helpdeskpro_articles` ADD  `$fieldName` TEXT NULL;";
					$db->setQuery($sql);
					$db->execute();
				}

				if (!in_array('alias_' . $prefix, $fields))
				{
					$fieldName = 'alias_' . $prefix;
					$sql       = "ALTER TABLE  `#__helpdeskpro_articles` ADD  `$fieldName` VARCHAR( 255 );";
					$db->setQuery($sql);
					$db->execute();
				}
			}
		}
	}

	/**
	 * Process BB code for the given message
	 *
	 * @param   string  $message
	 *
	 * @return string
	 */
	public static function processBBCode($message)
	{
		require_once JPATH_ROOT . '/administrator/components/com_helpdeskpro/libraries/bbcodeparser.php';

		$config = self::getConfig();

		if (!$config->use_html_editor)
		{
			$message = nl2br($message);
		}

		return BBCodeParser::parse($message);
	}

	/**
	 *  Method to add ticket or comment attachments to mailer for sending emails
	 *
	 * @param   Mail      $mailer
	 * @param   stdClass  $row
	 */
	protected static function addAttachmentsToMailer($mailer, $row)
	{
		$originalFileNames = explode('|', $row->original_filenames);
		$attachments       = explode('|', $row->attachments);

		foreach ($attachments as $i => $attachment)
		{
			$originalFileName = $originalFileNames[$i];
			if (file_exists(JPATH_ROOT . '/media/com_helpdeskpro/attachments/' . $attachment))
			{
				$mailer->addAttachment(
					JPATH_ROOT . '/media/com_helpdeskpro/attachments/' . $attachment,
					$originalFileName
				);
			}
		}
	}

	/**
	 * Send email to super administrator and user
	 *
	 * @param   stdClass  $row     The message object
	 * @param   stdClass  $ticket  The ticket object
	 * @param   Config    $config
	 */
	public static function sendTicketUpdatedEmailToCustomer($row, $ticket, $config)
	{
		$user = Factory::getApplication()->getIdentity();

		// This is the case comment is added using reply via email feature, set $user to the user who added the comment
		if (!$user->id && $row->user_id)
		{
			$user = Factory::getContainer()->get(UserFactoryInterface::class)->loadUserById($row->user_id);
		}

		$mailer = static::getMailer();

		/* @var DatabaseDriver $db */
		$db          = Factory::getContainer()->get('db');
		$message     = self::getEmailMessages();
		$fieldSuffix = self::getFieldSuffix($ticket->language);

		$query = $db->getQuery(true)
			->select('name')
			->from('#__users')
			->where('id = ' . (int) $row->user_id);
		$db->setQuery($query);
		$manageName = $db->loadResult();

		$replaces = self::getCommonTicketTags($ticket, $config);

		if ($config->use_html_editor)
		{
			$replaces['ticket_comment'] = $row->message;
		}
		else
		{
			$replaces['ticket_comment'] = nl2br($row->message);
		}

		// Process BBCODE before sending email message
		if ($config->process_bb_code)
		{
			$replaces['ticket_comment'] = self::processBBCode($replaces['ticket_comment']);
		}

		$replaces['manager_name'] = $manageName;

		if ($fieldSuffix && $message->{'ticket_updated_user_email_subject' . $fieldSuffix})
		{
			$subject = $message->{'ticket_updated_user_email_subject' . $fieldSuffix};
		}
		else
		{
			$subject = $message->ticket_updated_user_email_subject;
		}

		if ($fieldSuffix && strlen(strip_tags($message->{'ticket_updated_user_email_body' . $fieldSuffix})))
		{
			$body = $message->{'ticket_updated_user_email_body' . $fieldSuffix};
		}
		else
		{
			$body = $message->ticket_updated_user_email_body;
		}

		foreach ($replaces as $key => $value)
		{
			$key     = strtoupper($key);
			$body    = str_replace("[$key]", $value, $body);
			$subject = str_replace("[$key]", $value, $subject);
		}

		if ($config->notify_manager_when_staff_reply && ($ticket->staff_id == $user->id))
		{
			// Staff reply to ticket
			$emails = self::callOverridableHelperMethod('Helper', 'getTicketNotificationEmails', [$ticket]);
			$mailer->addBcc($emails);
		}
		elseif ($config->notify_other_managers_when_a_manager_reply && $ticket->staff_id != $user->id)
		{
			// Admin or manager reply to a ticket
			$emails = self::callOverridableHelperMethod('Helper', 'getTicketNotificationEmails', [$ticket]);

			// Exclude email of the current manager from receiving notification
			$emails = array_diff($emails, [$user->email]);

			if (count($emails))
			{
				$mailer->addBcc($emails);
			}
		}

		// Add ticket attachments to email if configured
		if ($config->send_ticket_attachments_to_email && $row->attachments)
		{
			static::addAttachmentsToMailer($mailer, $row);
		}

		if ($ticket->user_id)
		{
			$userEmail = static::getUserEmail($ticket->user_id);

			if ($userEmail != $ticket->email)
			{
				$ticket->email = $userEmail;

				$query->clear()
					->update('#__helpdeskpro_tickets')
					->set('email = ' . $db->quote($userEmail))
					->where('user_id = ' . $ticket->user_id);
				$db->setQuery($query)
					->execute();
			}
		}

		$userEmails = static::getReceiveConfirmationEmails($ticket);

		if (MailHelper::isEmailAddress($ticket->email))
		{
			$userEmails[] = $ticket->email;
		}

		$userEmails = array_unique($userEmails);

		static::send($mailer, $userEmails, $subject, $body, 2, 'ticket_updated_email');
	}

	/**
	 * Send email to super administrator and user
	 *
	 * @param   stdClass  $row     The message object
	 * @param   stdClass  $ticket  The ticket object
	 * @param   Config    $config
	 */
	public static function sendTicketUpdatedEmailToManagers($row, $ticket, $config)
	{
		$mailer      = static::getMailer();
		$message     = self::getEmailMessages();
		$fieldSuffix = self::getFieldSuffix($ticket->language);

		$replaces = self::getCommonTicketTags($ticket, $config);

		if ($config->use_html_editor)
		{
			$replaces['ticket_comment'] = $row->message;
		}
		else
		{
			$replaces['ticket_comment'] = nl2br($row->message);
		}

		// Process BBCODE before sending email message
		if ($config->process_bb_code)
		{
			$replaces['ticket_comment'] = self::processBBCode($replaces['ticket_comment']);
		}

		$emails = self::callOverridableHelperMethod('Helper', 'getTicketNotificationEmails', [$ticket, true]);

		if ($message->{'ticket_updated_admin_email_subject' . $fieldSuffix})
		{
			$subject = $message->{'ticket_updated_admin_email_subject' . $fieldSuffix};
		}
		else
		{
			$subject = $message->ticket_updated_admin_email_subject;
		}

		if (strlen(strip_tags($message->{'ticket_updated_admin_email_body' . $fieldSuffix})))
		{
			$body = $message->{'ticket_updated_admin_email_body' . $fieldSuffix};
		}
		else
		{
			$body = $message->ticket_updated_admin_email_body;
		}

		foreach ($replaces as $key => $value)
		{
			$key     = strtoupper($key);
			$body    = str_replace("[$key]", $value, $body);
			$subject = str_replace("[$key]", $value, $subject);
		}

		// Add ticket attachments to email if configured
		if ($config->send_ticket_attachments_to_email && $row->attachments)
		{
			static::addAttachmentsToMailer($mailer, $row);
		}

		static::send($mailer, $emails, $subject, $body, 1, 'ticket_updated_email');
	}

	/**
	 * Send email to super administrator, managers, staffs when an internal comment is added
	 *
	 * @param   Message  $row     The message object
	 * @param   Ticket   $ticket  The ticket object
	 * @param   Config   $config
	 */
	public static function sendInternalCommentAddedEmail($row, $ticket, $config)
	{
		$mailer      = static::getMailer();
		$message     = self::getEmailMessages();
		$fieldSuffix = self::getFieldSuffix($ticket->language);
		$user        = Factory::getApplication()->getIdentity();

		$replaces = self::getCommonTicketTags($ticket, $config);

		if ($config->use_html_editor)
		{
			$replaces['ticket_comment'] = $row->message;
		}
		else
		{
			$replaces['ticket_comment'] = nl2br($row->message);
		}

		$replaces['user_username'] = $user->username;
		$replaces['user_name']     = $user->name;
		$replaces['user_email']    = $user->email;

		$emails = self::callOverridableHelperMethod('Helper', 'getTicketNotificationEmails', [$ticket, true]);

		if ($fieldSuffix && $message->{'internal_comment_added_email_subject' . $fieldSuffix})
		{
			$subject = $message->{'internal_comment_added_email_subject' . $fieldSuffix};
		}
		else
		{
			$subject = $message->internal_comment_added_email_subject;
		}

		if ($fieldSuffix && strlen(strip_tags($message->{'internal_comment_added_email_body' . $fieldSuffix})))
		{
			$body = $message->{'internal_comment_added_email_body' . $fieldSuffix};
		}
		else
		{
			$body = $message->internal_comment_added_email_body;
		}

		foreach ($replaces as $key => $value)
		{
			$key     = strtoupper($key);
			$body    = str_replace("[$key]", $value, $body);
			$subject = str_replace("[$key]", $value, $subject);
		}

		// Add ticket attachments to email if configured
		if ($config->send_ticket_attachments_to_email && $row->attachments)
		{
			static::addAttachmentsToMailer($mailer, $row);
		}

		$emails = array_diff($emails, [$user->email]);

		if (count($emails))
		{
			static::send($mailer, $emails, $subject, $body, 1, 'internal_comment_added_email');
		}
	}

	/**
	 * Send email to super administrator and user
	 *
	 * @param   object  $row     The message object
	 * @param   object  $ticket  The ticket object
	 * @param   object  $config
	 */
	public static function sendNewTicketNotificationEmails($row, $config)
	{
		$mailer      = static::getMailer();
		$message     = self::getEmailMessages();
		$fieldSuffix = self::getFieldSuffix($row->language);

		$replaces = self::getCommonTicketTags($row, $config);

		$emails = self::callOverridableHelperMethod('Helper', 'getTicketNotificationEmails', [$row]);

		if ($message->{'new_ticket_admin_email_subject' . $fieldSuffix})
		{
			$subject = $message->{'new_ticket_admin_email_subject' . $fieldSuffix};
		}
		else
		{
			$subject = $message->new_ticket_admin_email_subject;
		}

		if (strlen(strip_tags($message->{'new_ticket_admin_email_body' . $fieldSuffix})))
		{
			$body = $message->{'new_ticket_admin_email_body' . $fieldSuffix};
		}
		else
		{
			$body = $message->new_ticket_admin_email_body;
		}

		foreach ($replaces as $key => $value)
		{
			$body    = str_ireplace("[$key]", $value, $body);
			$subject = str_ireplace("[$key]", $value, $subject);
		}

		// Add ticket attachments to email if configured
		if ($config->send_ticket_attachments_to_email && $row->attachments)
		{
			static::addAttachmentsToMailer($mailer, $row);
		}

		static::send($mailer, $emails, $subject, $body, 1, 'new_ticket_notification_email');

		$mailer->clearAllRecipients();
		$mailer->clearAttachments();

		$userEmails = static::getReceiveConfirmationEmails($row);

		if (MailHelper::isEmailAddress($row->email))
		{
			$userEmails[] = $row->email;
		}

		$userEmails = array_unique($userEmails);

		if (!count($userEmails))
		{
			return;
		}

		//Send email to user
		if ($message->{'new_ticket_user_email_subject' . $fieldSuffix})
		{
			$subject = $message->{'new_ticket_user_email_subject' . $fieldSuffix};
		}
		else
		{
			$subject = $message->new_ticket_user_email_subject;
		}

		if (strlen(strip_tags($message->{'new_ticket_user_email_body' . $fieldSuffix})))
		{
			$body = $message->{'new_ticket_user_email_body' . $fieldSuffix};
		}
		else
		{
			$body = $message->new_ticket_user_email_body;
		}

		foreach ($replaces as $key => $value)
		{
			$body    = str_ireplace("[$key]", $value, $body);
			$subject = str_ireplace("[$key]", $value, $subject);
		}

		if ($row->user_id)
		{
			$userEmail = static::getUserEmail($row->user_id);

			if ($userEmail != $row->email)
			{
				$row->email = $userEmail;

				/* @var DatabaseDriver $db */
				$db    = Factory::getContainer()->get('db');
				$query = $db->getQuery(true)
					->update('#__helpdeskpro_tickets')
					->set('email = ' . $db->quote($userEmail))
					->where('user_id = ' . $row->user_id);
				$db->setQuery($query)
					->execute();
			}
		}

		static::send($mailer, $userEmails, $subject, $body, 2, 'new_ticket_notification_email');
	}

	/**
	 * Send Ticket assigned email
	 *
	 * @param   object  $row
	 * @param   object  $config
	 */
	public static function sendTicketAssignedEmails($row, $config)
	{
		if (!$row->staff_id)
		{
			return;
		}

		$mailer                   = static::getMailer();
		$message                  = self::getEmailMessages();
		$fieldSuffix              = self::getFieldSuffix($row->language);
		$replaces                 = self::getCommonTicketTags($row, $config);
		$replaces['MANAGER_NAME'] = Factory::getApplication()->getIdentity()->name;

		if ($message->{'customer_ticket_assigned_email_subject' . $fieldSuffix})
		{
			$subject = $message->{'customer_ticket_assigned_email_subject' . $fieldSuffix};
		}
		else
		{
			$subject = $message->customer_ticket_assigned_email_subject;
		}

		if (strlen(strip_tags($message->{'customer_ticket_assigned_email_body' . $fieldSuffix})))
		{
			$body = $message->{'customer_ticket_assigned_email_body' . $fieldSuffix};
		}
		else
		{
			$body = $message->customer_ticket_assigned_email_body;
		}

		foreach ($replaces as $key => $value)
		{
			$body    = str_ireplace("[$key]", $value, $body);
			$subject = str_ireplace("[$key]", $value, $subject);
		}

		static::send($mailer, [$row->email], $subject, $body, 2, 'ticket_assigned_email');

		$mailer->clearAllRecipients();

		// Get staff email

		/* @var DatabaseDriver $db */
		$db    = Factory::getContainer()->get('db');
		$query = $db->getQuery(true)
			->select('email')
			->from('#__users')
			->where('id = ' . $row->staff_id);
		$db->setQuery($query);
		$staffEmail = $db->loadResult();

		if (!MailHelper::isEmailAddress($staffEmail))
		{
			return;
		}

		//Send email to staff
		if ($message->{'ticket_assiged_email_subject' . $fieldSuffix})
		{
			$subject = $message->{'ticket_assiged_email_subject' . $fieldSuffix};
		}
		else
		{
			$subject = $message->ticket_assiged_email_subject;
		}

		if (strlen(strip_tags($message->{'ticket_assiged_email_body' . $fieldSuffix})))
		{
			$body = $message->{'ticket_assiged_email_body' . $fieldSuffix};
		}
		else
		{
			$body = $message->ticket_assiged_email_body;
		}

		foreach ($replaces as $key => $value)
		{
			$body    = str_ireplace("[$key]", $value, $body);
			$subject = str_ireplace("[$key]", $value, $subject);
		}

		static::send($mailer, [$staffEmail], $subject, $body, 1, 'ticket_assigned_email');
	}


	/**
	 * Send ticket closed email to customer
	 *
	 * @param $row
	 * @param $config
	 */
	public static function sendTicketClosedEmail($row, $config)
	{
		$user = Factory::getApplication()->getIdentity();

		$mailer      = static::getMailer();
		$message     = self::getEmailMessages();
		$fieldSuffix = self::getFieldSuffix($row->language);
		$replaces    = self::getCommonTicketTags($row, $config);

		// Customer close ticket, sending email to managers
		if ($user->id == $row->user_id)
		{
			if ($message->{'manager_ticket_closed_email_subject' . $fieldSuffix})
			{
				$subject = $message->{'manager_ticket_closed_email_subject' . $fieldSuffix};
			}
			else
			{
				$subject = $message->manager_ticket_closed_email_subject;
			}

			if (strlen(strip_tags($message->{'manager_ticket_closed_email_body' . $fieldSuffix})))
			{
				$body = $message->{'manager_ticket_closed_email_body' . $fieldSuffix};
			}
			else
			{
				$body = $message->manager_ticket_closed_email_body;
			}

			foreach ($replaces as $key => $value)
			{
				$body    = str_ireplace("[$key]", $value, $body);
				$subject = str_ireplace("[$key]", $value, $subject);
			}

			// Get all managers + staff email
			$emails = self::callOverridableHelperMethod('Helper', 'getTicketNotificationEmails', [$row, true]);

			static::send($mailer, $emails, $subject, $body, 1, 'ticket_closed_email');

			return;
		}

		// Manager or staff closes ticket, send notification email to customer
		$replaces['MANAGER_NAME'] = Factory::getApplication()->getIdentity()->name;

		if ($message->{'customer_ticket_closed_email_subject' . $fieldSuffix})
		{
			$subject = $message->{'customer_ticket_closed_email_subject' . $fieldSuffix};
		}
		else
		{
			$subject = $message->customer_ticket_closed_email_subject;
		}

		if (strlen(strip_tags($message->{'customer_ticket_closed_email_body' . $fieldSuffix})))
		{
			$body = $message->{'customer_ticket_closed_email_body' . $fieldSuffix};
		}
		else
		{
			$body = $message->customer_ticket_closed_email_body;
		}

		foreach ($replaces as $key => $value)
		{
			$body    = str_ireplace("[$key]", $value, $body);
			$subject = str_ireplace("[$key]", $value, $subject);
		}

		$userEmails = static::getReceiveConfirmationEmails($row);

		if (MailHelper::isEmailAddress($row->email))
		{
			$userEmails[] = $row->email;
		}

		$userEmails = array_unique($userEmails);

		if (count($userEmails))
		{
			static::send($mailer, $userEmails, $subject, $body, 2, 'ticket_closed_email');
		}
	}

	/**
	 * Send ticket status change email to customer
	 *
	 * @param $row
	 * @param $config
	 */
	public static function sendTicketStatusChangeEmail($row, $oldStatus, $newStatus, $config)
	{
		$user = Factory::getApplication()->getIdentity();

		$mailer = static::getMailer();

		/* @var DatabaseDriver $db */
		$db          = Factory::getContainer()->get('db');
		$message     = self::getEmailMessages();
		$fieldSuffix = self::getFieldSuffix($row->language);
		$replaces    = self::getCommonTicketTags($row, $config);

		$query = $db->getQuery(true)
			->select('title')
			->from('#__helpdeskpro_statuses')
			->where('id = ' . (int) $oldStatus);

		if ($fieldSuffix)
		{
			DatabaseUtils::getMultilingualFields($query, ['title'], $fieldSuffix);
		}

		$db->setQuery($query);

		$replaces['old_status'] = $db->loadResult();

		$query->clear('where')
			->where('id = ' . (int) $newStatus);
		$db->setQuery($query);

		$replaces['new_status'] = $db->loadResult();

		// Customer close ticket, sending email to managers
		if ($user->id == $row->user_id)
		{
			if ($message->{'manager_ticket_status_changed_email_subject' . $fieldSuffix})
			{
				$subject = $message->{'manager_ticket_status_changed_email_subject' . $fieldSuffix};
			}
			else
			{
				$subject = $message->manager_ticket_status_changed_email_subject;
			}

			if (strlen(strip_tags($message->{'manager_ticket_status_changed_email_body' . $fieldSuffix})))
			{
				$body = $message->{'manager_ticket_status_changed_email_body' . $fieldSuffix};
			}
			else
			{
				$body = $message->manager_ticket_status_changed_email_body;
			}

			foreach ($replaces as $key => $value)
			{
				$body    = str_ireplace("[$key]", $value, $body);
				$subject = str_ireplace("[$key]", $value, $subject);
			}

			// Get all managers + staff emails
			$emails = self::callOverridableHelperMethod('Helper', 'getTicketNotificationEmails', [$row, true]);

			static::send($mailer, $emails, $subject, $body, 1, 'ticket_status_changed_email');

			return;
		}

		// Manager or staff change ticket status, send notification email to customer
		$replaces['MANAGER_NAME'] = Factory::getApplication()->getIdentity()->name;

		if ($message->{'customer_ticket_status_changed_email_subject' . $fieldSuffix})
		{
			$subject = $message->{'customer_ticket_status_changed_email_subject' . $fieldSuffix};
		}
		else
		{
			$subject = $message->customer_ticket_status_changed_email_subject;
		}

		if (strlen(strip_tags($message->{'customer_ticket_status_changed_email_body' . $fieldSuffix})))
		{
			$body = $message->{'customer_ticket_status_changed_email_body' . $fieldSuffix};
		}
		else
		{
			$body = $message->customer_ticket_status_changed_email_body;
		}

		foreach ($replaces as $key => $value)
		{
			$body    = str_ireplace("[$key]", $value, $body);
			$subject = str_ireplace("[$key]", $value, $subject);
		}

		$userEmails = static::getReceiveConfirmationEmails($row);

		if (MailHelper::isEmailAddress($row->email))
		{
			$userEmails[] = $row->email;
		}

		$userEmails = array_unique($userEmails);

		static::send($mailer, $userEmails, $subject, $body, 2, 'ticket_status_changed_email');
	}

	/**
	 * Get common tags related to ticket and use it for sending emails
	 *
	 * @param   stdClass  $row
	 * @param   Config    $config
	 *
	 * @return array
	 */
	public static function getCommonTicketTags($row, $config)
	{
		$siteUrl     = Uri::root();
		$fieldSuffix = self::getFieldSuffix($row->language);

		/* @var DatabaseDriver $db */
		$db       = Factory::getContainer()->get('db');
		$query    = $db->getQuery(true);
		$replaces = [];

		$replaces['ticket_id']      = $row->id;
		$replaces['ticket_subject'] = $row->subject;
		$replaces['name']           = $row->name;
		$replaces['email']          = $row->email;

		if ($row->staff_id)
		{
			$query->clear()
				->select('name')
				->from('#__users')
				->where('id = ' . (int) $row->staff_id);
			$db->setQuery($query);
			$replaces['staff_name'] = $db->loadResult();
		}
		else
		{
			$replaces['staff_name'] = '';
		}

		if ($config->use_html_editor)
		{
			$replaces['ticket_message'] = $row->message;
		}
		else
		{
			$replaces['ticket_message'] = nl2br($row->message);
		}

		// Process BBCODE before sending email message
		if ($config->process_bb_code)
		{
			$replaces['ticket_message'] = self::processBBCode($replaces['ticket_message']);
		}

		$replaces['frontend_link']               = $siteUrl . RouteHelper::getTicketRoute($row->id);
		$replaces['backend_link']                = $siteUrl . 'administrator/index.php?option=com_helpdeskpro&view=ticket&id=' . $row->id;
		$replaces['frontend_link_without_login'] = $siteUrl . RouteHelper::getTicketRoute($row->ticket_code, false);

		$query->clear()
			->select('id, name, title, receive_confirmation_email')
			->from('#__helpdeskpro_fields')
			->where('published=1')
			->where(
				'(category_id = -1 OR id IN (SELECT field_id FROM #__helpdeskpro_field_categories WHERE category_id=' . $row->category_id . '))'
			)
			->order('ordering');
		$db->setQuery($query);
		$rowFields = $db->loadObjectList();
		$fields    = [];

		foreach ($rowFields as $rowField)
		{
			$fields[$rowField->id] = $rowField->name;
		}

		$query->clear()
			->select('*')
			->from('#__helpdeskpro_field_value')
			->where('ticket_id = ' . $row->id);
		$db->setQuery($query);
		$rowValues = $db->loadObjectList();

		foreach ($rowValues as $rowValue)
		{
			$replaces[$fields[$rowValue->field_id]] = $rowValue->field_value;
		}

		$query->clear()
			->select('name')
			->from('#__helpdeskpro_fields');
		$db->setQuery($query);
		$fields = $db->loadColumn();

		foreach ($fields as $field)
		{
			if (!isset($replaces[$field]))
			{
				$replaces[$field] = '';
			}
		}

		$query->clear()
			->select($db->quoteName('title' . $fieldSuffix))
			->from('#__helpdeskpro_categories')
			->where('id = ' . $row->category_id);
		$db->setQuery($query);
		$replaces['ticket_category'] = $replaces['category_title'] = $db->loadResult();

		// Ticket status
		$query->clear()
			->select($db->quoteName('title' . $fieldSuffix))
			->from('#__helpdeskpro_statuses')
			->where('id = ' . (int) $row->status_id);
		$db->setQuery($query);
		$replaces['ticket_status'] = $db->loadResult();

		// Ticket priority
		$query->clear()
			->select($db->quoteName('title' . $fieldSuffix))
			->from('#__helpdeskpro_priorities')
			->where('id = ' . (int) $row->priority_id);
		$db->setQuery($query);
		$replaces['ticket_priority'] = $db->loadResult();

		// Ticket comments
		$query->clear()
			->select($db->quoteName('message'))
			->from('#__helpdeskpro_messages')
			->where('ticket_id = ' . $row->id);
		$db->setQuery($query);
		$comments = $db->loadColumn();

		$replaces['comments'] = implode('<br />***<br />', $comments);

		return $replaces;
	}

	/**
	 * Function to get all available languages except the default language
	 *
	 * @return array object list
	 */
	public static function getLanguages()
	{
		$languages = LanguageHelper::getLanguages('lang_code');

		unset($languages[self::getDefaultLanguage()]);

		return array_values($languages);
	}

	/**
	 * Get front-end default language
	 * @return string
	 */
	public static function getDefaultLanguage()
	{
		return ComponentHelper::getParams('com_languages')->get('site', 'en-GB');
	}

	/**
	 * Get formatted file size of a file
	 *
	 * @param   string  $filePath
	 *
	 * @return string
	 */
	public static function getSize($filePath)
	{
		$kb   = 1024;
		$mb   = 1024 * $kb;
		$gb   = 1024 * $mb;
		$tb   = 1024 * $gb;
		$size = @filesize($filePath);

		if ($size)
		{
			if ($size < $kb)
			{
				$final    = round($size, 2);
				$fileSize = $final . ' ' . 'Byte';
			}
			elseif ($size < $mb)
			{
				$final    = round($size / $kb, 2);
				$fileSize = $final . ' ' . 'KB';
			}
			elseif ($size < $gb)
			{
				$final    = round($size / $mb, 2);
				$fileSize = $final . ' ' . 'MB';
			}
			elseif ($size < $tb)
			{
				$final    = round($size / $gb, 2);
				$fileSize = $final . ' ' . 'GB';
			}
			else
			{
				$final    = round($size / $tb, 2);
				$fileSize = $final . ' ' . 'TB';
			}
		}
		else
		{
			if ($size == 0)
			{
				$fileSize = 'EMPTY';
			}
			else
			{
				$fileSize = 'ERROR';
			}
		}

		return $fileSize;
	}

	/**
	 * Store ticket from input data
	 *
	 * @param   array  $data
	 *
	 * @return bool
	 */
	public static function storeTicket($data)
	{
		/* @var DatabaseDriver $db */
		$db     = Factory::getContainer()->get('db');
		$user   = Factory::getApplication()->getIdentity();
		$config = Helper::getConfig();

		$row = new Ticket($db);
		$row->bind($data);

		if ($user->id)
		{
			$row->user_id = $user->id;
		}
		else
		{
			$sql = 'SELECT id FROM #__users WHERE email=' . $db->quote($data['email']);
			$db->setQuery($sql);
			$row->user_id = $db->loadResult();
		}

		$row->status_id = $config->new_ticket_status_id;

		while (true)
		{
			$ticketCode = strtolower(UserHelper::genRandomPassword(10));
			$sql        = 'SELECT COUNT(*) FROM #__helpdeskpro_tickets WHERE ticket_code=' . $db->quote($ticketCode);
			$db->setQuery($sql);
			$total = $db->loadResult();

			if (!$total)
			{
				break;
			}
		}

		$row->ticket_code  = $ticketCode;
		$row->created_date = $row->modified_date = gmdate('Y-m-d H:i:s');
		$row->store(); //Store custom fields information for this ticket	

		Helper::sendNewTicketNotificationEmails($row, $config);

		return true;
	}

	/**
	 * Get email of user
	 *
	 * @param   int  $userId
	 *
	 * @return string
	 */
	public static function getUserEmail($userId)
	{
		/* @var DatabaseDriver $db */
		$db    = Factory::getContainer()->get('db');
		$query = $db->getQuery(true)
			->select('email')
			->from('#__users')
			->where('id = ' . (int) $userId);
		$db->setQuery($query);

		return $db->loadResult();
	}

	/**
	 * Get avatar of user
	 *
	 * @param   int  $userId
	 *
	 * @return string
	 */
	public static function getUserAvatar($userId)
	{
		static $avatars = [];

		if (!isset($avatars[$userId]))
		{
			// Default avatar
			$avatars[$userId] = 'media/com_helpdeskpro/assets/images/avatars/icon-user.jpeg';

			$role = self::getUserRole($userId);

			if (file_exists(JPATH_ROOT . '/media/com_helpdeskpro/assets/images/avatars/' . $role . '.svg'))
			{
				$avatars[$userId] = 'media/com_helpdeskpro/assets/images/avatars/' . $role . '.svg';
			}

			/* @var DatabaseDriver $db */
			$db    = Factory::getContainer()->get('db');
			$query = $db->getQuery(true);
			$user  = Factory::getContainer()->get(UserFactoryInterface::class)->loadUserById($userId);

			if (file_exists(JPATH_ROOT . '/media/com_helpdeskpro/assets/images/avatars/' . $user->username . '.png'))
			{
				$avatars[$userId] = 'media/com_helpdeskpro/assets/images/avatars/' . $user->username . '.png';
			}
			elseif (ComponentHelper::isInstalled('com_comprofiler') && ComponentHelper::isEnabled('com_comprofiler'))
			{
				$query->select('avatar')
					->from('#__comprofiler')
					->where('user_id=' . $userId . ' AND avatarapproved = 1');
				$db->setQuery($query);
				$avatar = $db->loadResult();

				if ($avatar && file_exists(JPATH_ROOT . '/images/comprofiler/' . $avatar))
				{
					$avatars[$userId] = 'images/comprofiler/' . $avatar;
				}
			}
			elseif (ComponentHelper::isInstalled('com_kunena') && ComponentHelper::isEnabled('com_kunena'))
			{
				$query->select('avatar')
					->from('#__kunena_users')
					->where('userid=' . $userId);
				$db->setQuery($query);
				$avatar = $db->loadResult();

				if ($avatar && file_exists(JPATH_ROOT . '/media/kunena/avatars/resized/size90/' . $avatar))
				{
					$avatars[$userId] = 'media/kunena/avatars/resized/size90/' . $avatar;
				}
				elseif ($avatar && file_exists(JPATH_ROOT . '/media/kunena/avatars/resized/size72/' . $avatar))
				{
					$avatars[$userId] = 'media/kunena/avatars/resized/size72/' . $avatar;
				}
			}
		}

		return $avatars[$userId];
	}

	/**
	 * Return list of emails which are configured to receive confirmation email
	 *
	 * @param   stdClass  $ticket
	 *
	 * @return array
	 */
	public static function getReceiveConfirmationEmails($ticket)
	{
		/* @var DatabaseDriver $db */
		$db    = Factory::getContainer()->get('db');
		$query = $db->getQuery(true)
			->select('b.field_value')
			->from('#__helpdeskpro_fields AS a')
			->innerJoin('#__helpdeskpro_field_value AS b ON a.id = b.field_id')
			->where('a.receive_confirmation_email = 1')
			->where('b.ticket_id = ' . $ticket->id);
		$db->setQuery($query);
		$fieldValues = $db->loadColumn();

		$emails = [];

		foreach ($fieldValues as $fieldValue)
		{
			if (is_string($fieldValue) && strlen(trim($fieldValue)) > 0
				&& MailHelper::isEmailAddress($fieldValue))
			{
				$emails[] = $fieldValue;
			}
		}

		return $emails;
	}

	/**
	 * Get Notification Emails for a given ticket
	 *
	 * @param   Table  $ticket
	 * @param   bool   $includeStaffEmail
	 *
	 * @return array
	 */
	public static function getTicketNotificationEmails($ticket, $includeStaffEmail = false)
	{
		$config = static::getConfig();

		/* @var DatabaseDriver $db */
		$db    = Factory::getContainer()->get('db');
		$query = $db->getQuery(true)
			->select('managers')
			->from('#__helpdeskpro_categories')
			->where('id = ' . (int) $ticket->category_id);
		$db->setQuery($query);
		$managers = trim($db->loadResult());

		if ($managers)
		{
			$managers = explode(',', $managers);
			$managers = array_map('trim', $managers);
			$query->clear()
				->select('email')
				->from('#__users')
				->whereIn('username', $managers, ParameterType::STRING);
			$db->setQuery($query);
			$emails = $db->loadColumn();
		}
		else
		{
			$config = static::getConfig();
			$emails = array_map('trim', explode(',', $config->notification_emails));
		}

		if ($includeStaffEmail && $config->notify_staff_when_customer_reply && $ticket->staff_id)
		{
			$query->clear()
				->select('email')
				->from('#__users')
				->where('id = ' . (int) $ticket->staff_id);
			$db->setQuery($query);
			$email = $db->loadResult();

			if (MailHelper::isEmailAddress($email))
			{
				$emails[] = $email;
			}
		}

		return $emails;
	}

	/**
	 * Create and initialize mailer object from configuration data
	 *
	 * @return Mail
	 */
	public static function getMailer()
	{
		$mailer = Factory::getMailer();
		$config = static::getConfig();

		if ($config->reply_to_email)
		{
			$mailer->addReplyTo($config->reply_to_email);
		}

		if ($config->from_name)
		{
			$fromName = $config->from_name;
		}
		else
		{
			$fromName = Factory::getApplication()->get('fromname');
		}

		if ($config->from_email)
		{
			$fromEmail = $config->from_email;
		}
		else
		{
			$fromEmail = Factory::getApplication()->get('mailfrom');
		}

		$mailer->setSender([$fromEmail, $fromName]);

		$mailer->isHtml(true);

		return $mailer;
	}

	/**
	 * Process sending after all the data has been initialized
	 *
	 * @param   Mail    $mailer
	 * @param   array   $emails
	 * @param   string  $subject
	 * @param   string  $body
	 * @param   int     $sentTo
	 * @param   string  $emailType
	 *
	 */
	public static function send($mailer, $emails, $subject, $body, $sentTo = 0, $emailType = '')
	{
		if (empty($subject))
		{
			return;
		}

		$emails = (array) $emails;

		$emails = array_map('trim', $emails);

		for ($i = 0, $n = count($emails); $i < $n; $i++)
		{
			if (!MailHelper::isEmailAddress($emails[$i]))
			{
				unset($emails[$i]);
			}
		}

		$emails = array_unique($emails);

		if (count($emails) == 0)
		{
			return;
		}

		$config = self::getConfig();

		$emails = array_values($emails);

		$body = MailHelper::convertRelativeToAbsoluteUrls($body);

		foreach ($emails as $email)
		{
			if ($config->get('log_emails', 0))
			{
				$db              = Factory::getDbo();
				$row             = new EmailLog($db);
				$row->sent_at    = Factory::getDate()->toSql();
				$row->email      = $email;
				$row->subject    = $subject;
				$row->body       = $body;
				$row->sent_to    = $sentTo;
				$row->email_type = $emailType;

				$row->store();
			}

			try
			{
				$mailer->addRecipient($email);

				$mailer->setSubject($subject)
					->setBody($body)
					->Send();

				$mailer->clearAddresses();
			}
			catch (Exception $e)
			{
				Factory::getApplication()->enqueueMessage($e->getMessage(), 'warning');
			}
		}
	}

	/**
	 * Convert all img tags to use absolute URL
	 *
	 * @param   string  $text
	 *
	 * @return string
	 */
	public static function convertImgTags($text)
	{
		$app = Factory::getApplication();

		$siteUrl    = Uri::root();
		$rootURL    = rtrim(Uri::root(), '/');
		$subpathURL = Uri::root(true);

		if (!empty($subpathURL) && ($subpathURL != '/'))
		{
			$rootURL = substr($rootURL, 0, -1 * strlen($subpathURL));
		}

		// Replace index.php URI by SEF URI.
		if (strpos($text, 'href="index.php?') !== false)
		{
			preg_match_all('#href="index.php\?([^"]+)"#m', $text, $matches);

			foreach ($matches[1] as $urlQueryString)
			{
				if ($app->isClient('site'))
				{
					$text = str_replace(
						'href="index.php?' . $urlQueryString . '"',
						'href="' . $rootURL . Route::_('index.php?' . $urlQueryString) . '"',
						$text
					);
				}
				else
				{
					$text = str_replace(
						'href="index.php?' . $urlQueryString . '"',
						'href="' . $siteUrl . 'index.php?' . $urlQueryString . '"',
						$text
					);
				}
			}
		}

		$patterns     = [];
		$replacements = [];
		$i            = 0;
		$src_exp      = '/src="(.*?)"/';
		$link_exp     = "[^http:\/\/www\.|^www\.|^https:\/\/|^http:\/\/]";

		preg_match_all($src_exp, $text, $out, PREG_SET_ORDER);

		foreach ($out as $val)
		{
			$links = preg_match($link_exp, $val[1], $match, PREG_OFFSET_CAPTURE);

			if ($links == '0')
			{
				$patterns[$i]     = "\"$val[1]";
				$replacements[$i] = $siteUrl . $val[1];
				$replacements[$i] = "\"$replacements[$i]";
			}

			$i++;
		}

		$text = str_replace($patterns, $replacements, $text);

		return $text;
	}

	/**
	 * Check to see if the current user has permission to edit the given ticket
	 *
	 * @param   stdClass  $ticket
	 *
	 * @return bool
	 */
	public static function canEditTicket($ticket)
	{
		$config = self::getConfig();
		$role   = self::getUserRole();

		if (!in_array($role, explode(',', $config->get('edit_ticket_roles', 'admin,manager'))))
		{
			return false;
		}

		// Admin can edit any tickets
		if ($role === 'admin')
		{
			return true;
		}

		$user = Factory::getApplication()->getIdentity();

		// Managers can only edit tickets belong to the category which he managed
		if ($role == 'manager')
		{
			$managedCategoryIds = self::getTicketCategoryIds($user->username);

			if (in_array($ticket->category_id, $managedCategoryIds))
			{
				return true;
			}
		}

		if ($role == 'staff')
		{
			return $ticket->staff_id == $user->id;
		}

		if ($role == 'user')
		{
			return $user->id == $ticket->user_id || $user->email == $ticket->email;
		}

		return false;
	}

	/**
	 * Method to check to see if the current user can add internal comment to ticket
	 *
	 * @return bool
	 */
	public static function canAddInternalComment()
	{
		$config = self::getConfig();
		$role   = self::getUserRole();

		if (in_array($role, explode(',', $config->get('add_internal_comment_roles', 'admin,manager'))))
		{
			return true;
		}

		return false;
	}

	/**
	 * Method to check to see if the current user can view internal comments of ticket
	 *
	 * @return bool
	 */
	public static function canViewInternalComments()
	{
		$config = self::getConfig();
		$role   = self::getUserRole();

		if (in_array($role, explode(',', $config->get('view_internal_comment_roles', 'admin,manager,staff'))))
		{
			return true;
		}

		return false;
	}

	/**
	 * @param   mixed  $item  The ticket or message/comment record
	 *
	 * @return bool
	 */
	public static function canDeleteAttachment($item): bool
	{
		$role = self::getUserRole();

		// If this is admin or manage, he can delete attachments
		if (in_array($role, ['admin', 'manager']))
		{
			return true;
		}

		// For staffs and users, they can only delete attachments uploaded by themselves
		$user = Factory::getApplication()->getIdentity();

		if ($user->id == $item->id)
		{
			return true;
		}

		return false;
	}

	/**
	 * @param   stdClass  $item
	 *
	 * @return bool
	 */
	public static function isMessageRecord($item): bool
	{
		return property_exists($item, 'ticket_id');
	}

	/**
	 * Get duration
	 *
	 * @param   string  $duration
	 * @param   bool    $local  true: to local timezone, false: to UTC timezone
	 *
	 * @return array
	 *
	 * @throws Exception
	 */
	public static function getDateDuration($duration, $local = false)
	{
		$timezone = Factory::getApplication()->get('offset');

		switch ($duration)
		{
			case 'today':
				$date = Factory::getDate('now', $timezone);
				$date->setTime(0, 0, 0);
				$fromDate = $date->toSql($local);
				$date     = Factory::getDate('now', $timezone);
				$date->setTime(23, 59, 59);
				$toDate = $date->toSql($local);
				break;
			case 'tomorrow':
				$date = Factory::getDate('tomorrow', $timezone);
				$date->setTime(0, 0, 0);
				$fromDate = $date->toSql($local);
				$date     = Factory::getDate('tomorrow', $timezone);
				$date->setTime(23, 59, 59);
				$toDate = $date->toSql($local);
				break;
			case 'yesterday':
				$date = Factory::getDate('now', $timezone);
				$date->modify('-1 day');
				$date->setTime(0, 0, 0);
				$fromDate = $date->toSql($local);
				$date     = Factory::getDate('now', $timezone);
				$date->setTime(23, 59, 59);
				$date->modify('-1 day');
				$toDate = $date->toSql($local);
				break;
			case 'this_week':
				$date   = Factory::getDate('now', $timezone);
				$monday = $date->modify('Monday this week');
				$monday->setTime(0, 0, 0);
				$fromDate = $monday->toSql($local);
				$date     = Factory::getDate('now', $timezone);
				$sunday   = $date->modify('Sunday this week');
				$sunday->setTime(23, 59, 59);
				$toDate = $sunday->toSql($local);
				break;
			case 'next_week':
				$date   = Factory::getDate('now', $timezone);
				$monday = $date->modify('Monday next week');
				$monday->setTime(0, 0, 0);
				$fromDate = $monday->toSql($local);
				$date     = Factory::getDate('now', $timezone);
				$sunday   = $date->modify('Sunday next week');
				$sunday->setTime(23, 59, 59);
				$toDate = $sunday->toSql($local);
				break;
			case 'last_week':
				$date   = Factory::getDate('now', $timezone);
				$monday = $date->modify('Monday last week');
				$monday->setTime(0, 0, 0);
				$fromDate = $monday->toSql($local);
				$date     = Factory::getDate('now', $timezone);
				$sunday   = $date->modify('Sunday last week');
				$sunday->setTime(23, 59, 59);
				$toDate = $sunday->toSql($local);
				break;
			case 'this_month':
				$date = Factory::getDate('now', $timezone);
				$date->setDate($date->year, $date->month, 1);
				$date->setTime(0, 0, 0);
				$fromDate = $date->toSql($local);
				$date     = Factory::getDate('now', $timezone);
				$date->setDate($date->year, $date->month, $date->daysinmonth);
				$date->setTime(23, 59, 59);
				$toDate = $date->toSql($local);
				break;
			case 'next_month':
				$date = Factory::getDate('first day of next month', $timezone);
				$date->setTime(0, 0, 0);
				$fromDate = $date->toSql($local);
				$date     = Factory::getDate('last day of next month', $timezone);
				$date->setTime(23, 59, 59);
				$toDate = $date->toSql($local);
				break;
			case 'last_month':
				$date = Factory::getDate('first day of last month', $timezone);
				$date->setTime(0, 0, 0);
				$fromDate = $date->toSql($local);
				$date     = Factory::getDate('last day of last month', $timezone);
				$date->setTime(23, 59, 59);
				$toDate = $date->toSql($local);
				break;
			case 'this_year':
				// This year
				$date = Factory::getDate('now', $timezone);
				$date->setDate($date->year, 1, 1);
				$date->setTime(0, 0, 0);
				$fromDate = $date->toSql($local);
				$date     = Factory::getDate('now', $timezone);
				$date->setDate($date->year, 12, 31);
				$date->setTime(23, 59, 59);
				$toDate = $date->toSql($local);
				break;
			case 'last_year':
				$date = Factory::getDate('now', $timezone);
				$date->setDate($date->year - 1, 1, 1);
				$date->setTime(0, 0, 0);
				$date->setTimezone(new DateTimeZone('UCT'));
				$fromDate = $date->toSql($local);
				$date     = Factory::getDate('now', $timezone);
				$date->setDate($date->year - 1, 12, 31);
				$date->setTime(23, 59, 59);
				$date->setTimezone(new DateTimeZone('UCT'));
				$toDate = $date->toSql($local);
				break;
			case 'last_7_days':
				$date = Factory::getDate('now', $timezone);
				$date->modify('-7 days');
				$date->setTime(0, 0, 0);
				$fromDate = $date->toSql($local);
				$date     = Factory::getDate('now', $timezone);
				$date->setTime(23, 59, 59);
				$toDate = $date->toSql($local);
				break;
			case 'last_30_days':
				$date = Factory::getDate('now', $timezone);
				$date->setTime(0, 0, 0);
				$fromDate = $date->toSql($local);
				$date     = Factory::getDate('now', $timezone);
				$date->modify('-30 days');
				$date->setTime(23, 59, 59);
				$toDate = $date->toSql($local);
				break;
			default:
				$fromDate = '';
				$toDate   = '';
				break;
		}

		return [$fromDate, $toDate];
	}

	/**
	 * Check if the given message entered via HTML editor has actual data
	 *
	 * @param $string
	 *
	 * @return bool
	 */
	public static function isValidMessage($string)
	{
		if (!is_string($string) || strlen($string) === 0)
		{
			return false;
		}

		$string = strip_tags($string, '<img>');

		$string = str_replace('&nbsp;', '', $string);
		$string = str_replace("\xc2\xa0", ' ', $string);

		// Remove all special characters
		$string = str_replace(['.', ' ', "\n", "\t", "\r"], '', $string);

		$string = trim($string);

		if (strlen($string) > 10)
		{
			return true;
		}

		return false;
	}

	/**
	 * Method to get file extension
	 *
	 * @param   string  $file
	 *
	 * @return string
	 */
	public static function getFileExt($file)
	{
		// Use framework code if available
		if (is_callable([File::class, 'getExt']))
		{
			return File::getExt($file);
		}

		// String manipulation should be faster than pathinfo() on newer PHP versions.
		$dot = strrpos($file, '.');

		if ($dot === false)
		{
			return '';
		}

		$ext = substr($file, $dot + 1);

		// Extension cannot contain slashes.
		if (strpos($ext, '/') !== false || (DIRECTORY_SEPARATOR === '\\' && strpos($ext, '\\') !== false))
		{
			return '';
		}

		return $ext;
	}
}