<?php

/**
 * @package     MPF
 * @subpackage  UI
 *
 * @copyright   Copyright (C) 2016 - 2018 Ossolution Team, Inc. All rights reserved.
 * @license     GNU General Public License version 2 or later; see LICENSE
 */
defined('_JEXEC') or die;

abstract class HDPUiAbstract implements HDPUiInterface
{
	/**
	 * Css class map array
	 *
	 * @var array
	 */
	protected $classMaps;

	/**
	 * Framework own css classes
	 *
	 * @var array
	 */
	protected $frameworkClasses = [];

	/**
	 * Method to add a new class to class mapping
	 *
	 * @param $class
	 * @param $mappedClass
	 *
	 * @return $this
	 */
	public function addClassMapping($class, $mappedClass)
	{
		$class       = trim($class);
		$mappedClass = trim($mappedClass);

		$this->classMaps[$class] = $mappedClass;

		return $this;
	}

	/**
	 * Get the mapping of a given class
	 *
	 * @param   string  $class  The input class
	 *
	 * @return string The mapped class
	 */
	public function getClassMapping($class)
	{
		$class = trim($class);

		// Direct css class mapping
		if (isset($this->classMaps[$class]))
		{
			return $this->classMaps[$class];
		}

		if (strpos($class, ' ') !== false)
		{
			$classes       = explode(' ', $class);
			$mappedClasses = [];

			foreach ($classes as $cssClass)
			{
				if (isset($this->classMaps[$cssClass]))
				{
					// Special case for btn class in class group such as btn btn-primary
					if ($cssClass == 'btn')
					{
						$mappedClass     = $this->classMaps[$cssClass];
						$mappedClasses[] = explode(' ', $mappedClass)[0];
					}
					else
					{
						$mappedClasses[] = $this->classMaps[$cssClass];
					}
				}
				else
				{
					$mappedClasses[] = $cssClass;
				}
			}

			return implode(' ', $mappedClasses);
		}

		return $class;
	}

	/**
	 * Get framework own css class
	 *
	 * @param   string  $class
	 * @param   int     $behavior
	 *
	 * @return string
	 */
	public function getFrameworkClass($class, $behavior = 0)
	{
		$class = trim($class);

		$cssClasses    = explode(' ', $class);
		$outputClasses = [];

		foreach ($cssClasses as $cssClass)
		{
			if (!in_array($cssClass, $this->frameworkClasses))
			{
				continue;
			}

			$outputClasses[] = $cssClass;
		}

		if (count($outputClasses) == 0)
		{
			return null;
		}

		$class = implode(' ', $outputClasses);

		switch ($behavior)
		{
			case 1:
				return ' ' . $class;
			case 2;
				return $class . ' ';
			case 3:
				return ' class="' . $class . '"';
			default:
				return $class;
		}
	}
}