<?php
/**
 * @package   ShackErrorNotify
 * @contact   www.joomlashack.com, help@joomlashack.com
 * @copyright 2019-2023 Joomlashack.com. All rights reserved
 * @license   http://www.gnu.org/licenses/gpl.html GNU/GPL
 *
 * This file is part of ShackErrorNotify.
 *
 * ShackErrorNotify is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 *
 * ShackErrorNotify is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with ShackErrorNotify.  If not, see <http://www.gnu.org/licenses/>.
 */

use Joomla\CMS\Application\CMSApplication;
use Joomla\CMS\Factory;
use Joomla\CMS\Language\Text;
use Joomla\CMS\Layout\LayoutHelper;
use Joomla\CMS\Plugin\CMSPlugin;
use Joomla\CMS\Plugin\PluginHelper;
use Joomla\CMS\Response\JsonResponse;
use Joomla\CMS\Uri\Uri;
use Joomla\CMS\Version;
use Joomla\Event\DispatcherInterface;

defined('_JEXEC') or die();

class PlgSystemShackerrornotify extends CMSPlugin
{
    /**
     * @inheritdoc
     */
    protected $autoloadLanguage = true;

    /**
     * @var CMSApplication
     */
    protected $app = null;

    /**
     * @var mixed
     */
    protected $previousExceptionHandler = null;

    /**
     * @var string[]
     */
    protected $emails = null;

    /**
     * @var string[]
     */
    protected static $ajaxTest = [
        'option' => 'com_ajax',
        'plugin' => 'errorNotifyTest',
        'group'  => 'system',
        'format' => 'json'
    ];

    /**
     * @var JsonResponse
     */
    protected $testResponse = null;

    /**
     * PlgSystemShackerrornotify constructor.
     *
     * @param JEventDispatcher|DispatcherInterface $subject
     * @param array                                $config
     */
    public function __construct(&$subject, $config = [])
    {
        parent::__construct($subject, $config);

        if (
            $this->getEmails()
            && ($this->params->get('errors.enabled', 1)
                || $this->params->get('exceptions.enabled', 0))
            && Version::MAJOR_VERSION < 4
        ) {
            $this->previousExceptionHandler = set_exception_handler([$this, 'exceptionHandler']);
        }
    }

    /**
     * We want to avoid com_ajax exception handling, so
     * we short circuit the process and call the test method
     * directly
     *
     * @return void
     */
    public function onAfterInitialise()
    {
        $uri = Uri::getInstance();

        if ($this->app->isClient('administrator') && $uri->getQuery(true) == static::$ajaxTest) {
            header('content-type: application/json; charset=utf-8');

            $this->onAjaxErrorNotifyTest();

            jexit();
        }
    }

    /**
     * In Joomla 4, the only way to catch errors is
     * with the new onError event
     *
     * @param Throwable $event
     *
     * @return void
     */
    public function onError(Throwable $event)
    {
        $this->exceptionHandler($event);
    }

    /**
     * @param Throwable $error
     *
     * @return void
     */
    public function exceptionHandler(Throwable $error)
    {
        if ($this->exceptionEnabled($error)) {
            $displayData = [
                'error'     => $error,
                'trace'     => null,
                'dataLines' => $this->getInfo($error)
            ];
            if ($this->params->get('trace', 0)) {
                $displayData['trace'] = $error->getTrace();
            }

            $layoutPath = dirname(PluginHelper::getLayoutPath('system', 'shackerrornotify'));
            $message    = LayoutHelper::render('email', $displayData, $layoutPath);

            $this->sendEmail($message);
        }

        if ($this->previousExceptionHandler) {
            call_user_func($this->previousExceptionHandler, $error);
        }
    }

    /**
     * @return string[]
     */
    protected function getEmails(): array
    {
        if ($this->emails === null) {
            $addresses = preg_split('/[\n,]\r?/', $this->params->get('notify', ''));

            $this->emails = array_filter(array_map('trim', $addresses));
        }

        return $this->emails;
    }

    /**
     * @param string $body
     *
     * @return void
     */
    protected function sendEmail(string $body)
    {
        if ($recipients = $this->getEmails()) {
            $email = $this->app->get('mailfrom');
            $name  = $this->app->get('fromname');

            $subject = Text::sprintf('PLG_SYSTEM_SHACKERRORNOTIFY_EMAIL_SUBJECT', $this->app->get('sitename'));

            $mailer = Factory::getMailer();
            $mailer->isHtml();
            $mailer->setFrom($email, $name);
            $mailer->setSubject($subject);
            $mailer->setBody($body);

            foreach ($recipients as $recipient) {
                $mailer->addRecipient($recipient);
            }

            try {
                $response = $mailer->Send();
                if ($response instanceof Exception) {
                    throw $response;

                } elseif ($response !== true) {
                    $error = $mailer->ErrorInfo ?: '';
                    throw new Exception($error);
                }

            } catch (Throwable $mailError) {
                if ($this->testResponse) {
                    // When testing, this needs to be reported
                    $error = $mailError->getMessage();

                    $this->testResponse->message = $error
                        ? Text::sprintf('PLG_SYSTEM_SHACKERRORNOTIFY_ERROR_TEST_MAILER', $mailError->getMessage())
                        : Text::_('PLG_SYSTEM_SHACKERRORNOTIFY_ERROR_TEST_MAILER_0');

                    echo $this->testResponse;
                    jexit();
                }
            }

        }
    }

    /**
     * @param Throwable $error
     *
     * @return string[]
     */
    protected function getInfo(Throwable $error): array
    {
        $server = $this->app->input->server;

        $request  = $server->getString('REQUEST_URI') ?: Text::_('PLG_SYSTEM_SHACKERRORNOTIFY_UNKNOWN');
        $referrer = $server->getString('HTTP_REFERER') ?: Text::_('PLG_SYSTEM_SHACKERRORNOTIFY_DIRECT');
        $method   = $server->getString('REQUEST_METHOD');
        $host     = $server->getString('HTTP_HOST')
            ?: $server->getString('SERVER_NAME')
                ?: Text::_('PLG_SYSTEM_SHACKERRORNOTIFY_EMAIL_HOST_UNREPORTED');
        $protocol = $server->getString('HTTPS') !== 'on' ? 'http' : 'https';

        $lineItems = [
            Text::_('PLG_SYSTEM_SHACKERRORNOTIFY_EMAIL_CODE')     => $error->getCode(),
            Text::_('PLG_SYSTEM_SHACKERRORNOTIFY_EMAIL_TYPE')     => get_class($error),
            Text::_('PLG_SYSTEM_SHACKERRORNOTIFY_EMAIL_MESSAGE')  => $error->getMessage(),
            Text::_('PLG_SYSTEM_SHACKERRORNOTIFY_EMAIL_FILE')     => $error->getFile(),
            Text::_('PLG_SYSTEM_SHACKERRORNOTIFY_EMAIL_LINE')     => $error->getLine(),
            Text::_('PLG_SYSTEM_SHACKERRORNOTIFY_EMAIL_PROTOCOL') => $protocol,
            Text::_('PLG_SYSTEM_SHACKERRORNOTIFY_EMAIL_HOST')     => $host,
            Text::_('PLG_SYSTEM_SHACKERRORNOTIFY_EMAIL_REQUEST')  => $request,
            Text::_('PLG_SYSTEM_SHACKERRORNOTIFY_EMAIL_REFERRER') => $referrer,
            Text::_('PLG_SYSTEM_SHACKERRORNOTIFY_EMAIL_METHOD')   => $method,
        ];

        // Remote IP info
        $sources = [
            'HTTP_X_SUCURI_CLIENTIP',
            'HTTP_X_FORWARDED_FOR',
            'HTTP_X_FORWARDED',
            'REMOTE_ADDR'
        ];
        foreach ($sources as $serverVar) {
            if ($ip = $server->getString($serverVar)) {
                $host = gethostbyaddr($ip) ?: '';

                $label             = Text::_('PLG_SYSTEM_SHACKERRORNOTIFY_SERVER_' . $serverVar);
                $lineItems[$label] = sprintf('%s (%s)', $ip, $host);
            }
        }

        $user = Factory::getUser();
        if ($user->id) {
            $lineItems[Text::_('PLG_SYSTEM_SHACKERRORNOTIFY_EMAIL_USER')] = sprintf(
                '%s (%s)',
                $user->username,
                $user->id
            );
        }

        return $lineItems;
    }

    /**
     * @param Throwable $error
     *
     * @return bool
     */
    protected function exceptionEnabled(Throwable $error): bool
    {
        if ($this->exceptionCodeEnabled($error->getCode())) {
            $errorsList     = (array)$this->params->get('errors.list', []);
            $exceptionsList = (array)$this->params->get('exceptions.list', []);

            $errors     = $this->params->get('errors.enabled', 1);
            $exceptions = $this->params->get('exceptions.enabled', 0);

            $errorClass  = get_class($error);
            $isError     = ($errorClass == 'Error') || is_subclass_of($error, 'Error');
            $isException = ($errorClass == 'Exception') || is_subclass_of($error, 'Exception');

            return ($errors == 1 && $isError)
                || ($errors == 2 && in_array($errorClass, $errorsList))
                || ($exceptions == 1 && $isException)
                || ($exceptions == 2 && in_array($errorClass, $exceptionsList));
        }

        return false;
    }

    /**
     * @return string[]
     */
    public static function getAjaxTest()
    {
        return static::$ajaxTest;
    }

    /**
     * @return void
     */
    public function onAjaxErrorNotifyTest()
    {
        $this->testResponse = new JsonResponse(null, null, true);

        if ($this->params->get('errors.enabled') > 0) {
            $selected  = (array)$this->params->get('errors.list');
            $testClass = reset($selected) ?: TypeError::class;

        } elseif ($this->params->get('exceptions.enabled') > 0) {
            $selected  = (array)$this->params->get('exceptions.list');
            $testClass = reset($selected) ?: RuntimeException::class;
        }

        if (empty($testClass) == false) {
            $testError = new $testClass(Text::_('PLG_SYSTEM_SHACKERRORNOTIFY_ERROR_TEST_MESSAGE'));
        }

        if ($this->getEmails() == false) {
            $this->testResponse->message = Text::_('PLG_SYSTEM_SHACKERRORNOTIFY_ERROR_NOEMAILS');

        } elseif (empty($testError) || $this->exceptionEnabled($testError) == false) {
            $this->testResponse->message = Text::_('PLG_SYSTEM_SHACKERRORNOTIFY_ERROR_NOTEST');

        } else {
            // This should fail by design
            throw $testError;
        }

        echo $this->testResponse;
    }

    /**
     * @param int $code
     *
     * @return bool
     */
    protected function exceptionCodeEnabled(int $code): bool
    {
        $codeList = $this->params->get('ignoreCodes', '');
        if ($ignoreCodes = array_filter(array_map('trim', explode(',', $codeList)))) {
            foreach ($ignoreCodes as $ignoreCode) {
                $range = array_map('intval', explode('-', $ignoreCode));
                if (
                    (count($range) == 1 && $code == $range[0])
                    || (count($range) && $code >= $range[0] && $code <= $range[1])
                ) {
                    return false;
                }
            }
        }

        return true;
    }
}
