<?php

/**
 * @package   OSDownloads-Pro
 * @contact   www.joomlashack.com, help@joomlashack.com
 * @copyright 2022-2025 Joomlashack.com. All rights reserved
 * @license   https://www.gnu.org/licenses/gpl.html GNU/GPL
 *
 * This file is part of OSDownloads-Pro.
 *
 * OSDownloads-Pro is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 *
 * OSDownloads-Pro is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with OSDownloads-Pro.  If not, see <https://www.gnu.org/licenses/>.
 */

use Alledia\OSDownloads\Factory;
use Joomla\CMS\Categories\Categories;
use Joomla\CMS\Component\ComponentHelper;
use Joomla\CMS\Table\Table;
use Joomla\CMS\Version;
use Joomla\Component\Finder\Administrator\Indexer\Adapter;
use Joomla\Component\Finder\Administrator\Indexer\Result;
use Joomla\Database\DatabaseQuery;

// phpcs:disable PSR1.Files.SideEffects
defined('_JEXEC') or die();

if (class_exists(Adapter::class) == false) {
    // Allow usage on Joomla 3
    require_once JPATH_ADMINISTRATOR . '/components/com_finder/helpers/indexer/adapter.php';

    class_alias(FinderIndexerAdapter::class, Adapter::class);
    class_alias(FinderIndexerResult::class, Result::class);
}

// phpcs:enable PSR1.Files.SideEffects
// phpcs:disable PSR1.Classes.ClassDeclaration.MissingNamespace

class PlgFinderOsdownloads extends Adapter
{
    /**
     * @inheritdoc
     */
    protected $context = 'OSDownloads';

    /**
     * @inheritdoc
     */
    protected $extension = 'com_osdownloads';

    /**
     * @inheritdoc
     */
    protected $layout = 'item';

    /**
     * @inheritdoc
     */
    protected $type_title = 'Download Item';

    /**
     * @inheritdoc
     */
    protected $table = '#__osdownloads_documents';

    /**
     * @inheritdoc
     */
    protected $state_field = 'published';

    /**
     * @inheritdoc
     */
    protected $autoloadLanguage = true;

    /**
     * @var bool
     */
    protected $enabled = null;

    /**
     * @inheritDoc
     */
    protected function setup()
    {
        if ($this->enabled === null) {
            if (ComponentHelper::isEnabled($this->extension)) {
                $includePath   = JPATH_ADMINISTRATOR . '/components/com_osdownloads/include.php';
                $this->enabled = is_file($includePath) && include $includePath;

            } else {
                $this->enabled = false;
            }
        }

        return $this->enabled;
    }

    /**
     * @return bool
     */
    protected function isEnabled(): bool
    {
        try {
            return $this->setup();

        } catch (Throwable $exception) {
            return false;
        }
    }

    /**
     * @inheritDoc
     */
    protected function index(Result $item)
    {
        if ($this->isEnabled() === false) {
            return;
        }

        $item->setLanguage();

        $item->context = 'com_osdownloads.item';

        $item->url = $this->getUrl($item->id, $this->extension, $this->layout);

        $item->route = Factory::getPimpleContainer()
            ->helperRoute
            ->getViewItemRoute($item->id);

        $item->title = $this->getItemMenuTitle($item->url) ?: $item->title;

        if (Version::MAJOR_VERSION > 3) {
            // Add the category taxonomy data.
            $categories = Categories::getInstance('osdownloads', ['published' => false, 'access' => false]);
            $category   = $categories->get($item->catid);

            // Category does not exist, stop here
            if ($category == false) {
                return;
            }

            $item->addNestedTaxonomy(
                'Category',
                $category,
                $this->translateState($category->published),
                $category->access,
                $category->language
            );
        }

        $this->indexer->index($item);
    }

    /**
     * @inheritDoc
     */
    protected function getListQuery($query = null)
    {
        $db = $this->db;

        $query = $query instanceof DatabaseQuery ? $query : $db->getQuery(true);
        $query->select([
            'a.id',
            'a.name AS title',
            'a.alias',
            'a.brief AS summary',
            'a.published AS state',
            'a.catid',
            'a.created_time AS start_date',
            'c.alias AS category',
            'a.access',
        ])
            ->from('#__osdownloads_documents AS a')
            ->innerJoin('#__categories AS c ON c.id = a.catid');

        return $query;
    }

    /**
     * @param string $extension
     * @param int[]  $pks
     * @param int    $value
     *
     * @return void
     */
    public function onFinderCategoryChangeState($extension, $pks, $value): void
    {
        // Make sure we're handling com_content categories.
        if ($extension === 'com_osdownloads') {
            $this->categoryStateChange($pks, $value);
        }
    }

    /**
     * @param string $context
     * @param Table  $table
     *
     * @return void
     */
    public function onFinderAfterDelete($context, $table): void
    {
        try {
            if ($this->isItemContext($context)) {
                $id = $table->id;

            } elseif ($context === 'com_finder.index') {
                $id = $table->link_id ?? null;

            } else {
                return;
            }

            // Remove item from the index.
            $this->remove($id);

        } catch (Throwable $exception) {
            // ignore this
        }
    }

    /**
     * @param string $context
     * @param Table  $row
     * @param bool   $isNew
     *
     * @return void
     */
    public function onFinderAfterSave($context, $row, $isNew): void
    {
        try {
            if ($this->isItemContext($context)) {
                if (!$isNew && $this->old_access != $row->access) {
                    $this->itemAccessChange($row);
                }

                $this->reindex($row->id);
            }

            // Check for access changes in the category.
            if ($context === 'com_categories.category') {
                if (!$isNew && $this->old_cataccess != $row->access) {
                    $this->categoryAccessChange($row);
                }
            }

        } catch (Throwable $exception) {
            // ignore errors
        }
    }

    /**
     * @param string $context
     * @param Table  $row
     * @param bool   $isNew
     *
     * @return void
     */
    public function onFinderBeforeSave($context, $row, $isNew)
    {
        // We only want to handle items here.
        if ($this->isItemContext($context)) {
            // Query the database for the old access level if the item isn't new.
            if (!$isNew) {
                $this->checkItemAccess($row);
            }
        }

        // Check for access levels from the category.
        if ($context === 'com_categories.category') {
            // Query the database for the old access level if the item isn't new.
            if (!$isNew) {
                $this->checkCategoryAccess($row);
            }
        }
    }

    public function onFinderChangeState($context, $pks, $value)
    {
        // We only want to handle files here.
        if ($this->isItemContext($context)) {
            $this->itemStateChange($pks, $value);
        }

        // Handle when the plugin is disabled.
        if ($context === 'com_plugins.plugin' && $value === 0) {
            $this->pluginDisable($pks);
        }
    }

    /**
     * @param ?string $context
     *
     * @return bool
     */
    protected function isItemContext(?string $context): bool
    {
        return $context == 'com_osdownloads.item' || $context == 'com_osdownloads.form';
    }
}
