<?php
/**
 * @package     OSL
 * @subpackage  View
 *
 * @copyright   Copyright (C) 2016 Ossolution Team, Inc. All rights reserved.
 * @license     GNU General Public License version 2 or later; see LICENSE
 */

namespace OSL\View;

use Joomla\CMS\HTML\HTMLHelper;
use Joomla\CMS\Language\Text;
use Joomla\CMS\Pagination\Pagination;
use Joomla\CMS\Toolbar\ToolbarHelper;
use OSL\Container\Container;
use OSL\Model\ListModel;
use OSL\Model\State;
use Joomla\CMS\Router\Route;
use Joomla\CMS\Factory;

/**
 * Joomla CMS View List class, used to render list of records from front-end or back-end of your component
 *
 * @package      OSF
 * @subpackage   View
 * @since        1.0
 *
 * @property ListModel $model
 */
class ListView extends HtmlView
{
	/**
	 * The model state
	 *
	 * @var State
	 */
	protected $state;

	/**
	 * List of records which will be displayed
	 *
	 * @var array
	 */
	protected $items;

	/**
	 * The pagination object
	 *
	 * @var Pagination
	 */
	protected $pagination;

	/**
	 * The array which keeps list of "list" options which will used to display as the filter on the list
	 *
	 * @var array $lists
	 */
	protected $lists = [];

	/**
	 * The sidebar
	 *
	 * @var string
	 */
	protected $sidebar = '';

	/**
	 * View item name, used to build edit item link
	 *
	 * @var string
	 */
	protected $viewItem = '';

	/**
	 * Save order activated ?
	 *
	 * @var bool
	 */
	protected $saveOrder;

	/**
	 * The ajax save ordering URL
	 *
	 * @var string
	 */
	protected $saveOrderingUrl;

	/**
	 * Constructor
	 *
	 * @param   Container  $container  The DI container object
	 * @param   array      $config     An optional associative array of configuration settings.
	 *
	 * @see    \Joomla\CMS\MVC\View\HtmlView::__construct()
	 */
	public function __construct(Container $container, $config = [])
	{
		parent::__construct($container, $config);

		$this->viewItem = $this->container->inflector->singularize($this->name);

		$this->saveOrderingUrl = Route::_(
			'index.php?option=' . $this->option . '&task=' . $this->viewItem . '.save_order_ajax',
			false
		);
	}

	/**
	 * Prepare the view before it is displayed
	 *
	 */
	protected function beforeRender()
	{
		$this->state      = $this->model->getState();
		$this->items      = $this->model->getData();
		$this->pagination = $this->model->getPagination();
		$this->saveOrder  = $this->state->filter_order === 'tbl.ordering';

		if ($this->isAdminView)
		{
			$this->lists['filter_state']    = str_replace(
				'class="inputbox"',
				'class="input-medium"',
				HTMLHelper::_('grid.state', $this->state->filter_state)
			);
			$this->lists['filter_access']   = HTMLHelper::_(
				'access.level',
				'filter_access',
				$this->state->filter_access,
				'class="input-medium" onchange="submit();"',
				true
			);
			$this->lists['filter_language'] = HTMLHelper::_(
				'select.genericlist',
				HTMLHelper::_('contentlanguage.existing', true, true),
				'filter_language',
				' onchange="submit();" ',
				'value',
				'text',
				$this->state->filter_language
			);

			$this->addToolbar();
		}
	}

	/**
	 * Method to add toolbar buttons
	 *
	 */
	protected function addToolbar()
	{
		$helperClass = $this->container->componentNamespace . '\\Site\\Helper\\Helper';

		if (is_callable($helperClass . '::getActions'))
		{
			$canDo = call_user_func([$helperClass, 'getActions'], $this->name, $this->state);
		}
		else
		{
			$canDo = call_user_func(['\\OSL\Utils\\Helper', 'getActions'], $this->container->option);
		}

		$languagePrefix = $this->container->languagePrefix;

		ToolbarHelper::title(
			Text::_(
				strtoupper(
					$languagePrefix . '_' . $this->container->inflector->singularize($this->name) . '_MANAGEMENT'
				)
			),
			'link ' . $this->name
		);

		if ($canDo->get('core.create') && !in_array('add', $this->hideButtons))
		{
			ToolbarHelper::addNew('add', 'JTOOLBAR_NEW');
		}

		if ($canDo->get('core.edit') && isset($this->items[0]) && !in_array('edit', $this->hideButtons))
		{
			ToolbarHelper::editList('edit', 'JTOOLBAR_EDIT');
		}

		if ($canDo->get('core.delete') && isset($this->items[0]) && !in_array('delete', $this->hideButtons))
		{
			ToolbarHelper::deleteList(Text::_($languagePrefix . '_DELETE_CONFIRM'), 'delete');
		}

		if ($canDo->get('core.edit.state') && !in_array(
				'publish',
				$this->hideButtons
			) && (isset($this->items[0]->published) || isset($this->items[0]->state)))
		{
			ToolbarHelper::publish('publish', 'JTOOLBAR_PUBLISH', true);
			ToolbarHelper::unpublish('unpublish', 'JTOOLBAR_UNPUBLISH', true);
		}

		$this->addCustomToolbarButtons();

		if ($canDo->get('core.admin'))
		{
			ToolbarHelper::preferences($this->container->option);
		}
	}

	/**
	 * Allow adding custom toolbar buttons before Options button
	 *
	 * @return void
	 */
	protected function addCustomToolbarButtons()
	{
	}

	/**
	 * Load search tools
	 *
	 * @return void
	 * @throws \Exception
	 */
	protected function loadSearchTools(string $formId = '#adminForm'): void
	{
		$customOptions = [
			'filtersHidden'       => true,
			'defaultLimit'        => Factory::getApplication()->get('list_limit', 20),
			'searchFieldSelector' => '#filter_search',
			'orderFieldSelector'  => '#filter_full_ordering',
		];

		HTMLHelper::_('searchtools.form', $formId, $customOptions);
	}

	/**
	 * Load Draggable script uses to allow drag and drop re-ordering items
	 *
	 * @return void
	 */
	protected function loadDraggableLib()
	{
		if ($this->state->filter_order === 'tbl.ordering')
		{
			HTMLHelper::_('draggablelist.draggable');
		}
	}

	/**
	 * Method to allow adding sortable header to the list view
	 *
	 * @param   string  $title
	 * @param   string  $sortBy
	 *
	 * @return string
	 */
	protected function gridSort(string $title, string $sortBy): string
	{
		return HTMLHelper::_(
			'grid.sort',
			Text::_($title),
			$sortBy,
			$this->state->filter_order_Dir,
			$this->state->filter_order
		);
	}

	/**
	 * Method to allow adding searchtools sort header to the list view
	 *
	 * @param   string  $title
	 * @param   string  $sortBy
	 *
	 * @return string
	 */
	protected function searchToolsSort(string $title, string $sortBy): string
	{
		return HTMLHelper::_(
			'searchtools.sort',
			$title,
			$sortBy,
			$this->state->filter_order_Dir,
			$this->state->filter_order
		);
	}

	/**
	 * Method to render hidden variables for management list view
	 *
	 * @return void
	 */
	protected function renderFormHiddenVariables(): void
	{
		include __DIR__ . '/tmpl/form_list_hidden_vars.php';
	}

	/**
	 * Method to render hidden variables for management list view
	 *
	 * @return void
	 */
	protected function renderSearchBar(string $filterSearchLabel = '', string $filterSearchDescription = ''): void
	{
		$languagePrefix = $this->container->languagePrefix;

		if ($filterSearchLabel === '')
		{
			$filterSearchLabel = strtoupper(
				$languagePrefix . '_SEARCH_' . $this->name . '_DESC'
			);
		}

		if ($filterSearchDescription === '')
		{
			$filterSearchDescription = strtoupper(
				$languagePrefix . '_FILTER_SEARCH_' . $this->name . '_DESC'
			);
		}

		include __DIR__ . '/tmpl/form_list_search_bar.php';
	}

	/**
	 * Method to render hidden variables for management list view
	 *
	 * @return void
	 */
	protected function renderSearchTools(string $filterSearchLabel = '', string $filterSearchDescription = ''): void
	{
		$languagePrefix = $this->container->languagePrefix;

		if ($filterSearchLabel === '')
		{
			$filterSearchLabel = strtoupper(
				$languagePrefix . '_FILTER_SEARCH_' . $this->name . '_DESC'
			);
		}

		if ($filterSearchDescription === '')
		{
			$filterSearchDescription = strtoupper(
				$languagePrefix . '_SEARCH_' . $this->name . '_DESC'
			);
		}

		include __DIR__ . '/tmpl/form_list_search_tools.php';
	}

	/**
	 * Render re-order cell
	 *
	 * @param   \stdClass  $row
	 *
	 * @return void
	 */
	protected function reOrderCell($row): void
	{
		include __DIR__ . '/tmpl/reorder_cell.php';
	}

	/**
	 * Search tools ordering header
	 *
	 * @return string
	 */
	protected function searchToolsSortHeader(): string
	{
		return HTMLHelper::_(
			'searchtools.sort',
			'',
			'tbl.ordering',
			$this->state->filter_order_Dir,
			$this->state->filter_order,
			null,
			'asc',
			'JGRID_HEADING_ORDERING',
			'icon-menu-2'
		);
	}

	/**
	 * Get form action
	 *
	 * @return string
	 */
	protected function getFormAction(): string
	{
		return Route::_('index.php?option=' . $this->container->option . '&view=' . $this->name, false);
	}

	/**
	 * Get link to edit item
	 *
	 * @param   \stdClass  $row
	 * @param   string     $append
	 *
	 * @return string
	 */
	protected function getEditItemLink($row, $append = ''): string
	{
		return Route::_(
			'index.php?option=' . $this->container->option . '&view=' . $this->viewItem . '&id=' . $row->id . $append,
			false
		);
	}
}