<?php
/**
 * @package     OSL
 * @subpackage  Container
 *
 * @copyright   Copyright (C) 2016 Ossolution Team, Inc. All rights reserved.
 * @license     GNU General Public License version 2 or later; see LICENSE
 */

namespace OSL\Container;

use Composer\Autoload\ClassLoader;
use Exception;
use JEventDispatcher;
use Joomla\CMS\Application\CMSApplication;
use Joomla\CMS\Document\HtmlDocument;
use Joomla\CMS\Factory;
use Joomla\CMS\Language\Language;
use Joomla\CMS\Mail\Mail;
use Joomla\CMS\Session\Session;
use Joomla\CMS\User\User;
use Joomla\Database\DatabaseDriver;
use Joomla\DI\Container as DIContainer;
use OSL\Inflector\Inflector;
use OSL\Input\Input;
use OSL\Provider\SystemProvider;


/**
 * DI Container for component
 *
 * @property  string                    $option                                           The name of the component (com_eventbooking)
 * @property  string                    $componentName                                    The name of the component without com_ (eventbooking)
 * @property  string                    $componentNamespace                               The namespace of the component's classes (Ossolution\EventBooking)
 * @property  string                    $feNamespace                                      The frontend namespace of the component's classes (Ossolution\EventBooking\Site)
 * @property  string                    $beNamespace                                      The backend namespace of the component's classes (Ossolution\EventBooking\Admin)
 * @property  string                    $namespace                                        The namespace of the component's classes (Ossolution\EventBooking\Admin or Ossolution\EventBooking\Site),
 *                                                                                         depends on the current application being executued
 *
 * @property-read  \OSL\Factory\Factory $factory                                          The application object
 * @property-read  CMSApplication       $app                                              The application object
 * @property-read  HtmlDocument         $document                                         The event dispatcher
 * @property-read  Session              $session                                          The session object
 * @property-read  Language             $language                                         The language object
 * @property-read  Input                $input                                            The input object
 * @property-read  DatabaseDriver       $db                                               The database connection object
 * @property-read  Inflector            $inflector                                        The string inflector
 * @property-read  User                 $user                                             The user
 *
 * @property  string                    $defaultView                                      Default view
 * @property  string                    $tablePrefix                                      Database table prefix
 * @property  string                    $languagePrefix                                   Language Prefix
 */
class Container extends DIContainer
{
	/**
	 * Store configuration values for container, use for child container
	 *
	 * @var array
	 */
	protected $values = [];
	/**
	 * Cache of created container instances
	 *
	 * @var   array
	 */
	protected static $containerStore = [];

	/**
	 * Returns the container instance for a specific component. If the instance exists it returns the one already
	 * constructed.
	 *
	 * @param   string     $option  The name of the component, e.g. com_example
	 * @param   array      $values  Configuration override values
	 * @param   Container  $parent  Optional. Parent container (default: application container)
	 *
	 * @return  Container
	 *
	 * @throws Exception
	 */
	public static function &getInstance($option, array $values = [], ?Container $parent = null)
	{
		if (!isset(static::$containerStore[$option]))
		{
			$defaultValues = [
				'option'             => $option,
				'componentName'      => substr($option, 4),
				'componentNamespace' => 'Joomla\\Component\\' . ucfirst(substr($option, 4)),
				'frontEndPath'       => JPATH_ROOT . '/components/' . $option,
				'backEndPath'        => JPATH_ADMINISTRATOR . '/components/' . $option,
				'defaultView'        => substr($option, 4),
				'tablePrefix'        => '#__' . substr($option, 4) . '_',
				'languagePrefix'     => substr($option, 4),
			];

			$values = array_merge($defaultValues, $values);

			if (empty($values['feNamespace']))
			{
				$values['feNamespace'] = $values['componentNamespace'] . '\\Site';
			}

			if (empty($values['beNamespace']))
			{
				$values['beNamespace'] = $values['componentNamespace'] . '\\Admin';
			}

			$namespace           = Factory::getApplication()->isClient('administrator') ? $values['beNamespace'] : $values['feNamespace'];
			$values['namespace'] = $namespace;

			$className = $namespace . '\\Container';

			if (!class_exists($className, true))
			{
				$className = __CLASS__;
			}

			self::$containerStore[$option] = new $className($values, $parent);
		}

		return self::$containerStore[$option];
	}

	/**
	 * Creates a new DI container.
	 *
	 * @param   DIContainer  $parent  The parent DI container, optional
	 * @param   array        $values
	 */
	public function __construct(array $values = [], ?DIContainer $parent = null)
	{
		parent::__construct($parent);

		// Register the component to the PSR-4 auto-loader

		/** @var ClassLoader $autoLoader */
		$autoLoader = include JPATH_LIBRARIES . '/vendor/autoload.php';
		$autoLoader->setPsr4($values['feNamespace'] . '\\', $values['frontEndPath']);
		$autoLoader->setPsr4($values['beNamespace'] . '\\', $values['backEndPath']);

		foreach ($values as $key => $value)
		{
			$this->set($key, $value);
		}

		// Store the values, use for creating child container if needed
		$this->values = $values;

		// Register common used Joomla objects
		$this->registerServiceProvider(new SystemProvider());
	}

	/**
	 * Create a child Container with a new property scope that
	 * that has the ability to access the parent scope when resolving.
	 *
	 * @param   array  $values
	 *
	 * @return  Container  This object for chaining.
	 *
	 */
	public function createChild($values = [])
	{
		$values = array_merge($this->values, $values);

		return new static($values, $this);
	}

	/**
	 * Magic getter for alternative syntax, e.g. $container->foo instead of $container->get('foo'). Allows for type
	 * hinting using the property and property-read PHPdoc macros at the top of the container.
	 *
	 * @param   string  $name  The key of the data storage
	 *
	 * @return  mixed
	 */
	public function __get($name)
	{
		return $this->get($name);
	}

	/**
	 * Magic setter for alternative syntax, e.g. $container->foo = $value instead of $container->set('foo', $value).
	 * Allows for type hinting using the property and property-read PHPdoc macros at the top of the container.
	 *
	 * @param   string  $name   The key of the data storage
	 * @param   mixed   $value  The value to set in the data storage
	 */
	public function __set($name, $value)
	{
		$this->set($name, $value);
	}
}