<?php
/**
 * @package        Joomla
 * @subpackage     Helpdesk Pro
 * @author         Tuan Pham Ngoc
 * @copyright      Copyright (C) 2013 - 2026 Ossolution Team
 * @license        GNU/GPL, see LICENSE.php
 */

namespace OSSolution\HelpdeskPro\Site\View\Ticket;

use Exception;
use HelpdeskProHelperBootstrap;
use Joomla\CMS\Application\SiteApplication;
use Joomla\CMS\Captcha\Captcha;
use Joomla\CMS\Factory;
use Joomla\CMS\Language\Text;
use Joomla\CMS\Plugin\PluginHelper;
use Joomla\CMS\Router\Route;
use Joomla\CMS\Uri\Uri;
use Joomla\Registry\Registry;
use Joomla\Utilities\ArrayHelper;
use OSL\Config\Config;
use OSL\View\HtmlView;
use OSSolution\HelpdeskPro\Admin\Model\Ticket;
use OSSolution\HelpdeskPro\Admin\View\Ticket\Common;
use OSSolution\HelpdeskPro\Site\Helper\Database as HelpdeskProHelperDatabase;
use OSSolution\HelpdeskPro\Site\Helper\Helper as HelpdeskProHelper;
use OSSolution\HelpdeskPro\Site\Helper\Jquery as HelpdeskProHelperJquery;
use OSSolution\HelpdeskPro\Site\Helper\Route as RouteHelper;
use stdClass;

defined('_JEXEC') or die;

/**
 * Class Html
 *
 * @property Ticket $model
 */
class Html extends HtmlView
{
	use Common;

	/**
	 * The ticket
	 *
	 * @var stdClass
	 */
	protected $item;
	/**
	 * Flag to mark if the current user is customer
	 *
	 * @var int
	 */
	protected $isCustomer;

	/**
	 * Flag to mark if current user can add comment to the ticket
	 *
	 * @var bool
	 */
	protected $canComment;

	/**
	 * The component messages
	 *
	 * @var Config
	 */
	protected $message;

	/**
	 * List of select lists used on form
	 *
	 * @var array
	 */
	protected $lists = [];

	/**
	 * The current User ID
	 * @var int
	 */
	protected $userId;

	/**
	 * Category ID
	 *
	 * @var int
	 */
	protected $categoryId;

	/**
	 * The active menu item params
	 *
	 * @var Registry
	 */
	protected $params;

	/**
	 * Categories
	 *
	 * @var array
	 */
	protected $rowCategories;

	/**
	 * Captcha output
	 *
	 * @var string
	 */
	protected $captcha;

	/**
	 * The captcha plugin
	 *
	 * @var string
	 */
	protected $captchaPlugin;

	/**
	 * Flag to mark if we should show captcha
	 *
	 * @var int
	 */
	protected $showCaptcha;

	/**
	 * Captcha invalid. This property is used in layout, don't remove it
	 *
	 * @var bool
	 */
	protected $captchaInvalid;

	/**
	 * Prepare view data
	 *
	 * @return void
	 * @throws Exception
	 */
	protected function beforeRender()
	{
		// Load jQuery validation engine
		HelpdeskProHelperJquery::validateForm();

		/* @var SiteApplication $app */
		$app    = Factory::getApplication();
		$config = HelpdeskProHelper::getConfig();

		// Remove the uploaded files data from section
		$session = $app->getSession();
		$session->clear('hdp_uploaded_files');
		$session->clear('hdp_uploaded_files_original_names');
		$this->bootstrapHelper = HelpdeskProHelperBootstrap::getInstance();

		$layout = $this->getLayout();

		if ($layout !== 'print')
		{
			$this->loadCaptcha($app, $config);

			$this->captchaInvalid = ($this->input->getMethod() === 'POST');
		}

		if ($layout == 'form')
		{
			// Render form to allow submitting ticket
			$this->beforeRenderTicketForm();

			return;
		}
		elseif ($layout == 'edit')
		{
			// Render form to allow editing ticket
			$this->beforeRenderTicketEditForm();

			return;
		}
		elseif ($layout == 'print')
		{
			// Render form to allow printing ticket
			$this->beforeRenderPrintTicket();

			return;
		}

		// Render ticket information
		$user        = $app->getIdentity();
		$config      = HelpdeskProHelper::getConfig();
		$fieldSuffix = HelpdeskProHelper::getFieldSuffix();
		$item        = $this->model->getData();

		if (!$item->id)
		{
			$app->enqueueMessage(Text::_('HDP_TICKET_NOT_EXISTS'));
			$app->redirect(Route::_(RouteHelper::getTicketsRoute(), false));
		}

		// Require users to login if they try to access support ticket via public link
		if (!$config->allow_public_user_submit_ticket && $item->is_ticket_code)
		{
			$redirectUrl = RouteHelper::getTicketRoute($item->id);

			if ($user->id)
			{
				$app->redirect(Route::_($redirectUrl, false));
			}
			else
			{
				$app->enqueueMessage(Text::_('HDP_PLEASE_LOGIN_TO_CONTINUE'));
				$app->redirect(Route::_('index.php?option=com_users&view=login&return=' . base64_encode($redirectUrl)));
			}
		}

		$canAccess = HelpdeskProHelper::checkTicketAccess($item);

		if (!$canAccess)
		{
			if (!$user->id && !$item->is_ticket_code)
			{
				$redirectUrl = RouteHelper::getTicketRoute($item->id);
				$app->enqueueMessage(Text::_('HDP_PLEASE_LOGIN_TO_CONTINUE'));
				$app->redirect(Route::_('index.php?option=com_users&view=login&return=' . base64_encode($redirectUrl)));
			}
			else
			{
				$app->enqueueMessage(Text::_('HDP_INVALID_TICKET'), 'warning');
				$app->redirect(Uri::root());
			}
		}

		if ($user->id == $item->user_id || $item->is_ticket_code)
		{
			$isCustomer = 1;
		}
		else
		{
			$isCustomer = 0;
		}

		if ($isCustomer && ($item->status_id == $config->closed_ticket_status))
		{
			$canComment = false;
		}
		else
		{
			$canComment = true;
		}

		$message = $this->input->getHtml('message');

		$rows = HelpdeskProHelperDatabase::getAllCategories(
			'ordering',
			['access IN (' . implode(',', $user->getAuthorisedViewLevels()) . ')'],
			$fieldSuffix
		);

		$this->prepareDisplayTicketData($item, $rows);

		// Pathway
		$app->getPathway()->addItem(Text::sprintf('HDP_TICKET_NUMBER', $item->id));

		$this->item       = $item;
		$this->isCustomer = $isCustomer;
		$this->canComment = $canComment;
		$this->message    = $message;
	}

	/**
	 * Prepare data to render submit ticket form
	 *
	 * @throws Exception
	 */
	private function beforeRenderTicketForm()
	{
		/* @var SiteApplication $app */
		$app         = Factory::getApplication();
		$user        = $app->getIdentity();
		$config      = HelpdeskProHelper::getConfig();
		$fieldSuffix = HelpdeskProHelper::getFieldSuffix();
		$userId      = $user->id;

		// Require users to login before submitting ticket
		if (!$userId && !$config->allow_public_user_submit_ticket)
		{
			//Redirect user to login page
			$app->enqueueMessage(Text::_('HDP_LOGIN_TO_SUBMIT_TICKET'));
			$app->redirect(
				Route::_(
					'index.php?option=com_users&view=login&return=' . base64_encode(Uri::getInstance()->toString())
				)
			);
		}

		$active = $app->getMenu()->getActive();

		if ($active && isset($active->query['view'], $active->query['layout'])
			&& $active->query['view'] == 'ticket' && $active->query['layout'] == 'form')
		{
			$params = $active->getParams();
		}
		else
		{
			$params = new Registry;
		}

		$categoryId = $this->input->getInt('category_id', (int) $params->get('default_category_id'));

		$priorityId = $this->input->getInt('priority_id', 0);


		$filters = ['access IN (' . implode(',', $user->getAuthorisedViewLevels()) . ')'];

		if ($params->get('category_ids'))
		{
			$categoryIds = array_filter(ArrayHelper::toInteger(explode(',', $params->get('category_ids'))));

			if (count($categoryIds))
			{
				$filters[] = ['id IN (' . implode(',', $categoryIds) . ')'];
			}
		}

		$rows = HelpdeskProHelperDatabase::getAllCategories(
			'ordering',
			$filters,
			$fieldSuffix
		);

		if (count($rows) == 1)
		{
			$categoryId = $rows[0]->id;
		}

		if (!$categoryId && $categoryAlias = $this->input->getCmd('category'))
		{
			$categoryId = HelpdeskProHelperDatabase::getCategoryIdByAlias($categoryAlias);
		}

		$this->prepareSubmitTicketFormData($rows, $categoryId, $priorityId);

		// Pathway
		$pathway = $app->getPathway();
		$pathway->addItem(Text::_('HDP_NEW_TICKET'));

		$this->userId        = $userId;
		$this->categoryId    = $categoryId;
		$this->params        = $params;
		$this->rowCategories = $rows;
	}

	/**
	 * Generate form allows admin edit a ticket
	 *
	 * @return void
	 */
	private function beforeRenderTicketEditForm()
	{
		/* @var SiteApplication $app */
		$app        = Factory::getApplication();
		$config     = HelpdeskProHelper::getConfig();
		$this->item = $this->model->getData();

		if (!$this->item)
		{
			throw new Exception(404, Text::_('HDP_TICKET_NOT_FOUND'));
		}

		if (!HelpdeskProHelper::canEditTicket($this->item))
		{
			throw new Exception(403, Text::_('HDP_NO_EDIT_TICKET_PERMISSION'));
		}

		$fieldSuffix = HelpdeskProHelper::getFieldSuffix($this->item->language);
		$filters     = ['access IN (' . implode(',', $app->getIdentity()->getAuthorisedViewLevels()) . ')'];

		$rows = HelpdeskProHelperDatabase::getAllCategories(
			'ordering',
			$filters,
			$fieldSuffix
		);

		$this->prepareEditTicketFormData($this->item, $rows);
	}

	/**
	 * Render print ticket layout
	 */
	protected function beforeRenderPrintTicket()
	{
		/* @var SiteApplication $app */
		$app         = Factory::getApplication();
		$user        = $app->getIdentity();
		$config      = HelpdeskProHelper::getConfig();
		$fieldSuffix = HelpdeskProHelper::getFieldSuffix();
		$item        = $this->model->getData();

		if (!$item->id)
		{
			$app->enqueueMessage(Text::_('HDP_TICKET_NOT_EXISTS'));
			$app->redirect(Route::_(RouteHelper::getTicketsRoute(), false));
		}

		$canAccess = HelpdeskProHelper::checkTicketAccess($item);

		if (!$canAccess)
		{
			if (!$user->id && !$item->is_ticket_code)
			{
				$app->enqueueMessage(Text::_('HDP_PLEASE_LOGIN_TO_CONTINUE'));
				$app->redirect(
					Route::_(
						'index.php?option=com_users&view=login&return=' . base64_encode(Uri::getInstance()->toString())
					)
				);
			}
			else
			{
				$app->enqueueMessage(Text::_('HDP_INVALID_TICKET'), 'warning');
				$app->redirect(Uri::root());
			}
		}

		// Pathway
		$pathway = $app->getPathway();
		$pathway->addItem(Text::sprintf('HDP_TICKET_NUMBER', $item->id));

		$this->fields      = HelpdeskProHelper::getFields($item->category_id, [], $fieldSuffix);
		$this->messages    = $this->model->getMessages();
		$this->fieldValues = $this->model->getFieldsValue();
		$this->config      = $config;
		$this->item        = $item;
	}

	/**
	 * Load captcha for submit ticket and add comment form
	 *
	 * @param   SiteApplication  $app
	 * @param   Config           $config
	 */
	protected function loadCaptcha($app, $config)
	{
		// Captcha
		$showCaptcha   = 0;
		$captchaPlugin = '';
		$user          = Factory::getApplication()->getIdentity();

		if (!$config->enable_captcha || ($user->id && $config->enable_captcha == '2'))
		{
			$enableCaptcha = false;
		}
		else
		{
			$enableCaptcha = true;
		}

		if ($enableCaptcha)
		{
			$captchaPlugin = $app->get('captcha', $app->get('captcha'));

			if (!$captchaPlugin)
			{
				// Hardcode to recaptcha, reduce support request
				$captchaPlugin = 'recaptcha';
			}

			// Check to make sure Captcha is enabled
			$plugin = PluginHelper::getPlugin('captcha', $captchaPlugin);

			if ($plugin)
			{
				$showCaptcha   = 1;
				$this->captcha = Captcha::getInstance($captchaPlugin)->display(
					'dynamic_recaptcha_1',
					'dynamic_recaptcha_1',
					'required'
				);
			}
			else
			{
				$app->enqueueMessage(Text::_('HDP_CAPTCHA_NOT_ACTIVATED_IN_YOUR_SITE'), 'error');
			}
		}

		$this->captchaPlugin = $captchaPlugin;
		$this->showCaptcha   = $showCaptcha;
	}
}