<?php

/**
 * @package   OSDownloads-Pro
 * @contact   www.joomlashack.com, help@joomlashack.com
 * @copyright 2005-2025 Joomlashack.com. All rights reserved
 * @license   https://www.gnu.org/licenses/gpl.html GNU/GPL
 *
 * This file is part of OSDownloads-Pro.
 *
 * OSDownloads-Pro is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 *
 * OSDownloads-Pro is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with OSDownloads-Pro.  If not, see <https://www.gnu.org/licenses/>.
 */

namespace Alledia\OSDownloads\Pro\Joomla\Model;

use Alledia\Framework\Joomla\Model\AbstractBaseDatabaseModel;
use Alledia\OSDownloads\Factory;
use Exception;
use Joomla\Database\DatabaseQuery;

// phpcs:disable PSR1.Files.SideEffects
defined('_JEXEC') or die();

// phpcs:enable PSR1.Files.SideEffects

class Categories extends AbstractBaseDatabaseModel
{
    /**
     * Get the categories query
     *
     * @param ?int $catId
     *
     * @return DatabaseQuery
     * @throws Exception
     */
    public function getItemQuery(?int $catId = null): DatabaseQuery
    {
        $app    = Factory::getApplication();
        $db     = $this->getDbo();
        $user   = Factory::getUser();
        $groups = $user->getAuthorisedViewLevels();

        $filterOrder    = $app->getUserStateFromRequest(
            'com_osdownloads.category.filter_order',
            'filter_order',
            'cat.lft',
            ''
        );
        $filterOrderDir = $app->getUserStateFromRequest(
            'com_osdownloads.category.filter_order_Dir',
            'filter_order_Dir',
            'asc',
            'word'
        );

        $query = $db->getQuery(true)
            ->select('cat.*')
            ->from('#__categories AS cat')
            ->where([
                'cat.published = 1',
                'cat.access IN (' . implode(',', $groups) . ')',
                'cat.extension = ' . $db->quote('com_osdownloads'),
            ])
            ->order($db->quoteName($filterOrder) . ' ' . $filterOrderDir);

        if ($catId) {
            $query->where('cat.id = ' . $catId);
        }

        return $query;
    }

    /**
     * Get the child categories query
     *
     * @param ?int $parentId
     *
     * @return DatabaseQuery
     * @throws Exception
     */
    public function getChildCategoriesQuery(?int $parentId = null): DatabaseQuery
    {
        $app    = Factory::getApplication();
        $db     = $this->getDbo();
        $user   = Factory::getUser();
        $groups = array_map('intval', array_unique($user->getAuthorisedViewLevels()));

        // Use the root category if nothing specified
        $parentId = $parentId ?: 1;

        $filterOrder    = $app->getUserStateFromRequest(
            'com_osdownloads.category.filter_order',
            'filter_order',
            'cat.lft',
            ''
        );
        $filterOrderDir = $app->getUserStateFromRequest(
            'com_osdownloads.category.filter_order_Dir',
            'filter_order_Dir',
            'asc',
            'word'
        );

        $orWheres = [
            'cat.parent_id = ' . $parentId,
            'cat.id = ' . $parentId,
        ];

        return $db->getQuery(true)
            ->select('cat.*')
            ->from('#__categories AS cat')
            ->where([
                'cat.published = 1',
                sprintf('cat.access IN (%s)', implode(',', $groups)),
                'cat.extension = ' . $db->quote('com_osdownloads'),
                sprintf('(%s)', join(' OR ', $orWheres)),
            ])
            ->order($db->quoteName($filterOrder) . ' ' . $filterOrderDir);
    }

    /**
     * Get the total of files a category have
     *
     * @param int $catId
     *
     * @return int
     */
    public function getTotalCategoryFiles(int $catId): int
    {
        $db     = $this->getDbo();
        $user   = Factory::getUser();
        $groups = $user->getAuthorisedViewLevels();

        $query = $db->getQuery(true)
            ->select('count(*)')
            ->from('#__osdownloads_documents')
            ->where([
                'published = 1',
                'access IN (' . implode(',', $groups) . ')',
                'catid = ' . $catId,
            ]);

        return (int)$db->setQuery($query)->loadResult();
    }

    /**
     * Get the total of child categories a category has.
     *
     * @param int $parentId
     *
     * @return int
     */
    public function getTotalCategoryChildren(int $parentId): int
    {
        $db     = $this->getDbo();
        $user   = Factory::getUser();
        $groups = $user->getAuthorisedViewLevels();

        $query = $db->getQuery(true)
            ->select('count(*)')
            ->from('#__categories AS cat')
            ->where([
                'cat.published = 1',
                'cat.access IN (' . implode(',', $groups) . ')',
                'cat.extension = ' . $db->quote('com_osdownloads'),
                'cat.parent_id = ' . $parentId,
            ]);

        return (int)$db->setQuery($query)->loadResult();
    }
}
