<?php
/**
 * @package        Joomla
 * @subpackage     Helpdesk Pro
 * @author         Tuan Pham Ngoc
 * @copyright      Copyright (C) 2013 - 2026 Ossolution Team
 * @license        GNU/GPL, see LICENSE.php
 */

namespace OSSolution\HelpdeskPro\Plugin\User\HelpdeskPro\Extension;

use Joomla\CMS\Application\CMSApplicationInterface;
use Joomla\CMS\Component\ComponentHelper;
use Joomla\CMS\Language\Text;
use Joomla\CMS\Mail\MailHelper;
use Joomla\CMS\Plugin\CMSPlugin;
use Joomla\Database\DatabaseAwareTrait;
use Joomla\Database\DatabaseInterface;
use Joomla\Event\DispatcherInterface;
use Joomla\Event\Event;
use Joomla\Event\SubscriberInterface;

defined('_JEXEC') or die;

class HelpdeskPro extends CMSPlugin implements SubscriberInterface
{
	use DatabaseAwareTrait;

	public function __construct(
		DispatcherInterface $dispatcher,
		array $config,
		CMSApplicationInterface $application,
		DatabaseInterface $db
	) {
		parent::__construct($dispatcher, $config);

		$this->setDatabase($db);
		$this->setApplication($application);
	}

	/**
	 * Returns an array of events this subscriber will listen to.
	 *
	 * @return array
	 *
	 */
	public static function getSubscribedEvents(): array
	{
		return [
			'onUserAfterSave' => 'onUserAfterSave',
		];
	}

	/**
	 * Utility method to act on a user after it has been saved.
	 *
	 * This method updates email addresses in tickets table when user email is updated
	 *
	 * @param   Event  $event  The event object
	 *
	 * @return  void
	 */
	public function onUserAfterSave(Event $event): void
	{
		// Get the user data
		[$user, $isnew, $success, $msg] = array_values($event->getArguments());

		// If the user wasn't stored we don't sync
		if (!$success)
		{
			return;
		}

		// If the user is new we don't sync
		if ($isnew)
		{
			return;
		}

		// Ensure the user id is really an int
		$userId = (int) ($user['id'] ?? 0);

		// If the user id appears invalid then bail out just in case
		if ($userId <= 0)
		{
			return;
		}

		$app    = $this->getApplication();
		$option = $app->getInput()->getCmd('option');
		$task   = $app->getInput()->getCmd('task');

		// Only update data if data is updated via com_users
		if ($option !== 'com_users')
		{
			return;
		}

		if ($app->isClient('administrator') && !in_array($task, ['save', 'apply', 'save2new']))
		{
			return;
		}

		if ($app->isClient('site') && $task !== 'save')
		{
			return;
		}

		// Get the new email address
		$email = $user['email'] ?? '';

		if (empty($email))
		{
			return;
		}

		// Validate email format
		if (!MailHelper::isEmailAddress($email))
		{
			return;
		}

		// Update all tickets for this user with the new email address
		$db    = $this->getDatabase();
		$query = $db->getQuery(true)
			->update('#__helpdeskpro_tickets')
			->set($db->quoteName('email') . ' = ' . $db->quote($email))
			->where($db->quoteName('user_id') . ' = ' . (int) $userId);

		try
		{
			$db->setQuery($query)->execute();
		}
		catch (\RuntimeException $e)
		{
			// Log error but don't throw exception to prevent user save from failing
			$app->enqueueMessage(Text::_('PLG_USER_HELPDESKPRO_ERROR_UPDATE_TICKET_EMAILS'), 'warning');
		}
	}

	/**
	 * Override registerListeners to only enable the plugin if Helpdesk Pro is installed
	 *
	 * @return void
	 */
	public function registerListeners(): void
	{
		if (!ComponentHelper::isEnabled('com_helpdeskpro'))
		{
			return;
		}

		parent::registerListeners();
	}
}
