<?php
/**
 * @package   ShackErrorNotify
 * @contact   www.joomlashack.com, help@joomlashack.com
 * @copyright 2019-2023 Joomlashack.com. All rights reserved
 * @license   http://www.gnu.org/licenses/gpl.html GNU/GPL
 *
 * This file is part of ShackErrorNotify.
 *
 * ShackErrorNotify is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 *
 * ShackErrorNotify is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with ShackErrorNotify.  If not, see <http://www.gnu.org/licenses/>.
 */

use Joomla\CMS\Factory;
use Joomla\CMS\Form\FormField;
use Joomla\CMS\HTML\HTMLHelper;
use Joomla\CMS\Language\Text;
use Joomla\CMS\Plugin\PluginHelper;
use Joomla\Utilities\ArrayHelper;

defined('_JEXEC') or die();

class ShackformFieldTest extends FormField
{
    /**
     * @inheritDoc
     */
    public function setup(SimpleXMLElement $element, $value, $group = null)
    {
        if (parent::setup($element, $value, $group)) {
            $this->label = (string)$element['label'] ?: Text::_('PLG_SYSTEM_SHACKERRORNOTIFY_PARAMS_TESTBUTTON_LABEL');

            return true;
        }

        return false;
    }

    /**
     * @inheritDoc
     */
    protected function getInput()
    {
        if (class_exists('PlgSystemShackerrornotify')) {
            $attributes = [
                'id'    => $this->id,
                'type'  => 'button',
                'class' => 'btn btn-primary'
            ];

            $this->loadJS();

            return sprintf(
                '<button %s>%s</button>'
                . '<br><div id="%s_result" style="margin-top: 3px; float: left;"></div>',
                ArrayHelper::toString($attributes),
                $this->label,
                $this->id
            );
        }

        $enabled = PluginHelper::isEnabled('System', 'Shackerrornotify');
        return sprintf(
            '<span class="alert alert-warning">%s</span>',
            $enabled
                ? Text::_('PLG_SYSTEM_SHACKERRORNOTIFY_WARNING_TEST')
                : Text::_('PLG_SYSTEM_SHACKERRORNOTIFY_WARNING_ENABLE')
        );
    }

    /**
     * @inheritDoc
     */
    protected function getLabel()
    {
        return '';
    }

    /**
     * @return void
     */
    protected function loadJS()
    {
        HTMLHelper::_('jquery.framework');

        $urlVars = json_encode(PlgSystemShackerrornotify::getAjaxTest());

        $notifyField = $this->form->getField('notify', 'params');
        $emails      = array_unique(array_filter(preg_split('/[\s,]/', $notifyField->value)));

        $runningTest    = Text::_('PLG_SYSTEM_SHACKERRORNOTIFY_RUNNING_TEST');
        $successMessage = Text::sprintf('PLG_SYSTEM_SHACKERRORNOTIFY_CHECK_EMAIL', join(', ', $emails));
        $saveMessage    = Text::_('PLG_SYSTEM_SHACKERRORNOTIFY_PARAMS_SAVE_CHANGES');

        Text::script('PLG_SYSTEM_SHACKERRORNOTIFY_ERROR_TEST_SERVER');

        $jscript = <<<JSCRIPT
;jQuery(document).ready(function($) {
    let \$testButton = $('#{$this->id}'),
        \$form       = \$testButton.closest('form'),
        \$result     = $('#{$this->id}_result');
    
    let showMessage = function(message, messageType) {
        \$result
            .html('')
            .removeClass();
            
            if (message && messageType) {
                \$result
                    .addClass('alert alert-' + messageType)
                    .html(message);
            }
    };
    
    \$form.find(':input').on('change', function (evt) {
        if (\$form.data('changed') !== true) {
            showMessage('{$saveMessage}', 'info');
            \$form.data('changed', true);
            \$testButton.attr('disabled', true);
        }
    });
    
    \$testButton.on('click', function(evt) {
        evt.preventDefault();
        
        showMessage();
        
        let \$this = $(this);
        \$this.attr('disabled', true).html('{$runningTest}');
        
        $.get('index.php', {$urlVars})
            .always(function(response, status, error) {
                if (status === 'success') {
                    // This means something went wrong
                    showMessage(response.message, 'error');
                    
                } else if (response.status >= 400) {
                    // Just what we were expecting
                    showMessage('{$successMessage}', 'success');

                } else {
                    // This should neve happen, but we know about 'assume'
                    showMessage(
                        Joomla.JText._('PLG_SYSTEM_SHACKERRORNOTIFY_ERROR_TEST_SERVER'),
                        'error'
                    );
               }
                      
                \$this.attr('disabled', false).html('{$this->label}');
            });
    });
});
JSCRIPT;

        Factory::getDocument()->addScriptDeclaration($jscript);
    }
}
