<?php
/**
 * Project:                 4LOGS
 *
 * @package                 4LOGS
 * @copyright               Copyright Weeblr llc - 2021
 * @author                  Yannick Gaultier - Weeblr llc
 * @license                 GNU General Public License version 3; see LICENSE.md
 * @version                 @build_version_full_build@
 *
 * 2021-08-20
 */

namespace Weeblr\Wblib\V_FORLOGS_217\Platform;

use Weeblr\Wblib\V_FORLOGS_217\Wb;
use Weeblr\Wblib\V_FORLOGS_217\System;

use Joomla\Database\ParameterType;
use Joomla\CMS\Table\Table;
use Joomla\CMS\Factory;
use Joomla\CMS\Filesystem\Folder;
use Joomla\CMS\Filesystem\File;
use Joomla\CMS\Filesystem\Path;
use Joomla\CMS\Uri\Uri;
use Joomla\CMS\Language\Text;
use Joomla\CMS\Http\HttpFactory;
use Joomla\CMS\Cache\Cache;
use Joomla\Registry\Registry;
use Joomla\CMS\Plugin;
use Joomla\CMS\Router\Route;
use Joomla\CMS\Session\Session;
use Joomla\CMS\Toolbar\ToolbarHelper;
use Joomla\CMS\Filter;
use Joomla\String\StringHelper;
use Joomla\CMS\Language;

// Security check to ensure this file is being included by a parent file.
defined('_JEXEC') || defined('WBLIB_V_FORLOGS_217_ROOT_PATH') || die;

/**
 */
class JoomlaPlatform extends Platform implements Platforminterface
{
	protected $_name = 'joomla';

	protected $app = null;

	private static $hooks = array();

	private static $hooksStack = array();

	private static $hooksRuns = array();

	private static $canonicalRoot = null;


	public function __construct()
	{
		$this->app = Factory::getApplication();
	}

	/**
	 * Run any initialization code.
	 */
	public function boot()
	{

	}

	/**
	 * Detects whether running on Joomla.
	 *
	 * @return bool
	 */
	public function detect()
	{
		$detected =
			defined('_JEXEC')
			&&
			defined('JPATH_BASE');

		return $detected;
	}

	/**
	 * Returns major version of the running platform
	 */
	public function majorVersion()
	{
		$version = new \JVersion;

		return $version::MAJOR_VERSION;
	}

	/**
	 * Return full version of the running platform
	 */
	public function version()
	{
		$version = new \JVersion;

		return $version->getShortVersion();
	}

	public function getLayoutOverridePath()
	{
		return array();
	}

	public function getUserImagesPath()
	{
		return JPATH_ROOT . '/images';
	}

	public function getConfig()
	{
		return Factory::getConfig();
	}

	public function getUniqueId()
	{
		return Factory::getConfig()->get('secret');
	}

	public function getUser($id = null)
	{
		return Factory::getUser($id);
	}

	public function isGuest()
	{
		return (bool) Factory::getUser()->guest;
	}

	public function sanitizeInput($type, $input)
	{
		switch ($type)
		{
			case 'string':
				$output = Filter\InputFilter::getInstance()->clean($input, $type);
				break;
			case 'html':
				$output = Filter\InputFilter::getInstance(null, null, 1, 1)->clean($input, $type);
				break;
			default:
				$output = $input;
				break;
		}

		return $output;
	}

	public function getCSRFToken($forceNew = false)
	{
		return Session::getFormToken($forceNew);
	}

	public function checkCSRFToken($method = 'post')
	{
		$method = strtolower($method);
		$token  = $this->getCSRFToken();

		// Check from header first
		if ($token === System\Http::getRequestHeader('x-wblr-csrf-token'))
		{
			return true;
		}

		// Then fallback to HTTP query
		if (!$this->app->input->$method->get($token, '', 'alnum'))
		{
			return false;
		}

		return true;
	}

	public function getCurrentUrl()
	{
		return Uri::getInstance()->toString();
	}

	public function getCurrentContentType()
	{
		return $this->app->input->getCmd('option');
	}

	public function getCurrentRequestCategory()
	{
		$input       = $this->app->input;
		$contentType = $input->getCmd('option');
		$view        = $input->getCmd('view');
		$layout      = $input->getCmd('layout');
		$id          = $input->getInt('id');
		$catid       = $input->getInt('catid');

		if ('category' == $view)
		{
			$catid = $id;
		}

		if (
			empty($catid)
			&&
			'article' == $view
		)
		{
			switch ($contentType)
			{
				case 'com_banners':
					Table::addIncludePath(JPATH_ROOT . '/administrator/components/com_banners/tables');
					$table = Table::getInstance('Banners');
					$table->load($id);
					$catid = $table->catid;
					break;
				case 'com_content':
					$table = Table::getInstance('Content');
					$table->load($id);
					$catid = $table->catid;
					break;
				case 'com_contact':
					Table::addIncludePath(JPATH_ROOT . '/administrator/components/com_contact/tables');
					$table = Table::getInstance('Contact', 'ContactTable');
					$table->load($id);
					$catid = $table->catid;
					break;
				case 'com_newsfeeds':
					Table::addIncludePath(JPATH_ROOT . '/administrator/components/com_banners/tables');
					$table = Table::getInstance('Newsfeeds');
					$table->load($id);
					$catid = $table->catid;
					break;
			}
		}

		if (!empty($catid))
		{
			$allCategories = $this->getCategories($contentType);
			foreach ($allCategories as $category)
			{
				if ($category->id == $catid)
				{
					return $category;
				}
			}
		}

		return false;
	}

	public function getSitename()
	{
		return $this->app->get('sitename');
	}

	/**
	 * Maybe enforce presence of www prefix.
	 *
	 * @param           $url
	 * @param   bool    $enforce
	 * @param   string  $prefix
	 *
	 * @return mixed|string
	 */
	public function enforceUrlPrefix($url, $enforce = false, $prefix = 'www')
	{
		if (!empty($url) && !is_null($enforce))
		{
			$tls    = Wb\startsWith($url, 'https://');
			$url    = Wb\lTrim(
				$url,
				[
					'https://',
					'http://'
				]
			);
			$url    = Wb\lTrim($url, 'www.'); // make sure no prefix
			$prefix = empty($prefix) ? '' : $prefix . '.';
			$url    = 'http' . ($tls ? 's' : '') . '://' . $prefix . $url;
		}

		return $url;
	}

	public function getBaseUrl($pathOnly = true)
	{
		return Uri::base($pathOnly);
	}

	public function getRootUrl($pathOnly = true)
	{
		return Uri::root($pathOnly);;
	}

	public function getHomeUrl($normalized = false)
	{
		static $homeUrl = null;
		static $normalizedHomeUrl = null;

		if (is_null($homeUrl))
		{
			$lang_code = $this->app->getLanguage()->getDefault();
			$item      = $this->app->getMenu('site')->getDefault($lang_code);
			$homeUrl   = StringHelper::trim(
				$this->route('index.php?Itemid=' . $item->id),
				'/'
			);
			$homeUrl   = Wb\rTrim($homeUrl, '/index.php');

			// suppress the base path
			$normalizedHomeUrl = $this->normalizeUrl($homeUrl);
		}

		return $normalized
			? $normalizedHomeUrl
			: $homeUrl;
	}

	public function isHomePage()
	{
		static $isHomePage = null;

		if (is_null($isHomePage))
		{
			$currentUrl = Uri::getInstance()->toString(
				[
					'path',
					'query'
				]
			);

			$currentUrl = $this->normalizeUrl(
				StringHelper::trim($currentUrl, '/')
			);
			$isHomePage = $this->getHomeUrl(true) == $currentUrl;
		}

		return $isHomePage;
	}

	public function normalizeUrl($url, $removeLeadingSlash = true)
	{
		static $baseUrl = null;
		static $fqdnBaseUrl = null;

		if (is_null($baseUrl))
		{
			$baseUrl = Wb\rTrim(
				$this->getBaseUrl(true),
				'/'
			);
			$baseUrl = StringHelper::trim(
				Wb\rTrim(
					$baseUrl,
					'administrator'
				),
				'/'
			);

			$fqdnBaseUrl = StringHelper::trim(
				$this->getBaseUrl(false),
				'/'
			);
			$fqdnBaseUrl = Wb\rTrim(
				$fqdnBaseUrl,
				'administrator'
			);
		}

		$url = Wb\lTrim(
			$url,
			$fqdnBaseUrl
		);

		$url = Wb\lTrim(
			Wb\lTrim($url, '/'),
			$baseUrl
		);

		$url = $this->stripRewritePrefix($url);

		return $removeLeadingSlash
			? Wb\ltrim($url, '/')
			: $url;
	}

	public function stripRewritePrefix($url)
	{
		static $prefix = null;

		if (is_null($prefix))
		{
			$prefix = Factory::getApplication()->get('sef_rewrite');
		}
		if (empty($prefix))
		{
			$url = Wb\lTrim(
				Wb\lTrim($url, '/'),
				'index.php'
			);
		}

		return $url;
	}

	public function getUrlRewritingPrefix()
	{
		static $sefRewritePrefix = null;

		if (is_null($sefRewritePrefix))
		{
			$sefRewrite       = $this->app->get('sef_rewrite');
			$sefRewritePrefix = empty($sefRewrite)
				? '/index.php'
				: '';
		}

		return $sefRewritePrefix;
	}

	/**
	 * Builds and return the canonical domain of the page, taking into account
	 * the optional canonical domain in SEF plugin, and including the base path, if any
	 * Can be called from admin side, to get front end links, by passing true as param
	 *
	 * @param   null  $isAdmin  If === true or === false, disable using JApplication::isAdmin, for testing
	 *
	 * @return null|string
	 */
	public function getCanonicalRoot($isAdmin = null)
	{
		if (is_null(self::$canonicalRoot))
		{
			$sefPlugin      = Plugin\PluginHelper::getPlugin('system', 'sef');
			$sefPlgParams   = new Registry($sefPlugin->params);
			$canonicalParam = StringHelper::trim($sefPlgParams->get('domain', ''));
			if (empty($canonicalParam))
			{
				$base = $this->getRootUrl(false);
				if ($isAdmin === true || ($isAdmin !== false && $this->isBackend()))
				{
					$base = Wb\lTrim(
						$base,
						[
							'administrator/',
							'administrator',
						]
					);
				}
				self::$canonicalRoot = $base;
			}
			else
			{
				self::$canonicalRoot = $canonicalParam;
			}
			self::$canonicalRoot = StringHelper::rtrim(self::$canonicalRoot, '/') . '/';
		}

		return self::$canonicalRoot;
	}

	public function getExtensions($type)
	{
		static $extensions = [];

		if (!isset($extensions[$type]))
		{
			switch ($type)
			{
				case 'components':
					$db    = Factory::getDbo();
					$query = $db->getQuery(true)
						->select($db->quoteName(['extension_id', 'name', 'type', 'element', 'folder', 'enabled', 'params'], ['id', null, null, null, null, null, null]))
						->from($db->quoteName('#__extensions'))
						->where($db->quoteName('type') . ' = ' . $db->quote('component'))
						->where($db->quoteName('state') . ' = 0');
					$db->setQuery($query);

					$extensionsList = $db->loadObjectList('name');
					break;
			}

			$extensions[$type] = empty($extensionsList)
				? []
				: $extensionsList;
		}

		return $extensions[$type];
	}

	public function isPluginEnabled($group, $name)
	{
		return Plugin\PluginHelper::isEnabled($group, $name);
	}

	/**
	 * Save a joomla parameters object to the #__extensions table.
	 *
	 * @param   Registry  $params
	 * @param   array     $options
	 *
	 * @return bool
	 */
	public function saveExtensionParams($params, $options)
	{
		$db    = Factory::getDbo();
		$query = $db->getQuery(true)
			->update($db->quoteName('#__extensions'))
			->set($db->quoteName('params') . ' = ' . $db->quote((string) $params));
		foreach ($options as $key => $value)
		{
			$query->where($db->quoteName($key) . ' = ' . $db->quote($value));
		}
		$db->setQuery($query);
		$db->execute();

		return true;
	}

	public function getCategories($extensions = [], $language = '')
	{
		static $allCategories = null;

		if (is_null($allCategories))
		{
			$db    = Factory::getDbo();
			$query = $db->getQuery(true)
				->select($db->quoteName(['id', 'parent_id', 'lft', 'rgt', 'level', 'path', 'extension', 'title', 'alias', 'published', 'metadesc', 'language']))
				->where($db->quoteName('extension') . ' != ' . $db->quote('system'))
				->from($db->quoteName('#__categories'));
			$db->setQuery($query);
			$allCategories = $db->loadObjectList();
		}

		$extensions = Wb\arrayEnsure($extensions);

		// filter out by extension
		if (!empty($extensions))
		{
			$categories = array_values(
				array_filter(
					$allCategories,
					function ($category) use ($extensions) {
						return in_array(
							$category->extension,
							$extensions
						);
					}
				)
			);
		}
		else
		{
			$categories = $allCategories;
		};

		if (!empty($language))
		{
			$categories = array_values(
				array_filter(
					$categories,
					function ($category) use ($language) {
						return $category->language == $language;
					}
				)
			);
		}

		return empty($categories)
			? []
			: $categories;
	}

	public function getHttpInput()
	{
		return $this->app->input;
	}

	public function getCookiesManager()
	{
		return $this->app->input->cookie;
	}

	public function getHttpClient($options = array())
	{
		return HttpFactory::getHttp(
			new Registry(
				$options
			)
		);
	}

	public function getCache($type, $options = array())
	{
		return Cache::getInstance(
			$type,
			$options
		);
	}

	/**
	 * Get current request http method in uppercase.
	 *
	 * @return string
	 * @throws \Exception
	 */
	public function getMethod()
	{
		return strtoupper($this->app->input->getMethod());
	}

	public function getHost()
	{
		return Uri::getInstance()->getHost();
	}

	public function getScheme()
	{
		return Uri::getInstance()->getScheme();
	}

	public function getRootPath()
	{
		return JPATH_ROOT;
	}

	public function getLogsPath()
	{
		return Factory::getConfig()->get('log_path');
	}

	/**
	 * Builds the full path to a standard cache folder.
	 * Cached content is stored in:
	 *
	 * {platform_root}/media/com_forseo/cache/{$type}/{secret_key}
	 *
	 * @param   string  $type
	 *
	 * @return string
	 */
	public function getCachePath($type)
	{
		return Wb\slashTrimJoin(
			$this->getRootPath(),
			'media/com_forseo/cache',
			strtolower($type),
			$this->getUniqueId()
		);
	}

	public function getLayoutOverridesPath()
	{
		return array();
	}

	public function getDefaultLanguageTag($full = true)
	{
		return Factory::getLanguage()->getDefault();
	}

	public function getCurrentLanguageTag($full = true)
	{
		return Factory::getLanguage()->getTag();
	}

	public function getCurrentLanguageDirection()
	{
		return Factory::getLanguage()->isRtl() ? 'rtl' : 'ltr';
	}

	public function getLanguageDirection($lang)
	{
		static $directions;

		if (is_null($directions))
		{
			$languages = Language\LanguageHelper::getLanguages('lang_code');
			foreach ($languages as $lang => $def)
			{
				$languageObject    = Language\Language::getInstance($lang);
				$directions[$lang] = $languageObject->isRTL()
					? 'rtl'
					: 'ltr';
			}
		}

		return empty($directions[$lang]) ? 'ltr' : $directions[$lang];
	}

	public function getLanguageOverrides($extension)
	{
		if (empty($extension))
		{
			return array();
		}
		$extension = strtoupper($extension . '_');
		try
		{
			$language      = Factory::getLanguage();
			$r             = new \ReflectionClass('\Joomla\CMS\Language\Language');
			$overridesProp = $r->getProperty('override');
			$overridesProp->setAccessible(true);
			$overrides = $overridesProp->getValue($language);
			// only keep our overrides
			$mergedOverrides = [];
			// Keys can have a sub-level. In Overrides this is marked with a double underscore:
			// COM_FORSEO_MAIN_MENU__DASHBOARD
			foreach ($overrides as $key => $langString)
			{
				if (Wb\startsWith($key, $extension))
				{
					$keys = explode('__', strtolower(Wb\lTrim($key, $extension)));
					if (count($keys) == 1)
					{
						$mergedOverrides[] = $langString;
					}
					else
					{
						$mergedOverrides[$keys[0]]           = isset($mergedOverrides[$keys[0]]) ? $mergedOverrides[$keys[0]] : array();
						$mergedOverrides[$keys[0]][$keys[1]] = $langString;
					}
				}
			}

			return $mergedOverrides;
		}
		catch (\Throwable $e)
		{
			return array();
		}
		catch (\Exception $e)
		{
			return array();
		}
	}

	public function loadLanguageFile($name, $location = '')
	{
		$language = Factory::getLanguage();
		$location = 'admin' == $location ? JPATH_ADMINISTRATOR : '';
		$language->load($name, $location);
	}

	public function t($key, $options = array('js_safe' => false, 'lang' => ''))
	{
		$options['jsSafe'] = !empty($options['js_safe']);

		return Text::_($key, $options);
	}

	public function tprintf($key)
	{
		$args = func_get_args();

		return call_user_func_array('\Joomla\CMS\Language\Text::sprintf', $args);
	}

	public function getTimezone()
	{
		return $this->app->get('offset');
	}

	// html operations
	public function setHttpStatus($code, $message)
	{
		$this->app->setHeader('status', $code);

		return $this;
	}

	public function getHttpStatus()
	{
		$status  = System\Http::RETURN_OK;
		$headers = $this->app->getHeaders();
		foreach ($headers as $header)
		{
			if ('status' == strtolower($header['name']))
			{
				$status = (int) $header['value'];
			}
		}

		return empty($status)
			? System\Http::RETURN_OK
			: $status;
	}

	//public function addScript($url, $type = "text/javascript", $defer = false, $async = false);
	public function addScript($url, $options = array(), $attribs = array())
	{
		Factory::getDocument()->addScript($url, $options, $attribs);

		return $this;
	}

	public function addScripts($scripts)
	{
		if (!is_array($scripts))
		{
			return $this;
		}

		foreach ($scripts as $script)
		{
			$this->addScript(
				Wb\arrayGet($script, 'url', ''),
				Wb\arrayGet($script, 'options', array()),
				Wb\arrayGet($script, 'attr', array())
			);
		}

		return $this;
	}

	public function addScriptDeclaration($content, $type = 'text/javascript')
	{
		Factory::getDocument()->addScriptDeclaration($content, $type);

		return $this;
	}

	//public function addStyleSheet($url, $type = 'text/css', $media = null, $attribs = array());
	public function addStyleSheet($url, $options = array(), $attribs = array())
	{
		Factory::getDocument()->addStyleSheet($url, $options, $attribs);

		return $this;
	}

	public function addStyleSheets($stylesheets)
	{
		if (!is_array($stylesheets))
		{
			return $this;
		}

		foreach ($stylesheets as $stylesheet)
		{
			$this->addStyleSheet(
				Wb\arrayGet($stylesheet, 'url', ''),
				Wb\arrayGet($stylesheet, 'options', array()),
				Wb\arrayGet($stylesheet, 'attr', array())
			);
		}

		return $this;
	}

	public function addStyleDeclaration($content, $type = 'text/css')
	{

		Factory::getDocument()->addStyleDeclaration($content, $type);

		return $this;
	}

	public function isMultipageContent($contentData)
	{
		$context = Wb\arrayGet($contentData, 'context', '');

		if ('com_content.article' != $context)
		{
			return false;
		}

		$content = Wb\arrayGet($contentData, 'content');
		if (empty($content) || empty($content->text))
		{
			return false;
		}

		return Wb\contains(
			$content->text,
			'class="system-pagebreak'
		);

	}

	public function setTitle($title)
	{
		Factory::getDocument()->setTitle($title);

		return $this;
	}

	public function getTitle()
	{
		$title = Factory::getDocument()->getTitle();

		return empty($title)
			? ''
			: $title;
	}

	public function setAdminTitle($title)
	{
		ToolbarHelper::title($title);

		return $this;
	}

	public function setDescription($description)
	{
		Factory::getDocument()->setDescription($description);

		return $this;
	}

	public function getDescription()
	{
		$desc = Factory::getDocument()->getDescription();

		return empty($desc)
			? ''
			: $desc;
	}

	public function getCanonical()
	{
		$headLinks = Wb\arrayGet(
			Factory::getDocument()->getHeadData(),
			[
				'links',
			],
			[]
		);
		foreach ($headLinks as $href => $headLink)
		{
			if (Wb\arrayGet($headLink, 'relation', '') == 'canonical')
			{
				return System\Strings::pr('~([^:])(/{2,})~', '$1/', $href);
			}
		}

		return '';
	}

	public function addHeadLink($href, $relation, $relType = 'rel', $attribs = array(), $replace = true)
	{
		$document = Factory::getDocument();
		if ($replace)
		{
			$links = Wb\arrayGet(
				$document->getHeadData(),
				'links',
				[]
			);
			$links = array_filter(
				$links,
				function ($link) {
					return Wb\arrayGet($link, 'relation', '') != 'canonical';
				}
			);
			$document->setHeadData(
				[
					'links' => $links
				]
			);
		}

		$document->addHeadLink($href, $relation, $relType, $attribs);

		return $this;
	}

	public function setMetaData($name, $content, $attribute = 'name')
	{
		Factory::getDocument()->setMetaData($name, $content, $attribute);

		return $this;
	}

	public function getMetaData($name, $attribute = 'name')
	{
		return Factory::getDocument()->getMetaData($name, $attribute);
	}

	public function addCustomTag($html)
	{
		Factory::getDocument()->addCustomTag($html);

		return $this;
	}

	public function setHeader($name, $value, $override = false)
	{
		$this->app->setHeader($name, $value, $override);

		return $this;
	}

	public function setResponseType($type = 'html', $filename = 'document')
	{
		switch ($type)
		{
			case 'json':
				Factory::getDocument()
					->setType('json')
					->setMimeEncoding('application/json');
				break;
			case 'js':
				Factory::getDocument()
					->setType('text/html')
					->setMimeEncoding('application/javascript');
				break;
			case 'css':
				Factory::getDocument()
					->setType('text/css')
					->setMimeEncoding('text/css');
				break;
			case 'raw':
			case 'html':
			default:
				break;
		}

		return $this;
	}

	// workflow operations
	public function triggerEvent($event, $args = [])
	{
		if (version_compare(JVERSION, '4.0', '<'))
		{
			$dispatcher = \JEventDispatcher::getInstance();
			$dispatcher->trigger(
				$event,
				$args
			);
		}
		else
		{
			Factory::getApplication()->triggerEvent(
				$event,
				$args
			);
		}

		return $this;
	}

	/**
	 * Register an event handler with the application.
	 *
	 * @param   string     $eventName
	 * @param   \callable  $eventHandler
	 *
	 * @return bool
	 */
	public function registerEventHandler($eventName, $eventHandler)
	{
		if (version_compare(JVERSION, '4', '<'))
		{
			\JEventDispatcher::getInstance()->register(
				$eventName,
				$eventHandler
			);
		}
		else
		{
			Factory::getApplication()
				->getDispatcher()
				->addListener(
					$eventName,
					$eventHandler
				);
		}
	}

	public function isFrontend()
	{
		return $this->app->isClient('site');
	}

	public function isBackend()
	{
		return $this->app->isClient('administrator');
	}

	/**
	 * Whether the current page is an edit page.
	 * Note that we don't memoize the value in case this is called prior to
	 * onAfterRoute and the request has not been parsed yet. We should not do
	 * that but we may do it anyway.
	 *
	 * @return bool
	 */
	public function isFrontendEditPage()
	{
		// unlikely to be editing without being logged in
		// also rules out admin
		if ($this->isGuest())
		{
			return false;
		}

		$input  = $this->app->input;
		$option = $input->getCmd('option');
		$view   = $input->get('view');
		if (Wb\contains($view, '.'))
		{
			$view = explode('.', $view);
			$view = array_pop($view);
		}
		$task = $input->getCmd('task');
		if (Wb\contains($task, '.'))
		{
			$task = explode('.', $task);
			$task = array_pop($task);
		}

		$layout = $input->getCmd('layout');

		$isEditPage =
			in_array($option, ['com_config', 'com_contentsubmit', 'com_cckjseblod'])
			||
			($option == 'com_comprofiler' && in_array($task, ['', 'userdetails']))
			||
			in_array($task, ['edit', 'form', 'submission'])
			||
			in_array($view, ['edit', 'form'])
			||
			in_array($layout, ['edit', 'form', 'write']);

		return
			$this->app->isClient('site')
			&&
			$isEditPage;
	}

	public function isOffline()
	{
		return $this->app->get('offline');
	}

	public function enableOfflineMode()
	{
		return $this->app->set('offline', 1);
	}

	public function disableOfflineMode()
	{
		return $this->app->set('offline', 0);
	}

	public function isHtmlDocument()
	{
		return 'html' == Factory::getDocument()->getType();
	}

	public function isFeedDocument()
	{
		return 'feed' == Factory::getDocument()->getType();
	}

	public function isDocumentType()
	{
		return Factory::getDocument()->getType();
	}

	public function isLikelyEditPage()
	{
		$view = strtolower($this->app->input->getCmd('view'));
		if (in_array($view, ['form', 'edit']))
		{
			return true;
		}
		$layout = strtolower($this->app->input->getCmd('layout'));
		if (in_array($layout, ['form', 'edit']))
		{
			return true;
		}

		return false;
	}

	public function canCompress()
	{
		return $this->app->get('gzip')
			&& !ini_get('zlib.output_compression')
			&& (ini_get('output_handler') != 'ob_gzhandler');
	}

	public function getDocument()
	{
		return $this->app->getDocument();
	}

	public function getDocumentContent()
	{
		return $this->app->getBody();
	}

	public function isDebugEnabled()
	{
		return (bool) $this->app->get('debug');
	}

	// hooks
	public function getHooksPath()
	{
		return JPATH_ROOT . '/libraries/weeblr';
	}

	public function addHook($id, $callback, $priority = 100)
	{
		$added = false;

		if (!empty($id) && is_string($id) && is_callable($callback))
		{
			$priority                      = (int) $priority;
			self::$hooks[$id]              = empty(self::$hooks[$id]) ? array() : self::$hooks[$id];
			self::$hooks[$id][$priority]   = empty(self::$hooks[$id][$priority]) ? array() : self::$hooks[$id][$priority];
			self::$hooks[$id][$priority][] = array(
				'callback' => $callback,
				'hash'     => System\Auth::callbackUniqueId($callback)
			);

			// re-order by priority
			ksort(self::$hooks[$id]);

			$added = true;
		}

		return $added;
	}

	//
	public function removeHook($id, $callback, $priority = null)
	{
		$removed = false;

		// cannot remove, hook does not exist
		if (!array_key_exists($id, self::$hooks))
		{
			return $removed;
		}

		// do not remove a hook that is being executed
		if (in_array($id, self::$hooksStack))
		{
			return $removed;
		}

		$hash = $this->callbackUniqueId($callback);
		if (is_null($priority))
		{
			// if no priority was specified, we remove the hook
			// callback from all priority levels
			foreach (self::$hooks[$id] as $priority => $hookRecord)
			{
				$removed = $this->removeHookCallback($id, $priority, $hash);
				if ($removed)
				{
					break;
				}
			}
		}
		else
		{
			// a priority was specified, we only remove the callback
			// from that priority
			$removed = $this->removeHookCallback($id, $priority, $hash);
		}

		return $removed;
	}

	/**
	 * @param   string  $id        Dot-joined unique identifier for the hook
	 * @param   int     $priority  Restrict removal to a given priority level
	 * @param   string  $hash
	 *
	 * @return bool true if callback was remo
	 */
	private function removeCallback($id, $priority, $hash)
	{
		$removed = false;
		foreach (self::$hooks[$id][$priority] as $index => $hookRecord)
		{
			if ($hash == $hookRecord['hash'])
			{
				$removed = true;
				unset(self::$hooks[$id][$priority][$index]);
			}
		}

		return $removed;
	}

	public function executeHook($filter, $params)
	{
		// remove the filter id from params array
		$id = array_shift($params);

		// default returned value
		$currentValue = null;
		if (count($params) > 0)
		{
			$currentValue = $params[0];
		}

		// invalid hook id
		if (!is_string($id))
		{
			return $currentValue;
		}

		// no hook registered
		if (empty(self::$hooks[$id]))
		{
			return $currentValue;
		}

		// already running. We don't allow nesting
		self::$hooksStack[] = $id;

		// increase run counter
		self::$hooksRuns[$id] = isset(self::$hooksRuns[$id]) ? self::$hooksRuns[$id]++ : 1;

		// iterate over registered hook handlers
		foreach (self::$hooks[$id] as $priority => $callbackList)
		{
			foreach ($callbackList as $callbackRecord)
			{
				if ($filter)
				{
					$params[0] = call_user_func_array($callbackRecord['callback'], $params);
				}
				else
				{
					call_user_func_array($callbackRecord['callback'], $params);
				}
			}
		}

		$newValue = null;
		if ($filter)
		{
			$newValue = isset($params[0]) ? $params[0] : null;
		}

		array_pop(self::$hooksStack);

		return $newValue;
	}

	public function hasHook($id)
	{
		$hasHook = false;
		if (!empty($id) && is_string($id))
		{
			$hasHook = !empty(self::$hooks[$id]);
		}

		return $hasHook;
	}

	// display, or handle error
	public function handleError($request)
	{
		return $this;
	}

	public function handleMessage($msg, $type = 'info')
	{
		$this->app->enqueueMessage($msg, $type);

		return $this;
	}

	// routing, redirect
	public function route($url, $xhtml = true, $ssl = null)
	{
		return Route::link('site', $url, $xhtml, $ssl);
	}

	public function redirectTo($redirectTo, $redirectMethod = 301)
	{
		if (!empty($redirectTo))
		{
			// redirect to target $redirectTo
			$this->app->redirect(
				$redirectTo,
				$redirectMethod);
		}
	}

	// authorization
	public function authorize($action, $subject, $userId = null)
	{
		return Factory::getUser($userId)->authorise($action, $subject);
	}

	// filesystem

	public function moveFile($src, $dest, $path = '')
	{
		return File::move($src, $dest);
	}

	public function moveFiles($files, $path = '')
	{
		$files = Wb\arrayEnsure($files);
		$moved = [];
		foreach ($files as $src => $dest)
		{
			$moved[$src] = $this->moveFile($src, $dest, $path);
		}

		return $moved;
	}

	public function moveToFolder($files, $src, $dest)
	{
		$files = Wb\arrayEnsure($files);
		$moved = [];
		foreach ($files as $file)
		{
			$moved[$file] = $this->moveFile(
				Path::clean($src . '/' . $file),
				Path::clean($dest . '/' . $file)
			);
		}

		return $moved;
	}

	public function deleteFile($file)
	{
		if (is_file(Path::clean($file)))
		{
			return File::delete($file);
		}

		return true;
	}

	public function deleteFiles($files)
	{
		$files   = Wb\arrayEnsure($files);
		$deleted = [];
		foreach ($files as $file)
		{
			$deleted[$file] = $this->deleteFile($file);
		}

		return $deleted;
	}

	public function listFiles($path, $filter = '.', $recurse = false, $full = false, $exclude = array('.svn', 'CVS', '.DS_Store', '__MACOSX'),
	                          $excludeFilter = array('^\..*', '.*~'), $naturalSort = false)
	{
		if (!is_dir(Path::clean($path)))
		{
			return [];
		}

		return Folder::files($path, $filter, $recurse, $full, $exclude, $excludeFilter, $naturalSort);
	}

	public function createFolders($folders)
	{
		$folders = Wb\arrayEnsure($folders);
		foreach ($folders as $folder)
		{
			Folder::create($folder);
		}
	}

	public function deleteFolders($folders, $exclude = [])
	{
		$folders = Wb\arrayEnsure($folders);
		$folders = array_diff(
			$folders,
			$exclude
		);
		foreach ($folders as $folder)
		{
			if (is_dir(Path::clean($folder)))
			{
				Folder::delete($folder);
			}
		}
	}

	public function moveFolders($moves)
	{
		$moves = Wb\arrayEnsure($moves);
		foreach ($moves as $source => $destination)
		{
			Folder::move($source, $destination);
		}
	}

	public function listFolders($path, $filter = '.', $recurse = false, $full = false, $exclude = array('.svn', 'CVS', '.DS_Store', '__MACOSX'), $excludefilter = array('^\..*'))
	{
		if (!is_dir(Path::clean($path)))
		{
			return [];
		}

		return Folder::folders($path, $filter, $recurse, $full, $exclude, $excludefilter);
	}

	// Display
	public function defaultItemsPerPage()
	{
		return $this->app->get('list_limit');
	}

	/**
	 * Joomla 4 and up only. Get update credentials
	 *
	 * @param $extensionType
	 * @param $extensionElement
	 *
	 * @return string
	 * @throws \Exception
	 */
	public function getUpdateId($extensionType, $extensionElement)
	{
		$updateSiteId = $this->getUpdateSiteId($extensionType, $extensionElement);

		$updateSite = Table::getInstance('UpdateSite');
		$updateSite->load($updateSiteId);

		if (empty($updateSite) || empty($updateSite->extra_query))
		{
			// no update id recorded yet.
			return '';
		}

		// extract download id
		parse_str($updateSite->extra_query, $parsedBits);
		$parsedBits = Wb\arrayEnsure($parsedBits);

		return Wb\arrayGet($parsedBits, 'wblr_k', '');
	}

	/**
	 * Joomla 4 and up only. Set dlid.
	 *
	 * @param   string  $id
	 *
	 * @throws \Exception
	 */
	public function setUpdateId($extensionType, $extensionElement, $id)
	{
		$updateSiteId  = $this->getUpdateSiteId($extensionType, $extensionElement);
		$newExtraQuery = empty($id)
			? ''
			: 'wblr_k=' . $id . '&provider=weeblr.zip';
		$updateSite    = Table::getInstance('UpdateSite');
		$updateSite->load($updateSiteId);
		$updateSite->bind(
			[
				'extra_query' => $newExtraQuery
			]
		);
		$updateSite->store();
	}

	private function getUpdateSiteId($extensionType, $extensionElement)
	{
		static $updateSiteId = null;

		if (is_null($updateSiteId))
		{
			$db = Factory::getDbo();

			$query = $db->getQuery(true)
				->select($db->quoteName('extension_id'))
				->from($db->quoteName('#__extensions'))
				->where($db->quoteName('type') . ' = ' . $db->quote($extensionType))
				->where($db->quoteName('element') . ' = ' . $db->quote($extensionElement));
			$db->setQuery($query);
			$extensionId = $db->loadResult();

			if (empty($extensionId))
			{
				throw  new \Exception('Joomla platform, getUpdateSiteId: ' . $extensionType . '::' . $extensionElement . ' not found in Extensions table.', 500);
			}

			$query = $db->getQuery(true)
				->select($db->quoteName('update_site_id'))
				->from($db->quoteName('#__update_sites_extensions'))
				->where($db->quoteName('extension_id') . ' = ' . $db->quote($extensionId));
			$db->setQuery($query);
			$updateSiteId = $db->loadResult();

			if (empty($updateSiteId))
			{
				throw  new \Exception('Joomla platform, getUpdateSiteId: ' . $extensionType . '::' . $extensionElement . ' has not registered any update site.', 500);
			}
		}

		return $updateSiteId;
	}

	/**
	 * Disable plugins execution, by detaching them from the Joomla! dispatcher
	 *
	 * @param   string        $type
	 * @param   string|array  $pluginsNames
	 */
	public function disablePlugins($type, $pluginsNames)
	{
		if (empty($type) || empty($pluginsNames))
		{
			return;
		}

		if (!is_array($pluginsNames))
		{
			$pluginsNames = [$pluginsNames];
		}

		// load the plugins of this type
		Plugin\PluginHelper::importPlugin($type);

		$platformVersion = $this->majorVersion();
		$methodName      = 'disableJ' . $platformVersion;

		if (is_callable([$this, $methodName]))
		{
			$this->{$methodName}($type, $pluginsNames);
		}
	}

	private function disableJ4($type, $pluginsNames)
	{
		$dispatcher = Factory::getContainer()->get('dispatcher');

		// iterate over plugins named in the list
		foreach ($pluginsNames as $pluginName)
		{
			// disable this plugin
			$className = 'Plg' . ucfirst($type) . ucfirst($pluginName);
			if (class_exists($className))
			{
				$listenersList = $dispatcher->getListeners();
				foreach ($listenersList as $event => $listeners)
				{
					foreach ($listeners as $index => $listener)
					{
						if (is_array($listener) && !empty($listener[0]) && is_array($listener[0]) && is_callable($listener[0]))
						{
							// 0: object 1: method
							$object = $listener[0];
							if ($object instanceof $className)
							{
								$dispatcher->removeListener(
									$event,
									$listener
								);
							}
						}
						if (is_object($listener) && is_callable($listener)) // closure
						{
							$class = get_class($listener);
							if ('Closure' == $class)
							{
								// container / method / serviceId
								$function = new ReflectionFunction($listener);
								$object   = $function->getClosureThis();
								if ($object instanceof $className)
								{
									$dispatcher->removeListener(
										$event,
										$listener
									);
								}
							}
						}
						if (is_object($listener))
						{
							// container / method / serviceId
						}
					}
				}
			}
		}
	}

	private function disableJ3($type, $pluginsNames)
	{
		$dispatcher = \JEventDispatcher::getInstance();

		// iterate over plugins named in the list
		foreach ($pluginsNames as $pluginName)
		{
			// disable this plugin
			$className = 'Plg' . ucfirst($type) . ucfirst($pluginName);
			if (class_exists($className))
			{
				$pluginDetails  = Plugin\PluginHelper::getPlugin(
					$type,
					strtolower($pluginName)
				);
				$pluginInstance = new $className(
					$dispatcher,
					(array) $pluginDetails
				);
				$dispatcher->detach($pluginInstance);
			}
		}
	}
}
