<?php
/**
 * 4LOGS
 *
 * @package          4LOGS
 * @copyright        Copyright Weeblr llc - 2021
 * @author           Yannick Gaultier - Weeblr llc
 * @license          GNU General Public License version 3; see LICENSE.md
 * @version          1.1.2.217
 * @date        2021-08-20
 */

namespace Weeblr\Forlogs\Model;

use Weeblr\Forlogs\Helper;
use Weeblr\Wblib\V_FORLOGS_217\Wb;
use Weeblr\Wblib\V_FORLOGS_217\System;
use Weeblr\Wblib\V_FORLOGS_217\Base;
use Weeblr\Wblib\V_FORLOGS_217\Fs;

// Security check to ensure this file is being included by a parent file.
defined('_JEXEC') || defined('WBLIB_EXEC') || die;

class Files extends Base\Base
{
	/**
	 * @var string Santized root path we are processing.
	 */
	private $path = '';

	/**
	 * @var Helper\File Convenience instance of a file helper.
	 */
	private $fileHelper;

	/**
	 * @var string[] log files extension we can process.
	 */
	private $allowedExtensions = [
		'php'
	];

	/**
	 * Store a sitemap config for convenience.
	 */
	public function __construct($options = [])
	{
		parent::__construct();

		$this->fileHelper = $this->factory
			->getA(Helper\File::class);

		$this->path = $this->fileHelper
			->sanitizePath(
				Wb\arrayGet($options, 'path', ''),
				Wb\arrayGet($options, 'root', '')
			);
	}

	public function files()
	{
		$files = $this->filesFromFolder(
			$this->path,
			[]
		);

		$count = count($files);

		return [
			'data'  => [
				'files' => $files,
			],
			'count' => $count,
			'total' => $count
		];
	}

	/**
	 * Get files from a folder, recursively.
	 *
	 * @param string $path
	 * @param array  $files
	 * @return array
	 */
	protected function filesFromFolder($path, $files)
	{
		$rootPath = Wb\slashForward($this->path);

		if (!is_dir($path))
		{
			return $files;
		}

		foreach (new \DirectoryIterator($path) as $fileInfo)
		{
			if ($fileInfo->isDot())
			{
				// never include dot files
				continue;
			}

			// folder
			if ($fileInfo->isDir())
			{
				$subFolderFiles = $this->filesFromFolder(
					$fileInfo->getPathName(),
					$files
				);

				if (!empty($subFolderFiles))
				{
					$files = array_merge(
						$files,
						$subFolderFiles
					);
				}
			}
			else // files
			{
				// regular files
				if (!in_array(
					$fileInfo->getExtension(),
					$this->allowedExtensions)
				)
				{
					continue;
				}

				$cleanPath = Wb\lTrim(Wb\slashForward($fileInfo->getPath()), $rootPath);
				if (empty($files[$cleanPath]))
				{
					$files[$cleanPath] = [];
				}

				$files[$cleanPath][] = [
					'folder'    => Wb\lTrim(Wb\slashForward($fileInfo->getPath()), $rootPath),
					'file'      => Wb\lTrim(Wb\slashForward($fileInfo->getFileName()), $rootPath),
					'path'      => Wb\lTrim(Wb\slashForward($fileInfo->getPathName()), $rootPath),
					'size'      => System\Convert::displayableFileSize($fileInfo->getSize()),
					'bytesSize' => $fileInfo->getSize(),
					'mtime'     => $fileInfo->getMTime()
				];
			}
		}

		return $files;
	}

	/**
	 * List all the folders found under the folder
	 * specified in the path option.
	 *
	 * @param array $options
	 *
	 * @return array
	 * @throws \Exception
	 */
	public function folders($options)
	{
		$path = $this->fileHelper
			->sanitizePath(
				Wb\arrayGet(
					$options,
					'path',
					''
				),
				$this->rootPath
			);

		$folders = [];

		foreach (new \DirectoryIterator($path) as $fileInfo)
		{
			if (
				$fileInfo->isDot()
				||
				$fileInfo->isFile()
			)
			{
				continue;
			}

			$folders[] = $fileInfo->getFilename();
		}

		return [
			'data'  => $folders,
			'count' => count($folders),
			'total' => count($folders),
		];
	}

	/**
	 * Delete one or more log files.
	 *
	 * @return array|\Exception
	 */
	public function delete()
	{
		try
		{
			// prepare for when we'll delete a group of files instead of single ones.
			$paths = Wb\arrayEnsure($this->path);
			foreach ($paths as $path)
			{
				if (is_dir($path))
				{
					$this->platform->deleteFolders($path);
				}
				else
				{
					Fs\File::forceDelete($path);
				}
			}
		}
		catch (\Throwable $e)
		{
			return new \Exception($e->getMessage(), System\Http::RETURN_NOT_FOUND);
		}

		return [
			'data'  => null,
			'count' => count($paths),
			'total' => count($paths),
		];
	}
}
