<?php
/**
 * 4LOGS
 *
 * @package          4LOGS
 * @copyright        Copyright Weeblr llc - 2021
 * @author           Yannick Gaultier - Weeblr llc
 * @license          GNU General Public License version 3; see LICENSE.md
 * @version          1.1.2.217
 * @date        2021-08-20
 */

namespace Weeblr\Forlogs\Helper;

use Weeblr\Wblib\V_FORLOGS_217\Wb;
use Weeblr\Wblib\V_FORLOGS_217\Base;

// Security check to ensure this file is being included by a parent file.
defined('_JEXEC') || defined('WBLIB_EXEC') || die;

class Language extends Base\Base
{
	/**
	 * Builds a record with the current language code to use and the corresponding strings.
	 *
	 * 1. Load default language file {APP_ROOT}/locales/en/{file_name}.json
	 * 2. Look for a json file with strings in {APP_ROOT}/locales/{current_language}/{file_name}.json
	 * 3. Merge the translated strings over the original EN language as our translation system only
	 *    include translated strings
	 * 3. Get language strings overrides from the platform for {current_language}
	 * 4. Merge overrides with built-in strings
	 * 4. Returns an array with strings
	 *
	 * @param []|string $files
	 * @param string $languageTag
	 *
	 * @return array
	 */

	public function getJsLanguageStrings($files, $languageTag)
	{
		$languageStrings = [];

		foreach (Wb\arrayEnsure($files) as $file)
		{
			$languageStrings = array_replace_recursive(
				$languageStrings,
				$this->doGetJsLanguageStrings($file, $languageTag)
			);
		}

		return $languageStrings;
	}

	/**
	 * Builds a record with the language code to use and the corresponding strings.
	 *
	 * 1. Load default language file {APP_ROOT}/locales/en/{file_name}.json
	 * 2. Look for a json file with strings in {APP_ROOT}/locales/{current_language}/{file_name}.json
	 * 3. Merge the translated strings over the original EN language as our translation system only
	 *    include translated strings
	 * 3. Get language strings overrides from the platform for {current_language}
	 * 4. Merge overrides with built-in strings
	 * 4. Returns and array with tag and strings
	 *
	 * @param string $file
	 * @param string $languageTag
	 *
	 * @return array
	 */
	private function doGetJsLanguageStrings($file, $languageTag)
	{
		// 1 - Source
		$localeFile         = FORLOGS_APP_PATH . '/locales/en/' . $file . '.json';
		$rawlanguageStrings = file_get_contents($localeFile);
		$sourceStrings      = json_decode($rawlanguageStrings, true);

		// 2 - Desired translations, if any
		$bits               = explode('-', $languageTag);
		$languageFamily     = $bits[0];
		$localeFile         = FORLOGS_APP_PATH . '/locales/' . $languageTag . '/' . $file . '.json';
		$familyLanguageFile = FORLOGS_APP_PATH . '/locales/' . $languageFamily . '/' . $file . '.json';
		if (file_exists($localeFile))
		{
			$rawlanguageStrings = file_get_contents($localeFile);
		}
		else if (file_exists($familyLanguageFile))
		{
			$rawlanguageStrings = file_get_contents($familyLanguageFile);
		}

		$translatedStrings = json_decode($rawlanguageStrings, true);


		// 3 - Possible user overrides from Joomla language feature.
		$overrides = $this->platform->getLanguageOverrides('PLG_SYSTEM_FORLOGS');

		// ready to inject in page
		return array_replace_recursive(
			Wb\arrayEnsure($sourceStrings),
			Wb\arrayEnsure($translatedStrings),
			$overrides
		);
	}
}