<?php
/**
 * @package        Joomla
 * @subpackage     Helpdesk Pro
 * @author         Tuan Pham Ngoc
 * @copyright      Copyright (C) 2013 - 2026 Ossolution Team
 * @license        GNU/GPL, see LICENSE.php
 */

namespace OSSolution\HelpdeskPro\Plugin\System\AutoCloseTicket\Extension;

use Exception;
use Joomla\CMS\Application\CMSApplicationInterface;
use Joomla\CMS\Cache\Cache;
use Joomla\CMS\Factory;
use Joomla\CMS\Plugin\CMSPlugin;
use Joomla\Database\DatabaseAwareTrait;
use Joomla\Database\DatabaseInterface;
use Joomla\Event\DispatcherInterface;
use Joomla\Event\Event;
use Joomla\Event\SubscriberInterface;
use OSSolution\HelpdeskPro\Site\Helper\Helper as HelpdeskProHelper;

defined('_JEXEC') or die;

class AutoCloseTicket extends CMSPlugin implements SubscriberInterface
{
	use DatabaseAwareTrait;

	public function __construct(
		DispatcherInterface $dispatcher,
		array $config,
		CMSApplicationInterface $application,
		DatabaseInterface $db
	) {
		parent::__construct($dispatcher, $config);

		$this->setDatabase($db);
		$this->setApplication($application);
	}

	/**
	 * Returns an array of events this subscriber will listen to.
	 *
	 * @return array
	 *
	 */
	public static function getSubscribedEvents(): array
	{
		return [
			'onAfterRoute' => 'onAfterRoute',
		];
	}

	/**
	 * Check and close ticket
	 *
	 * @param   Event  $event
	 *
	 * @return void
	 * @throws Exception
	 */
	public function onAfterRoute(Event $event): void
	{
		$lastRun   = (int) $this->params->get('last_run', 0);
		$now       = time();
		$cacheTime = (int) $this->params->get('cache_time', 2) * 3600; // The plugin should be run every 2 hours

		if (($now - $lastRun) < $cacheTime)
		{
			return;
		}

		//Store last run time
		$this->params->set('last_run', $now);
		$db    = $this->getDatabase();
		$query = $db->getQuery(true)
			->update('#__extensions')
			->set('params = ' . $db->quote($this->params->toString()))
			->where($db->quoteName('element') . '=' . $db->quote('autocloseticket'))
			->where($db->quoteName('folder') . '=' . $db->quote('system'));

		try
		{
			// Lock the tables to prevent multiple plugin executions causing a race condition
			$db->lockTable('#__extensions');
		}
		catch (Exception $e)
		{
			// If we can't lock the tables it's too risk continuing execution
			return;
		}

		try
		{
			// Update the plugin parameters
			$result = $db->setQuery($query)->execute();
			$this->clearCacheGroups(['com_plugins'], [0, 1]);
		}
		catch (Exception $exc)
		{
			// If we failed to execite
			$db->unlockTables();
			$result = false;
		}

		try
		{
			// Unlock the tables after writing
			$db->unlockTables();
		}
		catch (Exception $e)
		{
			// If we can't lock the tables assume we have somehow failed
			$result = false;
		}

		// Abort on failure
		if (!$result)
		{
			return;
		}

		// Bootstrap the component
		require JPATH_ADMINISTRATOR . '/components/com_helpdeskpro/bootstrap.php';

		$numberDay          = $this->params->get('number_days', 14);
		$config             = HelpdeskProHelper::getConfig();
		$closedTicketStatus = $config->closed_ticket_status;

		$currentDate = $db->quote(Factory::getDate()->toSql());
		$query->clear()
			->update('#__helpdeskpro_tickets')
			->set('status_id = ' . (int) $closedTicketStatus)
			->where("DATEDIFF($currentDate , modified_date) >= " . (int) $numberDay)
			->where('status_id != ' . (int) $closedTicketStatus);
		$db->setQuery($query)
			->execute();
	}

	/**
	 * Override registerListeners to only enable the plugin if
	 *
	 * @return void
	 */
	public function registerListeners()
	{
		if (!file_exists(JPATH_ROOT . '/components/com_helpdeskpro/helpdeskpro.php'))
		{
			return;
		}

		parent::registerListeners();
	}

	/**
	 * Clears cache groups. We use it to clear the plugins cache after we update the last run timestamp.
	 *
	 * @param   array  $clearGroups   The cache groups to clean
	 * @param   array  $cacheClients  The cache clients (site, admin) to clean
	 *
	 * @return  void
	 */
	private function clearCacheGroups(array $clearGroups, array $cacheClients = [0, 1])
	{
		$cachePath = $this->getApplication()->get('cache_path', JPATH_SITE . '/cache');

		foreach ($clearGroups as $group)
		{
			foreach ($cacheClients as $clientId)
			{
				try
				{
					$options = [
						'defaultgroup' => $group,
						'cachebase'    => ($clientId) ? JPATH_ADMINISTRATOR . '/cache' : $cachePath,
					];
					$cache   = Cache::getInstance('callback', $options);
					$cache->clean();
				}
				catch (Exception $e)
				{
					// Ignore it
				}
			}
		}
	}
}