<?php

/**
 * @package   OSDownloads-Pro
 * @contact   www.joomlashack.com, help@joomlashack.com
 * @copyright 2005-2025 Joomlashack.com. All rights reserved
 * @license   https://www.gnu.org/licenses/gpl.html GNU/GPL
 *
 * This file is part of OSDownloads-Pro.
 *
 * OSDownloads-Pro is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 *
 * OSDownloads-Pro is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with OSDownloads-Pro.  If not, see <https://www.gnu.org/licenses/>.
 */

use Joomla\CMS\Factory;
use Joomla\CMS\Language\Text;
use Joomla\CMS\Plugin\CMSPlugin;
use Joomla\Database\DatabaseInterface;

// phpcs:disable PSR1.Files.SideEffects
defined('_JEXEC') or die();
// phpcs:enable PSR1.Files.SideEffects
// phpcs:disable PSR1.Classes.ClassDeclaration.MissingNamespace

class PlgSearchOSDownloads extends CMSPlugin
{
    /**
     * @inheritdoc
     */
    protected $autoloadLanguage = true;

    /**
     * @var string[]
     */
    protected $areas = [
        'osdownloads' => 'Downloads',
    ];

    /**
     * @var DatabaseInterface|JDatabaseDriver
     */
    protected $db = null;

    /**
     * @var bool
     */
    protected $enabled = null;

    /**
     * @return bool
     */
    protected function isEnabled(): bool
    {
        if ($this->enabled === null) {
            $includePath   = JPATH_ADMINISTRATOR . '/components/com_osdownloads/include.php';
            $this->enabled = is_file($includePath) && include $includePath;
        }

        return $this->enabled;
    }

    /**
     * @return string[]
     */
    public function onContentSearchAreas(): array
    {
        return $this->isEnabled() ? $this->areas : [];
    }

    /**
     * @param string  $text
     * @param ?string $phrase
     * @param ?string $ordering
     * @param ?array  $areas
     *
     * @return object[]
     */
    public function onContentSearch(
        string $text,
        ?string $phrase = '',
        ?string $ordering = '',
        ?array $areas = null
    ): array {
        if (
            $this->isEnabled() == false
            || $text == false
            || ($areas && array_intersect($areas, array_keys($this->areas)) == false)
        ) {
            return [];
        }

        $db     = $this->db;
        $user   = Factory::getUser();
        $groups = array_map('intval', array_unique($user->getAuthorisedViewLevels()));
        $limit  = $this->params->get('search_limit', '50');

        // Find a menu item for downloads
        $menuId = $db->setQuery(
            $db->getQuery(true)
                ->select('m.id')
                ->from('#__menu AS m')
                ->where([
                    'm.link = ' . $db->quote('index.php?option=com_osdownloads&view=downloads'),
                    'm.published > 0',
                ])
        )
            ->loadResult();

        $section = Text::_('PLG_SEARCH_OSDOWNLOADS_SECTION');

        $query = $db->getQuery(true);
        $query->select([
            'a.name AS title',
            'a.brief AS text',
            'a.id AS eventID',
            'c.alias AS catslug',
            'a.created_time AS created',
            'a.alias AS slug',
            $query->concatenate([$db->quote($section), 'c.title'], ' / ') . ' AS section',
            '0 AS browsernav',
        ])
            ->from('#__osdownloads_documents AS a')
            ->innerJoin('#__categories AS c ON c.id = a.catid')
            ->where([
                'a.published = 1',
                'c.published = 1',
                sprintf('c.access IN (%s)', join(',', $groups)),
            ]);

        // The fields we want to search in
        $searchFields = [
            'a.name',
            'a.brief',
            'a.description_1',
            'a.description_2',
            'a.description_3',
        ];

        switch ($phrase) {
            case 'exact':
                $text = $db->quote('%' . $db->escape($text, true) . '%', false);

                $ors = [];
                foreach ($searchFields as $field) {
                    $ors[] = sprintf('%s LIKE %s', $db->quoteName($field), $text);
                }
                $query->where(sprintf('(%s)', implode(' OR ', $ors)));
                break;

            case 'all':
            case 'any':
            default:
                $words = array_map(
                    function ($item) use ($db) {
                        return $db->quote('%' . $item . '%');
                    },
                    preg_split('/\s/', $text)
                );

                $ors  = [];
                $type = $phrase == 'all' ? ' AND ' : ' OR ';
                foreach ($searchFields as $field) {
                    $fieldWhere = [];
                    foreach ($words as $word) {
                        $fieldWhere[] = sprintf('%s LIKE %s', $db->quoteName($field), $word);
                    }
                    $ors[] = sprintf('(%s)', join($type, $fieldWhere));
                }
                $query->where(sprintf('(%s)', join(' OR ', $ors)));
                break;
        }

        // Ordering of the results
        switch ($ordering) {
            case 'alpha':
            case 'oldest':
            case 'popular':
            case 'newest':
            default:
                $ordering = 'a.name';
        }
        $query->order($ordering . ' ASC');

        $rows = $db->setQuery($query, 0, $limit)->loadObjectList();


        foreach ($rows as $row) {
            $linkQuery = [
                'option' => 'com_osdownloads',
                'view'   => 'item',
                'id'     => (int)$row->eventID,
                'Itemid' => $menuId,
            ];
            $row->href = 'index.php?' . http_build_query($linkQuery);
        }

        return $rows;
    }
}
