<?php
/**
 * @package        Joomla
 * @subpackage     Membership Pro
 * @author         Tuan Pham Ngoc
 * @copyright      Copyright (C) 2012 - 2021 Ossolution Team
 * @license        GNU/GPL, see LICENSE.php
 */

defined('_JEXEC') or die;

use Joomla\Registry\Registry;
use Joomla\Utilities\ArrayHelper;

class plgOSMembershipAcymailing extends JPlugin
{
	/**
	 * Application object.
	 *
	 * @var    JApplicationCms
	 */
	protected $app;

	/**
	 * Database object.
	 *
	 * @var    JDatabaseDriver
	 */
	protected $db;

	/**
	 * Make language files will be loaded automatically.
	 *
	 * @var bool
	 */
	protected $autoloadLanguage = true;

	/**
	 * Constructor
	 *
	 * @param   object &$subject  The object to observe
	 * @param   array   $config   An optional associative array of configuration settings.
	 */
	public function __construct($subject, array $config = [])
	{
		if (!file_exists(JPATH_ROOT . '/components/com_acymailing/acymailing.php'))
		{
			return;
		}

		parent::__construct($subject, $config);
	}

	/**
	 * Return list of custom fields in ACYMailing which will be used to map with fields in Events Booking
	 *
	 * @return array
	 */
	public function onGetNewsletterFields()
	{
		$db    = $this->db;
		$query = $db->getQuery(true)
			->select($db->quoteName(['namekey', 'namekey'], ['value', 'text']))
			->from('#__acymailing_fields')
			->where('namekey NOT IN ("name", "email", "html")');
		$db->setQuery($query);

		return $db->loadObjectList();
	}

	/**
	 * Render setting form
	 *
	 * @param   OSMembershipTablePlan  $row
	 *
	 * @return array
	 */
	public function onEditSubscriptionPlan($row)
	{
		if (!$this->isExecutable())
		{
			return [];
		}

		ob_start();
		$this->drawSettingForm($row);
		$form = ob_get_clean();

		return ['title' => JText::_('PLG_OSMEMBERSHIP_ACYMAILING_LIST_SETTINGS'),
		        'form'  => $form,
		];
	}

	/**
	 * Store setting into database, in this case, use params field of plans table
	 *
	 * @param   OSMembershipTablePlan  $row
	 * @param   bool                   $isNew  true if create new plan, false if edit
	 */
	public function onAfterSaveSubscriptionPlan($context, $row, $data, $isNew)
	{
		if (!$this->isExecutable())
		{
			return;
		}

		$params = new Registry($row->params);

		// Prevent PHP notices/warnings
		$keys = ['acymailing_list_ids', 'acymailing_active_remove_list_ids', 'subscription_expired_acymailing_list_ids', 'acymailing_expired_assign_list_ids'];

		foreach ($keys as $key)
		{
			if (!isset($data[$key]))
			{
				$data[$key] = [];
			}
		}

		$params->set('acymailing_list_ids', implode(',', $data['acymailing_list_ids']));
		$params->set('acymailing_active_remove_list_ids', implode(',', $data['acymailing_active_remove_list_ids']));
		$params->set('subscription_expired_acymailing_list_ids', implode(',', $data['subscription_expired_acymailing_list_ids']));
		$params->set('acymailing_expired_assign_list_ids', implode(',', $data['acymailing_expired_assign_list_ids']));
		$params->set('mailing_list_custom_field', isset($data['mailing_list_custom_field']) ? $data['mailing_list_custom_field'] : 0);
		$row->params = $params->toString();

		$row->store();
	}

	/**
	 * Run when a membership activated
	 *
	 * @param   OSMembershipTableSubscriber  $row
	 */
	public function onMembershipActive($row)
	{
		if (!JMailHelper::isEmailAddress($row->email))
		{
			return;
		}

		$config = OSMembershipHelper::getConfig();

		// In case subscriber doesn't want to subscribe to newsleter, stop
		if ($config->show_subscribe_newsletter_checkbox && empty($row->subscribe_newsletter))
		{
			return;
		}

		$db    = $this->db;
		$query = $db->getQuery(true);

		/* @var OSMembershipTablePlan $plan */
		$plan = JTable::getInstance('Plan', 'OSMembershipTable');
		$plan->load($row->plan_id);
		$params = new Registry($plan->params);

		if ($fieldId = (int) $params->get('mailing_list_custom_field'))
		{
			$query->select('field_value')
				->from('#__osmembership_field_value')
				->where('subscriber_id = ' . $row->id)
				->where('field_id = ' . $fieldId);
			$db->setQuery($query);
			$fieldValue = $db->loadResult();

			if ($fieldValue && is_array(json_decode($fieldValue)))
			{
				$listNames = array_map('trim', json_decode($fieldValue));
			}
            elseif (is_string($fieldValue) && strpos($fieldValue, ', ') !== false)
			{
				$listNames = explode(', ', $fieldValue);
			}
            elseif (is_string($fieldValue))
			{
				$listNames = [$fieldValue];
			}
			else
			{
				$listNames = [];
			}

			if (!empty($listNames))
			{
				$listNames = array_map([$db, 'quote'], $listNames);
				$query->clear()
					->select('listid')
					->from('#__acymailing_list')
					->where('published = 1')
					->where('(name = ' . implode(' OR name = ', $listNames) . ')');
				$db->setQuery($query);
				$listIds = implode(',', $db->loadColumn());
			}
			else
			{
				$listIds = '';
			}
		}
		else
		{
			$listIds = trim($params->get('acymailing_list_ids', ''));
		}

		$removeListIds = trim($params->get('acymailing_active_remove_list_ids'));

		if ($listIds || $removeListIds)
		{
			require_once JPATH_ADMINISTRATOR . '/components/com_acymailing/helpers/helper.php';

			/* @var subscriberClass $userClass */
			$userClass               = acymailing_get('class.subscriber');
			$userClass->checkAccess  = false;
			$userClass->checkVisitor = false;
			$subId                   = $userClass->subid($row->email);

			if (!$subId)
			{
				$myUser         = new stdClass();
				$myUser->email  = $row->email;
				$myUser->name   = $row->first_name . ' ' . $row->last_name;
				$myUser->userid = $row->user_id;

				$subId = $userClass->save($myUser); //this
			}

			// Update data from Membership Pro to ACYMailing
			$this->updateAcyMailingFieldsData($row, $subId);

			$listIds = explode(',', $listIds);
			$listIds = ArrayHelper::toInteger($listIds);
			$listIds = array_filter($listIds);

			$removeListIds = explode(',', $removeListIds);
			$removeListIds = ArrayHelper::toInteger($removeListIds);
			$removeListIds = array_filter($removeListIds);

			$newSubscriptions = [];

			foreach ($listIds as $listId)
			{
				$newSubscriptions[$listId] = ['status' => 1];
			}

			foreach ($removeListIds as $listId)
			{
				$newSubscriptions[$listId] = ['status' => 0];
			}

			$userClass->checkAccess  = false;
			$userClass->checkVisitor = false;

			try
			{
				$userClass->saveSubscription($subId, $newSubscriptions);
			}
			catch (Exception $e)
			{

			}
		}
	}

	/**
	 * Plugin triggered when user update his profile
	 *
	 * @param   OSMembershipTableSubscriber  $row  The subscription record
	 */
	public function onProfileUpdate($row)
	{
		if (!JMailHelper::isEmailAddress($row->email))
		{
			return;
		}

		$query = $this->db->getQuery(true);
		$user  = JFactory::getUser($row->user_id);
		$query->update('#__acymailing_subscriber')
			->set('email = ' . $this->db->quote($row->email))
			->where('email = ' . $this->db->quote($user->email));
		$this->db->setQuery($query);

		try
		{
			$this->db->execute();
		}
		catch (Exception $e)
		{
			// There is another ACYMailing user uses this email, ignore

			return;
		}

		require_once JPATH_ADMINISTRATOR . '/components/com_acymailing/helpers/helper.php';

		/* @var subscriberClass $userClass */
		$userClass               = acymailing_get('class.subscriber');
		$userClass->checkAccess  = false;
		$userClass->checkVisitor = false;
		$subId                   = $userClass->subid($row->email);

		if ($subId)
		{
			$this->updateAcyMailingFieldsData($row, $subId);
		}
	}

	/**
	 * Run when a membership expiried die
	 *
	 * @param   OSMembershipTableSubscriber  $row
	 */
	public function onMembershipExpire($row)
	{
		if (!JMailHelper::isEmailAddress($row->email))
		{
			return;
		}

		$config = OSMembershipHelper::getConfig();

		// In case subscriber doesn't want to subscribe to newsleter, stop
		if ($config->show_subscribe_newsletter_checkbox && empty($row->subscribe_newsletter))
		{
			return;
		}

		/* @var OSMembershipTablePlan $plan */
		$plan = JTable::getInstance('Plan', 'OSMembershipTable');
		$plan->load($row->plan_id);
		$params        = new Registry($plan->params);
		$listIds       = trim($params->get('subscription_expired_acymailing_list_ids', ''));
		$assignListIds = trim($params->get('acymailing_expired_assign_list_ids', ''));

		if ($row->user_id)
		{
			$activePlans = OSMembershipHelperSubscription::getActiveMembershipPlans($row->user_id, [$row->id]);

			// He renewed his subscription before, so don't remove him from the lists
			if (in_array($row->plan_id, $activePlans))
			{
				return;
			}
		}

		if ($listIds != '' || $assignListIds != '')
		{
			require_once JPATH_ADMINISTRATOR . '/components/com_acymailing/helpers/helper.php';

			/* @var subscriberClass $userClass */
			$userClass               = acymailing_get('class.subscriber');
			$userClass->checkAccess  = false;
			$userClass->checkVisitor = false;

			$subId = $userClass->subid($row->email);

			if (!$subId && $assignListIds)
			{
				// Create new subscriber as it is needed to assign user to the lists
				$myUser         = new stdClass();
				$myUser->email  = $row->email;
				$myUser->name   = $row->first_name . ' ' . $row->last_name;
				$myUser->userid = $row->user_id;
				$subId          = $userClass->save($myUser); //this

				// Update custom fields
				$this->updateAcyMailingFieldsData($row, $subId);
			}


			if ($subId)
			{

				$listIds = explode(',', $listIds);
				$listIds = ArrayHelper::toInteger($listIds);
				$listIds = array_filter($listIds);

				$assignListIds = explode(',', $assignListIds);
				$assignListIds = ArrayHelper::toInteger($assignListIds);
				$assignListIds = array_filter($assignListIds);

				$newSubscriptions = [];

				foreach ($listIds as $listId)
				{
					$newSubscriptions[$listId] = ['status' => 0];
				}

				foreach ($assignListIds as $listId)
				{
					$newSubscriptions[$listId] = ['status' => 1];
				}

				try
				{
					$userClass->saveSubscription($subId, $newSubscriptions);
				}
				catch (Exception $e)
				{
					// Ignore error
				}
			}
		}
	}

	/**
	 * Method to synchronize custom fields data from Membership Pro to ACYMailing
	 *
	 * @param   OSMembershipTableSubscriber  $row
	 * @param   int                          $subId
	 */
	private function updateAcyMailingFieldsData($row, $subId)
	{
		if (!$subId)
		{
			return;
		}

		$db        = $this->db;
		$query     = $db->getQuery(true);
		$rowFields = OSMembershipHelper::getProfileFields($row->plan_id, true, null, $row->act);
		$data      = OSMembershipHelper::getProfileData($row, $row->plan_id, $rowFields);

		$updateFields = [];

		foreach ($rowFields as $rowField)
		{
			if (!$rowField->newsletter_field_mapping)
			{
				continue;
			}

			$fieldValue                                        = isset($data[$rowField->name]) ? $data[$rowField->name] : '';
			$updateFields[$rowField->newsletter_field_mapping] = $fieldValue;
		}

		if (count($updateFields))
		{
			$query->clear()
				->update('#__acymailing_subscriber')
				->where('subid = ' . (int) $subId);

			foreach ($updateFields as $updateFieldName => $updateFieldValue)
			{
				$query->set($db->quoteName($updateFieldName) . '=' . $db->quote($updateFieldValue));
			}

			$db->setQuery($query)
				->execute();
		}
	}

	/**
	 * Method to check if the plugin is executable
	 *
	 * @return bool
	 */
	private function isExecutable()
	{
		if ($this->app->isClient('site') && !$this->params->get('show_on_frontend'))
		{
			return false;
		}

		return true;
	}

	/**
	 * Display form allows users to change settings on subscription plan add/edit screen
	 *
	 * @param   OSMembershipTablePlan  $row
	 */
	private function drawSettingForm($row)
	{
		require_once JPATH_ADMINISTRATOR . '/components/com_acymailing/helpers/helper.php';

		$params               = new Registry($row->params);
		$activeAssignListIds  = explode(',', $params->get('acymailing_list_ids', ''));
		$activeRemoveListIds  = explode(',', $params->get('acymailing_active_remove_list_ids', ''));
		$expiredRemoveListIds = explode(',', $params->get('subscription_expired_acymailing_list_ids', ''));
		$expiredAssignListIds = explode(',', $params->get('acymailing_expired_assign_list_ids', ''));
		$listClass            = acymailing_get('class.list');
		$allLists             = $listClass->getLists();

		$query = $this->db->getQuery(true);
		$query->select('id, name')
			->from('#__osmembership_fields')
			->where('published = 1')
			->where('fieldtype = "Checkboxes"')
			->order('name');
		$this->db->setQuery($query);
		$mailingListFields = $this->db->loadObjectList();
		?>
        <div class="row-fluid">
            <div class="span6 pull-left">
                <fieldset class="adminform">
                    <legend><?php echo JText::_('OSM_WHEN_SUBSCRIPTION_ACTIVE'); ?></legend>
                    <div class="control-group">
                        <div class="control-label">
							<?php echo OSMembershipHelperHtml::getFieldLabel('acymailing_list_ids', JText::_('OSM_ASSIGN_TO_MAILING_LISTS'), JText::_('OSM_ASSIGN_TO_MAILING_LISTS_EXPLAIN')); ?>
                        </div>
                        <div class="controls">
							<?php echo JHtml::_('select.genericlist', $allLists, 'acymailing_list_ids[]', 'class="inputbox" multiple="multiple" size="10"', 'listid', 'name', $activeAssignListIds) ?>
                        </div>
                    </div>
                    <div class="control-group">
                        <div class="control-label">
							<?php echo OSMembershipHelperHtml::getFieldLabel('acymailing_active_remove_list_ids', JText::_('OSM_REMOVE_FROM_MAILING_LISTS'), JText::_('OSM_REMOVE_FROM_MAILING_LISTS_EXPLAIN')); ?>
                        </div>
                        <div class="controls">
							<?php
							echo JHtml::_('select.genericlist', $allLists, 'acymailing_active_remove_list_ids[]', 'class="inputbox" multiple="multiple" size="10"', 'listid', 'name', $activeRemoveListIds);
							?>
                        </div>
                    </div>
					<?php
					if (count($mailingListFields))
					{
						$options   = [];
						$options[] = JHtml::_('select.option', '', 'Select Field', 'id', 'name');
						$options   = array_merge($options, $mailingListFields);
						?>
                        <div class="control-group">
                            <div class="control-label">
								<?php echo OSMembershipHelperHtml::getFieldLabel('mailing_list_custom_field', JText::_('Mailing Lists Custom Field'), JText::_('If you select a custom field here, subscribers will be assigned to the mailist list which he choose on the options of this custom field instead of the lists you select for this plan')); ?>
                            </div>
                            <div class="controls">
								<?php echo JHtml::_('select.genericlist', $options, 'mailing_list_custom_field', '', 'id', 'name', (int) $params->get('mailing_list_custom_field')); ?>
                            </div>
                        </div>
						<?php
					}
					?>
                </fieldset>
            </div>
            <div class="span6 pull-left">
                <fieldset class="adminform">
                    <legend><?php echo JText::_('OSM_WHEN_SUBSCRIPTION_EXPIRED'); ?></legend>
                    <div class="control-group">
                        <div class="control-label">
							<?php echo OSMembershipHelperHtml::getFieldLabel('subscription_expired_acymailing_list_ids', JText::_('OSM_REMOVE_FROM_MAILING_LISTS'), JText::_('OSM_REMOVE_FROM_MAILING_LISTS_EXPLAIN')); ?>
                        </div>
                        <div class="controls">
							<?php echo JHtml::_('select.genericlist', $allLists, 'subscription_expired_acymailing_list_ids[]', 'class="inputbox" multiple="multiple" size="10"', 'listid', 'name', $expiredRemoveListIds) ?>
                        </div>
                    </div>
                    <div class="control-group">
                        <div class="control-label">
							<?php echo OSMembershipHelperHtml::getFieldLabel('acymailing_expired_assign_list_ids', JText::_('OSM_ASSIGN_TO_MAILING_LISTS'), JText::_('OSM_ASSIGN_TO_MAILING_LISTS_EXPLAIN')); ?>
                        </div>
                        <div class="controls">
							<?php echo JHtml::_('select.genericlist', $allLists, 'acymailing_expired_assign_list_ids[]', 'class="inputbox" multiple="multiple" size="10"', 'listid', 'name', $expiredAssignListIds); ?>
                        </div>
                    </div>
            </div>
        </div>
		<?php
	}
}
