<?php
/**
 * @package         Joomla.Plugin
 * @subpackage      Captcha.jdcaptcha
 * @author          Tuan Pham Ngoc
 *
 * @copyright       Copyright (C) 2024 JoomDonation Team
 * @license         GNU General Public License version 3, or later
 */

namespace JoomDonation\Plugin\Captcha\JDCaptcha\Extension;

use Composer\Autoload\ClassLoader;
use Gregwar\Captcha\CaptchaBuilder;
use Joomla\CMS\Application\CMSApplication;
use Joomla\CMS\Application\CMSApplicationInterface;
use Joomla\CMS\Application\CMSWebApplicationInterface;
use Joomla\CMS\Factory;
use Joomla\CMS\Filesystem\File;
use Joomla\CMS\Filesystem\Folder;
use Joomla\CMS\Language\Text;
use Joomla\CMS\Log\Log;
use Joomla\CMS\Plugin\CMSPlugin;
use Joomla\CMS\Plugin\PluginHelper;
use Joomla\CMS\Session\Session;
use Joomla\CMS\Uri\Uri;
use Joomla\CMS\User\UserHelper;
use Joomla\Event\DispatcherInterface;

\defined('_JEXEC') or die;

final class JDCaptcha extends CMSPlugin
{
	/**
	 * Captcha instance index
	 *
	 * @var int
	 */
	private static $index = 0;

	/**
	 * Load the language file on instantiation.
	 *
	 * @var    boolean
	 * @since  1.0.0
	 */
	protected $autoloadLanguage = true;

	/**
	 * Captcha type
	 *
	 * @var int
	 */
	protected $captchaType = 1;

	/**
	 * Constructor
	 *
	 * @param   DispatcherInterface  $dispatcher
	 * @param   CMSApplication       $app
	 * @param   array                $config
	 */
	public function __construct(DispatcherInterface $dispatcher, CMSApplication $app, array $config = [])
	{
		parent::__construct($dispatcher, $config);

		$this->setApplication($app);
	}

	/**
	 * Initializes the captcha
	 *
	 * @param   string  $id  The id of the field.
	 *
	 * @return  bool True on success, false otherwise
	 *
	 * @throws  \RuntimeException
	 * @since   2.5
	 */
	public function onInit($id = 'dynamic_recaptcha_1')
	{
		$app = $this->getApplication();

		if (!$app instanceof CMSWebApplicationInterface)
		{
			return false;
		}

		$rootUri  = Uri::root(true);
		$document = $this->getApplication()->getDocument();
		$document
			->addStyleSheet($rootUri . '/media/plg_captcha_jdcaptcha/css/style.min.css')
			->addScript($rootUri . '/media/plg_captcha_jdcaptcha/js/script.min.js', [], ['defer' => true])
			->addScriptOptions(
				'jdReloadCaptchaURL',
				$rootUri . '/index.php?option=com_ajax&plugin=jdcaptcha&group=captcha&format=raw&' . Session::getFormToken() . '=1'
			);
	}

	/**
	 * Gets the challenge HTML
	 *
	 * @param   string  $name   The name of the field. Not Used.
	 * @param   string  $id     The id of the field.
	 * @param   string  $class  The class of the field.
	 *
	 * @return  string  The HTML to be embedded in the form.
	 */
	public function onDisplay()
	{
		self::$index++;

		$index = self::$index;

		/* @var Session $session */
		$session           = $this->getApplication()->getSession();
		$honeypotFieldName = $this->getHoneypotFieldName();

		if ($this->params->get('min_time') > 0)
		{
			$session->set($this->getSessionTimeStartKey($index), time());
		}

		$session->set($this->getSessionHoneypotKey($index), $honeypotFieldName);

		if ($this->getApplication()->getInput()->getCmd('option') !== 'com_kunena')
		{
			$this->params->set('enable_affects', 0);
		}

		$source = $this->getCaptchaImageSource($index);

		ob_start();

		require PluginHelper::getLayoutPath($this->_type, $this->_name);

		return ob_get_clean();
	}

	/**
	 * Calls an HTTP POST function to verify if the user's guess was correct
	 *
	 * @param   string  $code  Answer provided by user. Not needed for the Recaptcha implementation
	 *
	 * @return  bool True if the answer is correct, false otherwise
	 *
	 * @throws  \RuntimeException
	 * @since   2.5
	 */
	public function onCheckAnswer($code = null)
	{
		$app = $this->getApplication();

		/* @var Session $session */
		$session = $app->getSession();
		$index   = $app->input->getInt('jd_captcha_index', 0);

		if (!$this->validateMinTime($session, $index))
		{
			$app->enqueueMessage(Text::_('JD_CAPTCHA_MIN_TIME_VALIDATION_ERROR'), 'error');

			return false;
		}

		if (!$this->validateHoneypot($session, $index))
		{
			return false;
		}

		return $this->validateCaptcha($session, $index);
	}

	/**
	 * Reload captcha image
	 *
	 * @return void
	 */
	public function onAjaxJdcaptcha()
	{
		Session::checkToken('GET');

		$app   = $this->getApplication();
		$index = $app->input->getInt('captcha_index', 1);

		$response = new \stdClass();

		$response->imageData = $this->getCaptchaImageSource($index);

		if ($this->params->get('type', 1) == 0)
		{
			if ($this->captchaType == 1)
			{
				$response->instruction = Text::_('JD_CAPTCHA_TEXT_INSTRUCTION');
			}
			else
			{
				$response->instruction = Text::_('JD_CAPTCHA_MATH_INSTRUCTION');
			}
		}
		else
		{
			$response->instruction = '';
		}

		echo json_encode($response);

		$app->close();
	}

	/**
	 * Get content of the captcha image
	 *
	 * @return string
	 * @var int $index
	 */
	private function getCaptchaImageSource($index)
	{
		/** @var ClassLoader $autoLoader */
		$autoLoader = include JPATH_LIBRARIES . '/vendor/autoload.php';

		$autoLoader->setPsr4('Gregwar\\', JPATH_PLUGINS . '/captcha/jdcaptcha/lib/gregwar/captcha/src/Gregwar');

		$builder = $this->getCaptchaBuilder();

		[$phrase, $answer] = $this->generatePhraseAndSolution();
		$builder->setPhrase($phrase);

		$width  = $this->params->get('width', 150);
		$height = $this->params->get('height', 40);
		$font   = $this->getCaptchaFont();
		$builder->build($width, $height, $font);

		$this->getApplication()->getSession()->set('jd_captcha_' . $index . '_answer', $answer);

		return $builder->inline();
	}

	/**
	 * Generate phrase and solution depends on captcha type
	 *
	 * @return string[]
	 */
	private function generatePhraseAndSolution(): array
	{
		if ($this->params->get('type', 1) == 0)
		{
			$type = mt_rand(1, 2);
		}
		else
		{
			$type = $this->params->get('type', 1);
		}

		$this->captchaType = $type;

		if ($type == 1)
		{
			$phrase = UserHelper::genRandomPassword($this->params->get('length', 5));
			$answer = $phrase;
		}
		else
		{
			$operator = mt_rand(1, 4);

			switch ($operator)
			{
				case 1:
					$x      = mt_rand(10, 60);
					$y      = mt_rand(1, 9);
					$phrase = "$x + $y = ";
					$answer = $x + $y;
					break;
				case 2:
					$x      = mt_rand(10, 60);
					$y      = mt_rand(1, 9);
					$phrase = "$x - $y = ";
					$answer = $x - $y;
					break;
				case 3:
					$x      = mt_rand(1, 9);
					$y      = mt_rand(1, 6);
					$phrase = "$x x $y = ";
					$answer = $x * $y;
					break;
				default:
					$x       = mt_rand(1, 9);
					$y       = mt_rand(1, 6);
					$divided = $x * $y;
					$phrase  = "$divided : $x = ";
					$answer  = $y;
					break;
			}
		}

		return [$phrase, (string) $answer];
	}

	/**
	 * Validate min time
	 *
	 * @param   Session  $session
	 * @param   int      $index
	 *
	 * @return bool
	 */
	private function validateMinTime($session, $index): bool
	{
		$minTime        = $this->params->get('min_time');
		$sessionKey     = $this->getSessionTimeStartKey($index);
		$formRenderTime = $session->get($sessionKey, 0);
		$session->remove($sessionKey);

		if ($minTime > 0 && (time() - $formRenderTime < $minTime))
		{
			$this->logFailure(Text::_('JD_CAPTCHA_MIN_TIME_VALIDATION_ERROR'));

			return false;
		}

		return true;
	}

	/**
	 * Validate honeypot
	 *
	 * @param   Session  $session
	 * @param   int      $index
	 *
	 * @return bool
	 */
	private function validateHoneypot($session, $index): bool
	{
		$sessionKey    = $this->getSessionHoneypotKey($index);
		$honeypotName  = $session->get($sessionKey, '');
		$honeypotValue = $this->getApplication()->input->getString($honeypotName, '');
		$session->remove($sessionKey);

		if ($honeypotName && !empty($honeypotValue))
		{
			$option = $this->getApplication()->getInput()->getCmd('option');
			$task   = $this->getApplication()->getInput()->getCmd('task');
			$this->logFailure(Text::sprintf('JD_CAPTCHA_HONEYPOT_VALIDATION_ERROR', $option, $task, $honeypotName, $honeypotValue));

			return false;
		}

		return true;
	}

	/**
	 * Validate captcha response
	 *
	 * @param   Session  $session
	 * @param   int      $index
	 *
	 * @return bool
	 */
	private function validateCaptcha($session, $index): bool
	{
		$option      = $this->getApplication()->input->getCmd('option', '');
		$task        = $this->getApplication()->input->getCmd('task', '');
		$sessionKey  = 'jd_captcha_' . $index . '_answer';
		$captchaData = $session->get($sessionKey, '');
		$session->remove($sessionKey);

		$response = $this->getApplication()->input->getString('jd_captcha_input', '');

		$result = strlen($response) > 0 && $this->niceize($captchaData, $option) === $this->niceize($response, $option);

		if (!$result)
		{
			$this->logFailure(Text::sprintf('JD_CAPTCHA_FAILED_CAPTCHA_VALIDATION', $option, $task, $response, $captchaData));

			if ($option === 'com_kunena')
			{
				$this->getApplication()->enqueueMessage('Invalid Captcha. Please try again', CMSApplicationInterface::MSG_ERROR);
			}
		}
		elseif ($option === 'com_kunena')
		{
			$this->logFailure(
				Text::sprintf(
					'Captcha validation success in component %s, task:%s, response:%s, expected:%s',
					$option,
					$task,
					$response,
					$captchaData
				),
				false
			);
		}

		return $result;
	}

	/**
	 * Niceize a code
	 *
	 * @param   string  $str
	 *
	 * @return string
	 */
	public function niceize(string $str, string $option): string
	{
		if ($option === 'com_kunena')
		{
			return $str;
		}

		return strtr(strtolower($str), '01', 'ol');
	}

	/**
	 * Get captcha builder
	 *
	 * @return CaptchaBuilder
	 */
	private function getCaptchaBuilder()
	{
		$builder = new CaptchaBuilder();

		if ($maxAngle = $this->params->get('max_angle', '8'))
		{
			$builder->setMaxAngle($maxAngle);
		}

		if ($maxOffset = $this->params->get('max_offset', 5))
		{
			$builder->setMaxOffset($maxOffset);
		}

		if ($textColor = $this->params->get('text_color'))
		{
			$rgb = $this->convertHexToRgb($textColor);

			$builder->setTextColor(...$rgb);
		}

		if ($lineColor = $this->params->get('line_color'))
		{
			$rgb = $this->convertHexToRgb($lineColor);

			$builder->setLineColor(...$rgb);
		}

		if ($backgroundColor = $this->params->get('background_color'))
		{
			$rgb = $this->convertHexToRgb($backgroundColor);

			$builder->setBackgroundColor(...$rgb);
		}

		// Set background images uses for captcha
		$builder->setBackgroundImages($this->getBackgroundImages());

		if (!$this->params->get('enable_affects', 1))
		{
			$builder->setIgnoreAllEffects(true);

			// Early return because the other parameters do not take an effect
			return $builder;
		}

		$builder->setDistortion((bool) $this->params->get('distortion', 1));
		$builder->setInterpolation((bool) $this->params->get('interpolation', 1));

		if ($maxFrontLines = $this->params->get('max_front_lines'))
		{
			$builder->setMaxFrontLines($maxFrontLines);
		}

		if ($maxBehindLines = $this->params->get('max_behind_lines'))
		{
			$builder->setMaxBehindLines($maxBehindLines);
		}

		return $builder;
	}

	/**
	 * Method to get background images use for captcha
	 *
	 * @return array
	 */
	private function getBackgroundImages(): array
	{
		$backgroundImage = $this->params->get('background_image', '-1');

		if (empty($backgroundImage))
		{
			return Folder::files(JPATH_ROOT . '/media/plg_captcha_jdcaptcha/background', '\.png$|\.jpg$|\.jpeg$', false, true);
		}

		if (File::exists(JPATH_ROOT . '/' . $backgroundImage))
		{
			return [JPATH_ROOT . '/' . $backgroundImage];
		}

		return [];
	}

	/**
	 * Get path to the font which is used to render captcha image
	 *
	 * @return string
	 */
	private function getCaptchaFont(): string
	{
		if ($this->params->get('captcha_font', '') !== '')
		{
			return JPATH_ROOT . '/media/plg_captcha_jdcaptcha/fonts/' . $this->params->get('captcha_font');
		}

		$fonts = Folder::files(JPATH_ROOT . '/media/plg_captcha_jdcaptcha/fonts', '.ttf$', false, true);

		return $fonts[mt_rand(0, count($fonts) - 1)];
	}


	/**
	 * Method to convert a color code from hex to rgb
	 *
	 * @param   string  $hex
	 *
	 * @return array
	 */
	private function convertHexToRgb(string $hex): array
	{
		// Remove '#' if present
		$hex = str_replace('#', '', $hex);

		// Convert 3-character hex to 6-character hex
		if (strlen($hex) == 3)
		{
			$hex = str_repeat(substr($hex, 0, 1), 2) .
				str_repeat(substr($hex, 1, 1), 2) .
				str_repeat(substr($hex, 2, 1), 2);
		}

		// Separate the components
		$r = hexdec(substr($hex, 0, 2));
		$g = hexdec(substr($hex, 2, 2));
		$b = hexdec(substr($hex, 4, 2));

		// Return an array with r, g, b values
		return ['r' => $r, 'g' => $g, 'b' => $b];
	}

	/**
	 * Return a md5 randomized string which can be used as honeypot field name
	 *
	 * @return string
	 */
	private function getHoneypotFieldName(): string
	{
		$fieldName = $this->params->get('honeypot_field_name', $this->getRandomHoneypotFieldName());

		// Add random prefix to make sure the field is unique on the form
		return $this->generateHoneypotPrefix() . '_' . $fieldName;
	}

	/**
	 * Get random honeypot field name
	 *
	 * @return string
	 */
	private function getRandomHoneypotFieldName(): string
	{
		$fieldNames = [
			'email',           // Commonly targeted by bots
			'website',         // Bots may try to input website URLs
			'username',        // Bots might attempt to input usernames
			'phone',           // Bots may try to input phone numbers
			'address',         // Bots might attempt to input addresses
			'message',         // Bots may try to input messages or comments
			'comment',         // Similar to 'message'
			'text',            // Generic field name
			'name',            // Bots might attempt to input names
			'subject',         // Bots may try to input subjects or topics
			'body',            // Bots might attempt to input message bodies
			'captcha',         // Bots may attempt to bypass captchas
			'security_code',   // Similar to 'captcha'
			'spam',            // Bots might attempt to input spam messages
			'newsletter',      // Bots may try to subscribe to newsletters
			'subscribe',       // Similar to 'newsletter'
			'comment',         // Bots may try to post comments
			'url',             // Bots might attempt to input URLs
			'link',            // Similar to 'url'
			'website_url',     // Similar to 'url' or 'website'
			'credit_card',     // Bots may try to input credit card numbers
			'password',        // Bots might attempt to input passwords
			'confirm_password',// Similar to 'password'
			'terms',           // Bots may try to agree to terms without reading
			'agree',           // Similar to 'terms'
			'accept',          // Similar to 'terms' or 'agree'
		];

		return $fieldNames[array_rand($fieldNames)];
	}

	/**
	 * Name of the input for captcha
	 *
	 * @return string
	 */
	private function getCaptchaInputName(): string
	{
		return md5('jd_captcha_' . $this->getApplication()->get('secret'));
	}

	/**
	 * Get the session key uses to store the start time of the form
	 *
	 * @param   int  $index
	 *
	 * @return string
	 */
	private function getSessionTimeStartKey(int $index): string
	{
		return 'jd_captcha_' . $index . '_start';
	}

	/**
	 * Get the session key uses to store the start time of the form
	 *
	 * @param   int  $index
	 *
	 * @return string
	 */
	private function getSessionHoneypotKey(int $index): string
	{
		return 'jd_captcha_' . $index . '_honeypot';
	}

	/**
	 * Log failure message
	 *
	 * @param   string  $message
	 *
	 * @return void
	 */
	private function logFailure(string $message, $reject = true): void
	{
		if (!$this->params->get('enable_log'))
		{
			return;
		}

		$logPath = $this->getApplication()->get('log_path');

		if (!$logPath)
		{
			return;
		}

		Log::addLogger(
			[
				'text_file'         => 'jdcaptcha.php',
				'text_file_path'    => $logPath,
				'text_entry_format' => "{DATE} {TIME}\t{CLIENTIP}\t" . "{MESSAGE}",
			], Log::ALL, ['jdcaptcha']
		);

		try
		{
			if ($reject)
			{
				$logText = "REJECT: $message";
			}
			else
			{
				$logText = "Success: $message";
			}

			Log::add(
				$logText,
				'INFO',
				'jdcaptcha',
				Factory::getDate('now', $this->getApplication()->get('offset'))->format('Y-m-d H:i:s', true)
			);
		}
		catch (\Exception $e)
		{
			// Do-nothing
		}
	}

	/**
	 * Generate prefix uses for honeypot input field name
	 *
	 * @return string
	 */
	private function generateHoneypotPrefix(): string
	{
		$consonants = range('a', 'h');
		$vowels     = ['a', 'e', 'i', 'o', 'u'];
		$digits     = range('0', '9');

		return $consonants [array_rand($consonants)]
			. $vowels [array_rand($vowels)]
			. $consonants [array_rand($consonants)]
			. $consonants [array_rand($consonants)]
			. $vowels [array_rand($vowels)]
			. $consonants [array_rand($consonants)]
			. $vowels [array_rand($vowels)]
			. $digits[array_rand($digits)];
	}
}
