var app = (function () {
    'use strict';

    function noop$1() { }
    const identity = x => x;
    function assign(tar, src) {
        // @ts-ignore
        for (const k in src)
            tar[k] = src[k];
        return tar;
    }
    function add_location(element, file, line, column, char) {
        element.__svelte_meta = {
            loc: { file, line, column, char }
        };
    }
    function run(fn) {
        return fn();
    }
    function blank_object() {
        return Object.create(null);
    }
    function run_all(fns) {
        fns.forEach(run);
    }
    function is_function(thing) {
        return typeof thing === 'function';
    }
    function safe_not_equal(a, b) {
        return a != a ? b == b : a !== b || ((a && typeof a === 'object') || typeof a === 'function');
    }
    let src_url_equal_anchor;
    function src_url_equal(element_src, url) {
        if (!src_url_equal_anchor) {
            src_url_equal_anchor = document.createElement('a');
        }
        src_url_equal_anchor.href = url;
        return element_src === src_url_equal_anchor.href;
    }
    function is_empty(obj) {
        return Object.keys(obj).length === 0;
    }
    function validate_store(store, name) {
        if (store != null && typeof store.subscribe !== 'function') {
            throw new Error(`'${name}' is not a store with a 'subscribe' method`);
        }
    }
    function subscribe(store, ...callbacks) {
        if (store == null) {
            return noop$1;
        }
        const unsub = store.subscribe(...callbacks);
        return unsub.unsubscribe ? () => unsub.unsubscribe() : unsub;
    }
    function get_store_value(store) {
        let value;
        subscribe(store, _ => value = _)();
        return value;
    }
    function component_subscribe(component, store, callback) {
        component.$$.on_destroy.push(subscribe(store, callback));
    }
    function create_slot(definition, ctx, $$scope, fn) {
        if (definition) {
            const slot_ctx = get_slot_context(definition, ctx, $$scope, fn);
            return definition[0](slot_ctx);
        }
    }
    function get_slot_context(definition, ctx, $$scope, fn) {
        return definition[1] && fn
            ? assign($$scope.ctx.slice(), definition[1](fn(ctx)))
            : $$scope.ctx;
    }
    function get_slot_changes(definition, $$scope, dirty, fn) {
        if (definition[2] && fn) {
            const lets = definition[2](fn(dirty));
            if ($$scope.dirty === undefined) {
                return lets;
            }
            if (typeof lets === 'object') {
                const merged = [];
                const len = Math.max($$scope.dirty.length, lets.length);
                for (let i = 0; i < len; i += 1) {
                    merged[i] = $$scope.dirty[i] | lets[i];
                }
                return merged;
            }
            return $$scope.dirty | lets;
        }
        return $$scope.dirty;
    }
    function update_slot_base(slot, slot_definition, ctx, $$scope, slot_changes, get_slot_context_fn) {
        if (slot_changes) {
            const slot_context = get_slot_context(slot_definition, ctx, $$scope, get_slot_context_fn);
            slot.p(slot_context, slot_changes);
        }
    }
    function get_all_dirty_from_scope($$scope) {
        if ($$scope.ctx.length > 32) {
            const dirty = [];
            const length = $$scope.ctx.length / 32;
            for (let i = 0; i < length; i++) {
                dirty[i] = -1;
            }
            return dirty;
        }
        return -1;
    }
    function set_store_value(store, ret, value) {
        store.set(value);
        return ret;
    }
    function action_destroyer(action_result) {
        return action_result && is_function(action_result.destroy) ? action_result.destroy : noop$1;
    }

    const is_client = typeof window !== 'undefined';
    let now = is_client
        ? () => window.performance.now()
        : () => Date.now();
    let raf = is_client ? cb => requestAnimationFrame(cb) : noop$1;

    const tasks = new Set();
    function run_tasks(now) {
        tasks.forEach(task => {
            if (!task.c(now)) {
                tasks.delete(task);
                task.f();
            }
        });
        if (tasks.size !== 0)
            raf(run_tasks);
    }
    /**
     * Creates a new task that runs on each raf frame
     * until it returns a falsy value or is aborted
     */
    function loop(callback) {
        let task;
        if (tasks.size === 0)
            raf(run_tasks);
        return {
            promise: new Promise(fulfill => {
                tasks.add(task = { c: callback, f: fulfill });
            }),
            abort() {
                tasks.delete(task);
            }
        };
    }
    function append(target, node) {
        target.appendChild(node);
    }
    function get_root_for_style(node) {
        if (!node)
            return document;
        const root = node.getRootNode ? node.getRootNode() : node.ownerDocument;
        if (root.host) {
            return root;
        }
        return document;
    }
    function append_empty_stylesheet(node) {
        const style_element = element('style');
        append_stylesheet(get_root_for_style(node), style_element);
        return style_element;
    }
    function append_stylesheet(node, style) {
        append(node.head || node, style);
    }
    function insert(target, node, anchor) {
        target.insertBefore(node, anchor || null);
    }
    function detach(node) {
        node.parentNode.removeChild(node);
    }
    function destroy_each(iterations, detaching) {
        for (let i = 0; i < iterations.length; i += 1) {
            if (iterations[i])
                iterations[i].d(detaching);
        }
    }
    function element(name) {
        return document.createElement(name);
    }
    function svg_element(name) {
        return document.createElementNS('http://www.w3.org/2000/svg', name);
    }
    function text(data) {
        return document.createTextNode(data);
    }
    function space() {
        return text(' ');
    }
    function empty() {
        return text('');
    }
    function listen(node, event, handler, options) {
        node.addEventListener(event, handler, options);
        return () => node.removeEventListener(event, handler, options);
    }
    function prevent_default(fn) {
        return function (event) {
            event.preventDefault();
            // @ts-ignore
            return fn.call(this, event);
        };
    }
    function stop_propagation(fn) {
        return function (event) {
            event.stopPropagation();
            // @ts-ignore
            return fn.call(this, event);
        };
    }
    function attr(node, attribute, value) {
        if (value == null)
            node.removeAttribute(attribute);
        else if (node.getAttribute(attribute) !== value)
            node.setAttribute(attribute, value);
    }
    function children(element) {
        return Array.from(element.childNodes);
    }
    function set_style(node, key, value, important) {
        node.style.setProperty(key, value, important ? 'important' : '');
    }
    function custom_event(type, detail, bubbles = false) {
        const e = document.createEvent('CustomEvent');
        e.initCustomEvent(type, bubbles, false, detail);
        return e;
    }
    class HtmlTag {
        constructor() {
            this.e = this.n = null;
        }
        c(html) {
            this.h(html);
        }
        m(html, target, anchor = null) {
            if (!this.e) {
                this.e = element(target.nodeName);
                this.t = target;
                this.c(html);
            }
            this.i(anchor);
        }
        h(html) {
            this.e.innerHTML = html;
            this.n = Array.from(this.e.childNodes);
        }
        i(anchor) {
            for (let i = 0; i < this.n.length; i += 1) {
                insert(this.t, this.n[i], anchor);
            }
        }
        p(html) {
            this.d();
            this.h(html);
            this.i(this.a);
        }
        d() {
            this.n.forEach(detach);
        }
    }

    const active_docs = new Set();
    let active = 0;
    // https://github.com/darkskyapp/string-hash/blob/master/index.js
    function hash(str) {
        let hash = 5381;
        let i = str.length;
        while (i--)
            hash = ((hash << 5) - hash) ^ str.charCodeAt(i);
        return hash >>> 0;
    }
    function create_rule(node, a, b, duration, delay, ease, fn, uid = 0) {
        const step = 16.666 / duration;
        let keyframes = '{\n';
        for (let p = 0; p <= 1; p += step) {
            const t = a + (b - a) * ease(p);
            keyframes += p * 100 + `%{${fn(t, 1 - t)}}\n`;
        }
        const rule = keyframes + `100% {${fn(b, 1 - b)}}\n}`;
        const name = `__svelte_${hash(rule)}_${uid}`;
        const doc = get_root_for_style(node);
        active_docs.add(doc);
        const stylesheet = doc.__svelte_stylesheet || (doc.__svelte_stylesheet = append_empty_stylesheet(node).sheet);
        const current_rules = doc.__svelte_rules || (doc.__svelte_rules = {});
        if (!current_rules[name]) {
            current_rules[name] = true;
            stylesheet.insertRule(`@keyframes ${name} ${rule}`, stylesheet.cssRules.length);
        }
        const animation = node.style.animation || '';
        node.style.animation = `${animation ? `${animation}, ` : ''}${name} ${duration}ms linear ${delay}ms 1 both`;
        active += 1;
        return name;
    }
    function delete_rule(node, name) {
        const previous = (node.style.animation || '').split(', ');
        const next = previous.filter(name
            ? anim => anim.indexOf(name) < 0 // remove specific animation
            : anim => anim.indexOf('__svelte') === -1 // remove all Svelte animations
        );
        const deleted = previous.length - next.length;
        if (deleted) {
            node.style.animation = next.join(', ');
            active -= deleted;
            if (!active)
                clear_rules();
        }
    }
    function clear_rules() {
        raf(() => {
            if (active)
                return;
            active_docs.forEach(doc => {
                const stylesheet = doc.__svelte_stylesheet;
                let i = stylesheet.cssRules.length;
                while (i--)
                    stylesheet.deleteRule(i);
                doc.__svelte_rules = {};
            });
            active_docs.clear();
        });
    }

    let current_component;
    function set_current_component(component) {
        current_component = component;
    }
    function get_current_component() {
        if (!current_component)
            throw new Error('Function called outside component initialization');
        return current_component;
    }
    function beforeUpdate(fn) {
        get_current_component().$$.before_update.push(fn);
    }
    function onMount(fn) {
        get_current_component().$$.on_mount.push(fn);
    }
    function onDestroy(fn) {
        get_current_component().$$.on_destroy.push(fn);
    }
    function setContext(key, context) {
        get_current_component().$$.context.set(key, context);
    }
    function getContext(key) {
        return get_current_component().$$.context.get(key);
    }
    // TODO figure out if we still want to support
    // shorthand events, or if we want to implement
    // a real bubbling mechanism
    function bubble(component, event) {
        const callbacks = component.$$.callbacks[event.type];
        if (callbacks) {
            // @ts-ignore
            callbacks.slice().forEach(fn => fn.call(this, event));
        }
    }

    const dirty_components = [];
    const binding_callbacks = [];
    const render_callbacks = [];
    const flush_callbacks = [];
    const resolved_promise = Promise.resolve();
    let update_scheduled = false;
    function schedule_update() {
        if (!update_scheduled) {
            update_scheduled = true;
            resolved_promise.then(flush);
        }
    }
    function add_render_callback(fn) {
        render_callbacks.push(fn);
    }
    function add_flush_callback(fn) {
        flush_callbacks.push(fn);
    }
    let flushing = false;
    const seen_callbacks = new Set();
    function flush() {
        if (flushing)
            return;
        flushing = true;
        do {
            // first, call beforeUpdate functions
            // and update components
            for (let i = 0; i < dirty_components.length; i += 1) {
                const component = dirty_components[i];
                set_current_component(component);
                update(component.$$);
            }
            set_current_component(null);
            dirty_components.length = 0;
            while (binding_callbacks.length)
                binding_callbacks.pop()();
            // then, once components are updated, call
            // afterUpdate functions. This may cause
            // subsequent updates...
            for (let i = 0; i < render_callbacks.length; i += 1) {
                const callback = render_callbacks[i];
                if (!seen_callbacks.has(callback)) {
                    // ...so guard against infinite loops
                    seen_callbacks.add(callback);
                    callback();
                }
            }
            render_callbacks.length = 0;
        } while (dirty_components.length);
        while (flush_callbacks.length) {
            flush_callbacks.pop()();
        }
        update_scheduled = false;
        flushing = false;
        seen_callbacks.clear();
    }
    function update($$) {
        if ($$.fragment !== null) {
            $$.update();
            run_all($$.before_update);
            const dirty = $$.dirty;
            $$.dirty = [-1];
            $$.fragment && $$.fragment.p($$.ctx, dirty);
            $$.after_update.forEach(add_render_callback);
        }
    }

    let promise;
    function wait() {
        if (!promise) {
            promise = Promise.resolve();
            promise.then(() => {
                promise = null;
            });
        }
        return promise;
    }
    function dispatch(node, direction, kind) {
        node.dispatchEvent(custom_event(`${direction ? 'intro' : 'outro'}${kind}`));
    }
    const outroing = new Set();
    let outros;
    function group_outros() {
        outros = {
            r: 0,
            c: [],
            p: outros // parent group
        };
    }
    function check_outros() {
        if (!outros.r) {
            run_all(outros.c);
        }
        outros = outros.p;
    }
    function transition_in(block, local) {
        if (block && block.i) {
            outroing.delete(block);
            block.i(local);
        }
    }
    function transition_out(block, local, detach, callback) {
        if (block && block.o) {
            if (outroing.has(block))
                return;
            outroing.add(block);
            outros.c.push(() => {
                outroing.delete(block);
                if (callback) {
                    if (detach)
                        block.d(1);
                    callback();
                }
            });
            block.o(local);
        }
    }
    const null_transition = { duration: 0 };
    function create_in_transition(node, fn, params) {
        let config = fn(node, params);
        let running = false;
        let animation_name;
        let task;
        let uid = 0;
        function cleanup() {
            if (animation_name)
                delete_rule(node, animation_name);
        }
        function go() {
            const { delay = 0, duration = 300, easing = identity, tick = noop$1, css } = config || null_transition;
            if (css)
                animation_name = create_rule(node, 0, 1, duration, delay, easing, css, uid++);
            tick(0, 1);
            const start_time = now() + delay;
            const end_time = start_time + duration;
            if (task)
                task.abort();
            running = true;
            add_render_callback(() => dispatch(node, true, 'start'));
            task = loop(now => {
                if (running) {
                    if (now >= end_time) {
                        tick(1, 0);
                        dispatch(node, true, 'end');
                        cleanup();
                        return running = false;
                    }
                    if (now >= start_time) {
                        const t = easing((now - start_time) / duration);
                        tick(t, 1 - t);
                    }
                }
                return running;
            });
        }
        let started = false;
        return {
            start() {
                if (started)
                    return;
                started = true;
                delete_rule(node);
                if (is_function(config)) {
                    config = config();
                    wait().then(go);
                }
                else {
                    go();
                }
            },
            invalidate() {
                started = false;
            },
            end() {
                if (running) {
                    cleanup();
                    running = false;
                }
            }
        };
    }
    function create_out_transition(node, fn, params) {
        let config = fn(node, params);
        let running = true;
        let animation_name;
        const group = outros;
        group.r += 1;
        function go() {
            const { delay = 0, duration = 300, easing = identity, tick = noop$1, css } = config || null_transition;
            if (css)
                animation_name = create_rule(node, 1, 0, duration, delay, easing, css);
            const start_time = now() + delay;
            const end_time = start_time + duration;
            add_render_callback(() => dispatch(node, false, 'start'));
            loop(now => {
                if (running) {
                    if (now >= end_time) {
                        tick(0, 1);
                        dispatch(node, false, 'end');
                        if (!--group.r) {
                            // this will result in `end()` being called,
                            // so we don't need to clean up here
                            run_all(group.c);
                        }
                        return false;
                    }
                    if (now >= start_time) {
                        const t = easing((now - start_time) / duration);
                        tick(1 - t, t);
                    }
                }
                return running;
            });
        }
        if (is_function(config)) {
            wait().then(() => {
                // @ts-ignore
                config = config();
                go();
            });
        }
        else {
            go();
        }
        return {
            end(reset) {
                if (reset && config.tick) {
                    config.tick(1, 0);
                }
                if (running) {
                    if (animation_name)
                        delete_rule(node, animation_name);
                    running = false;
                }
            }
        };
    }

    const globals = (typeof window !== 'undefined'
        ? window
        : typeof globalThis !== 'undefined'
            ? globalThis
            : global);
    function outro_and_destroy_block(block, lookup) {
        transition_out(block, 1, 1, () => {
            lookup.delete(block.key);
        });
    }
    function update_keyed_each(old_blocks, dirty, get_key, dynamic, ctx, list, lookup, node, destroy, create_each_block, next, get_context) {
        let o = old_blocks.length;
        let n = list.length;
        let i = o;
        const old_indexes = {};
        while (i--)
            old_indexes[old_blocks[i].key] = i;
        const new_blocks = [];
        const new_lookup = new Map();
        const deltas = new Map();
        i = n;
        while (i--) {
            const child_ctx = get_context(ctx, list, i);
            const key = get_key(child_ctx);
            let block = lookup.get(key);
            if (!block) {
                block = create_each_block(key, child_ctx);
                block.c();
            }
            else if (dynamic) {
                block.p(child_ctx, dirty);
            }
            new_lookup.set(key, new_blocks[i] = block);
            if (key in old_indexes)
                deltas.set(key, Math.abs(i - old_indexes[key]));
        }
        const will_move = new Set();
        const did_move = new Set();
        function insert(block) {
            transition_in(block, 1);
            block.m(node, next);
            lookup.set(block.key, block);
            next = block.first;
            n--;
        }
        while (o && n) {
            const new_block = new_blocks[n - 1];
            const old_block = old_blocks[o - 1];
            const new_key = new_block.key;
            const old_key = old_block.key;
            if (new_block === old_block) {
                // do nothing
                next = new_block.first;
                o--;
                n--;
            }
            else if (!new_lookup.has(old_key)) {
                // remove old block
                destroy(old_block, lookup);
                o--;
            }
            else if (!lookup.has(new_key) || will_move.has(new_key)) {
                insert(new_block);
            }
            else if (did_move.has(old_key)) {
                o--;
            }
            else if (deltas.get(new_key) > deltas.get(old_key)) {
                did_move.add(new_key);
                insert(new_block);
            }
            else {
                will_move.add(old_key);
                o--;
            }
        }
        while (o--) {
            const old_block = old_blocks[o];
            if (!new_lookup.has(old_block.key))
                destroy(old_block, lookup);
        }
        while (n)
            insert(new_blocks[n - 1]);
        return new_blocks;
    }
    function validate_each_keys(ctx, list, get_context, get_key) {
        const keys = new Set();
        for (let i = 0; i < list.length; i++) {
            const key = get_key(get_context(ctx, list, i));
            if (keys.has(key)) {
                throw new Error('Cannot have duplicate keys in a keyed each');
            }
            keys.add(key);
        }
    }

    function bind(component, name, callback) {
        const index = component.$$.props[name];
        if (index !== undefined) {
            component.$$.bound[index] = callback;
            callback(component.$$.ctx[index]);
        }
    }
    function create_component(block) {
        block && block.c();
    }
    function mount_component(component, target, anchor, customElement) {
        const { fragment, on_mount, on_destroy, after_update } = component.$$;
        fragment && fragment.m(target, anchor);
        if (!customElement) {
            // onMount happens before the initial afterUpdate
            add_render_callback(() => {
                const new_on_destroy = on_mount.map(run).filter(is_function);
                if (on_destroy) {
                    on_destroy.push(...new_on_destroy);
                }
                else {
                    // Edge case - component was destroyed immediately,
                    // most likely as a result of a binding initialising
                    run_all(new_on_destroy);
                }
                component.$$.on_mount = [];
            });
        }
        after_update.forEach(add_render_callback);
    }
    function destroy_component(component, detaching) {
        const $$ = component.$$;
        if ($$.fragment !== null) {
            run_all($$.on_destroy);
            $$.fragment && $$.fragment.d(detaching);
            // TODO null out other refs, including component.$$ (but need to
            // preserve final state?)
            $$.on_destroy = $$.fragment = null;
            $$.ctx = [];
        }
    }
    function make_dirty(component, i) {
        if (component.$$.dirty[0] === -1) {
            dirty_components.push(component);
            schedule_update();
            component.$$.dirty.fill(0);
        }
        component.$$.dirty[(i / 31) | 0] |= (1 << (i % 31));
    }
    function init(component, options, instance, create_fragment, not_equal, props, append_styles, dirty = [-1]) {
        const parent_component = current_component;
        set_current_component(component);
        const $$ = component.$$ = {
            fragment: null,
            ctx: null,
            // state
            props,
            update: noop$1,
            not_equal,
            bound: blank_object(),
            // lifecycle
            on_mount: [],
            on_destroy: [],
            on_disconnect: [],
            before_update: [],
            after_update: [],
            context: new Map(parent_component ? parent_component.$$.context : options.context || []),
            // everything else
            callbacks: blank_object(),
            dirty,
            skip_bound: false,
            root: options.target || parent_component.$$.root
        };
        append_styles && append_styles($$.root);
        let ready = false;
        $$.ctx = instance
            ? instance(component, options.props || {}, (i, ret, ...rest) => {
                const value = rest.length ? rest[0] : ret;
                if ($$.ctx && not_equal($$.ctx[i], $$.ctx[i] = value)) {
                    if (!$$.skip_bound && $$.bound[i])
                        $$.bound[i](value);
                    if (ready)
                        make_dirty(component, i);
                }
                return ret;
            })
            : [];
        $$.update();
        ready = true;
        run_all($$.before_update);
        // `false` as a special case of no DOM component
        $$.fragment = create_fragment ? create_fragment($$.ctx) : false;
        if (options.target) {
            if (options.hydrate) {
                const nodes = children(options.target);
                // eslint-disable-next-line @typescript-eslint/no-non-null-assertion
                $$.fragment && $$.fragment.l(nodes);
                nodes.forEach(detach);
            }
            else {
                // eslint-disable-next-line @typescript-eslint/no-non-null-assertion
                $$.fragment && $$.fragment.c();
            }
            if (options.intro)
                transition_in(component.$$.fragment);
            mount_component(component, options.target, options.anchor, options.customElement);
            flush();
        }
        set_current_component(parent_component);
    }
    /**
     * Base class for Svelte components. Used when dev=false.
     */
    class SvelteComponent {
        $destroy() {
            destroy_component(this, 1);
            this.$destroy = noop$1;
        }
        $on(type, callback) {
            const callbacks = (this.$$.callbacks[type] || (this.$$.callbacks[type] = []));
            callbacks.push(callback);
            return () => {
                const index = callbacks.indexOf(callback);
                if (index !== -1)
                    callbacks.splice(index, 1);
            };
        }
        $set($$props) {
            if (this.$$set && !is_empty($$props)) {
                this.$$.skip_bound = true;
                this.$$set($$props);
                this.$$.skip_bound = false;
            }
        }
    }

    function dispatch_dev(type, detail) {
        document.dispatchEvent(custom_event(type, Object.assign({ version: '3.41.0' }, detail), true));
    }
    function append_dev(target, node) {
        dispatch_dev('SvelteDOMInsert', { target, node });
        append(target, node);
    }
    function insert_dev(target, node, anchor) {
        dispatch_dev('SvelteDOMInsert', { target, node, anchor });
        insert(target, node, anchor);
    }
    function detach_dev(node) {
        dispatch_dev('SvelteDOMRemove', { node });
        detach(node);
    }
    function listen_dev(node, event, handler, options, has_prevent_default, has_stop_propagation) {
        const modifiers = options === true ? ['capture'] : options ? Array.from(Object.keys(options)) : [];
        if (has_prevent_default)
            modifiers.push('preventDefault');
        if (has_stop_propagation)
            modifiers.push('stopPropagation');
        dispatch_dev('SvelteDOMAddEventListener', { node, event, handler, modifiers });
        const dispose = listen(node, event, handler, options);
        return () => {
            dispatch_dev('SvelteDOMRemoveEventListener', { node, event, handler, modifiers });
            dispose();
        };
    }
    function attr_dev(node, attribute, value) {
        attr(node, attribute, value);
        if (value == null)
            dispatch_dev('SvelteDOMRemoveAttribute', { node, attribute });
        else
            dispatch_dev('SvelteDOMSetAttribute', { node, attribute, value });
    }
    function prop_dev(node, property, value) {
        node[property] = value;
        dispatch_dev('SvelteDOMSetProperty', { node, property, value });
    }
    function set_data_dev(text, data) {
        data = '' + data;
        if (text.wholeText === data)
            return;
        dispatch_dev('SvelteDOMSetData', { node: text, data });
        text.data = data;
    }
    function validate_each_argument(arg) {
        if (typeof arg !== 'string' && !(arg && typeof arg === 'object' && 'length' in arg)) {
            let msg = '{#each} only iterates over array-like objects.';
            if (typeof Symbol === 'function' && arg && Symbol.iterator in arg) {
                msg += ' You can use a spread to convert this iterable into an array.';
            }
            throw new Error(msg);
        }
    }
    function validate_slots(name, slot, keys) {
        for (const slot_key of Object.keys(slot)) {
            if (!~keys.indexOf(slot_key)) {
                console.warn(`<${name}> received an unexpected slot "${slot_key}".`);
            }
        }
    }
    /**
     * Base class for Svelte components with some minor dev-enhancements. Used when dev=true.
     */
    class SvelteComponentDev extends SvelteComponent {
        constructor(options) {
            if (!options || (!options.target && !options.$$inline)) {
                throw new Error("'target' is a required option");
            }
            super();
        }
        $destroy() {
            super.$destroy();
            this.$destroy = () => {
                console.warn('Component was already destroyed'); // eslint-disable-line no-console
            };
        }
        $capture_state() { }
        $inject_state() { }
    }

    /**
     * Layouts utilities.
     *
     * @author       Weeblr,llc
     * @copyright    Weeblr,llc - 2020
     * @package      @weeblr/translate
     * @license      GPL-3.0-or-later
     *
     */

    /**
     * Based on undescore.js
     *
     * @type {{"&": string, "<": string, ">": string, "\"": string, "'": string, "`": string}}
     */
    const escapeMap = {
        '&': '&amp;',
        '<': '&lt;',
        '>': '&gt;',
        '"': '&quot;',
        "'": '&#x27;',
        '`': '&#x60;'
    };

    const escapeAttrMap = {
        '"': '&quot;',
        "'": '&#x27;',
        '`': '&#x60;'
    };

    /**
     * Functions for escaping a string
     *
     * @param map
     * @returns {function(*=)}
     */
    const wblCreateEscaper = (map) => {
        const escaper = (match) => {
            return map[match];
        };
        // Regexes for identifying a key that needs to be escaped.
        const source = '(?:' + Object.keys(map).join('|') + ')';
        const testRegexp = RegExp(source);
        const replaceRegexp = RegExp(source, 'g');
        return (string) => {
            string = string == null ? '' : '' + string;
            return testRegexp.test(string) ? string.replace(replaceRegexp, escaper) : string;
        };
    };
    /**
     * Functions for escaping a string to HTML
     *
     * @returns {function(*=)}
     */
    const escape = wblCreateEscaper(escapeMap);

    /**
     * Functions for escaping a string to an HTML attribute
     *
     * @returns {function(*=)}
     */
    wblCreateEscaper(escapeAttrMap);

    /***********************************************************************************************************************
     * String utilities
     **********************************************************************************************************************/

    /**
     * Format a value into a string, an integer or a float. See sprintf ofr format.
     *
     * @param {mixed} value
     * @param {string} format
     * @param {int} precision
     * @returns {string|*}
     */
    const stringFormat = (value, format, precision) => {
        switch (format) {
            case 'd':
                return Number.parseFloat(value).toFixed()
            case 'f':
                return Number.parseFloat(value).toFixed(precision)
        }

        return value
    };
    /**
     * Insert params into a text with numbered placeholders. Simplified version of sprintf.
     *
     * If more than one value, they must be passed in an array.
     *
     * - Numbered placeholders:
     *   sprintf('Some %1 sample text %3 example %2'), ['foo', 'bar', 'baz'])
     *   => Some foo sample text baz example bar
     *
     * - Formatting: can append s, d or f to numbered placeholder:
     *   sprintf('Some %1s sample text %3d example %2f'), [123, 123.456, 123.456])
     *   => Some 123 sample text 123 example 123.456
     *
     * - Floating point can have a precision:
     *   sprintf('Some floating number: %1f.3'), [123.456789])
     *   => Some floating number: 123.456
     *
     * @param {string} text
     * @param {array} params
     * @param {boolean} escape
     */
    const sprintf = (text, params, escape$1) => {
        text = ensureString(text);
        if (params === null) {
            return escape$1 ? escape(text) : text
        }
        params = Array.isArray(params) ? params : [params];
        if (params.length < 1) {
            return escape$1 ? escape(text) : text
        }

        escape$1 = !!escape$1;
        text = text.replace(new RegExp('%([0-9]+)([sdf])?(\.[0-9]+)?', 'g'),
            (...matches) => {
                if (!matches[1]) {
                    return matches[0]
                }
                let value = params[matches[1] - 1] ? params[matches[1] - 1] : '';
                if (matches[2]) {
                    let precision = matches[3] ? matches[3].slice(1) : null;
                    value = stringFormat(value, matches[2], precision);
                }
                return escape$1 ? escape(value) : value
            }
        );

        return text
    };

    /**
     * Ensure a thing is a string by prepending it
     *
     * @param {string} s
     * @returns {string}
     */
    const ensureString = s => '' + s;

    /**
     * Translation utilities.
     *
     * @author       Weeblr,llc
     * @copyright    Weeblr,llc - 2020
     * @package      @weeblr/translate
     * @license      GPL-3.0-or-later
     *
     */

    /**
     * Default language code
     *
     * @type {string}
     * @private
     */
    let _lang = 'en-GB';

    /**
     * Holds per language translations
     *
     * @type {{}}
     * @private
     */
    let storedStrings = {};

    /**
     * Split the requested key by the first dot. Allow one-level sub-key.
     *
     * @param {string} key
     * @returns {*|[parser.Node[], parser.Node[]]|string[]|void}
     */
    const splitKey = key => key.split('.', 2);

    /**
     * Lookup a translated string for a language identified by an array of keys.
     * Can handle only 1 level deep keys.
     *
     * @param {array} keys
     * @param {string} lang
     * @returns {null|string}
     */
    const getByKeys = (keys, lang) => {
        if (!storedStrings[lang] || !storedStrings[lang][keys[0]]) {
            return null
        }
        if (keys.length > 1) {
            return storedStrings[lang][keys[0]][keys[1]] ? storedStrings[lang][keys[0]][keys[1]] : null
        } else {
            return storedStrings[lang][keys[0]]
        }
    };
    /**
     * Returns the translation of a given key
     * in a given language (default language if no language supplied)
     *
     * @param key
     * @param lang
     * @returns {*}
     * @private
     */
    const t = (key, lang) => {
        if (!key) {
            return ''
        }

        let keys = splitKey(key);
        lang = lang || _lang;
        let translation = getByKeys(keys, lang);

        if (!translation) {
            translation = getByKeys(keys, _lang) ;
            translation = translation || key;
        }
        return translation
    };

    // lookup and format a key translation
    const tsprintf = (key, data, lang) => {
        return sprintf(t(key, lang), data, false)
    };

    // set a group of keys translation, for a given language
    const setObject = (lang, obj) => {
        storedStrings[lang] = storedStrings[lang] || {};
        storedStrings[lang] = {...storedStrings[lang], ...obj};
    };

    // set the default language
    const setLang = (lang) => {
        _lang = lang || _lang;
    };

    /**
     * Carbon Design icons integration.
     *
     * @author       Weeblr,llc
     * @copyright    IBM Corp, Weeblr,llc - 2020
     * @package      @weeblr/cicons
     * @license      Apache-2.0
     *
     */


    const iconCheckmarkOutline = '<svg aria-hidden="true" viewBox="0 0 32 32"><path d="M14 21.414l-5-5.001L10.413 15 14 18.586 21.585 11 23 12.415l-9 8.999z"/><path d="M16 2a14 14 0 1014 14A14 14 0 0016 2zm0 26a12 12 0 1112-12 12 12 0 01-12 12z"/></svg>';


    const iconCheckmark = '<svg aria-hidden="true" data-name="Layer 1" viewBox="0 0 32 32"><path d="M13 24l-9-9 1.414-1.414L13 21.171 26.586 7.586 28 9 13 24z"/></svg>';


    const iconClose = '<svg aria-hidden="true" viewBox="0 0 32 32"><path d="M24 9.4L22.6 8 16 14.6 9.4 8 8 9.4l6.6 6.6L8 22.6 9.4 24l6.6-6.6 6.6 6.6 1.4-1.4-6.6-6.6L24 9.4z"/><rect fill="none"/></svg>';


    const iconDelete = '<svg aria-hidden="true" viewBox="0 0 32 32"><path d="M12 12h2v12h-2zM18 12h2v12h-2z"/><path d="M4 6v2h2v20a2 2 0 002 2h16a2 2 0 002-2V8h2V6zm4 22V8h16v20z"/><rect x="12" y="2"/></svg>';


    const iconDownload = '<svg aria-hidden="true" viewBox="0 0 32 32"><path d="M26 15l-1.41-1.41L17 21.17V2h-2v19.17l-7.59-7.58L6 15l10 10 10-10z"/><path d="M26 24v4H6v-4H4v4a2 2 0 002 2h20a2 2 0 002-2v-4z"/><rect fill="none"/></svg>';


    const iconInProgress = '<svg aria-hidden="true" viewBox="0 0 32 32"><path d="M16 2a14 14 0 1014 14A14.016 14.016 0 0016 2zm0 26a12 12 0 010-24v12l8.481 8.481A11.963 11.963 0 0116 28z"/></svg>';


    const iconLight = '<svg aria-hidden="true" viewBox="0 0 32 32"><path d="M15 2h2v5h-2zM21.687 8.9l3.506-3.507 1.414 1.415-3.506 3.506zM25 15h5v2h-5zM21.687 23.1l1.414-1.414 3.506 3.506-1.414 1.415zM15 25h2v5h-2zM5.394 25.192L8.9 21.686l1.414 1.415-3.506 3.506zM2 15h5v2H2zM5.394 6.808l1.414-1.415L10.314 8.9 8.9 10.314zM16 12a4 4 0 11-4 4 4.005 4.005 0 014-4m0-2a6 6 0 106 6 6 6 0 00-6-6z"/></svg>';


    const iconNotification = '<svg aria-hidden="true" viewBox="0 0 32 32"><path d="M28.7 20.3L26 17.6V13a10.07 10.07 0 00-9-10V1h-2v2a10.15 10.15 0 00-9 10v4.6l-2.7 2.7a.91.91 0 00-.3.7v3a.94.94 0 001 1h7a5 5 0 0010 0h7a.94.94 0 001-1v-3a.91.91 0 00-.3-.7zM16 28a3 3 0 01-3-3h6a3 3 0 01-3 3zm11-5H5v-1.6l2.7-2.7A.91.91 0 008 18v-5a8 8 0 0116 0v5a.91.91 0 00.3.7l2.7 2.7z"/></svg>';


    const iconPause = '<svg aria-hidden="true" viewBox="0 0 32 32"><path d="M12 8v16H8V8h4m0-2H8a2 2 0 00-2 2v16a2 2 0 002 2h4a2 2 0 002-2V8a2 2 0 00-2-2zM24 8v16h-4V8h4m0-2h-4a2 2 0 00-2 2v16a2 2 0 002 2h4a2 2 0 002-2V8a2 2 0 00-2-2z"/></svg>';


    const iconRain = '<svg aria-hidden="true" viewBox="0 0 32 32"><path d="M23.5 22h-15A6.5 6.5 0 017.2 9.14a9 9 0 0117.6 0A6.5 6.5 0 0123.5 22zM16 4a7 7 0 00-6.94 6.14L9 11h-.86a4.5 4.5 0 00.36 9h15a4.5 4.5 0 00.36-9H23l-.1-.82A7 7 0 0016 4zM14 30a.93.93 0 01-.45-.11 1 1 0 01-.44-1.34l2-4a1 1 0 111.78.9l-2 4A1 1 0 0114 30zM20 30a.93.93 0 01-.45-.11 1 1 0 01-.44-1.34l2-4a1 1 0 111.78.9l-2 4A1 1 0 0120 30zM8 30a.93.93 0 01-.45-.11 1 1 0 01-.44-1.34l2-4a1 1 0 111.78.9l-2 4A1 1 0 018 30z"/><rect fill="none"/></svg>';


    const iconRenew = '<svg aria-hidden="true" viewBox="0 0 32 32"><path d="M12 10H6.78A11 11 0 0127 16h2A13 13 0 006 7.68V4H4v8h8zM20 22h5.22A11 11 0 015 16H3a13 13 0 0023 8.32V28h2v-8h-8z"/><rect fill="none" data-name="&lt;Transparent Rectangle&gt;"/></svg>';


    const iconThunderstorm = '<svg aria-hidden="true" viewBox="0 0 32 32"><path d="M23.5 22H23v-2h.5a4.5 4.5 0 00.36-9H23l-.1-.82a7 7 0 00-13.88 0L9 11h-.86a4.5 4.5 0 00.36 9H9v2h-.5A6.5 6.5 0 017.2 9.14a9 9 0 0117.6 0A6.5 6.5 0 0123.5 22z"/><path d="M15.87 30.5l-1.74-1 3.15-5.5h-6l4.85-8.5 1.74 1-3.15 5.5h6l-4.85 8.5z"/></svg>';


    const iconView = '<svg aria-hidden="true" viewBox="0 0 32 32"><path d="M30.94 15.66A16.69 16.69 0 0016 5 16.69 16.69 0 001.06 15.66a1 1 0 000 .68A16.69 16.69 0 0016 27a16.69 16.69 0 0014.94-10.66 1 1 0 000-.68zM16 25c-5.3 0-10.9-3.93-12.93-9C5.1 10.93 10.7 7 16 7s10.9 3.93 12.93 9C26.9 21.07 21.3 25 16 25z"/><path d="M16 10a6 6 0 106 6 6 6 0 00-6-6zm0 10a4 4 0 114-4 4 4 0 01-4 4z"/></svg>';

    /**
     * Application features flags
     * Keys are simple strings, values can be anything.
     *
     * @author       Weeblr,llc
     * @copyright    Weeblr,llc - 2020
     * @package      @weeblr/translate
     * @license      GPL-3.0-or-later
     *
     */

    let index = {};

    var features = {
        set: (key, value) => {
            index[key] = value;
        },
        setAll: (value) => {
            index = value;
        },
        get: (key, defaultValue) => {
            return index.hasOwnProperty(key)
                ? index[key]
                : defaultValue
        },
        has: (key) => {
            return index.hasOwnProperty(key)
        },
        isFalsy: (key) => {
            return index.hasOwnProperty(key) && !index[key]
        },
        isTruthy: (key) => {
            return index.hasOwnProperty(key) && index[key]
        }
    };

    /**
     * Application configuration
     *
     */

    const setFeatures = () => {
        features.setAll(
            {
                'ui.withShadow': true,
                'ui.autoclearDefault': 15000,
                'ui.autoclearShort': 7500,
                'ui.autoclearLong': 30000,
                'ui.fadeInDuration': 300,
                // true for Carbon, Ionic and Fluent, false for Tabler
                'ui.icons.withFillCurrent': true,
                // true for Ionic and Fluent, false for Tabler
                // tabler has stroke=current on svg element, removing it
                // causes the optimizer in our build process to remove
                // stroke=none in some sub-element (path, polyline) causing
                // wrong colors to be used.
                'ui.icons.withStrokeCurrent': false,
                'ui.pingInterval': 30000,
                'fetch.defaultTimeout': 10000,
                // max size in bytes for viewing files vs downloading them
                viewMaxFileSize: 50000000
            }
        );

        return features
    };

    const config = {
        // from server
        platformVersion: window.forLogsConfig.platformVersion,
        version: window.forLogsConfig.version,
        tz: window.forLogsConfig.tz,
        date: window.forLogsConfig.date,
        copyright: window.forLogsConfig.copyright,
        language: window.forLogsConfig.language,
        languageDirection: window.forLogsConfig.languageDirection,
        urls: window.forLogsConfig.urls,
        user: window.forLogsConfig.user,
        path: window.forLogsConfig.path,
        tokens: window.forLogsConfig.tokens,
        bundleFile: window.forLogsConfig.bundleFile,
        uiConstants: window.forLogsConfig.uiConstants
    };

    /**
     * Various utilities. Def. need to be broken down.
     *
     * @author       Weeblr,llc
     * @copyright    Weeblr,llc - 2020
     * @package      @weeblr/translate
     * @license      GPL-3.0-or-later
     *
     */

    /***********************************************************************************************************************
     * Utilities
     **********************************************************************************************************************/
    const noop = () => {
    };

    /***********************************************************************************************************************
     * Loop utilities
     **********************************************************************************************************************/

    /**
     * Create an alphabetical sorting function by a specific object property.
     *
     * @param propName
     * @param caseInsensitive
     * @returns {function(*, *)}
     */
    const propAlphaSort = (propName, caseInsensitive) => (a, b) => {
        caseInsensitive = caseInsensitive ? true : false;
        return (a, b) => {
            let aValue = caseInsensitive ? a[propName].toLowerCase() : a[propName];
            let bValue = caseInsensitive ? b[propName].toLowerCase() : b[propName];
            if (aValue < bValue) {
                return -1
            }
            if (aValue > bValue) {
                return 1
            }
            return 0
        }
    };

    /***********************************************************************************************************************
     * Promises utilities
     **********************************************************************************************************************/

    /**
     * Find out if a variable is a Promise. Seems harder than it should
     *
     * @param thing
     */
    const isPromise = (thing) => {
        return thing && typeof thing == 'object' && typeof thing.then == 'function'
        // alternate
        // return thing instanceof Promise
    };

    /**
     * Reject a promise after a time out.
     *
     * @param originalPromise
     * @param timeoutDelay
     * @returns {*}
     */
    const timedOutPromise = (originalPromise, timeoutDelay, callback) => {
        if (!callback) {
            callback = () => {
                try {
                    let e = Error('Operation timed-out after delay ' + timeoutDelay + 'ms');
                    e.name = 'WblTimeOutException';
                    return e
                } catch (e) {
                    console.error(e);
                }
            };
        }
        return Promise.race([
            originalPromise,
            new Promise(
                (resolve, reject) => {
                    setTimeout(
                        reject,
                        timeoutDelay,
                        callback()
                    );
                }
            )
        ])
    };

    /***********************************************************************************************************************
     * Debounce function
     **********************************************************************************************************************/

    const debounce = (fn, delay) => {
        let timer;
        return function () {
            const that = this;
            const args = arguments;
            clearTimeout(timer);
            timer = setTimeout(() => fn.apply(that, args), delay);
        };
    };

    /**
     * Filter out some properties from an object.
     *
     * @param {Object} o
     * @param string|array props
     */
    const removeObjectProperty = (o, props) => {
        props = Array.isArray(props)
            ? props
            : [props];
        let res = {};
        Object.keys(o).map(
            (key => {
                if (!props.includes(key)) {
                    res[key] = o[key];
                }
            })
        );

        return res
    };

    /***********************************************************************************************************************
     * DOM utilities
     **********************************************************************************************************************/

    const getFocusablesFromElement = (el) => {
        let focusables = el
            ? el.querySelectorAll('button:not([disabled]), a[href], area[href], input:not([disabled]), select:not([disabled]), textarea:not([disabled]), iframe, object, [tabindex]:not([tabindex="-1"]),[contenteditable]')
            : [];
        return focusables
    };

    const getFocusTrapElements = (el) => {
        let focusables = getFocusablesFromElement(el);
        if (!focusables.length) {
            return {
                first: null,
                last: null
            }
        }
        return {
            first: focusables[0],
            last: focusables[focusables.length - 1]
        }
    };

    const defaultFocusable = (el) => {
        let focusables = getFocusablesFromElement(el);
        if (!focusables.length) {
            return null
        }

        for (let i = 0; i < focusables.length; i++) {
            if (focusables[i].dataset.focusdefault === 'true') {
                return focusables[i]
            }
        }

        // default to first if none
        return focusables[0]
    };

    const focusOnDefault = (el) => {
        let target = defaultFocusable(el);
        target && target.focus();
    };

    const isEnterOrSpaceKey = e => e && e.keyCode && [13, 32].includes(e.keyCode);

    const isEscapeKey = e => e && e.keyCode && 27 == e.keyCode;

    /**
     * Wrapper around fetch.
     *
     * @author       Weeblr,llc
     * @copyright    Weeblr,llc - 2020
     * @package      @weeblr/translate
     * @license      GPL-3.0-or-later
     *
     */

    /**
     * Wraps a Fetch call in a timing-out wrapper. If timing out, the fetch is aborted by an
     * AbortController, either the provided one or one created automatically
     * if none provided.
     *
     * @param {string} url
     * @param {[]} options
     * @param {int} timeoutDelay
     * @param {AbortController} abortController
     */
    const timedOutFetch = (url, options, timeoutDelay, abortController) => {
        timeoutDelay = timeoutDelay || features.get('fetch.defaultTimeout') || 10000;
        abortController = abortController || new AbortController();
        options.signal = abortController.signal;
        let fetchCall = fetch(
            url,
            options
        );
        return timedOutPromise(
            fetchCall,
            timeoutDelay,
            () => {
                let e = Error('Fetch timed-out after delay ' + timeoutDelay + 'ms for ' + url);
                e.name = 'WblTimeOutException';
                return e
            }
        )
    };

    const defaultOptions$1 = {
        method: 'GET',
        credentials: 'same-origin',
        mode: 'same-origin',
        cache: 'no-cache',
        redirect: 'follow',
        asJson: true,
        headers: {
            'Content-Type': 'application/json',
        }
    };

    const getDefaultOptions = (method) => {
        switch (method) {
            case 'get':
                return defaultOptions$1
            case 'post':
                return {
                    ...defaultOptions$1,
                    ...{
                        method: 'POST'
                    }
                }
            case 'put':
            case 'patch':
            case 'delete':
                return {
                    ...defaultOptions$1,
                    ...{
                        method: 'POST',
                        headers: {
                            'x-wblr-http-method-override': method.toUpperCase(),
                        },
                    }
                }
        }
    };

    const doFetch = async (url, options) => {
        try {
            let csrfToken = options['csrfToken'] || '';
            if (csrfToken) {
                options.headers['x-wblr-csrf-token'] = csrfToken;
            }
            let timeoutDelay = options['timeoutDelay'] || 0;

            if (options['query']) {
                let separator;
                Object.keys(options['query']).map(
                    (key => {
                        separator = url.indexOf('?') < 0 ? '?' : '&';
                        url += separator + key + '=' + encodeURIComponent(options['query'][key]);
                    })
                );
            }

            let abortController = options['abortController'] || '';

            let asJson = options['asJson'] !== false;

            options = removeObjectProperty(
                options,
                ['csrfToken', 'timeoutDelay', 'query', 'data', 'abortController', 'asJson']
            );

            const response = await timedOutFetch(
                url,
                options,
                timeoutDelay,
                abortController
            );

            if (response.status >= 400) {
                return {
                    failed: true,
                    status: response.status,
                    message: 'backendComm.errorLoadingDataFromServer',
                    details: response.status + ' ' + response.statusText
                }
            }

            if (response.status >= 400) {
                let responseDetails = await response.json();
                let errors = responseDetails?.errors;
                let details = errors.length
                    ? errors.reduce((acc, error) => acc += error.code + ': ' + error.message, '')
                    : response.status + ' ' + response.statusText;
                return {
                    failed: true,
                    status: response.status,
                    message: 'backendComm.errorLoadingDataFromServer',
                    details
                }
            }

            let processedResponse = response.status == 204
                ? ''
                : (
                    asJson
                        ? await response.json()
                        : await response.text()
                );

            if (!asJson || response.ok) {
                return processedResponse
            } else {
                return {
                    failed: true,
                    status: response.status,
                    message: 'backendComm.errorLoadingDataFromServer',
                    details: response.status + ' ' + response.statusText
                }
            }
        } catch (e) {
            return {
                failed: true,
                status: 'error',
                message: 'backendComm.errorCannotCommunicateWithServer',
                details: e.message
            }
        }
    };

    const execute = async (method, url, options) => {
        options = options || {};
        options = {
            ...getDefaultOptions(method),
            ...options
        };

        if ('get' !== method && options.hasOwnProperty('data')) {
            options['body'] = JSON.stringify(options['data']);
        }

        return doFetch(
            url,
            options
        )
    };

    var wbFetch = {
        get: async (url, options) => execute('get', url, options),
        post: async (url, options) => execute('post', url, options),
        put: async (url, options) => execute('put', url, options),
        patch: async (url, options) => execute('patch', url, options),
        delete: async (url, options) => execute('delete', url, options),
    };

    // Implementation

    const endpoint = config.urls.api + '/' + config.version.api + '/files';

    // Interface
    const fetchList = async (search) => {

        let query = {};

        if (search) {
            query['search'] = search;
        }

        return wbFetch.get(
            endpoint,
            {
                query
            }
        )
    };

    const downloadOne = (path, name) => {
        const a = document.createElement('a');
        a.href = endpoint + '&path=' + encodeURIComponent(path) + '&download=true';
        a.setAttribute("download", name);
        a.click();
    };

    const fetchOne = async (path) => {

        let query = {
            path: encodeURIComponent(path)
        };
        const response = await wbFetch.get(
            endpoint,
            {
                query
            }
        );

        return response
    };

    const saveOne = async (path, content) => {

        let query = {
            path: encodeURIComponent(path)
        };

        const response = await wbFetch.patch(
            endpoint,
            {
                query,
                content,
                csrfToken: config.tokens.csrf
            }
        );

        return response
    };

    /**
     * Makes an API call to delete one item. Optionally calls
     * success or errors handlers.
     *
     * @param {Array} ids
     */
    const deleteFiles = async (path, successHandler = noop, failureHandler = noop) => {

        let query = {
            path: encodeURIComponent(path)
        };

        const response = await wbFetch.delete(
            endpoint,
            {
                query,
                csrfToken: config.tokens.csrf
            }
        );
        if (response['status'] && 'error' == response['status']) {
            failureHandler(response);
        } else {
            successHandler(response);
        }

        return response
    };

    var filesModel = {
        fetchList,
        fetchOne,
        downloadOne,
        saveOne,
        deleteFiles
    };

    /**
     * DateTime ouput utilities
     */

    const defaultLocale = 'en-GB';
    const defaultOptions = {
        timeZone: 'UTC',
        dateStyle: 'medium',
        timeStyle: 'short',
    };

    const unixToSiteTimeZone = (ts, locale = defaultLocale, options = defaultOptions) => {
        try {
            return new Date(ts * 1000).toLocaleString(
                locale,
                options
            )
        } catch (e) {
            console.error(e);
        }
    };

    /* node_modules\@weeblr\svelte\components\forms\Legend.svelte generated by Svelte v3.41.0 */

    const file$q = "node_modules\\@weeblr\\svelte\\components\\forms\\Legend.svelte";

    // (13:0) {#if legend}
    function create_if_block$k(ctx) {
    	let legend_1;
    	let legend_1_class_value;

    	const block = {
    		c: function create() {
    			legend_1 = element("legend");
    			attr_dev(legend_1, "class", legend_1_class_value = "" + (/*classes*/ ctx[1] + " " + /*textSize*/ ctx[3] + " " + /*extraClasses*/ ctx[2]));
    			add_location(legend_1, file$q, 13, 4, 322);
    		},
    		m: function mount(target, anchor) {
    			insert_dev(target, legend_1, anchor);
    			legend_1.innerHTML = /*legend*/ ctx[0];
    		},
    		p: function update(ctx, dirty) {
    			if (dirty & /*legend*/ 1) legend_1.innerHTML = /*legend*/ ctx[0];
    			if (dirty & /*classes, textSize, extraClasses*/ 14 && legend_1_class_value !== (legend_1_class_value = "" + (/*classes*/ ctx[1] + " " + /*textSize*/ ctx[3] + " " + /*extraClasses*/ ctx[2]))) {
    				attr_dev(legend_1, "class", legend_1_class_value);
    			}
    		},
    		d: function destroy(detaching) {
    			if (detaching) detach_dev(legend_1);
    		}
    	};

    	dispatch_dev("SvelteRegisterBlock", {
    		block,
    		id: create_if_block$k.name,
    		type: "if",
    		source: "(13:0) {#if legend}",
    		ctx
    	});

    	return block;
    }

    function create_fragment$r(ctx) {
    	let if_block_anchor;
    	let if_block = /*legend*/ ctx[0] && create_if_block$k(ctx);

    	const block = {
    		c: function create() {
    			if (if_block) if_block.c();
    			if_block_anchor = empty();
    		},
    		l: function claim(nodes) {
    			throw new Error("options.hydrate only works if the component was compiled with the `hydratable: true` option");
    		},
    		m: function mount(target, anchor) {
    			if (if_block) if_block.m(target, anchor);
    			insert_dev(target, if_block_anchor, anchor);
    		},
    		p: function update(ctx, [dirty]) {
    			if (/*legend*/ ctx[0]) {
    				if (if_block) {
    					if_block.p(ctx, dirty);
    				} else {
    					if_block = create_if_block$k(ctx);
    					if_block.c();
    					if_block.m(if_block_anchor.parentNode, if_block_anchor);
    				}
    			} else if (if_block) {
    				if_block.d(1);
    				if_block = null;
    			}
    		},
    		i: noop$1,
    		o: noop$1,
    		d: function destroy(detaching) {
    			if (if_block) if_block.d(detaching);
    			if (detaching) detach_dev(if_block_anchor);
    		}
    	};

    	dispatch_dev("SvelteRegisterBlock", {
    		block,
    		id: create_fragment$r.name,
    		type: "component",
    		source: "",
    		ctx
    	});

    	return block;
    }

    function instance$r($$self, $$props, $$invalidate) {
    	let { $$slots: slots = {}, $$scope } = $$props;
    	validate_slots('Legend', slots, []);
    	let { legend = '' } = $$props;
    	let { classes = 'wb-block wb-mb-4 wb-font-semibold wb-leading-normal wb-tracking-wide wb-border-0 wb-bg-transparent' } = $$props;
    	let { extraClasses = 'wb-text-gray-700' } = $$props;
    	let { textSize = 'wb-text-base' } = $$props;
    	const writable_props = ['legend', 'classes', 'extraClasses', 'textSize'];

    	Object.keys($$props).forEach(key => {
    		if (!~writable_props.indexOf(key) && key.slice(0, 2) !== '$$' && key !== 'slot') console.warn(`<Legend> was created with unknown prop '${key}'`);
    	});

    	$$self.$$set = $$props => {
    		if ('legend' in $$props) $$invalidate(0, legend = $$props.legend);
    		if ('classes' in $$props) $$invalidate(1, classes = $$props.classes);
    		if ('extraClasses' in $$props) $$invalidate(2, extraClasses = $$props.extraClasses);
    		if ('textSize' in $$props) $$invalidate(3, textSize = $$props.textSize);
    	};

    	$$self.$capture_state = () => ({ legend, classes, extraClasses, textSize });

    	$$self.$inject_state = $$props => {
    		if ('legend' in $$props) $$invalidate(0, legend = $$props.legend);
    		if ('classes' in $$props) $$invalidate(1, classes = $$props.classes);
    		if ('extraClasses' in $$props) $$invalidate(2, extraClasses = $$props.extraClasses);
    		if ('textSize' in $$props) $$invalidate(3, textSize = $$props.textSize);
    	};

    	if ($$props && "$$inject" in $$props) {
    		$$self.$inject_state($$props.$$inject);
    	}

    	return [legend, classes, extraClasses, textSize];
    }

    class Legend extends SvelteComponentDev {
    	constructor(options) {
    		super(options);

    		init(this, options, instance$r, create_fragment$r, safe_not_equal, {
    			legend: 0,
    			classes: 1,
    			extraClasses: 2,
    			textSize: 3
    		});

    		dispatch_dev("SvelteRegisterComponent", {
    			component: this,
    			tagName: "Legend",
    			options,
    			id: create_fragment$r.name
    		});
    	}

    	get legend() {
    		throw new Error("<Legend>: Props cannot be read directly from the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	set legend(value) {
    		throw new Error("<Legend>: Props cannot be set directly on the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	get classes() {
    		throw new Error("<Legend>: Props cannot be read directly from the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	set classes(value) {
    		throw new Error("<Legend>: Props cannot be set directly on the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	get extraClasses() {
    		throw new Error("<Legend>: Props cannot be read directly from the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	set extraClasses(value) {
    		throw new Error("<Legend>: Props cannot be set directly on the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	get textSize() {
    		throw new Error("<Legend>: Props cannot be read directly from the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	set textSize(value) {
    		throw new Error("<Legend>: Props cannot be set directly on the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}
    }

    /* node_modules\@weeblr\svelte\components\forms\Fieldset.svelte generated by Svelte v3.41.0 */
    const file$p = "node_modules\\@weeblr\\svelte\\components\\forms\\Fieldset.svelte";

    // (27:0) {:else}
    function create_else_block$9(ctx) {
    	let current;
    	const default_slot_template = /*#slots*/ ctx[7].default;
    	const default_slot = create_slot(default_slot_template, ctx, /*$$scope*/ ctx[6], null);

    	const block = {
    		c: function create() {
    			if (default_slot) default_slot.c();
    		},
    		m: function mount(target, anchor) {
    			if (default_slot) {
    				default_slot.m(target, anchor);
    			}

    			current = true;
    		},
    		p: function update(ctx, dirty) {
    			if (default_slot) {
    				if (default_slot.p && (!current || dirty & /*$$scope*/ 64)) {
    					update_slot_base(
    						default_slot,
    						default_slot_template,
    						ctx,
    						/*$$scope*/ ctx[6],
    						!current
    						? get_all_dirty_from_scope(/*$$scope*/ ctx[6])
    						: get_slot_changes(default_slot_template, /*$$scope*/ ctx[6], dirty, null),
    						null
    					);
    				}
    			}
    		},
    		i: function intro(local) {
    			if (current) return;
    			transition_in(default_slot, local);
    			current = true;
    		},
    		o: function outro(local) {
    			transition_out(default_slot, local);
    			current = false;
    		},
    		d: function destroy(detaching) {
    			if (default_slot) default_slot.d(detaching);
    		}
    	};

    	dispatch_dev("SvelteRegisterBlock", {
    		block,
    		id: create_else_block$9.name,
    		type: "else",
    		source: "(27:0) {:else}",
    		ctx
    	});

    	return block;
    }

    // (17:0) {#if legend}
    function create_if_block$j(ctx) {
    	let fieldset;
    	let t0;
    	let t1;
    	let fieldset_aria_label_value;
    	let current;
    	let if_block0 = /*legend*/ ctx[0] && /*legendBefore*/ ctx[2] && create_if_block_2$d(ctx);
    	const default_slot_template = /*#slots*/ ctx[7].default;
    	const default_slot = create_slot(default_slot_template, ctx, /*$$scope*/ ctx[6], null);
    	let if_block1 = /*legend*/ ctx[0] && !/*legendBefore*/ ctx[2] && create_if_block_1$f(ctx);

    	const block = {
    		c: function create() {
    			fieldset = element("fieldset");
    			if (if_block0) if_block0.c();
    			t0 = space();
    			if (default_slot) default_slot.c();
    			t1 = space();
    			if (if_block1) if_block1.c();
    			attr_dev(fieldset, "class", /*classes*/ ctx[3]);
    			attr_dev(fieldset, "aria-label", fieldset_aria_label_value = /*ariaLabel*/ ctx[1] || null);
    			add_location(fieldset, file$p, 17, 4, 328);
    		},
    		m: function mount(target, anchor) {
    			insert_dev(target, fieldset, anchor);
    			if (if_block0) if_block0.m(fieldset, null);
    			append_dev(fieldset, t0);

    			if (default_slot) {
    				default_slot.m(fieldset, null);
    			}

    			append_dev(fieldset, t1);
    			if (if_block1) if_block1.m(fieldset, null);
    			current = true;
    		},
    		p: function update(ctx, dirty) {
    			if (/*legend*/ ctx[0] && /*legendBefore*/ ctx[2]) {
    				if (if_block0) {
    					if_block0.p(ctx, dirty);

    					if (dirty & /*legend, legendBefore*/ 5) {
    						transition_in(if_block0, 1);
    					}
    				} else {
    					if_block0 = create_if_block_2$d(ctx);
    					if_block0.c();
    					transition_in(if_block0, 1);
    					if_block0.m(fieldset, t0);
    				}
    			} else if (if_block0) {
    				group_outros();

    				transition_out(if_block0, 1, 1, () => {
    					if_block0 = null;
    				});

    				check_outros();
    			}

    			if (default_slot) {
    				if (default_slot.p && (!current || dirty & /*$$scope*/ 64)) {
    					update_slot_base(
    						default_slot,
    						default_slot_template,
    						ctx,
    						/*$$scope*/ ctx[6],
    						!current
    						? get_all_dirty_from_scope(/*$$scope*/ ctx[6])
    						: get_slot_changes(default_slot_template, /*$$scope*/ ctx[6], dirty, null),
    						null
    					);
    				}
    			}

    			if (/*legend*/ ctx[0] && !/*legendBefore*/ ctx[2]) {
    				if (if_block1) {
    					if_block1.p(ctx, dirty);

    					if (dirty & /*legend, legendBefore*/ 5) {
    						transition_in(if_block1, 1);
    					}
    				} else {
    					if_block1 = create_if_block_1$f(ctx);
    					if_block1.c();
    					transition_in(if_block1, 1);
    					if_block1.m(fieldset, null);
    				}
    			} else if (if_block1) {
    				group_outros();

    				transition_out(if_block1, 1, 1, () => {
    					if_block1 = null;
    				});

    				check_outros();
    			}

    			if (!current || dirty & /*classes*/ 8) {
    				attr_dev(fieldset, "class", /*classes*/ ctx[3]);
    			}

    			if (!current || dirty & /*ariaLabel*/ 2 && fieldset_aria_label_value !== (fieldset_aria_label_value = /*ariaLabel*/ ctx[1] || null)) {
    				attr_dev(fieldset, "aria-label", fieldset_aria_label_value);
    			}
    		},
    		i: function intro(local) {
    			if (current) return;
    			transition_in(if_block0);
    			transition_in(default_slot, local);
    			transition_in(if_block1);
    			current = true;
    		},
    		o: function outro(local) {
    			transition_out(if_block0);
    			transition_out(default_slot, local);
    			transition_out(if_block1);
    			current = false;
    		},
    		d: function destroy(detaching) {
    			if (detaching) detach_dev(fieldset);
    			if (if_block0) if_block0.d();
    			if (default_slot) default_slot.d(detaching);
    			if (if_block1) if_block1.d();
    		}
    	};

    	dispatch_dev("SvelteRegisterBlock", {
    		block,
    		id: create_if_block$j.name,
    		type: "if",
    		source: "(17:0) {#if legend}",
    		ctx
    	});

    	return block;
    }

    // (19:8) {#if legend && legendBefore}
    function create_if_block_2$d(ctx) {
    	let legend_1;
    	let current;

    	legend_1 = new Legend({
    			props: {
    				legend: /*legend*/ ctx[0],
    				textSize: /*textSize*/ ctx[5],
    				legendClasses: /*legendClasses*/ ctx[4]
    			},
    			$$inline: true
    		});

    	const block = {
    		c: function create() {
    			create_component(legend_1.$$.fragment);
    		},
    		m: function mount(target, anchor) {
    			mount_component(legend_1, target, anchor);
    			current = true;
    		},
    		p: function update(ctx, dirty) {
    			const legend_1_changes = {};
    			if (dirty & /*legend*/ 1) legend_1_changes.legend = /*legend*/ ctx[0];
    			if (dirty & /*textSize*/ 32) legend_1_changes.textSize = /*textSize*/ ctx[5];
    			if (dirty & /*legendClasses*/ 16) legend_1_changes.legendClasses = /*legendClasses*/ ctx[4];
    			legend_1.$set(legend_1_changes);
    		},
    		i: function intro(local) {
    			if (current) return;
    			transition_in(legend_1.$$.fragment, local);
    			current = true;
    		},
    		o: function outro(local) {
    			transition_out(legend_1.$$.fragment, local);
    			current = false;
    		},
    		d: function destroy(detaching) {
    			destroy_component(legend_1, detaching);
    		}
    	};

    	dispatch_dev("SvelteRegisterBlock", {
    		block,
    		id: create_if_block_2$d.name,
    		type: "if",
    		source: "(19:8) {#if legend && legendBefore}",
    		ctx
    	});

    	return block;
    }

    // (23:8) {#if legend && !legendBefore}
    function create_if_block_1$f(ctx) {
    	let legend_1;
    	let current;

    	legend_1 = new Legend({
    			props: {
    				legend: /*legend*/ ctx[0],
    				textSize: /*textSize*/ ctx[5],
    				extraClasses: /*legendClasses*/ ctx[4]
    			},
    			$$inline: true
    		});

    	const block = {
    		c: function create() {
    			create_component(legend_1.$$.fragment);
    		},
    		m: function mount(target, anchor) {
    			mount_component(legend_1, target, anchor);
    			current = true;
    		},
    		p: function update(ctx, dirty) {
    			const legend_1_changes = {};
    			if (dirty & /*legend*/ 1) legend_1_changes.legend = /*legend*/ ctx[0];
    			if (dirty & /*textSize*/ 32) legend_1_changes.textSize = /*textSize*/ ctx[5];
    			if (dirty & /*legendClasses*/ 16) legend_1_changes.extraClasses = /*legendClasses*/ ctx[4];
    			legend_1.$set(legend_1_changes);
    		},
    		i: function intro(local) {
    			if (current) return;
    			transition_in(legend_1.$$.fragment, local);
    			current = true;
    		},
    		o: function outro(local) {
    			transition_out(legend_1.$$.fragment, local);
    			current = false;
    		},
    		d: function destroy(detaching) {
    			destroy_component(legend_1, detaching);
    		}
    	};

    	dispatch_dev("SvelteRegisterBlock", {
    		block,
    		id: create_if_block_1$f.name,
    		type: "if",
    		source: "(23:8) {#if legend && !legendBefore}",
    		ctx
    	});

    	return block;
    }

    function create_fragment$q(ctx) {
    	let current_block_type_index;
    	let if_block;
    	let if_block_anchor;
    	let current;
    	const if_block_creators = [create_if_block$j, create_else_block$9];
    	const if_blocks = [];

    	function select_block_type(ctx, dirty) {
    		if (/*legend*/ ctx[0]) return 0;
    		return 1;
    	}

    	current_block_type_index = select_block_type(ctx);
    	if_block = if_blocks[current_block_type_index] = if_block_creators[current_block_type_index](ctx);

    	const block = {
    		c: function create() {
    			if_block.c();
    			if_block_anchor = empty();
    		},
    		l: function claim(nodes) {
    			throw new Error("options.hydrate only works if the component was compiled with the `hydratable: true` option");
    		},
    		m: function mount(target, anchor) {
    			if_blocks[current_block_type_index].m(target, anchor);
    			insert_dev(target, if_block_anchor, anchor);
    			current = true;
    		},
    		p: function update(ctx, [dirty]) {
    			let previous_block_index = current_block_type_index;
    			current_block_type_index = select_block_type(ctx);

    			if (current_block_type_index === previous_block_index) {
    				if_blocks[current_block_type_index].p(ctx, dirty);
    			} else {
    				group_outros();

    				transition_out(if_blocks[previous_block_index], 1, 1, () => {
    					if_blocks[previous_block_index] = null;
    				});

    				check_outros();
    				if_block = if_blocks[current_block_type_index];

    				if (!if_block) {
    					if_block = if_blocks[current_block_type_index] = if_block_creators[current_block_type_index](ctx);
    					if_block.c();
    				} else {
    					if_block.p(ctx, dirty);
    				}

    				transition_in(if_block, 1);
    				if_block.m(if_block_anchor.parentNode, if_block_anchor);
    			}
    		},
    		i: function intro(local) {
    			if (current) return;
    			transition_in(if_block);
    			current = true;
    		},
    		o: function outro(local) {
    			transition_out(if_block);
    			current = false;
    		},
    		d: function destroy(detaching) {
    			if_blocks[current_block_type_index].d(detaching);
    			if (detaching) detach_dev(if_block_anchor);
    		}
    	};

    	dispatch_dev("SvelteRegisterBlock", {
    		block,
    		id: create_fragment$q.name,
    		type: "component",
    		source: "",
    		ctx
    	});

    	return block;
    }

    function instance$q($$self, $$props, $$invalidate) {
    	let { $$slots: slots = {}, $$scope } = $$props;
    	validate_slots('Fieldset', slots, ['default']);
    	let { legend = '' } = $$props;
    	let { ariaLabel = '' } = $$props;
    	let { legendBefore = false } = $$props;
    	let { classes = 'wb-border-0' } = $$props;
    	let { legendClasses = '' } = $$props;
    	let { textSize = 'wb-text-base' } = $$props;
    	const writable_props = ['legend', 'ariaLabel', 'legendBefore', 'classes', 'legendClasses', 'textSize'];

    	Object.keys($$props).forEach(key => {
    		if (!~writable_props.indexOf(key) && key.slice(0, 2) !== '$$' && key !== 'slot') console.warn(`<Fieldset> was created with unknown prop '${key}'`);
    	});

    	$$self.$$set = $$props => {
    		if ('legend' in $$props) $$invalidate(0, legend = $$props.legend);
    		if ('ariaLabel' in $$props) $$invalidate(1, ariaLabel = $$props.ariaLabel);
    		if ('legendBefore' in $$props) $$invalidate(2, legendBefore = $$props.legendBefore);
    		if ('classes' in $$props) $$invalidate(3, classes = $$props.classes);
    		if ('legendClasses' in $$props) $$invalidate(4, legendClasses = $$props.legendClasses);
    		if ('textSize' in $$props) $$invalidate(5, textSize = $$props.textSize);
    		if ('$$scope' in $$props) $$invalidate(6, $$scope = $$props.$$scope);
    	};

    	$$self.$capture_state = () => ({
    		legend,
    		ariaLabel,
    		legendBefore,
    		classes,
    		legendClasses,
    		textSize,
    		Legend
    	});

    	$$self.$inject_state = $$props => {
    		if ('legend' in $$props) $$invalidate(0, legend = $$props.legend);
    		if ('ariaLabel' in $$props) $$invalidate(1, ariaLabel = $$props.ariaLabel);
    		if ('legendBefore' in $$props) $$invalidate(2, legendBefore = $$props.legendBefore);
    		if ('classes' in $$props) $$invalidate(3, classes = $$props.classes);
    		if ('legendClasses' in $$props) $$invalidate(4, legendClasses = $$props.legendClasses);
    		if ('textSize' in $$props) $$invalidate(5, textSize = $$props.textSize);
    	};

    	if ($$props && "$$inject" in $$props) {
    		$$self.$inject_state($$props.$$inject);
    	}

    	return [
    		legend,
    		ariaLabel,
    		legendBefore,
    		classes,
    		legendClasses,
    		textSize,
    		$$scope,
    		slots
    	];
    }

    class Fieldset extends SvelteComponentDev {
    	constructor(options) {
    		super(options);

    		init(this, options, instance$q, create_fragment$q, safe_not_equal, {
    			legend: 0,
    			ariaLabel: 1,
    			legendBefore: 2,
    			classes: 3,
    			legendClasses: 4,
    			textSize: 5
    		});

    		dispatch_dev("SvelteRegisterComponent", {
    			component: this,
    			tagName: "Fieldset",
    			options,
    			id: create_fragment$q.name
    		});
    	}

    	get legend() {
    		throw new Error("<Fieldset>: Props cannot be read directly from the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	set legend(value) {
    		throw new Error("<Fieldset>: Props cannot be set directly on the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	get ariaLabel() {
    		throw new Error("<Fieldset>: Props cannot be read directly from the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	set ariaLabel(value) {
    		throw new Error("<Fieldset>: Props cannot be set directly on the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	get legendBefore() {
    		throw new Error("<Fieldset>: Props cannot be read directly from the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	set legendBefore(value) {
    		throw new Error("<Fieldset>: Props cannot be set directly on the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	get classes() {
    		throw new Error("<Fieldset>: Props cannot be read directly from the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	set classes(value) {
    		throw new Error("<Fieldset>: Props cannot be set directly on the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	get legendClasses() {
    		throw new Error("<Fieldset>: Props cannot be read directly from the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	set legendClasses(value) {
    		throw new Error("<Fieldset>: Props cannot be set directly on the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	get textSize() {
    		throw new Error("<Fieldset>: Props cannot be read directly from the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	set textSize(value) {
    		throw new Error("<Fieldset>: Props cannot be set directly on the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}
    }

    function cubicOut(t) {
        const f = t - 1.0;
        return f * f * f + 1.0;
    }
    function sineOut(t) {
        return Math.sin((t * Math.PI) / 2);
    }

    function fade(node, { delay = 0, duration = 400, easing = identity } = {}) {
        const o = +getComputedStyle(node).opacity;
        return {
            delay,
            duration,
            easing,
            css: t => `opacity: ${t * o}`
        };
    }
    function fly(node, { delay = 0, duration = 400, easing = cubicOut, x = 0, y = 0, opacity = 0 } = {}) {
        const style = getComputedStyle(node);
        const target_opacity = +style.opacity;
        const transform = style.transform === 'none' ? '' : style.transform;
        const od = target_opacity * (1 - opacity);
        return {
            delay,
            duration,
            easing,
            css: (t, u) => `
			transform: ${transform} translate(${(1 - t) * x}px, ${(1 - t) * y}px);
			opacity: ${target_opacity - (od * u)}`
        };
    }
    function slide(node, { delay = 0, duration = 400, easing = cubicOut } = {}) {
        const style = getComputedStyle(node);
        const opacity = +style.opacity;
        const height = parseFloat(style.height);
        const padding_top = parseFloat(style.paddingTop);
        const padding_bottom = parseFloat(style.paddingBottom);
        const margin_top = parseFloat(style.marginTop);
        const margin_bottom = parseFloat(style.marginBottom);
        const border_top_width = parseFloat(style.borderTopWidth);
        const border_bottom_width = parseFloat(style.borderBottomWidth);
        return {
            delay,
            duration,
            easing,
            css: t => 'overflow: hidden;' +
                `opacity: ${Math.min(t * 20, 1) * opacity};` +
                `height: ${t * height}px;` +
                `padding-top: ${t * padding_top}px;` +
                `padding-bottom: ${t * padding_bottom}px;` +
                `margin-top: ${t * margin_top}px;` +
                `margin-bottom: ${t * margin_bottom}px;` +
                `border-top-width: ${t * border_top_width}px;` +
                `border-bottom-width: ${t * border_bottom_width}px;`
        };
    }

    /* node_modules\@weeblr\svelte\components\Icon.svelte generated by Svelte v3.41.0 */
    const file$o = "node_modules\\@weeblr\\svelte\\components\\Icon.svelte";

    // (90:0) {:else }
    function create_else_block$8(ctx) {
    	let span;
    	let html_tag;
    	let t;
    	let current;
    	const default_slot_template = /*#slots*/ ctx[19].default;
    	const default_slot = create_slot(default_slot_template, ctx, /*$$scope*/ ctx[18], null);

    	const block = {
    		c: function create() {
    			span = element("span");
    			html_tag = new HtmlTag();
    			t = space();
    			if (default_slot) default_slot.c();
    			html_tag.a = t;
    			attr_dev(span, "class", /*wrapperClasses*/ ctx[0]);
    			attr_dev(span, "title", /*expandedTitle*/ ctx[9]);
    			attr_dev(span, "alt", /*expandedAlt*/ ctx[10]);
    			add_location(span, file$o, 90, 4, 3146);
    		},
    		m: function mount(target, anchor) {
    			insert_dev(target, span, anchor);
    			html_tag.m(/*processedIcon*/ ctx[11], span);
    			append_dev(span, t);

    			if (default_slot) {
    				default_slot.m(span, null);
    			}

    			current = true;
    		},
    		p: function update(ctx, dirty) {
    			if (!current || dirty & /*processedIcon*/ 2048) html_tag.p(/*processedIcon*/ ctx[11]);

    			if (default_slot) {
    				if (default_slot.p && (!current || dirty & /*$$scope*/ 262144)) {
    					update_slot_base(
    						default_slot,
    						default_slot_template,
    						ctx,
    						/*$$scope*/ ctx[18],
    						!current
    						? get_all_dirty_from_scope(/*$$scope*/ ctx[18])
    						: get_slot_changes(default_slot_template, /*$$scope*/ ctx[18], dirty, null),
    						null
    					);
    				}
    			}

    			if (!current || dirty & /*wrapperClasses*/ 1) {
    				attr_dev(span, "class", /*wrapperClasses*/ ctx[0]);
    			}

    			if (!current || dirty & /*expandedTitle*/ 512) {
    				attr_dev(span, "title", /*expandedTitle*/ ctx[9]);
    			}

    			if (!current || dirty & /*expandedAlt*/ 1024) {
    				attr_dev(span, "alt", /*expandedAlt*/ ctx[10]);
    			}
    		},
    		i: function intro(local) {
    			if (current) return;
    			transition_in(default_slot, local);
    			current = true;
    		},
    		o: function outro(local) {
    			transition_out(default_slot, local);
    			current = false;
    		},
    		d: function destroy(detaching) {
    			if (detaching) detach_dev(span);
    			if (default_slot) default_slot.d(detaching);
    		}
    	};

    	dispatch_dev("SvelteRegisterBlock", {
    		block,
    		id: create_else_block$8.name,
    		type: "else",
    		source: "(90:0) {:else }",
    		ctx
    	});

    	return block;
    }

    // (87:19) 
    function create_if_block_3$6(ctx) {
    	let span;
    	let html_tag;
    	let t;
    	let span_class_value;
    	let current;
    	const default_slot_template = /*#slots*/ ctx[19].default;
    	const default_slot = create_slot(default_slot_template, ctx, /*$$scope*/ ctx[18], null);

    	const block = {
    		c: function create() {
    			span = element("span");
    			html_tag = new HtmlTag();
    			t = space();
    			if (default_slot) default_slot.c();
    			html_tag.a = t;
    			attr_dev(span, "title", /*expandedTitle*/ ctx[9]);
    			attr_dev(span, "alt", /*expandedAlt*/ ctx[10]);
    			attr_dev(span, "class", span_class_value = "wb-opacity-50 " + /*wrapperClasses*/ ctx[0]);
    			add_location(span, file$o, 87, 4, 2999);
    		},
    		m: function mount(target, anchor) {
    			insert_dev(target, span, anchor);
    			html_tag.m(/*processedIcon*/ ctx[11], span);
    			append_dev(span, t);

    			if (default_slot) {
    				default_slot.m(span, null);
    			}

    			current = true;
    		},
    		p: function update(ctx, dirty) {
    			if (!current || dirty & /*processedIcon*/ 2048) html_tag.p(/*processedIcon*/ ctx[11]);

    			if (default_slot) {
    				if (default_slot.p && (!current || dirty & /*$$scope*/ 262144)) {
    					update_slot_base(
    						default_slot,
    						default_slot_template,
    						ctx,
    						/*$$scope*/ ctx[18],
    						!current
    						? get_all_dirty_from_scope(/*$$scope*/ ctx[18])
    						: get_slot_changes(default_slot_template, /*$$scope*/ ctx[18], dirty, null),
    						null
    					);
    				}
    			}

    			if (!current || dirty & /*expandedTitle*/ 512) {
    				attr_dev(span, "title", /*expandedTitle*/ ctx[9]);
    			}

    			if (!current || dirty & /*expandedAlt*/ 1024) {
    				attr_dev(span, "alt", /*expandedAlt*/ ctx[10]);
    			}

    			if (!current || dirty & /*wrapperClasses*/ 1 && span_class_value !== (span_class_value = "wb-opacity-50 " + /*wrapperClasses*/ ctx[0])) {
    				attr_dev(span, "class", span_class_value);
    			}
    		},
    		i: function intro(local) {
    			if (current) return;
    			transition_in(default_slot, local);
    			current = true;
    		},
    		o: function outro(local) {
    			transition_out(default_slot, local);
    			current = false;
    		},
    		d: function destroy(detaching) {
    			if (detaching) detach_dev(span);
    			if (default_slot) default_slot.d(detaching);
    		}
    	};

    	dispatch_dev("SvelteRegisterBlock", {
    		block,
    		id: create_if_block_3$6.name,
    		type: "if",
    		source: "(87:19) ",
    		ctx
    	});

    	return block;
    }

    // (80:25) 
    function create_if_block_2$c(ctx) {
    	let span;
    	let html_tag;
    	let t;
    	let span_tabindex_value;
    	let span_class_value;
    	let span_aria_expanded_value;
    	let span_aria_owns_value;
    	let current;
    	let mounted;
    	let dispose;
    	const default_slot_template = /*#slots*/ ctx[19].default;
    	const default_slot = create_slot(default_slot_template, ctx, /*$$scope*/ ctx[18], null);

    	const block = {
    		c: function create() {
    			span = element("span");
    			html_tag = new HtmlTag();
    			t = space();
    			if (default_slot) default_slot.c();
    			html_tag.a = t;
    			attr_dev(span, "title", /*expandedTitle*/ ctx[9]);
    			attr_dev(span, "alt", /*expandedAlt*/ ctx[10]);
    			attr_dev(span, "tabindex", span_tabindex_value = /*tabbable*/ ctx[6] ? 0 : -1);
    			attr_dev(span, "class", span_class_value = "wb-c-focusable " + (/*listenToClicks*/ ctx[4] ? 'wb-cursor-pointer' : '') + " " + /*wrapperClasses*/ ctx[0]);

    			attr_dev(span, "aria-expanded", span_aria_expanded_value = typeof /*ariaExpanded*/ ctx[7] == 'undefined'
    			? null
    			: /*ariaExpanded*/ ctx[7]);

    			attr_dev(span, "aria-owns", span_aria_owns_value = /*ariaOwns*/ ctx[8] || null);
    			add_location(span, file$o, 80, 4, 2589);
    		},
    		m: function mount(target, anchor) {
    			insert_dev(target, span, anchor);
    			html_tag.m(/*processedIcon*/ ctx[11], span);
    			append_dev(span, t);

    			if (default_slot) {
    				default_slot.m(span, null);
    			}

    			current = true;

    			if (!mounted) {
    				dispose = [
    					listen_dev(span, "click", /*click_handler_1*/ ctx[21], false, false, false),
    					listen_dev(span, "keydown", /*keydownHandler*/ ctx[12], false, false, false)
    				];

    				mounted = true;
    			}
    		},
    		p: function update(ctx, dirty) {
    			if (!current || dirty & /*processedIcon*/ 2048) html_tag.p(/*processedIcon*/ ctx[11]);

    			if (default_slot) {
    				if (default_slot.p && (!current || dirty & /*$$scope*/ 262144)) {
    					update_slot_base(
    						default_slot,
    						default_slot_template,
    						ctx,
    						/*$$scope*/ ctx[18],
    						!current
    						? get_all_dirty_from_scope(/*$$scope*/ ctx[18])
    						: get_slot_changes(default_slot_template, /*$$scope*/ ctx[18], dirty, null),
    						null
    					);
    				}
    			}

    			if (!current || dirty & /*expandedTitle*/ 512) {
    				attr_dev(span, "title", /*expandedTitle*/ ctx[9]);
    			}

    			if (!current || dirty & /*expandedAlt*/ 1024) {
    				attr_dev(span, "alt", /*expandedAlt*/ ctx[10]);
    			}

    			if (!current || dirty & /*tabbable*/ 64 && span_tabindex_value !== (span_tabindex_value = /*tabbable*/ ctx[6] ? 0 : -1)) {
    				attr_dev(span, "tabindex", span_tabindex_value);
    			}

    			if (!current || dirty & /*listenToClicks, wrapperClasses*/ 17 && span_class_value !== (span_class_value = "wb-c-focusable " + (/*listenToClicks*/ ctx[4] ? 'wb-cursor-pointer' : '') + " " + /*wrapperClasses*/ ctx[0])) {
    				attr_dev(span, "class", span_class_value);
    			}

    			if (!current || dirty & /*ariaExpanded*/ 128 && span_aria_expanded_value !== (span_aria_expanded_value = typeof /*ariaExpanded*/ ctx[7] == 'undefined'
    			? null
    			: /*ariaExpanded*/ ctx[7])) {
    				attr_dev(span, "aria-expanded", span_aria_expanded_value);
    			}

    			if (!current || dirty & /*ariaOwns*/ 256 && span_aria_owns_value !== (span_aria_owns_value = /*ariaOwns*/ ctx[8] || null)) {
    				attr_dev(span, "aria-owns", span_aria_owns_value);
    			}
    		},
    		i: function intro(local) {
    			if (current) return;
    			transition_in(default_slot, local);
    			current = true;
    		},
    		o: function outro(local) {
    			transition_out(default_slot, local);
    			current = false;
    		},
    		d: function destroy(detaching) {
    			if (detaching) detach_dev(span);
    			if (default_slot) default_slot.d(detaching);
    			mounted = false;
    			run_all(dispose);
    		}
    	};

    	dispatch_dev("SvelteRegisterBlock", {
    		block,
    		id: create_if_block_2$c.name,
    		type: "if",
    		source: "(80:25) ",
    		ctx
    	});

    	return block;
    }

    // (76:18) 
    function create_if_block_1$e(ctx) {
    	let span;
    	let html_tag;
    	let t;
    	let span_intro;
    	let current;
    	const default_slot_template = /*#slots*/ ctx[19].default;
    	const default_slot = create_slot(default_slot_template, ctx, /*$$scope*/ ctx[18], null);

    	const block = {
    		c: function create() {
    			span = element("span");
    			html_tag = new HtmlTag();
    			t = space();
    			if (default_slot) default_slot.c();
    			html_tag.a = t;
    			attr_dev(span, "class", /*wrapperClasses*/ ctx[0]);
    			attr_dev(span, "title", /*expandedTitle*/ ctx[9]);
    			attr_dev(span, "alt", /*expandedAlt*/ ctx[10]);
    			add_location(span, file$o, 76, 4, 2388);
    		},
    		m: function mount(target, anchor) {
    			insert_dev(target, span, anchor);
    			html_tag.m(/*processedIcon*/ ctx[11], span);
    			append_dev(span, t);

    			if (default_slot) {
    				default_slot.m(span, null);
    			}

    			current = true;
    		},
    		p: function update(new_ctx, dirty) {
    			ctx = new_ctx;
    			if (!current || dirty & /*processedIcon*/ 2048) html_tag.p(/*processedIcon*/ ctx[11]);

    			if (default_slot) {
    				if (default_slot.p && (!current || dirty & /*$$scope*/ 262144)) {
    					update_slot_base(
    						default_slot,
    						default_slot_template,
    						ctx,
    						/*$$scope*/ ctx[18],
    						!current
    						? get_all_dirty_from_scope(/*$$scope*/ ctx[18])
    						: get_slot_changes(default_slot_template, /*$$scope*/ ctx[18], dirty, null),
    						null
    					);
    				}
    			}

    			if (!current || dirty & /*wrapperClasses*/ 1) {
    				attr_dev(span, "class", /*wrapperClasses*/ ctx[0]);
    			}

    			if (!current || dirty & /*expandedTitle*/ 512) {
    				attr_dev(span, "title", /*expandedTitle*/ ctx[9]);
    			}

    			if (!current || dirty & /*expandedAlt*/ 1024) {
    				attr_dev(span, "alt", /*expandedAlt*/ ctx[10]);
    			}
    		},
    		i: function intro(local) {
    			if (current) return;
    			transition_in(default_slot, local);

    			if (!span_intro) {
    				add_render_callback(() => {
    					span_intro = create_in_transition(span, fly, {
    						duration: 400,
    						x: /*flyX*/ ctx[2],
    						y: /*flyY*/ ctx[3]
    					});

    					span_intro.start();
    				});
    			}

    			current = true;
    		},
    		o: function outro(local) {
    			transition_out(default_slot, local);
    			current = false;
    		},
    		d: function destroy(detaching) {
    			if (detaching) detach_dev(span);
    			if (default_slot) default_slot.d(detaching);
    		}
    	};

    	dispatch_dev("SvelteRegisterBlock", {
    		block,
    		id: create_if_block_1$e.name,
    		type: "if",
    		source: "(76:18) ",
    		ctx
    	});

    	return block;
    }

    // (68:0) {#if withFly && listenToClicks}
    function create_if_block$i(ctx) {
    	let span;
    	let html_tag;
    	let t;
    	let span_tabindex_value;
    	let span_class_value;
    	let span_aria_expanded_value;
    	let span_aria_owns_value;
    	let span_intro;
    	let current;
    	let mounted;
    	let dispose;
    	const default_slot_template = /*#slots*/ ctx[19].default;
    	const default_slot = create_slot(default_slot_template, ctx, /*$$scope*/ ctx[18], null);

    	const block = {
    		c: function create() {
    			span = element("span");
    			html_tag = new HtmlTag();
    			t = space();
    			if (default_slot) default_slot.c();
    			html_tag.a = t;
    			attr_dev(span, "tabindex", span_tabindex_value = /*tabbable*/ ctx[6] ? 0 : -1);
    			attr_dev(span, "class", span_class_value = "wb-c-focusable " + (/*listenToClicks*/ ctx[4] ? 'wb-cursor-pointer' : '') + " " + /*wrapperClasses*/ ctx[0]);
    			attr_dev(span, "title", /*expandedTitle*/ ctx[9]);
    			attr_dev(span, "alt", /*expandedAlt*/ ctx[10]);

    			attr_dev(span, "aria-expanded", span_aria_expanded_value = typeof /*ariaExpanded*/ ctx[7] == 'undefined'
    			? null
    			: /*ariaExpanded*/ ctx[7]);

    			attr_dev(span, "aria-owns", span_aria_owns_value = /*ariaOwns*/ ctx[8] || null);
    			add_location(span, file$o, 68, 4, 1928);
    		},
    		m: function mount(target, anchor) {
    			insert_dev(target, span, anchor);
    			html_tag.m(/*processedIcon*/ ctx[11], span);
    			append_dev(span, t);

    			if (default_slot) {
    				default_slot.m(span, null);
    			}

    			current = true;

    			if (!mounted) {
    				dispose = [
    					listen_dev(span, "click", /*click_handler*/ ctx[20], false, false, false),
    					listen_dev(span, "keydown", /*keydownHandler*/ ctx[12], false, false, false)
    				];

    				mounted = true;
    			}
    		},
    		p: function update(new_ctx, dirty) {
    			ctx = new_ctx;
    			if (!current || dirty & /*processedIcon*/ 2048) html_tag.p(/*processedIcon*/ ctx[11]);

    			if (default_slot) {
    				if (default_slot.p && (!current || dirty & /*$$scope*/ 262144)) {
    					update_slot_base(
    						default_slot,
    						default_slot_template,
    						ctx,
    						/*$$scope*/ ctx[18],
    						!current
    						? get_all_dirty_from_scope(/*$$scope*/ ctx[18])
    						: get_slot_changes(default_slot_template, /*$$scope*/ ctx[18], dirty, null),
    						null
    					);
    				}
    			}

    			if (!current || dirty & /*tabbable*/ 64 && span_tabindex_value !== (span_tabindex_value = /*tabbable*/ ctx[6] ? 0 : -1)) {
    				attr_dev(span, "tabindex", span_tabindex_value);
    			}

    			if (!current || dirty & /*listenToClicks, wrapperClasses*/ 17 && span_class_value !== (span_class_value = "wb-c-focusable " + (/*listenToClicks*/ ctx[4] ? 'wb-cursor-pointer' : '') + " " + /*wrapperClasses*/ ctx[0])) {
    				attr_dev(span, "class", span_class_value);
    			}

    			if (!current || dirty & /*expandedTitle*/ 512) {
    				attr_dev(span, "title", /*expandedTitle*/ ctx[9]);
    			}

    			if (!current || dirty & /*expandedAlt*/ 1024) {
    				attr_dev(span, "alt", /*expandedAlt*/ ctx[10]);
    			}

    			if (!current || dirty & /*ariaExpanded*/ 128 && span_aria_expanded_value !== (span_aria_expanded_value = typeof /*ariaExpanded*/ ctx[7] == 'undefined'
    			? null
    			: /*ariaExpanded*/ ctx[7])) {
    				attr_dev(span, "aria-expanded", span_aria_expanded_value);
    			}

    			if (!current || dirty & /*ariaOwns*/ 256 && span_aria_owns_value !== (span_aria_owns_value = /*ariaOwns*/ ctx[8] || null)) {
    				attr_dev(span, "aria-owns", span_aria_owns_value);
    			}
    		},
    		i: function intro(local) {
    			if (current) return;
    			transition_in(default_slot, local);

    			if (!span_intro) {
    				add_render_callback(() => {
    					span_intro = create_in_transition(span, fly, {
    						duration: 400,
    						x: /*flyX*/ ctx[2],
    						y: /*flyY*/ ctx[3]
    					});

    					span_intro.start();
    				});
    			}

    			current = true;
    		},
    		o: function outro(local) {
    			transition_out(default_slot, local);
    			current = false;
    		},
    		d: function destroy(detaching) {
    			if (detaching) detach_dev(span);
    			if (default_slot) default_slot.d(detaching);
    			mounted = false;
    			run_all(dispose);
    		}
    	};

    	dispatch_dev("SvelteRegisterBlock", {
    		block,
    		id: create_if_block$i.name,
    		type: "if",
    		source: "(68:0) {#if withFly && listenToClicks}",
    		ctx
    	});

    	return block;
    }

    function create_fragment$p(ctx) {
    	let current_block_type_index;
    	let if_block;
    	let if_block_anchor;
    	let current;

    	const if_block_creators = [
    		create_if_block$i,
    		create_if_block_1$e,
    		create_if_block_2$c,
    		create_if_block_3$6,
    		create_else_block$8
    	];

    	const if_blocks = [];

    	function select_block_type(ctx, dirty) {
    		if (/*withFly*/ ctx[1] && /*listenToClicks*/ ctx[4]) return 0;
    		if (/*withFly*/ ctx[1]) return 1;
    		if (/*listenToClicks*/ ctx[4]) return 2;
    		if (/*disabled*/ ctx[5]) return 3;
    		return 4;
    	}

    	current_block_type_index = select_block_type(ctx);
    	if_block = if_blocks[current_block_type_index] = if_block_creators[current_block_type_index](ctx);

    	const block = {
    		c: function create() {
    			if_block.c();
    			if_block_anchor = empty();
    		},
    		l: function claim(nodes) {
    			throw new Error("options.hydrate only works if the component was compiled with the `hydratable: true` option");
    		},
    		m: function mount(target, anchor) {
    			if_blocks[current_block_type_index].m(target, anchor);
    			insert_dev(target, if_block_anchor, anchor);
    			current = true;
    		},
    		p: function update(ctx, [dirty]) {
    			let previous_block_index = current_block_type_index;
    			current_block_type_index = select_block_type(ctx);

    			if (current_block_type_index === previous_block_index) {
    				if_blocks[current_block_type_index].p(ctx, dirty);
    			} else {
    				group_outros();

    				transition_out(if_blocks[previous_block_index], 1, 1, () => {
    					if_blocks[previous_block_index] = null;
    				});

    				check_outros();
    				if_block = if_blocks[current_block_type_index];

    				if (!if_block) {
    					if_block = if_blocks[current_block_type_index] = if_block_creators[current_block_type_index](ctx);
    					if_block.c();
    				} else {
    					if_block.p(ctx, dirty);
    				}

    				transition_in(if_block, 1);
    				if_block.m(if_block_anchor.parentNode, if_block_anchor);
    			}
    		},
    		i: function intro(local) {
    			if (current) return;
    			transition_in(if_block);
    			current = true;
    		},
    		o: function outro(local) {
    			transition_out(if_block);
    			current = false;
    		},
    		d: function destroy(detaching) {
    			if_blocks[current_block_type_index].d(detaching);
    			if (detaching) detach_dev(if_block_anchor);
    		}
    	};

    	dispatch_dev("SvelteRegisterBlock", {
    		block,
    		id: create_fragment$p.name,
    		type: "component",
    		source: "",
    		ctx
    	});

    	return block;
    }

    function instance$p($$self, $$props, $$invalidate) {
    	let processedIcon;
    	let expandedAlt;
    	let expandedTitle;
    	let { $$slots: slots = {}, $$scope } = $$props;
    	validate_slots('Icon', slots, ['default']);
    	let { classes = 'wb-max-w-none' } = $$props;
    	let { wrapperClasses = 'wb-inline-block wb-rounded-sm wb-align-middle' } = $$props;
    	let { title = '' } = $$props;
    	let { alt = '' } = $$props;
    	let { size = 'l' } = $$props; // s | xs | l | xl | xxl | css length
    	let { icon = '' } = $$props;
    	let { withFly = false } = $$props;
    	let { flyX = 0 } = $$props;
    	let { flyY = 0 } = $$props;
    	let { listenToClicks = false } = $$props;
    	let { disabled = false } = $$props;
    	let { tabbable = false } = $$props;
    	let { ariaExpanded = undefined } = $$props;
    	let { ariaOwns = '' } = $$props;

    	const sizes = {
    		xs: '0.75rem',
    		s: '1rem',
    		l: '1.25rem',
    		xl: '1.5rem',
    		xxl: '2rem',
    		xxxl: '4rem'
    	};

    	const keydownHandler = e => {
    		if (isEnterOrSpaceKey(e)) {
    			e.preventDefault();
    			e.stopPropagation();
    			e.target.click();
    		}
    	};

    	const features = getContext('features');

    	const expandClasses = classes => {
    		return 'class="' + (features.isTruthy('ui.icons.withFillCurrent')
    		? 'wb-fill-current'
    		: '') + (features.isTruthy('ui.icons.withStrokeCurrent')
    		? 'wb-stroke-current'
    		: '') + (classes ? ' ' + classes + ' ' : '') + '" ';
    	};

    	const expandSize = size => {
    		size = sizes[size] ? sizes[size] : size;
    		return ` width="${size}" height="${size}" `;
    	};

    	const expandAlt = (title, alt) => {
    		return alt || title || null;
    	};

    	const expandTitle = (title, alt) => {
    		return title || alt || null;
    	};

    	const writable_props = [
    		'classes',
    		'wrapperClasses',
    		'title',
    		'alt',
    		'size',
    		'icon',
    		'withFly',
    		'flyX',
    		'flyY',
    		'listenToClicks',
    		'disabled',
    		'tabbable',
    		'ariaExpanded',
    		'ariaOwns'
    	];

    	Object.keys($$props).forEach(key => {
    		if (!~writable_props.indexOf(key) && key.slice(0, 2) !== '$$' && key !== 'slot') console.warn(`<Icon> was created with unknown prop '${key}'`);
    	});

    	function click_handler(event) {
    		bubble.call(this, $$self, event);
    	}

    	function click_handler_1(event) {
    		bubble.call(this, $$self, event);
    	}

    	$$self.$$set = $$props => {
    		if ('classes' in $$props) $$invalidate(13, classes = $$props.classes);
    		if ('wrapperClasses' in $$props) $$invalidate(0, wrapperClasses = $$props.wrapperClasses);
    		if ('title' in $$props) $$invalidate(14, title = $$props.title);
    		if ('alt' in $$props) $$invalidate(15, alt = $$props.alt);
    		if ('size' in $$props) $$invalidate(16, size = $$props.size);
    		if ('icon' in $$props) $$invalidate(17, icon = $$props.icon);
    		if ('withFly' in $$props) $$invalidate(1, withFly = $$props.withFly);
    		if ('flyX' in $$props) $$invalidate(2, flyX = $$props.flyX);
    		if ('flyY' in $$props) $$invalidate(3, flyY = $$props.flyY);
    		if ('listenToClicks' in $$props) $$invalidate(4, listenToClicks = $$props.listenToClicks);
    		if ('disabled' in $$props) $$invalidate(5, disabled = $$props.disabled);
    		if ('tabbable' in $$props) $$invalidate(6, tabbable = $$props.tabbable);
    		if ('ariaExpanded' in $$props) $$invalidate(7, ariaExpanded = $$props.ariaExpanded);
    		if ('ariaOwns' in $$props) $$invalidate(8, ariaOwns = $$props.ariaOwns);
    		if ('$$scope' in $$props) $$invalidate(18, $$scope = $$props.$$scope);
    	};

    	$$self.$capture_state = () => ({
    		classes,
    		wrapperClasses,
    		title,
    		alt,
    		size,
    		icon,
    		withFly,
    		flyX,
    		flyY,
    		listenToClicks,
    		disabled,
    		tabbable,
    		ariaExpanded,
    		ariaOwns,
    		getContext,
    		fly,
    		isEnterOrSpaceKey,
    		sizes,
    		keydownHandler,
    		features,
    		expandClasses,
    		expandSize,
    		expandAlt,
    		expandTitle,
    		expandedTitle,
    		expandedAlt,
    		processedIcon
    	});

    	$$self.$inject_state = $$props => {
    		if ('classes' in $$props) $$invalidate(13, classes = $$props.classes);
    		if ('wrapperClasses' in $$props) $$invalidate(0, wrapperClasses = $$props.wrapperClasses);
    		if ('title' in $$props) $$invalidate(14, title = $$props.title);
    		if ('alt' in $$props) $$invalidate(15, alt = $$props.alt);
    		if ('size' in $$props) $$invalidate(16, size = $$props.size);
    		if ('icon' in $$props) $$invalidate(17, icon = $$props.icon);
    		if ('withFly' in $$props) $$invalidate(1, withFly = $$props.withFly);
    		if ('flyX' in $$props) $$invalidate(2, flyX = $$props.flyX);
    		if ('flyY' in $$props) $$invalidate(3, flyY = $$props.flyY);
    		if ('listenToClicks' in $$props) $$invalidate(4, listenToClicks = $$props.listenToClicks);
    		if ('disabled' in $$props) $$invalidate(5, disabled = $$props.disabled);
    		if ('tabbable' in $$props) $$invalidate(6, tabbable = $$props.tabbable);
    		if ('ariaExpanded' in $$props) $$invalidate(7, ariaExpanded = $$props.ariaExpanded);
    		if ('ariaOwns' in $$props) $$invalidate(8, ariaOwns = $$props.ariaOwns);
    		if ('expandedTitle' in $$props) $$invalidate(9, expandedTitle = $$props.expandedTitle);
    		if ('expandedAlt' in $$props) $$invalidate(10, expandedAlt = $$props.expandedAlt);
    		if ('processedIcon' in $$props) $$invalidate(11, processedIcon = $$props.processedIcon);
    	};

    	if ($$props && "$$inject" in $$props) {
    		$$self.$inject_state($$props.$$inject);
    	}

    	$$self.$$.update = () => {
    		if ($$self.$$.dirty & /*icon, classes, size*/ 204800) {
    			$$invalidate(11, processedIcon = icon.replace('<svg ', '<svg ' + expandClasses(classes) + expandSize(size)));
    		}

    		if ($$self.$$.dirty & /*title, alt*/ 49152) {
    			$$invalidate(10, expandedAlt = expandAlt(title, alt));
    		}

    		if ($$self.$$.dirty & /*title, alt*/ 49152) {
    			$$invalidate(9, expandedTitle = expandTitle(title, alt));
    		}
    	};

    	return [
    		wrapperClasses,
    		withFly,
    		flyX,
    		flyY,
    		listenToClicks,
    		disabled,
    		tabbable,
    		ariaExpanded,
    		ariaOwns,
    		expandedTitle,
    		expandedAlt,
    		processedIcon,
    		keydownHandler,
    		classes,
    		title,
    		alt,
    		size,
    		icon,
    		$$scope,
    		slots,
    		click_handler,
    		click_handler_1
    	];
    }

    class Icon extends SvelteComponentDev {
    	constructor(options) {
    		super(options);

    		init(this, options, instance$p, create_fragment$p, safe_not_equal, {
    			classes: 13,
    			wrapperClasses: 0,
    			title: 14,
    			alt: 15,
    			size: 16,
    			icon: 17,
    			withFly: 1,
    			flyX: 2,
    			flyY: 3,
    			listenToClicks: 4,
    			disabled: 5,
    			tabbable: 6,
    			ariaExpanded: 7,
    			ariaOwns: 8
    		});

    		dispatch_dev("SvelteRegisterComponent", {
    			component: this,
    			tagName: "Icon",
    			options,
    			id: create_fragment$p.name
    		});
    	}

    	get classes() {
    		throw new Error("<Icon>: Props cannot be read directly from the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	set classes(value) {
    		throw new Error("<Icon>: Props cannot be set directly on the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	get wrapperClasses() {
    		throw new Error("<Icon>: Props cannot be read directly from the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	set wrapperClasses(value) {
    		throw new Error("<Icon>: Props cannot be set directly on the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	get title() {
    		throw new Error("<Icon>: Props cannot be read directly from the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	set title(value) {
    		throw new Error("<Icon>: Props cannot be set directly on the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	get alt() {
    		throw new Error("<Icon>: Props cannot be read directly from the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	set alt(value) {
    		throw new Error("<Icon>: Props cannot be set directly on the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	get size() {
    		throw new Error("<Icon>: Props cannot be read directly from the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	set size(value) {
    		throw new Error("<Icon>: Props cannot be set directly on the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	get icon() {
    		throw new Error("<Icon>: Props cannot be read directly from the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	set icon(value) {
    		throw new Error("<Icon>: Props cannot be set directly on the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	get withFly() {
    		throw new Error("<Icon>: Props cannot be read directly from the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	set withFly(value) {
    		throw new Error("<Icon>: Props cannot be set directly on the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	get flyX() {
    		throw new Error("<Icon>: Props cannot be read directly from the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	set flyX(value) {
    		throw new Error("<Icon>: Props cannot be set directly on the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	get flyY() {
    		throw new Error("<Icon>: Props cannot be read directly from the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	set flyY(value) {
    		throw new Error("<Icon>: Props cannot be set directly on the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	get listenToClicks() {
    		throw new Error("<Icon>: Props cannot be read directly from the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	set listenToClicks(value) {
    		throw new Error("<Icon>: Props cannot be set directly on the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	get disabled() {
    		throw new Error("<Icon>: Props cannot be read directly from the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	set disabled(value) {
    		throw new Error("<Icon>: Props cannot be set directly on the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	get tabbable() {
    		throw new Error("<Icon>: Props cannot be read directly from the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	set tabbable(value) {
    		throw new Error("<Icon>: Props cannot be set directly on the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	get ariaExpanded() {
    		throw new Error("<Icon>: Props cannot be read directly from the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	set ariaExpanded(value) {
    		throw new Error("<Icon>: Props cannot be set directly on the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	get ariaOwns() {
    		throw new Error("<Icon>: Props cannot be read directly from the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	set ariaOwns(value) {
    		throw new Error("<Icon>: Props cannot be set directly on the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}
    }

    // Unique ID creation requires a high quality random # generator. In the browser we therefore
    // require the crypto API and do not support built-in fallback to lower quality random number
    // generators (like Math.random()).
    var getRandomValues;
    var rnds8 = new Uint8Array(16);
    function rng() {
      // lazy load so that environments that need to polyfill have a chance to do so
      if (!getRandomValues) {
        // getRandomValues needs to be invoked in a context where "this" is a Crypto implementation. Also,
        // find the complete implementation of crypto (msCrypto) on IE11.
        getRandomValues = typeof crypto !== 'undefined' && crypto.getRandomValues && crypto.getRandomValues.bind(crypto) || typeof msCrypto !== 'undefined' && typeof msCrypto.getRandomValues === 'function' && msCrypto.getRandomValues.bind(msCrypto);

        if (!getRandomValues) {
          throw new Error('crypto.getRandomValues() not supported. See https://github.com/uuidjs/uuid#getrandomvalues-not-supported');
        }
      }

      return getRandomValues(rnds8);
    }

    var REGEX = /^(?:[0-9a-f]{8}-[0-9a-f]{4}-[1-5][0-9a-f]{3}-[89ab][0-9a-f]{3}-[0-9a-f]{12}|00000000-0000-0000-0000-000000000000)$/i;

    function validate(uuid) {
      return typeof uuid === 'string' && REGEX.test(uuid);
    }

    /**
     * Convert array of 16 byte values to UUID string format of the form:
     * XXXXXXXX-XXXX-XXXX-XXXX-XXXXXXXXXXXX
     */

    var byteToHex = [];

    for (var i = 0; i < 256; ++i) {
      byteToHex.push((i + 0x100).toString(16).substr(1));
    }

    function stringify(arr) {
      var offset = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : 0;
      // Note: Be careful editing this code!  It's been tuned for performance
      // and works in ways you may not expect. See https://github.com/uuidjs/uuid/pull/434
      var uuid = (byteToHex[arr[offset + 0]] + byteToHex[arr[offset + 1]] + byteToHex[arr[offset + 2]] + byteToHex[arr[offset + 3]] + '-' + byteToHex[arr[offset + 4]] + byteToHex[arr[offset + 5]] + '-' + byteToHex[arr[offset + 6]] + byteToHex[arr[offset + 7]] + '-' + byteToHex[arr[offset + 8]] + byteToHex[arr[offset + 9]] + '-' + byteToHex[arr[offset + 10]] + byteToHex[arr[offset + 11]] + byteToHex[arr[offset + 12]] + byteToHex[arr[offset + 13]] + byteToHex[arr[offset + 14]] + byteToHex[arr[offset + 15]]).toLowerCase(); // Consistency check for valid UUID.  If this throws, it's likely due to one
      // of the following:
      // - One or more input array values don't map to a hex octet (leading to
      // "undefined" in the uuid)
      // - Invalid input values for the RFC `version` or `variant` fields

      if (!validate(uuid)) {
        throw TypeError('Stringified UUID is invalid');
      }

      return uuid;
    }

    function v4(options, buf, offset) {
      options = options || {};
      var rnds = options.random || (options.rng || rng)(); // Per 4.4, set bits for version and `clock_seq_hi_and_reserved`

      rnds[6] = rnds[6] & 0x0f | 0x40;
      rnds[8] = rnds[8] & 0x3f | 0x80; // Copy bytes to buffer, if provided

      if (buf) {
        offset = offset || 0;

        for (var i = 0; i < 16; ++i) {
          buf[offset + i] = rnds[i];
        }

        return buf;
      }

      return stringify(rnds);
    }

    /* node_modules\@weeblr\svelte\components\forms\Button.svelte generated by Svelte v3.41.0 */
    const file$n = "node_modules\\@weeblr\\svelte\\components\\forms\\Button.svelte";

    // (64:8) {#if withIcon && iconBefore}
    function create_if_block_3$5(ctx) {
    	let icon;
    	let current;

    	icon = new Icon({
    			props: {
    				classes: "wb-flex-shrink-0 " + (/*text*/ ctx[3] ? 'wb-mr-2' : '') + " wb-max-w-none",
    				icon: /*withIcon*/ ctx[15],
    				size: /*iconSize*/ ctx[18]
    			},
    			$$inline: true
    		});

    	const block = {
    		c: function create() {
    			create_component(icon.$$.fragment);
    		},
    		m: function mount(target, anchor) {
    			mount_component(icon, target, anchor);
    			current = true;
    		},
    		p: function update(ctx, dirty) {
    			const icon_changes = {};
    			if (dirty[0] & /*text*/ 8) icon_changes.classes = "wb-flex-shrink-0 " + (/*text*/ ctx[3] ? 'wb-mr-2' : '') + " wb-max-w-none";
    			if (dirty[0] & /*withIcon*/ 32768) icon_changes.icon = /*withIcon*/ ctx[15];
    			if (dirty[0] & /*iconSize*/ 262144) icon_changes.size = /*iconSize*/ ctx[18];
    			icon.$set(icon_changes);
    		},
    		i: function intro(local) {
    			if (current) return;
    			transition_in(icon.$$.fragment, local);
    			current = true;
    		},
    		o: function outro(local) {
    			transition_out(icon.$$.fragment, local);
    			current = false;
    		},
    		d: function destroy(detaching) {
    			destroy_component(icon, detaching);
    		}
    	};

    	dispatch_dev("SvelteRegisterBlock", {
    		block,
    		id: create_if_block_3$5.name,
    		type: "if",
    		source: "(64:8) {#if withIcon && iconBefore}",
    		ctx
    	});

    	return block;
    }

    // (68:8) {#if text}
    function create_if_block_2$b(ctx) {
    	let span;
    	let t;

    	const block = {
    		c: function create() {
    			span = element("span");
    			t = text(/*text*/ ctx[3]);
    			attr_dev(span, "class", /*textSize*/ ctx[4]);
    			add_location(span, file$n, 68, 12, 2636);
    		},
    		m: function mount(target, anchor) {
    			insert_dev(target, span, anchor);
    			append_dev(span, t);
    		},
    		p: function update(ctx, dirty) {
    			if (dirty[0] & /*text*/ 8) set_data_dev(t, /*text*/ ctx[3]);

    			if (dirty[0] & /*textSize*/ 16) {
    				attr_dev(span, "class", /*textSize*/ ctx[4]);
    			}
    		},
    		d: function destroy(detaching) {
    			if (detaching) detach_dev(span);
    		}
    	};

    	dispatch_dev("SvelteRegisterBlock", {
    		block,
    		id: create_if_block_2$b.name,
    		type: "if",
    		source: "(68:8) {#if text}",
    		ctx
    	});

    	return block;
    }

    // (74:42) 
    function create_if_block_1$d(ctx) {
    	let icon;
    	let current;

    	icon = new Icon({
    			props: {
    				classes: "wb-flex-shrink-0 " + (/*text*/ ctx[3] ? 'wb-ml-2' : '') + " wb-max-w-none",
    				icon: /*withIcon*/ ctx[15],
    				size: /*iconSize*/ ctx[18]
    			},
    			$$inline: true
    		});

    	const block = {
    		c: function create() {
    			create_component(icon.$$.fragment);
    		},
    		m: function mount(target, anchor) {
    			mount_component(icon, target, anchor);
    			current = true;
    		},
    		p: function update(ctx, dirty) {
    			const icon_changes = {};
    			if (dirty[0] & /*text*/ 8) icon_changes.classes = "wb-flex-shrink-0 " + (/*text*/ ctx[3] ? 'wb-ml-2' : '') + " wb-max-w-none";
    			if (dirty[0] & /*withIcon*/ 32768) icon_changes.icon = /*withIcon*/ ctx[15];
    			if (dirty[0] & /*iconSize*/ 262144) icon_changes.size = /*iconSize*/ ctx[18];
    			icon.$set(icon_changes);
    		},
    		i: function intro(local) {
    			if (current) return;
    			transition_in(icon.$$.fragment, local);
    			current = true;
    		},
    		o: function outro(local) {
    			transition_out(icon.$$.fragment, local);
    			current = false;
    		},
    		d: function destroy(detaching) {
    			destroy_component(icon, detaching);
    		}
    	};

    	dispatch_dev("SvelteRegisterBlock", {
    		block,
    		id: create_if_block_1$d.name,
    		type: "if",
    		source: "(74:42) ",
    		ctx
    	});

    	return block;
    }

    // (71:8) {#if withCloseIcon}
    function create_if_block$h(ctx) {
    	let icon;
    	let current;

    	icon = new Icon({
    			props: {
    				classes: "wb-flex-shrink-0 " + (/*text*/ ctx[3] ? 'wb-ml-2' : '') + " wb-max-w-none",
    				icon: iconClose,
    				size: /*iconSize*/ ctx[18]
    			},
    			$$inline: true
    		});

    	const block = {
    		c: function create() {
    			create_component(icon.$$.fragment);
    		},
    		m: function mount(target, anchor) {
    			mount_component(icon, target, anchor);
    			current = true;
    		},
    		p: function update(ctx, dirty) {
    			const icon_changes = {};
    			if (dirty[0] & /*text*/ 8) icon_changes.classes = "wb-flex-shrink-0 " + (/*text*/ ctx[3] ? 'wb-ml-2' : '') + " wb-max-w-none";
    			if (dirty[0] & /*iconSize*/ 262144) icon_changes.size = /*iconSize*/ ctx[18];
    			icon.$set(icon_changes);
    		},
    		i: function intro(local) {
    			if (current) return;
    			transition_in(icon.$$.fragment, local);
    			current = true;
    		},
    		o: function outro(local) {
    			transition_out(icon.$$.fragment, local);
    			current = false;
    		},
    		d: function destroy(detaching) {
    			destroy_component(icon, detaching);
    		}
    	};

    	dispatch_dev("SvelteRegisterBlock", {
    		block,
    		id: create_if_block$h.name,
    		type: "if",
    		source: "(71:8) {#if withCloseIcon}",
    		ctx
    	});

    	return block;
    }

    // (49:0) <Fieldset legend="{label}" textSize="{labelSize}" legendBefore="{labelBefore}" classes="wb-inline-block wb-border-0">
    function create_default_slot$5(ctx) {
    	let button;
    	let t0;
    	let t1;
    	let current_block_type_index;
    	let if_block2;
    	let t2;
    	let button_class_value;
    	let button_title_value;
    	let button_aria_label_value;
    	let button_aria_describedby_value;
    	let button_aria_owns_value;
    	let button_aria_haspopup_value;
    	let button_aria_expanded_value;
    	let current;
    	let mounted;
    	let dispose;
    	let if_block0 = /*withIcon*/ ctx[15] && /*iconBefore*/ ctx[16] && create_if_block_3$5(ctx);
    	let if_block1 = /*text*/ ctx[3] && create_if_block_2$b(ctx);
    	const if_block_creators = [create_if_block$h, create_if_block_1$d];
    	const if_blocks = [];

    	function select_block_type(ctx, dirty) {
    		if (/*withCloseIcon*/ ctx[17]) return 0;
    		if (/*withIcon*/ ctx[15] && !/*iconBefore*/ ctx[16]) return 1;
    		return -1;
    	}

    	if (~(current_block_type_index = select_block_type(ctx))) {
    		if_block2 = if_blocks[current_block_type_index] = if_block_creators[current_block_type_index](ctx);
    	}

    	const default_slot_template = /*#slots*/ ctx[30].default;
    	const default_slot = create_slot(default_slot_template, ctx, /*$$scope*/ ctx[31], null);

    	const block = {
    		c: function create() {
    			button = element("button");
    			if (if_block0) if_block0.c();
    			t0 = space();
    			if (if_block1) if_block1.c();
    			t1 = space();
    			if (if_block2) if_block2.c();
    			t2 = space();
    			if (default_slot) default_slot.c();
    			attr_dev(button, "id", /*id*/ ctx[0]);
    			attr_dev(button, "type", "button");

    			attr_dev(button, "class", button_class_value = "" + ((/*size*/ ctx[1] ? 'wb-h-' + /*size*/ ctx[1] : '') + " " + (/*minW*/ ctx[2] ? 'wb-min-w-' + /*minW*/ ctx[2] : '') + " " + /*displayClasses*/ ctx[20] + " wb-rounded " + (/*disabled*/ ctx[13]
    			? /*disabledClasses*/ ctx[25]
    			: /*enabledClasses*/ ctx[24]) + " " + (/*active*/ ctx[21]
    			? /*activeClasses*/ ctx[22]
    			: /*inactiveClasses*/ ctx[23]) + " " + /*classes*/ ctx[19]));

    			attr_dev(button, "tabindex", /*tabindex*/ ctx[26]);
    			attr_dev(button, "aria-disabled", /*disabled*/ ctx[13]);
    			attr_dev(button, "data-focusdefault", /*focusDefault*/ ctx[14]);
    			attr_dev(button, "title", button_title_value = /*title*/ ctx[5] || /*label*/ ctx[6] || /*ariaLabel*/ ctx[9]);
    			attr_dev(button, "aria-label", button_aria_label_value = /*ariaLabel*/ ctx[9] || /*label*/ ctx[6] || /*title*/ ctx[5] || null);
    			attr_dev(button, "aria-describedby", button_aria_describedby_value = /*ariaDescribedBy*/ ctx[10] || null);

    			attr_dev(button, "aria-owns", button_aria_owns_value = /*ariaExpanded*/ ctx[11]
    			? /*ariaOwns*/ ctx[12] || null
    			: null);

    			attr_dev(button, "aria-haspopup", button_aria_haspopup_value = !!/*ariaOwns*/ ctx[12]);

    			attr_dev(button, "aria-expanded", button_aria_expanded_value = null === /*ariaExpanded*/ ctx[11]
    			? null
    			: !!/*ariaExpanded*/ ctx[11]);

    			add_location(button, file$n, 49, 4, 1646);
    		},
    		m: function mount(target, anchor) {
    			insert_dev(target, button, anchor);
    			if (if_block0) if_block0.m(button, null);
    			append_dev(button, t0);
    			if (if_block1) if_block1.m(button, null);
    			append_dev(button, t1);

    			if (~current_block_type_index) {
    				if_blocks[current_block_type_index].m(button, null);
    			}

    			append_dev(button, t2);

    			if (default_slot) {
    				default_slot.m(button, null);
    			}

    			current = true;

    			if (!mounted) {
    				dispose = listen_dev(button, "click", stop_propagation(prevent_default(/*handlerWrapper*/ ctx[27])), false, true, true);
    				mounted = true;
    			}
    		},
    		p: function update(ctx, dirty) {
    			if (/*withIcon*/ ctx[15] && /*iconBefore*/ ctx[16]) {
    				if (if_block0) {
    					if_block0.p(ctx, dirty);

    					if (dirty[0] & /*withIcon, iconBefore*/ 98304) {
    						transition_in(if_block0, 1);
    					}
    				} else {
    					if_block0 = create_if_block_3$5(ctx);
    					if_block0.c();
    					transition_in(if_block0, 1);
    					if_block0.m(button, t0);
    				}
    			} else if (if_block0) {
    				group_outros();

    				transition_out(if_block0, 1, 1, () => {
    					if_block0 = null;
    				});

    				check_outros();
    			}

    			if (/*text*/ ctx[3]) {
    				if (if_block1) {
    					if_block1.p(ctx, dirty);
    				} else {
    					if_block1 = create_if_block_2$b(ctx);
    					if_block1.c();
    					if_block1.m(button, t1);
    				}
    			} else if (if_block1) {
    				if_block1.d(1);
    				if_block1 = null;
    			}

    			let previous_block_index = current_block_type_index;
    			current_block_type_index = select_block_type(ctx);

    			if (current_block_type_index === previous_block_index) {
    				if (~current_block_type_index) {
    					if_blocks[current_block_type_index].p(ctx, dirty);
    				}
    			} else {
    				if (if_block2) {
    					group_outros();

    					transition_out(if_blocks[previous_block_index], 1, 1, () => {
    						if_blocks[previous_block_index] = null;
    					});

    					check_outros();
    				}

    				if (~current_block_type_index) {
    					if_block2 = if_blocks[current_block_type_index];

    					if (!if_block2) {
    						if_block2 = if_blocks[current_block_type_index] = if_block_creators[current_block_type_index](ctx);
    						if_block2.c();
    					} else {
    						if_block2.p(ctx, dirty);
    					}

    					transition_in(if_block2, 1);
    					if_block2.m(button, t2);
    				} else {
    					if_block2 = null;
    				}
    			}

    			if (default_slot) {
    				if (default_slot.p && (!current || dirty[1] & /*$$scope*/ 1)) {
    					update_slot_base(
    						default_slot,
    						default_slot_template,
    						ctx,
    						/*$$scope*/ ctx[31],
    						!current
    						? get_all_dirty_from_scope(/*$$scope*/ ctx[31])
    						: get_slot_changes(default_slot_template, /*$$scope*/ ctx[31], dirty, null),
    						null
    					);
    				}
    			}

    			if (!current || dirty[0] & /*id*/ 1) {
    				attr_dev(button, "id", /*id*/ ctx[0]);
    			}

    			if (!current || dirty[0] & /*size, minW, displayClasses, disabled, disabledClasses, enabledClasses, active, activeClasses, inactiveClasses, classes*/ 66592774 && button_class_value !== (button_class_value = "" + ((/*size*/ ctx[1] ? 'wb-h-' + /*size*/ ctx[1] : '') + " " + (/*minW*/ ctx[2] ? 'wb-min-w-' + /*minW*/ ctx[2] : '') + " " + /*displayClasses*/ ctx[20] + " wb-rounded " + (/*disabled*/ ctx[13]
    			? /*disabledClasses*/ ctx[25]
    			: /*enabledClasses*/ ctx[24]) + " " + (/*active*/ ctx[21]
    			? /*activeClasses*/ ctx[22]
    			: /*inactiveClasses*/ ctx[23]) + " " + /*classes*/ ctx[19]))) {
    				attr_dev(button, "class", button_class_value);
    			}

    			if (!current || dirty[0] & /*tabindex*/ 67108864) {
    				attr_dev(button, "tabindex", /*tabindex*/ ctx[26]);
    			}

    			if (!current || dirty[0] & /*disabled*/ 8192) {
    				attr_dev(button, "aria-disabled", /*disabled*/ ctx[13]);
    			}

    			if (!current || dirty[0] & /*focusDefault*/ 16384) {
    				attr_dev(button, "data-focusdefault", /*focusDefault*/ ctx[14]);
    			}

    			if (!current || dirty[0] & /*title, label, ariaLabel*/ 608 && button_title_value !== (button_title_value = /*title*/ ctx[5] || /*label*/ ctx[6] || /*ariaLabel*/ ctx[9])) {
    				attr_dev(button, "title", button_title_value);
    			}

    			if (!current || dirty[0] & /*ariaLabel, label, title*/ 608 && button_aria_label_value !== (button_aria_label_value = /*ariaLabel*/ ctx[9] || /*label*/ ctx[6] || /*title*/ ctx[5] || null)) {
    				attr_dev(button, "aria-label", button_aria_label_value);
    			}

    			if (!current || dirty[0] & /*ariaDescribedBy*/ 1024 && button_aria_describedby_value !== (button_aria_describedby_value = /*ariaDescribedBy*/ ctx[10] || null)) {
    				attr_dev(button, "aria-describedby", button_aria_describedby_value);
    			}

    			if (!current || dirty[0] & /*ariaExpanded, ariaOwns*/ 6144 && button_aria_owns_value !== (button_aria_owns_value = /*ariaExpanded*/ ctx[11]
    			? /*ariaOwns*/ ctx[12] || null
    			: null)) {
    				attr_dev(button, "aria-owns", button_aria_owns_value);
    			}

    			if (!current || dirty[0] & /*ariaOwns*/ 4096 && button_aria_haspopup_value !== (button_aria_haspopup_value = !!/*ariaOwns*/ ctx[12])) {
    				attr_dev(button, "aria-haspopup", button_aria_haspopup_value);
    			}

    			if (!current || dirty[0] & /*ariaExpanded*/ 2048 && button_aria_expanded_value !== (button_aria_expanded_value = null === /*ariaExpanded*/ ctx[11]
    			? null
    			: !!/*ariaExpanded*/ ctx[11])) {
    				attr_dev(button, "aria-expanded", button_aria_expanded_value);
    			}
    		},
    		i: function intro(local) {
    			if (current) return;
    			transition_in(if_block0);
    			transition_in(if_block2);
    			transition_in(default_slot, local);
    			current = true;
    		},
    		o: function outro(local) {
    			transition_out(if_block0);
    			transition_out(if_block2);
    			transition_out(default_slot, local);
    			current = false;
    		},
    		d: function destroy(detaching) {
    			if (detaching) detach_dev(button);
    			if (if_block0) if_block0.d();
    			if (if_block1) if_block1.d();

    			if (~current_block_type_index) {
    				if_blocks[current_block_type_index].d();
    			}

    			if (default_slot) default_slot.d(detaching);
    			mounted = false;
    			dispose();
    		}
    	};

    	dispatch_dev("SvelteRegisterBlock", {
    		block,
    		id: create_default_slot$5.name,
    		type: "slot",
    		source: "(49:0) <Fieldset legend=\\\"{label}\\\" textSize=\\\"{labelSize}\\\" legendBefore=\\\"{labelBefore}\\\" classes=\\\"wb-inline-block wb-border-0\\\">",
    		ctx
    	});

    	return block;
    }

    function create_fragment$o(ctx) {
    	let fieldset;
    	let current;

    	fieldset = new Fieldset({
    			props: {
    				legend: /*label*/ ctx[6],
    				textSize: /*labelSize*/ ctx[7],
    				legendBefore: /*labelBefore*/ ctx[8],
    				classes: "wb-inline-block wb-border-0",
    				$$slots: { default: [create_default_slot$5] },
    				$$scope: { ctx }
    			},
    			$$inline: true
    		});

    	const block = {
    		c: function create() {
    			create_component(fieldset.$$.fragment);
    		},
    		l: function claim(nodes) {
    			throw new Error("options.hydrate only works if the component was compiled with the `hydratable: true` option");
    		},
    		m: function mount(target, anchor) {
    			mount_component(fieldset, target, anchor);
    			current = true;
    		},
    		p: function update(ctx, dirty) {
    			const fieldset_changes = {};
    			if (dirty[0] & /*label*/ 64) fieldset_changes.legend = /*label*/ ctx[6];
    			if (dirty[0] & /*labelSize*/ 128) fieldset_changes.textSize = /*labelSize*/ ctx[7];
    			if (dirty[0] & /*labelBefore*/ 256) fieldset_changes.legendBefore = /*labelBefore*/ ctx[8];

    			if (dirty[0] & /*id, size, minW, displayClasses, disabled, disabledClasses, enabledClasses, active, activeClasses, inactiveClasses, classes, tabindex, focusDefault, title, label, ariaLabel, ariaDescribedBy, ariaExpanded, ariaOwns, text, iconSize, withCloseIcon, withIcon, iconBefore, textSize*/ 134217343 | dirty[1] & /*$$scope*/ 1) {
    				fieldset_changes.$$scope = { dirty, ctx };
    			}

    			fieldset.$set(fieldset_changes);
    		},
    		i: function intro(local) {
    			if (current) return;
    			transition_in(fieldset.$$.fragment, local);
    			current = true;
    		},
    		o: function outro(local) {
    			transition_out(fieldset.$$.fragment, local);
    			current = false;
    		},
    		d: function destroy(detaching) {
    			destroy_component(fieldset, detaching);
    		}
    	};

    	dispatch_dev("SvelteRegisterBlock", {
    		block,
    		id: create_fragment$o.name,
    		type: "component",
    		source: "",
    		ctx
    	});

    	return block;
    }

    function instance$o($$self, $$props, $$invalidate) {
    	let { $$slots: slots = {}, $$scope } = $$props;
    	validate_slots('Button', slots, ['default']);
    	let { id = '' } = $$props;
    	let { size = '8' } = $$props;
    	let { minW = '' } = $$props;
    	let { text = '' } = $$props;
    	let { textSize = 'wb-text-base' } = $$props;
    	let { title = '' } = $$props;
    	let { label = '' } = $$props;
    	let { labelSize = 'wb-text-base' } = $$props;
    	let { labelBefore = false } = $$props;
    	let { ariaLabel = '' } = $$props;
    	let { ariaDescribedBy = '' } = $$props;
    	let { ariaExpanded = null } = $$props;
    	let { ariaOwns = null } = $$props;
    	let { ariaHasPopup = null } = $$props;
    	let { disabled = false } = $$props;
    	let { focusDefault = false } = $$props;
    	let { withIcon = false } = $$props;
    	let { iconBefore = false } = $$props;
    	let { withCloseIcon = false } = $$props;
    	let { iconSize = 'l' } = $$props;
    	let { classes = 'wb-px-4 wb-py-2' } = $$props;
    	let { displayClasses = 'wb-inline-flex wb-items-center' } = $$props;
    	let { active = false } = $$props;
    	let { activeClasses = '' } = $$props;
    	let { inactiveClasses = '' } = $$props;
    	let { enabledClasses = 'wb-bg-gray-700 wb-text-gray-200 hover:wb-bg-gray-900 focus:wb-bg-gray-900 wb-c-focusable wb-cursor-pointer' } = $$props;
    	let { disabledClasses = 'wb-opacity-50 wb-bg-gray-700 wb-text-gray-200 wb-cursor-default' } = $$props;
    	let { handler = undefined } = $$props;
    	let { tabindex = 0 } = $$props;
    	id = id || v4();

    	const handlerWrapper = e => {
    		return disabled || !handler ? false : handler(e);
    	};

    	const writable_props = [
    		'id',
    		'size',
    		'minW',
    		'text',
    		'textSize',
    		'title',
    		'label',
    		'labelSize',
    		'labelBefore',
    		'ariaLabel',
    		'ariaDescribedBy',
    		'ariaExpanded',
    		'ariaOwns',
    		'ariaHasPopup',
    		'disabled',
    		'focusDefault',
    		'withIcon',
    		'iconBefore',
    		'withCloseIcon',
    		'iconSize',
    		'classes',
    		'displayClasses',
    		'active',
    		'activeClasses',
    		'inactiveClasses',
    		'enabledClasses',
    		'disabledClasses',
    		'handler',
    		'tabindex'
    	];

    	Object.keys($$props).forEach(key => {
    		if (!~writable_props.indexOf(key) && key.slice(0, 2) !== '$$' && key !== 'slot') console.warn(`<Button> was created with unknown prop '${key}'`);
    	});

    	$$self.$$set = $$props => {
    		if ('id' in $$props) $$invalidate(0, id = $$props.id);
    		if ('size' in $$props) $$invalidate(1, size = $$props.size);
    		if ('minW' in $$props) $$invalidate(2, minW = $$props.minW);
    		if ('text' in $$props) $$invalidate(3, text = $$props.text);
    		if ('textSize' in $$props) $$invalidate(4, textSize = $$props.textSize);
    		if ('title' in $$props) $$invalidate(5, title = $$props.title);
    		if ('label' in $$props) $$invalidate(6, label = $$props.label);
    		if ('labelSize' in $$props) $$invalidate(7, labelSize = $$props.labelSize);
    		if ('labelBefore' in $$props) $$invalidate(8, labelBefore = $$props.labelBefore);
    		if ('ariaLabel' in $$props) $$invalidate(9, ariaLabel = $$props.ariaLabel);
    		if ('ariaDescribedBy' in $$props) $$invalidate(10, ariaDescribedBy = $$props.ariaDescribedBy);
    		if ('ariaExpanded' in $$props) $$invalidate(11, ariaExpanded = $$props.ariaExpanded);
    		if ('ariaOwns' in $$props) $$invalidate(12, ariaOwns = $$props.ariaOwns);
    		if ('ariaHasPopup' in $$props) $$invalidate(28, ariaHasPopup = $$props.ariaHasPopup);
    		if ('disabled' in $$props) $$invalidate(13, disabled = $$props.disabled);
    		if ('focusDefault' in $$props) $$invalidate(14, focusDefault = $$props.focusDefault);
    		if ('withIcon' in $$props) $$invalidate(15, withIcon = $$props.withIcon);
    		if ('iconBefore' in $$props) $$invalidate(16, iconBefore = $$props.iconBefore);
    		if ('withCloseIcon' in $$props) $$invalidate(17, withCloseIcon = $$props.withCloseIcon);
    		if ('iconSize' in $$props) $$invalidate(18, iconSize = $$props.iconSize);
    		if ('classes' in $$props) $$invalidate(19, classes = $$props.classes);
    		if ('displayClasses' in $$props) $$invalidate(20, displayClasses = $$props.displayClasses);
    		if ('active' in $$props) $$invalidate(21, active = $$props.active);
    		if ('activeClasses' in $$props) $$invalidate(22, activeClasses = $$props.activeClasses);
    		if ('inactiveClasses' in $$props) $$invalidate(23, inactiveClasses = $$props.inactiveClasses);
    		if ('enabledClasses' in $$props) $$invalidate(24, enabledClasses = $$props.enabledClasses);
    		if ('disabledClasses' in $$props) $$invalidate(25, disabledClasses = $$props.disabledClasses);
    		if ('handler' in $$props) $$invalidate(29, handler = $$props.handler);
    		if ('tabindex' in $$props) $$invalidate(26, tabindex = $$props.tabindex);
    		if ('$$scope' in $$props) $$invalidate(31, $$scope = $$props.$$scope);
    	};

    	$$self.$capture_state = () => ({
    		id,
    		size,
    		minW,
    		text,
    		textSize,
    		title,
    		label,
    		labelSize,
    		labelBefore,
    		ariaLabel,
    		ariaDescribedBy,
    		ariaExpanded,
    		ariaOwns,
    		ariaHasPopup,
    		disabled,
    		focusDefault,
    		withIcon,
    		iconBefore,
    		withCloseIcon,
    		iconSize,
    		classes,
    		displayClasses,
    		active,
    		activeClasses,
    		inactiveClasses,
    		enabledClasses,
    		disabledClasses,
    		handler,
    		tabindex,
    		Fieldset,
    		Icon,
    		iconClose,
    		uuidv4: v4,
    		handlerWrapper
    	});

    	$$self.$inject_state = $$props => {
    		if ('id' in $$props) $$invalidate(0, id = $$props.id);
    		if ('size' in $$props) $$invalidate(1, size = $$props.size);
    		if ('minW' in $$props) $$invalidate(2, minW = $$props.minW);
    		if ('text' in $$props) $$invalidate(3, text = $$props.text);
    		if ('textSize' in $$props) $$invalidate(4, textSize = $$props.textSize);
    		if ('title' in $$props) $$invalidate(5, title = $$props.title);
    		if ('label' in $$props) $$invalidate(6, label = $$props.label);
    		if ('labelSize' in $$props) $$invalidate(7, labelSize = $$props.labelSize);
    		if ('labelBefore' in $$props) $$invalidate(8, labelBefore = $$props.labelBefore);
    		if ('ariaLabel' in $$props) $$invalidate(9, ariaLabel = $$props.ariaLabel);
    		if ('ariaDescribedBy' in $$props) $$invalidate(10, ariaDescribedBy = $$props.ariaDescribedBy);
    		if ('ariaExpanded' in $$props) $$invalidate(11, ariaExpanded = $$props.ariaExpanded);
    		if ('ariaOwns' in $$props) $$invalidate(12, ariaOwns = $$props.ariaOwns);
    		if ('ariaHasPopup' in $$props) $$invalidate(28, ariaHasPopup = $$props.ariaHasPopup);
    		if ('disabled' in $$props) $$invalidate(13, disabled = $$props.disabled);
    		if ('focusDefault' in $$props) $$invalidate(14, focusDefault = $$props.focusDefault);
    		if ('withIcon' in $$props) $$invalidate(15, withIcon = $$props.withIcon);
    		if ('iconBefore' in $$props) $$invalidate(16, iconBefore = $$props.iconBefore);
    		if ('withCloseIcon' in $$props) $$invalidate(17, withCloseIcon = $$props.withCloseIcon);
    		if ('iconSize' in $$props) $$invalidate(18, iconSize = $$props.iconSize);
    		if ('classes' in $$props) $$invalidate(19, classes = $$props.classes);
    		if ('displayClasses' in $$props) $$invalidate(20, displayClasses = $$props.displayClasses);
    		if ('active' in $$props) $$invalidate(21, active = $$props.active);
    		if ('activeClasses' in $$props) $$invalidate(22, activeClasses = $$props.activeClasses);
    		if ('inactiveClasses' in $$props) $$invalidate(23, inactiveClasses = $$props.inactiveClasses);
    		if ('enabledClasses' in $$props) $$invalidate(24, enabledClasses = $$props.enabledClasses);
    		if ('disabledClasses' in $$props) $$invalidate(25, disabledClasses = $$props.disabledClasses);
    		if ('handler' in $$props) $$invalidate(29, handler = $$props.handler);
    		if ('tabindex' in $$props) $$invalidate(26, tabindex = $$props.tabindex);
    	};

    	if ($$props && "$$inject" in $$props) {
    		$$self.$inject_state($$props.$$inject);
    	}

    	return [
    		id,
    		size,
    		minW,
    		text,
    		textSize,
    		title,
    		label,
    		labelSize,
    		labelBefore,
    		ariaLabel,
    		ariaDescribedBy,
    		ariaExpanded,
    		ariaOwns,
    		disabled,
    		focusDefault,
    		withIcon,
    		iconBefore,
    		withCloseIcon,
    		iconSize,
    		classes,
    		displayClasses,
    		active,
    		activeClasses,
    		inactiveClasses,
    		enabledClasses,
    		disabledClasses,
    		tabindex,
    		handlerWrapper,
    		ariaHasPopup,
    		handler,
    		slots,
    		$$scope
    	];
    }

    class Button extends SvelteComponentDev {
    	constructor(options) {
    		super(options);

    		init(
    			this,
    			options,
    			instance$o,
    			create_fragment$o,
    			safe_not_equal,
    			{
    				id: 0,
    				size: 1,
    				minW: 2,
    				text: 3,
    				textSize: 4,
    				title: 5,
    				label: 6,
    				labelSize: 7,
    				labelBefore: 8,
    				ariaLabel: 9,
    				ariaDescribedBy: 10,
    				ariaExpanded: 11,
    				ariaOwns: 12,
    				ariaHasPopup: 28,
    				disabled: 13,
    				focusDefault: 14,
    				withIcon: 15,
    				iconBefore: 16,
    				withCloseIcon: 17,
    				iconSize: 18,
    				classes: 19,
    				displayClasses: 20,
    				active: 21,
    				activeClasses: 22,
    				inactiveClasses: 23,
    				enabledClasses: 24,
    				disabledClasses: 25,
    				handler: 29,
    				tabindex: 26
    			},
    			null,
    			[-1, -1]
    		);

    		dispatch_dev("SvelteRegisterComponent", {
    			component: this,
    			tagName: "Button",
    			options,
    			id: create_fragment$o.name
    		});
    	}

    	get id() {
    		throw new Error("<Button>: Props cannot be read directly from the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	set id(value) {
    		throw new Error("<Button>: Props cannot be set directly on the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	get size() {
    		throw new Error("<Button>: Props cannot be read directly from the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	set size(value) {
    		throw new Error("<Button>: Props cannot be set directly on the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	get minW() {
    		throw new Error("<Button>: Props cannot be read directly from the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	set minW(value) {
    		throw new Error("<Button>: Props cannot be set directly on the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	get text() {
    		throw new Error("<Button>: Props cannot be read directly from the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	set text(value) {
    		throw new Error("<Button>: Props cannot be set directly on the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	get textSize() {
    		throw new Error("<Button>: Props cannot be read directly from the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	set textSize(value) {
    		throw new Error("<Button>: Props cannot be set directly on the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	get title() {
    		throw new Error("<Button>: Props cannot be read directly from the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	set title(value) {
    		throw new Error("<Button>: Props cannot be set directly on the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	get label() {
    		throw new Error("<Button>: Props cannot be read directly from the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	set label(value) {
    		throw new Error("<Button>: Props cannot be set directly on the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	get labelSize() {
    		throw new Error("<Button>: Props cannot be read directly from the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	set labelSize(value) {
    		throw new Error("<Button>: Props cannot be set directly on the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	get labelBefore() {
    		throw new Error("<Button>: Props cannot be read directly from the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	set labelBefore(value) {
    		throw new Error("<Button>: Props cannot be set directly on the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	get ariaLabel() {
    		throw new Error("<Button>: Props cannot be read directly from the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	set ariaLabel(value) {
    		throw new Error("<Button>: Props cannot be set directly on the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	get ariaDescribedBy() {
    		throw new Error("<Button>: Props cannot be read directly from the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	set ariaDescribedBy(value) {
    		throw new Error("<Button>: Props cannot be set directly on the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	get ariaExpanded() {
    		throw new Error("<Button>: Props cannot be read directly from the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	set ariaExpanded(value) {
    		throw new Error("<Button>: Props cannot be set directly on the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	get ariaOwns() {
    		throw new Error("<Button>: Props cannot be read directly from the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	set ariaOwns(value) {
    		throw new Error("<Button>: Props cannot be set directly on the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	get ariaHasPopup() {
    		throw new Error("<Button>: Props cannot be read directly from the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	set ariaHasPopup(value) {
    		throw new Error("<Button>: Props cannot be set directly on the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	get disabled() {
    		throw new Error("<Button>: Props cannot be read directly from the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	set disabled(value) {
    		throw new Error("<Button>: Props cannot be set directly on the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	get focusDefault() {
    		throw new Error("<Button>: Props cannot be read directly from the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	set focusDefault(value) {
    		throw new Error("<Button>: Props cannot be set directly on the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	get withIcon() {
    		throw new Error("<Button>: Props cannot be read directly from the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	set withIcon(value) {
    		throw new Error("<Button>: Props cannot be set directly on the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	get iconBefore() {
    		throw new Error("<Button>: Props cannot be read directly from the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	set iconBefore(value) {
    		throw new Error("<Button>: Props cannot be set directly on the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	get withCloseIcon() {
    		throw new Error("<Button>: Props cannot be read directly from the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	set withCloseIcon(value) {
    		throw new Error("<Button>: Props cannot be set directly on the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	get iconSize() {
    		throw new Error("<Button>: Props cannot be read directly from the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	set iconSize(value) {
    		throw new Error("<Button>: Props cannot be set directly on the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	get classes() {
    		throw new Error("<Button>: Props cannot be read directly from the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	set classes(value) {
    		throw new Error("<Button>: Props cannot be set directly on the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	get displayClasses() {
    		throw new Error("<Button>: Props cannot be read directly from the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	set displayClasses(value) {
    		throw new Error("<Button>: Props cannot be set directly on the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	get active() {
    		throw new Error("<Button>: Props cannot be read directly from the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	set active(value) {
    		throw new Error("<Button>: Props cannot be set directly on the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	get activeClasses() {
    		throw new Error("<Button>: Props cannot be read directly from the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	set activeClasses(value) {
    		throw new Error("<Button>: Props cannot be set directly on the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	get inactiveClasses() {
    		throw new Error("<Button>: Props cannot be read directly from the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	set inactiveClasses(value) {
    		throw new Error("<Button>: Props cannot be set directly on the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	get enabledClasses() {
    		throw new Error("<Button>: Props cannot be read directly from the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	set enabledClasses(value) {
    		throw new Error("<Button>: Props cannot be set directly on the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	get disabledClasses() {
    		throw new Error("<Button>: Props cannot be read directly from the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	set disabledClasses(value) {
    		throw new Error("<Button>: Props cannot be set directly on the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	get handler() {
    		throw new Error("<Button>: Props cannot be read directly from the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	set handler(value) {
    		throw new Error("<Button>: Props cannot be set directly on the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	get tabindex() {
    		throw new Error("<Button>: Props cannot be read directly from the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	set tabindex(value) {
    		throw new Error("<Button>: Props cannot be set directly on the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}
    }

    /* node_modules\@weeblr\svelte\components\dashboard\StatusMessage.svelte generated by Svelte v3.41.0 */
    const file$m = "node_modules\\@weeblr\\svelte\\components\\dashboard\\StatusMessage.svelte";

    // (44:0) {#if text}
    function create_if_block$g(ctx) {
    	let div3;
    	let div0;
    	let icon_1;
    	let div0_class_value;
    	let t0;
    	let div2;
    	let div1;
    	let raw_value = t(/*text*/ ctx[0]) + "";
    	let t1;
    	let current;

    	icon_1 = new Icon({
    			props: { icon: /*icon*/ ctx[1] },
    			$$inline: true
    		});

    	const default_slot_template = /*#slots*/ ctx[7].default;
    	const default_slot = create_slot(default_slot_template, ctx, /*$$scope*/ ctx[6], null);

    	const block = {
    		c: function create() {
    			div3 = element("div");
    			div0 = element("div");
    			create_component(icon_1.$$.fragment);
    			t0 = space();
    			div2 = element("div");
    			div1 = element("div");
    			t1 = space();
    			if (default_slot) default_slot.c();
    			attr_dev(div0, "class", div0_class_value = "wb-flex-shrink-0 wb-w-6 wb-h-6 wb-mr-4 " + /*color*/ ctx[2]);
    			add_location(div0, file$m, 45, 8, 1050);
    			attr_dev(div1, "role", "status");
    			attr_dev(div1, "class", "wb-leading-6");
    			add_location(div1, file$m, 49, 12, 1224);
    			attr_dev(div2, "class", "wb-flex wb-flex-wrap wb-gap-4");
    			add_location(div2, file$m, 48, 8, 1168);
    			attr_dev(div3, "class", "wb-flex wb-items-start");
    			add_location(div3, file$m, 44, 4, 1005);
    		},
    		m: function mount(target, anchor) {
    			insert_dev(target, div3, anchor);
    			append_dev(div3, div0);
    			mount_component(icon_1, div0, null);
    			append_dev(div3, t0);
    			append_dev(div3, div2);
    			append_dev(div2, div1);
    			div1.innerHTML = raw_value;
    			append_dev(div2, t1);

    			if (default_slot) {
    				default_slot.m(div2, null);
    			}

    			current = true;
    		},
    		p: function update(ctx, dirty) {
    			const icon_1_changes = {};
    			if (dirty & /*icon*/ 2) icon_1_changes.icon = /*icon*/ ctx[1];
    			icon_1.$set(icon_1_changes);

    			if (!current || dirty & /*color*/ 4 && div0_class_value !== (div0_class_value = "wb-flex-shrink-0 wb-w-6 wb-h-6 wb-mr-4 " + /*color*/ ctx[2])) {
    				attr_dev(div0, "class", div0_class_value);
    			}

    			if ((!current || dirty & /*text*/ 1) && raw_value !== (raw_value = t(/*text*/ ctx[0]) + "")) div1.innerHTML = raw_value;
    			if (default_slot) {
    				if (default_slot.p && (!current || dirty & /*$$scope*/ 64)) {
    					update_slot_base(
    						default_slot,
    						default_slot_template,
    						ctx,
    						/*$$scope*/ ctx[6],
    						!current
    						? get_all_dirty_from_scope(/*$$scope*/ ctx[6])
    						: get_slot_changes(default_slot_template, /*$$scope*/ ctx[6], dirty, null),
    						null
    					);
    				}
    			}
    		},
    		i: function intro(local) {
    			if (current) return;
    			transition_in(icon_1.$$.fragment, local);
    			transition_in(default_slot, local);
    			current = true;
    		},
    		o: function outro(local) {
    			transition_out(icon_1.$$.fragment, local);
    			transition_out(default_slot, local);
    			current = false;
    		},
    		d: function destroy(detaching) {
    			if (detaching) detach_dev(div3);
    			destroy_component(icon_1);
    			if (default_slot) default_slot.d(detaching);
    		}
    	};

    	dispatch_dev("SvelteRegisterBlock", {
    		block,
    		id: create_if_block$g.name,
    		type: "if",
    		source: "(44:0) {#if text}",
    		ctx
    	});

    	return block;
    }

    function create_fragment$n(ctx) {
    	let if_block_anchor;
    	let current;
    	let if_block = /*text*/ ctx[0] && create_if_block$g(ctx);

    	const block = {
    		c: function create() {
    			if (if_block) if_block.c();
    			if_block_anchor = empty();
    		},
    		l: function claim(nodes) {
    			throw new Error("options.hydrate only works if the component was compiled with the `hydratable: true` option");
    		},
    		m: function mount(target, anchor) {
    			if (if_block) if_block.m(target, anchor);
    			insert_dev(target, if_block_anchor, anchor);
    			current = true;
    		},
    		p: function update(ctx, [dirty]) {
    			if (/*text*/ ctx[0]) {
    				if (if_block) {
    					if_block.p(ctx, dirty);

    					if (dirty & /*text*/ 1) {
    						transition_in(if_block, 1);
    					}
    				} else {
    					if_block = create_if_block$g(ctx);
    					if_block.c();
    					transition_in(if_block, 1);
    					if_block.m(if_block_anchor.parentNode, if_block_anchor);
    				}
    			} else if (if_block) {
    				group_outros();

    				transition_out(if_block, 1, 1, () => {
    					if_block = null;
    				});

    				check_outros();
    			}
    		},
    		i: function intro(local) {
    			if (current) return;
    			transition_in(if_block);
    			current = true;
    		},
    		o: function outro(local) {
    			transition_out(if_block);
    			current = false;
    		},
    		d: function destroy(detaching) {
    			if (if_block) if_block.d(detaching);
    			if (detaching) detach_dev(if_block_anchor);
    		}
    	};

    	dispatch_dev("SvelteRegisterBlock", {
    		block,
    		id: create_fragment$n.name,
    		type: "component",
    		source: "",
    		ctx
    	});

    	return block;
    }

    function instance$n($$self, $$props, $$invalidate) {
    	let color;
    	let icon;
    	let { $$slots: slots = {}, $$scope } = $$props;
    	validate_slots('StatusMessage', slots, ['default']);
    	let { status = 'info' } = $$props;
    	let { text = '' } = $$props;

    	const colors = {
    		ok: 'wb-text-green-500',
    		warning: 'wb-text-yellow-600',
    		danger: 'wb-text-red-500',
    		default: 'wb-text-gray-900'
    	};

    	const icons = {
    		ok: iconLight,
    		warning: iconRain,
    		danger: iconThunderstorm,
    		default: iconCheckmark
    	};

    	let { colorCustomizer = status => {
    		status = colors[status] ? status : 'default';
    		return colors[status];
    	} } = $$props;

    	let { iconCustomizer = status => {
    		status = icons[status] ? status : 'default';
    		return icons[status];
    	} } = $$props;

    	const writable_props = ['status', 'text', 'colorCustomizer', 'iconCustomizer'];

    	Object.keys($$props).forEach(key => {
    		if (!~writable_props.indexOf(key) && key.slice(0, 2) !== '$$' && key !== 'slot') console.warn(`<StatusMessage> was created with unknown prop '${key}'`);
    	});

    	$$self.$$set = $$props => {
    		if ('status' in $$props) $$invalidate(3, status = $$props.status);
    		if ('text' in $$props) $$invalidate(0, text = $$props.text);
    		if ('colorCustomizer' in $$props) $$invalidate(4, colorCustomizer = $$props.colorCustomizer);
    		if ('iconCustomizer' in $$props) $$invalidate(5, iconCustomizer = $$props.iconCustomizer);
    		if ('$$scope' in $$props) $$invalidate(6, $$scope = $$props.$$scope);
    	};

    	$$self.$capture_state = () => ({
    		status,
    		text,
    		t,
    		iconLight,
    		iconRain,
    		iconThunderstorm,
    		iconCheckmark,
    		Icon,
    		colors,
    		icons,
    		colorCustomizer,
    		iconCustomizer,
    		icon,
    		color
    	});

    	$$self.$inject_state = $$props => {
    		if ('status' in $$props) $$invalidate(3, status = $$props.status);
    		if ('text' in $$props) $$invalidate(0, text = $$props.text);
    		if ('colorCustomizer' in $$props) $$invalidate(4, colorCustomizer = $$props.colorCustomizer);
    		if ('iconCustomizer' in $$props) $$invalidate(5, iconCustomizer = $$props.iconCustomizer);
    		if ('icon' in $$props) $$invalidate(1, icon = $$props.icon);
    		if ('color' in $$props) $$invalidate(2, color = $$props.color);
    	};

    	if ($$props && "$$inject" in $$props) {
    		$$self.$inject_state($$props.$$inject);
    	}

    	$$self.$$.update = () => {
    		if ($$self.$$.dirty & /*colorCustomizer, status*/ 24) {
    			$$invalidate(2, color = colorCustomizer(status));
    		}

    		if ($$self.$$.dirty & /*iconCustomizer, status*/ 40) {
    			$$invalidate(1, icon = iconCustomizer(status));
    		}
    	};

    	return [text, icon, color, status, colorCustomizer, iconCustomizer, $$scope, slots];
    }

    class StatusMessage extends SvelteComponentDev {
    	constructor(options) {
    		super(options);

    		init(this, options, instance$n, create_fragment$n, safe_not_equal, {
    			status: 3,
    			text: 0,
    			colorCustomizer: 4,
    			iconCustomizer: 5
    		});

    		dispatch_dev("SvelteRegisterComponent", {
    			component: this,
    			tagName: "StatusMessage",
    			options,
    			id: create_fragment$n.name
    		});
    	}

    	get status() {
    		throw new Error("<StatusMessage>: Props cannot be read directly from the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	set status(value) {
    		throw new Error("<StatusMessage>: Props cannot be set directly on the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	get text() {
    		throw new Error("<StatusMessage>: Props cannot be read directly from the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	set text(value) {
    		throw new Error("<StatusMessage>: Props cannot be set directly on the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	get colorCustomizer() {
    		throw new Error("<StatusMessage>: Props cannot be read directly from the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	set colorCustomizer(value) {
    		throw new Error("<StatusMessage>: Props cannot be set directly on the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	get iconCustomizer() {
    		throw new Error("<StatusMessage>: Props cannot be read directly from the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	set iconCustomizer(value) {
    		throw new Error("<StatusMessage>: Props cannot be set directly on the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}
    }

    /* src\app\admin\components\List.svelte generated by Svelte v3.41.0 */

    const { console: console_1$2 } = globals;
    const file$l = "src\\app\\admin\\components\\List.svelte";

    function get_each_context$4(ctx, list, i) {
    	const child_ctx = ctx.slice();
    	child_ctx[21] = list[i];
    	return child_ctx;
    }

    function get_each_context_1$1(ctx, list, i) {
    	const child_ctx = ctx.slice();
    	child_ctx[24] = list[i];
    	child_ctx[26] = i;
    	return child_ctx;
    }

    // (160:0) {:else}
    function create_else_block$7(ctx) {
    	let div;
    	let statusmessage;
    	let current;

    	statusmessage = new StatusMessage({
    			props: {
    				status: "warning",
    				text: t('main.noFiles')
    			},
    			$$inline: true
    		});

    	const block = {
    		c: function create() {
    			div = element("div");
    			create_component(statusmessage.$$.fragment);
    			attr_dev(div, "class", "wb-flex wb-justify-center wb-mt-16");
    			add_location(div, file$l, 160, 4, 7193);
    		},
    		m: function mount(target, anchor) {
    			insert_dev(target, div, anchor);
    			mount_component(statusmessage, div, null);
    			current = true;
    		},
    		p: noop$1,
    		i: function intro(local) {
    			if (current) return;
    			transition_in(statusmessage.$$.fragment, local);
    			current = true;
    		},
    		o: function outro(local) {
    			transition_out(statusmessage.$$.fragment, local);
    			current = false;
    		},
    		d: function destroy(detaching) {
    			if (detaching) detach_dev(div);
    			destroy_component(statusmessage);
    		}
    	};

    	dispatch_dev("SvelteRegisterBlock", {
    		block,
    		id: create_else_block$7.name,
    		type: "else",
    		source: "(160:0) {:else}",
    		ctx
    	});

    	return block;
    }

    // (102:0) {#if $forlogsState.filteredFolders.length > 0}
    function create_if_block$f(ctx) {
    	let each_1_anchor;
    	let current;
    	let each_value = /*$forlogsState*/ ctx[2].filteredFolders;
    	validate_each_argument(each_value);
    	let each_blocks = [];

    	for (let i = 0; i < each_value.length; i += 1) {
    		each_blocks[i] = create_each_block$4(get_each_context$4(ctx, each_value, i));
    	}

    	const out = i => transition_out(each_blocks[i], 1, 1, () => {
    		each_blocks[i] = null;
    	});

    	const block = {
    		c: function create() {
    			for (let i = 0; i < each_blocks.length; i += 1) {
    				each_blocks[i].c();
    			}

    			each_1_anchor = empty();
    		},
    		m: function mount(target, anchor) {
    			for (let i = 0; i < each_blocks.length; i += 1) {
    				each_blocks[i].m(target, anchor);
    			}

    			insert_dev(target, each_1_anchor, anchor);
    			current = true;
    		},
    		p: function update(ctx, dirty) {
    			if (dirty & /*$forlogsState, unixToSiteTimeZone, config, t, iconDelete, deleteInProgress, confirmDeleteFile, iconDownload, downloadInProgress, download, iconView, features, view*/ 999) {
    				each_value = /*$forlogsState*/ ctx[2].filteredFolders;
    				validate_each_argument(each_value);
    				let i;

    				for (i = 0; i < each_value.length; i += 1) {
    					const child_ctx = get_each_context$4(ctx, each_value, i);

    					if (each_blocks[i]) {
    						each_blocks[i].p(child_ctx, dirty);
    						transition_in(each_blocks[i], 1);
    					} else {
    						each_blocks[i] = create_each_block$4(child_ctx);
    						each_blocks[i].c();
    						transition_in(each_blocks[i], 1);
    						each_blocks[i].m(each_1_anchor.parentNode, each_1_anchor);
    					}
    				}

    				group_outros();

    				for (i = each_value.length; i < each_blocks.length; i += 1) {
    					out(i);
    				}

    				check_outros();
    			}
    		},
    		i: function intro(local) {
    			if (current) return;

    			for (let i = 0; i < each_value.length; i += 1) {
    				transition_in(each_blocks[i]);
    			}

    			current = true;
    		},
    		o: function outro(local) {
    			each_blocks = each_blocks.filter(Boolean);

    			for (let i = 0; i < each_blocks.length; i += 1) {
    				transition_out(each_blocks[i]);
    			}

    			current = false;
    		},
    		d: function destroy(detaching) {
    			destroy_each(each_blocks, detaching);
    			if (detaching) detach_dev(each_1_anchor);
    		}
    	};

    	dispatch_dev("SvelteRegisterBlock", {
    		block,
    		id: create_if_block$f.name,
    		type: "if",
    		source: "(102:0) {#if $forlogsState.filteredFolders.length > 0}",
    		ctx
    	});

    	return block;
    }

    // (110:16) {#if folder}
    function create_if_block_1$c(ctx) {
    	let button;
    	let current;

    	function func() {
    		return /*func*/ ctx[10](/*folder*/ ctx[21]);
    	}

    	button = new Button({
    			props: {
    				text: t('main.deleteFolder'),
    				textSize: "wb-text-xs",
    				iconBefore: true,
    				withIcon: iconDelete,
    				iconSize: "xs",
    				enabledClasses: "wb-bg-red-700 wb-text-gray-200 hover:wb-bg-gray-900 focus:wb-bg-gray-900 wb-c-focusable wb-cursor-pointer",
    				disabled: /*deleteInProgress*/ ctx[1][/*folder*/ ctx[21]],
    				handler: func
    			},
    			$$inline: true
    		});

    	const block = {
    		c: function create() {
    			create_component(button.$$.fragment);
    		},
    		m: function mount(target, anchor) {
    			mount_component(button, target, anchor);
    			current = true;
    		},
    		p: function update(new_ctx, dirty) {
    			ctx = new_ctx;
    			const button_changes = {};
    			if (dirty & /*deleteInProgress, $forlogsState*/ 6) button_changes.disabled = /*deleteInProgress*/ ctx[1][/*folder*/ ctx[21]];
    			if (dirty & /*$forlogsState*/ 4) button_changes.handler = func;
    			button.$set(button_changes);
    		},
    		i: function intro(local) {
    			if (current) return;
    			transition_in(button.$$.fragment, local);
    			current = true;
    		},
    		o: function outro(local) {
    			transition_out(button.$$.fragment, local);
    			current = false;
    		},
    		d: function destroy(detaching) {
    			destroy_component(button, detaching);
    		}
    	};

    	dispatch_dev("SvelteRegisterBlock", {
    		block,
    		id: create_if_block_1$c.name,
    		type: "if",
    		source: "(110:16) {#if folder}",
    		ctx
    	});

    	return block;
    }

    // (122:12) {#each $forlogsState.filteredFiles[folder] as file, index}
    function create_each_block_1$1(ctx) {
    	let div;
    	let span0;
    	let t0_value = /*file*/ ctx[24].file + "";
    	let t0;
    	let t1;
    	let span1;
    	let button0;
    	let t2;
    	let button1;
    	let t3;
    	let button2;
    	let t4;
    	let span2;
    	let t5_value = /*file*/ ctx[24].size + "";
    	let t5;
    	let t6;
    	let span3;
    	let t7_value = unixToSiteTimeZone(/*file*/ ctx[24].mtime, /*config*/ ctx[5].language, { timeZone: /*config*/ ctx[5].tz }) + "";
    	let t7;
    	let current;

    	function func_1() {
    		return /*func_1*/ ctx[11](/*file*/ ctx[24]);
    	}

    	button0 = new Button({
    			props: {
    				text: t('main.view'),
    				textSize: "wb-text-xs",
    				iconBefore: true,
    				withIcon: iconView,
    				iconSize: "xs",
    				disabled: /*file*/ ctx[24].bytesSize > /*features*/ ctx[6].get('viewMaxFileSize'),
    				title: /*file*/ ctx[24].bytesSize > /*features*/ ctx[6].get('viewMaxFileSize')
    				? t('main.tooLargeToView')
    				: '',
    				handler: func_1
    			},
    			$$inline: true
    		});

    	function func_2() {
    		return /*func_2*/ ctx[12](/*file*/ ctx[24]);
    	}

    	button1 = new Button({
    			props: {
    				text: t('main.download'),
    				textSize: "wb-text-xs",
    				iconBefore: true,
    				withIcon: iconDownload,
    				iconSize: "xs",
    				disabled: /*downloadInProgress*/ ctx[0],
    				handler: func_2
    			},
    			$$inline: true
    		});

    	function func_3() {
    		return /*func_3*/ ctx[13](/*file*/ ctx[24]);
    	}

    	button2 = new Button({
    			props: {
    				text: t('main.delete'),
    				textSize: "wb-text-xs",
    				iconBefore: true,
    				withIcon: iconDelete,
    				iconSize: "xs",
    				enabledClasses: "wb-bg-red-700 wb-text-gray-200 hover:wb-bg-gray-900 focus:wb-bg-gray-900 wb-c-focusable wb-cursor-pointer",
    				disabled: /*deleteInProgress*/ ctx[1][/*file*/ ctx[24].path],
    				handler: func_3
    			},
    			$$inline: true
    		});

    	const block = {
    		c: function create() {
    			div = element("div");
    			span0 = element("span");
    			t0 = text(t0_value);
    			t1 = space();
    			span1 = element("span");
    			create_component(button0.$$.fragment);
    			t2 = space();
    			create_component(button1.$$.fragment);
    			t3 = space();
    			create_component(button2.$$.fragment);
    			t4 = space();
    			span2 = element("span");
    			t5 = text(t5_value);
    			t6 = space();
    			span3 = element("span");
    			t7 = text(t7_value);
    			attr_dev(span0, "class", "wb-w-96 wb-font-semibold wb-my-2");
    			add_location(span0, file$l, 123, 20, 4640);
    			attr_dev(span1, "class", "wb-flex wb-flex-shrink-0 wb-items-center wb-space-x-4 wb-my-2");
    			add_location(span1, file$l, 124, 20, 4726);
    			attr_dev(span2, "class", "wb-w-32 wb-px-4 wb-text-sm wb-text-right wb-my-2");
    			add_location(span2, file$l, 152, 20, 6887);
    			attr_dev(span3, "class", "wb-px-4 wb-text-sm wb-my-2");
    			add_location(span3, file$l, 153, 20, 6989);
    			attr_dev(div, "class", "wb-flex wb-flex-wrap wb-items-center wb-my-2 wb-text-sm");
    			add_location(div, file$l, 122, 16, 4550);
    		},
    		m: function mount(target, anchor) {
    			insert_dev(target, div, anchor);
    			append_dev(div, span0);
    			append_dev(span0, t0);
    			append_dev(div, t1);
    			append_dev(div, span1);
    			mount_component(button0, span1, null);
    			append_dev(span1, t2);
    			mount_component(button1, span1, null);
    			append_dev(span1, t3);
    			mount_component(button2, span1, null);
    			append_dev(div, t4);
    			append_dev(div, span2);
    			append_dev(span2, t5);
    			append_dev(div, t6);
    			append_dev(div, span3);
    			append_dev(span3, t7);
    			current = true;
    		},
    		p: function update(new_ctx, dirty) {
    			ctx = new_ctx;
    			if ((!current || dirty & /*$forlogsState*/ 4) && t0_value !== (t0_value = /*file*/ ctx[24].file + "")) set_data_dev(t0, t0_value);
    			const button0_changes = {};
    			if (dirty & /*$forlogsState*/ 4) button0_changes.disabled = /*file*/ ctx[24].bytesSize > /*features*/ ctx[6].get('viewMaxFileSize');

    			if (dirty & /*$forlogsState*/ 4) button0_changes.title = /*file*/ ctx[24].bytesSize > /*features*/ ctx[6].get('viewMaxFileSize')
    			? t('main.tooLargeToView')
    			: '';

    			if (dirty & /*$forlogsState*/ 4) button0_changes.handler = func_1;
    			button0.$set(button0_changes);
    			const button1_changes = {};
    			if (dirty & /*downloadInProgress*/ 1) button1_changes.disabled = /*downloadInProgress*/ ctx[0];
    			if (dirty & /*$forlogsState*/ 4) button1_changes.handler = func_2;
    			button1.$set(button1_changes);
    			const button2_changes = {};
    			if (dirty & /*deleteInProgress, $forlogsState*/ 6) button2_changes.disabled = /*deleteInProgress*/ ctx[1][/*file*/ ctx[24].path];
    			if (dirty & /*$forlogsState*/ 4) button2_changes.handler = func_3;
    			button2.$set(button2_changes);
    			if ((!current || dirty & /*$forlogsState*/ 4) && t5_value !== (t5_value = /*file*/ ctx[24].size + "")) set_data_dev(t5, t5_value);
    			if ((!current || dirty & /*$forlogsState*/ 4) && t7_value !== (t7_value = unixToSiteTimeZone(/*file*/ ctx[24].mtime, /*config*/ ctx[5].language, { timeZone: /*config*/ ctx[5].tz }) + "")) set_data_dev(t7, t7_value);
    		},
    		i: function intro(local) {
    			if (current) return;
    			transition_in(button0.$$.fragment, local);
    			transition_in(button1.$$.fragment, local);
    			transition_in(button2.$$.fragment, local);
    			current = true;
    		},
    		o: function outro(local) {
    			transition_out(button0.$$.fragment, local);
    			transition_out(button1.$$.fragment, local);
    			transition_out(button2.$$.fragment, local);
    			current = false;
    		},
    		d: function destroy(detaching) {
    			if (detaching) detach_dev(div);
    			destroy_component(button0);
    			destroy_component(button1);
    			destroy_component(button2);
    		}
    	};

    	dispatch_dev("SvelteRegisterBlock", {
    		block,
    		id: create_each_block_1$1.name,
    		type: "each",
    		source: "(122:12) {#each $forlogsState.filteredFiles[folder] as file, index}",
    		ctx
    	});

    	return block;
    }

    // (103:4) {#each $forlogsState.filteredFolders as folder}
    function create_each_block$4(ctx) {
    	let div1;
    	let div0;
    	let span2;
    	let span0;
    	let t2;
    	let span1;
    	let t3_value = (/*folder*/ ctx[21] ? /*folder*/ ctx[21] : '/') + "";
    	let t3;
    	let t4;
    	let t5;
    	let t6;
    	let current;
    	let if_block = /*folder*/ ctx[21] && create_if_block_1$c(ctx);
    	let each_value_1 = /*$forlogsState*/ ctx[2].filteredFiles[/*folder*/ ctx[21]];
    	validate_each_argument(each_value_1);
    	let each_blocks = [];

    	for (let i = 0; i < each_value_1.length; i += 1) {
    		each_blocks[i] = create_each_block_1$1(get_each_context_1$1(ctx, each_value_1, i));
    	}

    	const out = i => transition_out(each_blocks[i], 1, 1, () => {
    		each_blocks[i] = null;
    	});

    	const block = {
    		c: function create() {
    			div1 = element("div");
    			div0 = element("div");
    			span2 = element("span");
    			span0 = element("span");
    			span0.textContent = `${t('main.folder')}:`;
    			t2 = space();
    			span1 = element("span");
    			t3 = text(t3_value);
    			t4 = space();
    			if (if_block) if_block.c();
    			t5 = space();

    			for (let i = 0; i < each_blocks.length; i += 1) {
    				each_blocks[i].c();
    			}

    			t6 = space();
    			attr_dev(span0, "class", "wb-text-sm");
    			add_location(span0, file$l, 106, 20, 3643);
    			attr_dev(span1, "class", "wb-font-semibold");
    			add_location(span1, file$l, 107, 20, 3715);
    			add_location(span2, file$l, 105, 16, 3616);
    			attr_dev(div0, "class", "wb-flex wb-justify-between wb-items-center wb-bg-white wb-p-2 wb-rounded-md");
    			add_location(div0, file$l, 104, 12, 3510);
    			attr_dev(div1, "class", "wb-my-4 wb-p-4 wb-bg-gray-200 wb-rounded-md");
    			add_location(div1, file$l, 103, 8, 3440);
    		},
    		m: function mount(target, anchor) {
    			insert_dev(target, div1, anchor);
    			append_dev(div1, div0);
    			append_dev(div0, span2);
    			append_dev(span2, span0);
    			append_dev(span2, t2);
    			append_dev(span2, span1);
    			append_dev(span1, t3);
    			append_dev(div0, t4);
    			if (if_block) if_block.m(div0, null);
    			append_dev(div1, t5);

    			for (let i = 0; i < each_blocks.length; i += 1) {
    				each_blocks[i].m(div1, null);
    			}

    			append_dev(div1, t6);
    			current = true;
    		},
    		p: function update(ctx, dirty) {
    			if ((!current || dirty & /*$forlogsState*/ 4) && t3_value !== (t3_value = (/*folder*/ ctx[21] ? /*folder*/ ctx[21] : '/') + "")) set_data_dev(t3, t3_value);

    			if (/*folder*/ ctx[21]) {
    				if (if_block) {
    					if_block.p(ctx, dirty);

    					if (dirty & /*$forlogsState*/ 4) {
    						transition_in(if_block, 1);
    					}
    				} else {
    					if_block = create_if_block_1$c(ctx);
    					if_block.c();
    					transition_in(if_block, 1);
    					if_block.m(div0, null);
    				}
    			} else if (if_block) {
    				group_outros();

    				transition_out(if_block, 1, 1, () => {
    					if_block = null;
    				});

    				check_outros();
    			}

    			if (dirty & /*unixToSiteTimeZone, $forlogsState, config, t, iconDelete, deleteInProgress, confirmDeleteFile, iconDownload, downloadInProgress, download, iconView, features, view*/ 999) {
    				each_value_1 = /*$forlogsState*/ ctx[2].filteredFiles[/*folder*/ ctx[21]];
    				validate_each_argument(each_value_1);
    				let i;

    				for (i = 0; i < each_value_1.length; i += 1) {
    					const child_ctx = get_each_context_1$1(ctx, each_value_1, i);

    					if (each_blocks[i]) {
    						each_blocks[i].p(child_ctx, dirty);
    						transition_in(each_blocks[i], 1);
    					} else {
    						each_blocks[i] = create_each_block_1$1(child_ctx);
    						each_blocks[i].c();
    						transition_in(each_blocks[i], 1);
    						each_blocks[i].m(div1, t6);
    					}
    				}

    				group_outros();

    				for (i = each_value_1.length; i < each_blocks.length; i += 1) {
    					out(i);
    				}

    				check_outros();
    			}
    		},
    		i: function intro(local) {
    			if (current) return;
    			transition_in(if_block);

    			for (let i = 0; i < each_value_1.length; i += 1) {
    				transition_in(each_blocks[i]);
    			}

    			current = true;
    		},
    		o: function outro(local) {
    			transition_out(if_block);
    			each_blocks = each_blocks.filter(Boolean);

    			for (let i = 0; i < each_blocks.length; i += 1) {
    				transition_out(each_blocks[i]);
    			}

    			current = false;
    		},
    		d: function destroy(detaching) {
    			if (detaching) detach_dev(div1);
    			if (if_block) if_block.d();
    			destroy_each(each_blocks, detaching);
    		}
    	};

    	dispatch_dev("SvelteRegisterBlock", {
    		block,
    		id: create_each_block$4.name,
    		type: "each",
    		source: "(103:4) {#each $forlogsState.filteredFolders as folder}",
    		ctx
    	});

    	return block;
    }

    function create_fragment$m(ctx) {
    	let current_block_type_index;
    	let if_block;
    	let if_block_anchor;
    	let current;
    	const if_block_creators = [create_if_block$f, create_else_block$7];
    	const if_blocks = [];

    	function select_block_type(ctx, dirty) {
    		if (/*$forlogsState*/ ctx[2].filteredFolders.length > 0) return 0;
    		return 1;
    	}

    	current_block_type_index = select_block_type(ctx);
    	if_block = if_blocks[current_block_type_index] = if_block_creators[current_block_type_index](ctx);

    	const block = {
    		c: function create() {
    			if_block.c();
    			if_block_anchor = empty();
    		},
    		l: function claim(nodes) {
    			throw new Error("options.hydrate only works if the component was compiled with the `hydratable: true` option");
    		},
    		m: function mount(target, anchor) {
    			if_blocks[current_block_type_index].m(target, anchor);
    			insert_dev(target, if_block_anchor, anchor);
    			current = true;
    		},
    		p: function update(ctx, [dirty]) {
    			let previous_block_index = current_block_type_index;
    			current_block_type_index = select_block_type(ctx);

    			if (current_block_type_index === previous_block_index) {
    				if_blocks[current_block_type_index].p(ctx, dirty);
    			} else {
    				group_outros();

    				transition_out(if_blocks[previous_block_index], 1, 1, () => {
    					if_blocks[previous_block_index] = null;
    				});

    				check_outros();
    				if_block = if_blocks[current_block_type_index];

    				if (!if_block) {
    					if_block = if_blocks[current_block_type_index] = if_block_creators[current_block_type_index](ctx);
    					if_block.c();
    				} else {
    					if_block.p(ctx, dirty);
    				}

    				transition_in(if_block, 1);
    				if_block.m(if_block_anchor.parentNode, if_block_anchor);
    			}
    		},
    		i: function intro(local) {
    			if (current) return;
    			transition_in(if_block);
    			current = true;
    		},
    		o: function outro(local) {
    			transition_out(if_block);
    			current = false;
    		},
    		d: function destroy(detaching) {
    			if_blocks[current_block_type_index].d(detaching);
    			if (detaching) detach_dev(if_block_anchor);
    		}
    	};

    	dispatch_dev("SvelteRegisterBlock", {
    		block,
    		id: create_fragment$m.name,
    		type: "component",
    		source: "",
    		ctx
    	});

    	return block;
    }

    function instance$m($$self, $$props, $$invalidate) {
    	let $appState;
    	let $forlogsState;
    	let { $$slots: slots = {}, $$scope } = $$props;
    	validate_slots('List', slots, []);
    	let appState = getContext('appState');
    	validate_store(appState, 'appState');
    	component_subscribe($$self, appState, value => $$invalidate(14, $appState = value));
    	let busyState = getContext('busyState');
    	let oneoffNotifications = getContext('oneoffNotifications');
    	let forlogsState = getContext('forlogsState');
    	validate_store(forlogsState, 'forlogsState');
    	component_subscribe($$self, forlogsState, value => $$invalidate(2, $forlogsState = value));
    	let config = getContext('config');
    	const features = getContext('features');
    	let downloadInProgress = false;
    	let deleteInProgress = {};

    	const download = file => {
    		$$invalidate(0, downloadInProgress = true);
    		filesModel.downloadOne(file.path, file.name);
    		$$invalidate(0, downloadInProgress = false);
    	};

    	const view = file => {
    		set_store_value(forlogsState, $forlogsState.viewing = file, $forlogsState);
    	};

    	const getDeleteConfirmNotification = (file, doHandler, cancelHandler) => {
    		return {
    			type: '1_danger',
    			canClear: false,
    			title: tsprintf('main.confirmDelete', file.file),
    			autoclear: 0,
    			buttons: [
    				{
    					text: t('main.deleteFile'),
    					default: false,
    					handler: doHandler
    				},
    				{
    					text: t('main.keepFile'),
    					default: true,
    					handler: cancelHandler,
    					enabledClasses: 'wb-bg-gray-900 wb-text-gray-200 hover:wb-bg-gray-700 hover:wb-text-gray-200 focus:wb-bg-gray-900 focus:wb-text-gray-200 wb-c-focusable wb-cursor-pointer'
    				}
    			]
    		};
    	};

    	const confirmDeleteFile = file => {
    		$$invalidate(1, deleteInProgress[file.path] = true, deleteInProgress);

    		set_store_value(
    			forlogsState,
    			$forlogsState.confirmationDialog = getDeleteConfirmNotification(
    				file,
    				() => {
    					set_store_value(forlogsState, $forlogsState.confirmationDialog = false, $forlogsState);
    					deleteFile(file);
    				},
    				() => {
    					$$invalidate(1, deleteInProgress[file.path] = false, deleteInProgress);
    					set_store_value(forlogsState, $forlogsState.confirmationDialog = false, $forlogsState);
    				}
    			),
    			$forlogsState
    		);
    	};

    	const deleteFileError = (file, response) => {
    		console.error('Error deleting file', response);
    		oneoffNotifications.addError(t('main.errorDeletingFile'), response.details, true);
    		console.error('4LOGS: ', t('response.message'), ': ', response.details);
    		busyState.stopSpinner('delete');
    		$$invalidate(1, deleteInProgress[file.path] = false, deleteInProgress);
    	};

    	const deleteFileSuccess = file => {
    		oneoffNotifications.addInfo(t('main.fileDeleted'), file.path, true);
    		busyState.stopSpinner('delete');
    		$$invalidate(1, deleteInProgress[file.path] = false, deleteInProgress);
    		set_store_value(forlogsState, $forlogsState.nextAction = 'refresh', $forlogsState);
    		set_store_value(appState, $appState.status = 'ready', $appState);
    	};

    	const deleteFile = file => {
    		busyState.startSpinner('delete');
    		filesModel.deleteFiles([file.path], response => deleteFileSuccess(file), response => deleteFileError(file, response));
    	};

    	const writable_props = [];

    	Object.keys($$props).forEach(key => {
    		if (!~writable_props.indexOf(key) && key.slice(0, 2) !== '$$' && key !== 'slot') console_1$2.warn(`<List> was created with unknown prop '${key}'`);
    	});

    	const func = folder => confirmDeleteFile({ file: folder, path: folder });
    	const func_1 = file => view(file);
    	const func_2 = file => download(file);
    	const func_3 = file => confirmDeleteFile(file);

    	$$self.$capture_state = () => ({
    		getContext,
    		t,
    		tsprintf,
    		unixToSiteTimeZone,
    		iconView,
    		iconDownload,
    		iconDelete,
    		Button,
    		StatusMessage,
    		filesModel,
    		appState,
    		busyState,
    		oneoffNotifications,
    		forlogsState,
    		config,
    		features,
    		downloadInProgress,
    		deleteInProgress,
    		download,
    		view,
    		getDeleteConfirmNotification,
    		confirmDeleteFile,
    		deleteFileError,
    		deleteFileSuccess,
    		deleteFile,
    		$appState,
    		$forlogsState
    	});

    	$$self.$inject_state = $$props => {
    		if ('appState' in $$props) $$invalidate(3, appState = $$props.appState);
    		if ('busyState' in $$props) busyState = $$props.busyState;
    		if ('oneoffNotifications' in $$props) oneoffNotifications = $$props.oneoffNotifications;
    		if ('forlogsState' in $$props) $$invalidate(4, forlogsState = $$props.forlogsState);
    		if ('config' in $$props) $$invalidate(5, config = $$props.config);
    		if ('downloadInProgress' in $$props) $$invalidate(0, downloadInProgress = $$props.downloadInProgress);
    		if ('deleteInProgress' in $$props) $$invalidate(1, deleteInProgress = $$props.deleteInProgress);
    	};

    	if ($$props && "$$inject" in $$props) {
    		$$self.$inject_state($$props.$$inject);
    	}

    	return [
    		downloadInProgress,
    		deleteInProgress,
    		$forlogsState,
    		appState,
    		forlogsState,
    		config,
    		features,
    		download,
    		view,
    		confirmDeleteFile,
    		func,
    		func_1,
    		func_2,
    		func_3
    	];
    }

    class List extends SvelteComponentDev {
    	constructor(options) {
    		super(options);
    		init(this, options, instance$m, create_fragment$m, safe_not_equal, {});

    		dispatch_dev("SvelteRegisterComponent", {
    			component: this,
    			tagName: "List",
    			options,
    			id: create_fragment$m.name
    		});
    	}
    }

    /* node_modules\@weeblr\svelte\components\ContentLoading.svelte generated by Svelte v3.41.0 */

    const file$k = "node_modules\\@weeblr\\svelte\\components\\ContentLoading.svelte";

    function get_each_context$3(ctx, list, i) {
    	const child_ctx = ctx.slice();
    	child_ctx[5] = list[i];
    	return child_ctx;
    }

    // (27:2) {#each lines as line}
    function create_each_block$3(ctx) {
    	let line;

    	const block = {
    		c: function create() {
    			line = svg_element("line");
    			attr_dev(line, "x1", "6");
    			attr_dev(line, "y1", /*line*/ ctx[5].y);
    			attr_dev(line, "x2", /*line*/ ctx[5].x);
    			attr_dev(line, "y2", /*line*/ ctx[5].y);
    			attr_dev(line, "stroke-linecap", "round");
    			attr_dev(line, "class", "svelte-13xnxqb");
    			add_location(line, file$k, 27, 12, 733);
    		},
    		m: function mount(target, anchor) {
    			insert_dev(target, line, anchor);
    		},
    		p: noop$1,
    		d: function destroy(detaching) {
    			if (detaching) detach_dev(line);
    		}
    	};

    	dispatch_dev("SvelteRegisterBlock", {
    		block,
    		id: create_each_block$3.name,
    		type: "each",
    		source: "(27:2) {#each lines as line}",
    		ctx
    	});

    	return block;
    }

    function create_fragment$l(ctx) {
    	let section;
    	let svg;
    	let svg_class_value;
    	let section_class_value;
    	let each_value = /*lines*/ ctx[3];
    	validate_each_argument(each_value);
    	let each_blocks = [];

    	for (let i = 0; i < each_value.length; i += 1) {
    		each_blocks[i] = create_each_block$3(get_each_context$3(ctx, each_value, i));
    	}

    	const block = {
    		c: function create() {
    			section = element("section");
    			svg = svg_element("svg");

    			for (let i = 0; i < each_blocks.length; i += 1) {
    				each_blocks[i].c();
    			}

    			attr_dev(svg, "class", svg_class_value = "wb-w-full " + /*color*/ ctx[0] + " svelte-13xnxqb");
    			add_location(svg, file$k, 25, 4, 665);
    			attr_dev(section, "aria-hidden", "true");
    			attr_dev(section, "class", section_class_value = "" + (/*background*/ ctx[1] + " wb-overflow-y-auto wb-animate-pulse " + /*classes*/ ctx[2]));
    			add_location(section, file$k, 24, 0, 565);
    		},
    		l: function claim(nodes) {
    			throw new Error("options.hydrate only works if the component was compiled with the `hydratable: true` option");
    		},
    		m: function mount(target, anchor) {
    			insert_dev(target, section, anchor);
    			append_dev(section, svg);

    			for (let i = 0; i < each_blocks.length; i += 1) {
    				each_blocks[i].m(svg, null);
    			}
    		},
    		p: function update(ctx, [dirty]) {
    			if (dirty & /*lines*/ 8) {
    				each_value = /*lines*/ ctx[3];
    				validate_each_argument(each_value);
    				let i;

    				for (i = 0; i < each_value.length; i += 1) {
    					const child_ctx = get_each_context$3(ctx, each_value, i);

    					if (each_blocks[i]) {
    						each_blocks[i].p(child_ctx, dirty);
    					} else {
    						each_blocks[i] = create_each_block$3(child_ctx);
    						each_blocks[i].c();
    						each_blocks[i].m(svg, null);
    					}
    				}

    				for (; i < each_blocks.length; i += 1) {
    					each_blocks[i].d(1);
    				}

    				each_blocks.length = each_value.length;
    			}

    			if (dirty & /*color*/ 1 && svg_class_value !== (svg_class_value = "wb-w-full " + /*color*/ ctx[0] + " svelte-13xnxqb")) {
    				attr_dev(svg, "class", svg_class_value);
    			}

    			if (dirty & /*background, classes*/ 6 && section_class_value !== (section_class_value = "" + (/*background*/ ctx[1] + " wb-overflow-y-auto wb-animate-pulse " + /*classes*/ ctx[2]))) {
    				attr_dev(section, "class", section_class_value);
    			}
    		},
    		i: noop$1,
    		o: noop$1,
    		d: function destroy(detaching) {
    			if (detaching) detach_dev(section);
    			destroy_each(each_blocks, detaching);
    		}
    	};

    	dispatch_dev("SvelteRegisterBlock", {
    		block,
    		id: create_fragment$l.name,
    		type: "component",
    		source: "",
    		ctx
    	});

    	return block;
    }

    function instance$l($$self, $$props, $$invalidate) {
    	let { $$slots: slots = {}, $$scope } = $$props;
    	validate_slots('ContentLoading', slots, []);
    	let { linesCount = 4 } = $$props;
    	let { color = 'wb-text-gray-300' } = $$props;
    	let { background = 'wb-bg-transparent' } = $$props;
    	let { classes = 'wb-py-4' } = $$props;

    	// Implementation
    	let lines = [
    		{ x: 400, y: 6 },
    		{ x: 200, y: 36 },
    		{ x: 300, y: 66 },
    		{ x: 280, y: 96 },
    		{ x: 330, y: 126 },
    		{ x: 400, y: 156 },
    		{ x: 200, y: 186 },
    		{ x: 300, y: 216 },
    		{ x: 280, y: 246 },
    		{ x: 330, y: 276 }
    	];

    	linesCount = Math.min(linesCount, 10);
    	lines.splice(linesCount);
    	const writable_props = ['linesCount', 'color', 'background', 'classes'];

    	Object.keys($$props).forEach(key => {
    		if (!~writable_props.indexOf(key) && key.slice(0, 2) !== '$$' && key !== 'slot') console.warn(`<ContentLoading> was created with unknown prop '${key}'`);
    	});

    	$$self.$$set = $$props => {
    		if ('linesCount' in $$props) $$invalidate(4, linesCount = $$props.linesCount);
    		if ('color' in $$props) $$invalidate(0, color = $$props.color);
    		if ('background' in $$props) $$invalidate(1, background = $$props.background);
    		if ('classes' in $$props) $$invalidate(2, classes = $$props.classes);
    	};

    	$$self.$capture_state = () => ({
    		linesCount,
    		color,
    		background,
    		classes,
    		lines
    	});

    	$$self.$inject_state = $$props => {
    		if ('linesCount' in $$props) $$invalidate(4, linesCount = $$props.linesCount);
    		if ('color' in $$props) $$invalidate(0, color = $$props.color);
    		if ('background' in $$props) $$invalidate(1, background = $$props.background);
    		if ('classes' in $$props) $$invalidate(2, classes = $$props.classes);
    		if ('lines' in $$props) $$invalidate(3, lines = $$props.lines);
    	};

    	if ($$props && "$$inject" in $$props) {
    		$$self.$inject_state($$props.$$inject);
    	}

    	return [color, background, classes, lines, linesCount];
    }

    class ContentLoading extends SvelteComponentDev {
    	constructor(options) {
    		super(options);

    		init(this, options, instance$l, create_fragment$l, safe_not_equal, {
    			linesCount: 4,
    			color: 0,
    			background: 1,
    			classes: 2
    		});

    		dispatch_dev("SvelteRegisterComponent", {
    			component: this,
    			tagName: "ContentLoading",
    			options,
    			id: create_fragment$l.name
    		});
    	}

    	get linesCount() {
    		throw new Error("<ContentLoading>: Props cannot be read directly from the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	set linesCount(value) {
    		throw new Error("<ContentLoading>: Props cannot be set directly on the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	get color() {
    		throw new Error("<ContentLoading>: Props cannot be read directly from the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	set color(value) {
    		throw new Error("<ContentLoading>: Props cannot be set directly on the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	get background() {
    		throw new Error("<ContentLoading>: Props cannot be read directly from the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	set background(value) {
    		throw new Error("<ContentLoading>: Props cannot be set directly on the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	get classes() {
    		throw new Error("<ContentLoading>: Props cannot be read directly from the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	set classes(value) {
    		throw new Error("<ContentLoading>: Props cannot be set directly on the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}
    }

    /* src\app\admin\components\View.svelte generated by Svelte v3.41.0 */

    const { console: console_1$1 } = globals;
    const file$j = "src\\app\\admin\\components\\View.svelte";

    function get_each_context$2(ctx, list, i) {
    	const child_ctx = ctx.slice();
    	child_ctx[7] = list[i];
    	child_ctx[9] = i;
    	return child_ctx;
    }

    // (59:4) {:else}
    function create_else_block$6(ctx) {
    	let div;

    	const block = {
    		c: function create() {
    			div = element("div");
    			div.textContent = `${t('main.fileIsEmpty')}`;
    			attr_dev(div, "class", "wb-text-center wb-text-base wb-my-8");
    			add_location(div, file$j, 59, 8, 2477);
    		},
    		m: function mount(target, anchor) {
    			insert_dev(target, div, anchor);
    		},
    		p: noop$1,
    		i: noop$1,
    		o: noop$1,
    		d: function destroy(detaching) {
    			if (detaching) detach_dev(div);
    		}
    	};

    	dispatch_dev("SvelteRegisterBlock", {
    		block,
    		id: create_else_block$6.name,
    		type: "else",
    		source: "(59:4) {:else}",
    		ctx
    	});

    	return block;
    }

    // (55:26) 
    function create_if_block_2$a(ctx) {
    	let pre;
    	let t_1;

    	const block = {
    		c: function create() {
    			pre = element("pre");
    			t_1 = text(/*fileContent*/ ctx[0]);
    			attr_dev(pre, "class", "wb-flex wb-flex-col wb-w-full wb-font-mono wb-text-xs wb-leading-5 wb-m-0 wb-py-1 wb-px-2 wb-bg-gray-200 wb-text-gray-900 wb-break-all wb-whitespace-pre-wrap");
    			add_location(pre, file$j, 55, 9, 2242);
    		},
    		m: function mount(target, anchor) {
    			insert_dev(target, pre, anchor);
    			append_dev(pre, t_1);
    		},
    		p: function update(ctx, dirty) {
    			if (dirty & /*fileContent*/ 1) set_data_dev(t_1, /*fileContent*/ ctx[0]);
    		},
    		i: noop$1,
    		o: noop$1,
    		d: function destroy(detaching) {
    			if (detaching) detach_dev(pre);
    		}
    	};

    	dispatch_dev("SvelteRegisterBlock", {
    		block,
    		id: create_if_block_2$a.name,
    		type: "if",
    		source: "(55:26) ",
    		ctx
    	});

    	return block;
    }

    // (46:56) 
    function create_if_block_1$b(ctx) {
    	let div;
    	let each_value = /*fileContent*/ ctx[0];
    	validate_each_argument(each_value);
    	let each_blocks = [];

    	for (let i = 0; i < each_value.length; i += 1) {
    		each_blocks[i] = create_each_block$2(get_each_context$2(ctx, each_value, i));
    	}

    	const block = {
    		c: function create() {
    			div = element("div");

    			for (let i = 0; i < each_blocks.length; i += 1) {
    				each_blocks[i].c();
    			}

    			attr_dev(div, "class", "wb-flex wb-flex-col wb-w-full wb-font-mono wb-text-xs wb-leading-5 wb-bg-gray-200 wb-text-gray-900");
    			add_location(div, file$j, 46, 8, 1663);
    		},
    		m: function mount(target, anchor) {
    			insert_dev(target, div, anchor);

    			for (let i = 0; i < each_blocks.length; i += 1) {
    				each_blocks[i].m(div, null);
    			}
    		},
    		p: function update(ctx, dirty) {
    			if (dirty & /*fileContent*/ 1) {
    				each_value = /*fileContent*/ ctx[0];
    				validate_each_argument(each_value);
    				let i;

    				for (i = 0; i < each_value.length; i += 1) {
    					const child_ctx = get_each_context$2(ctx, each_value, i);

    					if (each_blocks[i]) {
    						each_blocks[i].p(child_ctx, dirty);
    					} else {
    						each_blocks[i] = create_each_block$2(child_ctx);
    						each_blocks[i].c();
    						each_blocks[i].m(div, null);
    					}
    				}

    				for (; i < each_blocks.length; i += 1) {
    					each_blocks[i].d(1);
    				}

    				each_blocks.length = each_value.length;
    			}
    		},
    		i: noop$1,
    		o: noop$1,
    		d: function destroy(detaching) {
    			if (detaching) detach_dev(div);
    			destroy_each(each_blocks, detaching);
    		}
    	};

    	dispatch_dev("SvelteRegisterBlock", {
    		block,
    		id: create_if_block_1$b.name,
    		type: "if",
    		source: "(46:56) ",
    		ctx
    	});

    	return block;
    }

    // (44:4) {#if fileContent === null}
    function create_if_block$e(ctx) {
    	let contentloading;
    	let current;

    	contentloading = new ContentLoading({
    			props: { linesCount: "8", classes: "wb-p-4" },
    			$$inline: true
    		});

    	const block = {
    		c: function create() {
    			create_component(contentloading.$$.fragment);
    		},
    		m: function mount(target, anchor) {
    			mount_component(contentloading, target, anchor);
    			current = true;
    		},
    		p: noop$1,
    		i: function intro(local) {
    			if (current) return;
    			transition_in(contentloading.$$.fragment, local);
    			current = true;
    		},
    		o: function outro(local) {
    			transition_out(contentloading.$$.fragment, local);
    			current = false;
    		},
    		d: function destroy(detaching) {
    			destroy_component(contentloading, detaching);
    		}
    	};

    	dispatch_dev("SvelteRegisterBlock", {
    		block,
    		id: create_if_block$e.name,
    		type: "if",
    		source: "(44:4) {#if fileContent === null}",
    		ctx
    	});

    	return block;
    }

    // (48:12) {#each fileContent as line, index}
    function create_each_block$2(ctx) {
    	let div;
    	let span;
    	let t0_value = /*index*/ ctx[9] + 1 + "";
    	let t0;
    	let t1;
    	let pre;
    	let t2_value = /*line*/ ctx[7] + "";
    	let t2;
    	let t3;

    	const block = {
    		c: function create() {
    			div = element("div");
    			span = element("span");
    			t0 = text(t0_value);
    			t1 = space();
    			pre = element("pre");
    			t2 = text(t2_value);
    			t3 = space();
    			attr_dev(span, "class", "wb-w-16 wb-py-1 wb-px-2 wb-flex-shrink-0 wb-text-right wb-bg-gray-700 wb-text-red-400");
    			add_location(span, file$j, 49, 20, 1891);
    			attr_dev(pre, "class", "wb-m-0 wb-py-1 wb-px-2 wb-bg-transparent wb-border-none wb-break-all wb-whitespace-pre-wrap");
    			add_location(pre, file$j, 50, 20, 2030);
    			attr_dev(div, "class", "wb-flex wb-w-full");
    			add_location(div, file$j, 48, 16, 1839);
    		},
    		m: function mount(target, anchor) {
    			insert_dev(target, div, anchor);
    			append_dev(div, span);
    			append_dev(span, t0);
    			append_dev(div, t1);
    			append_dev(div, pre);
    			append_dev(pre, t2);
    			append_dev(div, t3);
    		},
    		p: function update(ctx, dirty) {
    			if (dirty & /*fileContent*/ 1 && t2_value !== (t2_value = /*line*/ ctx[7] + "")) set_data_dev(t2, t2_value);
    		},
    		d: function destroy(detaching) {
    			if (detaching) detach_dev(div);
    		}
    	};

    	dispatch_dev("SvelteRegisterBlock", {
    		block,
    		id: create_each_block$2.name,
    		type: "each",
    		source: "(48:12) {#each fileContent as line, index}",
    		ctx
    	});

    	return block;
    }

    function create_fragment$k(ctx) {
    	let div;
    	let show_if;
    	let current_block_type_index;
    	let if_block;
    	let current;
    	const if_block_creators = [create_if_block$e, create_if_block_1$b, create_if_block_2$a, create_else_block$6];
    	const if_blocks = [];

    	function select_block_type(ctx, dirty) {
    		if (/*fileContent*/ ctx[0] === null) return 0;
    		if (dirty & /*fileContent*/ 1) show_if = !!(/*fileContent*/ ctx[0] && Array.isArray(/*fileContent*/ ctx[0]));
    		if (show_if) return 1;
    		if (/*fileContent*/ ctx[0]) return 2;
    		return 3;
    	}

    	current_block_type_index = select_block_type(ctx, -1);
    	if_block = if_blocks[current_block_type_index] = if_block_creators[current_block_type_index](ctx);

    	const block = {
    		c: function create() {
    			div = element("div");
    			if_block.c();
    			attr_dev(div, "class", "wb-mt-4 wb-w-full");
    			add_location(div, file$j, 42, 0, 1477);
    		},
    		l: function claim(nodes) {
    			throw new Error("options.hydrate only works if the component was compiled with the `hydratable: true` option");
    		},
    		m: function mount(target, anchor) {
    			insert_dev(target, div, anchor);
    			if_blocks[current_block_type_index].m(div, null);
    			current = true;
    		},
    		p: function update(ctx, [dirty]) {
    			let previous_block_index = current_block_type_index;
    			current_block_type_index = select_block_type(ctx, dirty);

    			if (current_block_type_index === previous_block_index) {
    				if_blocks[current_block_type_index].p(ctx, dirty);
    			} else {
    				group_outros();

    				transition_out(if_blocks[previous_block_index], 1, 1, () => {
    					if_blocks[previous_block_index] = null;
    				});

    				check_outros();
    				if_block = if_blocks[current_block_type_index];

    				if (!if_block) {
    					if_block = if_blocks[current_block_type_index] = if_block_creators[current_block_type_index](ctx);
    					if_block.c();
    				} else {
    					if_block.p(ctx, dirty);
    				}

    				transition_in(if_block, 1);
    				if_block.m(div, null);
    			}
    		},
    		i: function intro(local) {
    			if (current) return;
    			transition_in(if_block);
    			current = true;
    		},
    		o: function outro(local) {
    			transition_out(if_block);
    			current = false;
    		},
    		d: function destroy(detaching) {
    			if (detaching) detach_dev(div);
    			if_blocks[current_block_type_index].d();
    		}
    	};

    	dispatch_dev("SvelteRegisterBlock", {
    		block,
    		id: create_fragment$k.name,
    		type: "component",
    		source: "",
    		ctx
    	});

    	return block;
    }

    function instance$k($$self, $$props, $$invalidate) {
    	let $appState;
    	let $forlogsState;
    	let { $$slots: slots = {}, $$scope } = $$props;
    	validate_slots('View', slots, []);
    	let forlogsState = getContext('forlogsState');
    	validate_store(forlogsState, 'forlogsState');
    	component_subscribe($$self, forlogsState, value => $$invalidate(4, $forlogsState = value));
    	let appState = getContext('appState');
    	validate_store(appState, 'appState');
    	component_subscribe($$self, appState, value => $$invalidate(3, $appState = value));
    	let busyState = getContext('busyState');
    	let oneoffNotifications = getContext('oneoffNotifications');
    	let fileContent = null;

    	onMount(() => {
    		const spinnerName = 'file.' + $forlogsState.viewing.path;
    		busyState.startSpinner(spinnerName);

    		filesModel.fetchOne($forlogsState.viewing.path).then(json => {
    			busyState.stopSpinner(spinnerName);

    			if (json.failed || json.status && json.status == 'error') {
    				set_store_value(appState, $appState.status = 'error', $appState);
    				oneoffNotifications.addError(t(json.message), t('backendComm.errorViewDetailsInConsole'), true);
    				console.error('4LOGS: ', json.message, ': ', json.details);
    			} else {
    				$$invalidate(0, fileContent = json.data.content);
    			}
    		});
    	});

    	const writable_props = [];

    	Object.keys($$props).forEach(key => {
    		if (!~writable_props.indexOf(key) && key.slice(0, 2) !== '$$' && key !== 'slot') console_1$1.warn(`<View> was created with unknown prop '${key}'`);
    	});

    	$$self.$capture_state = () => ({
    		getContext,
    		onMount,
    		filesModel,
    		t,
    		ContentLoading,
    		forlogsState,
    		appState,
    		busyState,
    		oneoffNotifications,
    		fileContent,
    		$appState,
    		$forlogsState
    	});

    	$$self.$inject_state = $$props => {
    		if ('forlogsState' in $$props) $$invalidate(1, forlogsState = $$props.forlogsState);
    		if ('appState' in $$props) $$invalidate(2, appState = $$props.appState);
    		if ('busyState' in $$props) busyState = $$props.busyState;
    		if ('oneoffNotifications' in $$props) oneoffNotifications = $$props.oneoffNotifications;
    		if ('fileContent' in $$props) $$invalidate(0, fileContent = $$props.fileContent);
    	};

    	if ($$props && "$$inject" in $$props) {
    		$$self.$inject_state($$props.$$inject);
    	}

    	return [fileContent, forlogsState, appState];
    }

    class View extends SvelteComponentDev {
    	constructor(options) {
    		super(options);
    		init(this, options, instance$k, create_fragment$k, safe_not_equal, {});

    		dispatch_dev("SvelteRegisterComponent", {
    			component: this,
    			tagName: "View",
    			options,
    			id: create_fragment$k.name
    		});
    	}
    }

    /* src\app\admin\components\Title.svelte generated by Svelte v3.41.0 */
    const file$i = "src\\app\\admin\\components\\Title.svelte";

    function create_fragment$j(ctx) {
    	let h1;

    	const block = {
    		c: function create() {
    			h1 = element("h1");
    			attr_dev(h1, "class", "wb-inline-block wb-m-0 wb-text-2xl wb-font-semibold wb-leading-normal wb-tracking-widest wb-text-gray-900 wb-c-focusable");
    			add_location(h1, file$i, 12, 0, 169);
    		},
    		l: function claim(nodes) {
    			throw new Error("options.hydrate only works if the component was compiled with the `hydratable: true` option");
    		},
    		m: function mount(target, anchor) {
    			insert_dev(target, h1, anchor);
    			h1.innerHTML = /*title*/ ctx[0];
    		},
    		p: function update(ctx, [dirty]) {
    			if (dirty & /*title*/ 1) h1.innerHTML = /*title*/ ctx[0];		},
    		i: noop$1,
    		o: noop$1,
    		d: function destroy(detaching) {
    			if (detaching) detach_dev(h1);
    		}
    	};

    	dispatch_dev("SvelteRegisterBlock", {
    		block,
    		id: create_fragment$j.name,
    		type: "component",
    		source: "",
    		ctx
    	});

    	return block;
    }

    function instance$j($$self, $$props, $$invalidate) {
    	let { $$slots: slots = {}, $$scope } = $$props;
    	validate_slots('Title', slots, []);
    	let { title = '' } = $$props;
    	let appState = getContext('appState');
    	const writable_props = ['title'];

    	Object.keys($$props).forEach(key => {
    		if (!~writable_props.indexOf(key) && key.slice(0, 2) !== '$$' && key !== 'slot') console.warn(`<Title> was created with unknown prop '${key}'`);
    	});

    	$$self.$$set = $$props => {
    		if ('title' in $$props) $$invalidate(0, title = $$props.title);
    	};

    	$$self.$capture_state = () => ({ title, getContext, appState });

    	$$self.$inject_state = $$props => {
    		if ('title' in $$props) $$invalidate(0, title = $$props.title);
    		if ('appState' in $$props) appState = $$props.appState;
    	};

    	if ($$props && "$$inject" in $$props) {
    		$$self.$inject_state($$props.$$inject);
    	}

    	return [title];
    }

    class Title extends SvelteComponentDev {
    	constructor(options) {
    		super(options);
    		init(this, options, instance$j, create_fragment$j, safe_not_equal, { title: 0 });

    		dispatch_dev("SvelteRegisterComponent", {
    			component: this,
    			tagName: "Title",
    			options,
    			id: create_fragment$j.name
    		});
    	}

    	get title() {
    		throw new Error("<Title>: Props cannot be read directly from the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	set title(value) {
    		throw new Error("<Title>: Props cannot be set directly on the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}
    }

    /* src\assets\Logo.svelte generated by Svelte v3.41.0 */

    const file$h = "src\\assets\\Logo.svelte";

    // (14:0) {#if 'forlogsSymbol' == type && 'wb-dark' != theme}
    function create_if_block_3$4(ctx) {
    	let svg;
    	let path0;
    	let circle0;
    	let circle1;
    	let path1;
    	let path2;
    	let svg_class_value;
    	let svg_alt_value;
    	let svg_title_value;

    	const block = {
    		c: function create() {
    			svg = svg_element("svg");
    			path0 = svg_element("path");
    			circle0 = svg_element("circle");
    			circle1 = svg_element("circle");
    			path1 = svg_element("path");
    			path2 = svg_element("path");
    			attr_dev(path0, "fill", "none");
    			attr_dev(path0, "d", "M.019-.003h1181.105V1181.1H.02z");
    			add_location(path0, file$h, 14, 295, 618);
    			attr_dev(circle0, "cx", "1763.4");
    			attr_dev(circle0, "cy", "563.352");
    			attr_dev(circle0, "r", "460.653");
    			attr_dev(circle0, "fill", "#ff5259");
    			attr_dev(circle0, "transform", "translate(-1263.182 -1.681) scale(1.05123)");
    			add_location(circle0, file$h, 14, 350, 673);
    			attr_dev(circle1, "cx", "1058.3");
    			attr_dev(circle1, "cy", "1019.56");
    			attr_dev(circle1, "r", "736.788");
    			attr_dev(circle1, "fill", "#2d3748");
    			attr_dev(circle1, "stroke", "#2d3748");
    			attr_dev(circle1, "stroke-width", "2.24");
    			attr_dev(circle1, "transform", "translate(119.096 136.327) scale(.4455)");
    			add_location(circle1, file$h, 14, 466, 789);
    			attr_dev(path1, "d", "M20 21h-8c-1.097 0-2-.903-2-2v-2c0-1.097.903-2 2-2h8c1.097 0 2 .903 2 2v2c0 1.097-.903 2-2 2zm-8-4v2h8v-2h-8z");
    			attr_dev(path1, "fill", "#f8eed5");
    			attr_dev(path1, "fill-rule", "nonzero");
    			attr_dev(path1, "stroke", "#f8eed5");
    			attr_dev(path1, "stroke-width", ".08");
    			attr_dev(path1, "stroke-linecap", "butt");
    			attr_dev(path1, "stroke-miterlimit", "2");
    			attr_dev(path1, "transform", "matrix(13.18194 0 0 13.1819 379.66 366.5)");
    			add_location(path1, file$h, 14, 616, 939);
    			attr_dev(path2, "d", "M28 4H4c-1.097 0-2 .903-2 2v4c0 1.097.903 2 2 2v16c0 1.097.903 2 2 2h20c1.097 0 2-.903 2-2V12c1.097 0 2-.903 2-2V6c0-1.097-.903-2-2-2zm-2 24H6V12h20v16zm2-18H4V6h24v4z");
    			attr_dev(path2, "fill", "#f8eed5");
    			attr_dev(path2, "fill-rule", "nonzero");
    			attr_dev(path2, "stroke", "#f8eed5");
    			attr_dev(path2, "stroke-width", ".08");
    			attr_dev(path2, "stroke-linecap", "butt");
    			attr_dev(path2, "stroke-miterlimit", "2");
    			attr_dev(path2, "transform", "matrix(13.18194 0 0 13.1819 379.66 366.5)");
    			add_location(path2, file$h, 14, 906, 1229);
    			attr_dev(svg, "class", svg_class_value = /*classes*/ ctx[5] || null);
    			attr_dev(svg, "alt", svg_alt_value = /*alt*/ ctx[2] || /*title*/ ctx[3] || null);
    			attr_dev(svg, "title", svg_title_value = /*title*/ ctx[3] || /*alt*/ ctx[2] | null);
    			attr_dev(svg, "aria-hidden", /*ariaHidden*/ ctx[4]);
    			attr_dev(svg, "height", "100%");
    			attr_dev(svg, "viewBox", "0 0 1182 1182");
    			attr_dev(svg, "xmlns", "http://www.w3.org/2000/svg");
    			attr_dev(svg, "fill-rule", "evenodd");
    			attr_dev(svg, "clip-rule", "evenodd");
    			attr_dev(svg, "stroke-linecap", "round");
    			attr_dev(svg, "stroke-linejoin", "round");
    			attr_dev(svg, "stroke-miterlimit", "1.5");
    			add_location(svg, file$h, 14, 4, 327);
    		},
    		m: function mount(target, anchor) {
    			insert_dev(target, svg, anchor);
    			append_dev(svg, path0);
    			append_dev(svg, circle0);
    			append_dev(svg, circle1);
    			append_dev(svg, path1);
    			append_dev(svg, path2);
    		},
    		p: function update(ctx, dirty) {
    			if (dirty & /*classes*/ 32 && svg_class_value !== (svg_class_value = /*classes*/ ctx[5] || null)) {
    				attr_dev(svg, "class", svg_class_value);
    			}

    			if (dirty & /*alt, title*/ 12 && svg_alt_value !== (svg_alt_value = /*alt*/ ctx[2] || /*title*/ ctx[3] || null)) {
    				attr_dev(svg, "alt", svg_alt_value);
    			}

    			if (dirty & /*title, alt*/ 12 && svg_title_value !== (svg_title_value = /*title*/ ctx[3] || /*alt*/ ctx[2] | null)) {
    				attr_dev(svg, "title", svg_title_value);
    			}

    			if (dirty & /*ariaHidden*/ 16) {
    				attr_dev(svg, "aria-hidden", /*ariaHidden*/ ctx[4]);
    			}
    		},
    		d: function destroy(detaching) {
    			if (detaching) detach_dev(svg);
    		}
    	};

    	dispatch_dev("SvelteRegisterBlock", {
    		block,
    		id: create_if_block_3$4.name,
    		type: "if",
    		source: "(14:0) {#if 'forlogsSymbol' == type && 'wb-dark' != theme}",
    		ctx
    	});

    	return block;
    }

    // (18:0) {#if 'forlogsSymbol' == type && 'wb-dark' == theme}
    function create_if_block_2$9(ctx) {
    	let svg;
    	let path0;
    	let circle0;
    	let circle1;
    	let path1;
    	let path2;
    	let svg_class_value;
    	let svg_alt_value;
    	let svg_title_value;

    	const block = {
    		c: function create() {
    			svg = svg_element("svg");
    			path0 = svg_element("path");
    			circle0 = svg_element("circle");
    			circle1 = svg_element("circle");
    			path1 = svg_element("path");
    			path2 = svg_element("path");
    			attr_dev(path0, "fill", "none");
    			attr_dev(path0, "d", "M.019-.003h1181.105V1181.1H.02z");
    			add_location(path0, file$h, 18, 295, 1938);
    			attr_dev(circle0, "cx", "1763.4");
    			attr_dev(circle0, "cy", "563.352");
    			attr_dev(circle0, "r", "460.653");
    			attr_dev(circle0, "fill", "#ff5259");
    			attr_dev(circle0, "transform", "translate(-1263.182 -1.681) scale(1.05123)");
    			add_location(circle0, file$h, 18, 350, 1993);
    			attr_dev(circle1, "cx", "1058.3");
    			attr_dev(circle1, "cy", "1019.56");
    			attr_dev(circle1, "r", "736.788");
    			attr_dev(circle1, "fill", "#2d3748");
    			attr_dev(circle1, "stroke", "#2d3748");
    			attr_dev(circle1, "stroke-width", "2.24");
    			attr_dev(circle1, "transform", "translate(119.096 136.327) scale(.4455)");
    			add_location(circle1, file$h, 18, 466, 2109);
    			attr_dev(path1, "d", "M20 21h-8c-1.097 0-2-.903-2-2v-2c0-1.097.903-2 2-2h8c1.097 0 2 .903 2 2v2c0 1.097-.903 2-2 2zm-8-4v2h8v-2h-8z");
    			attr_dev(path1, "fill", "#f8eed5");
    			attr_dev(path1, "fill-rule", "nonzero");
    			attr_dev(path1, "stroke", "#f8eed5");
    			attr_dev(path1, "stroke-width", ".08");
    			attr_dev(path1, "stroke-linecap", "butt");
    			attr_dev(path1, "stroke-miterlimit", "2");
    			attr_dev(path1, "transform", "matrix(13.18194 0 0 13.1819 379.66 366.5)");
    			add_location(path1, file$h, 18, 616, 2259);
    			attr_dev(path2, "d", "M28 4H4c-1.097 0-2 .903-2 2v4c0 1.097.903 2 2 2v16c0 1.097.903 2 2 2h20c1.097 0 2-.903 2-2V12c1.097 0 2-.903 2-2V6c0-1.097-.903-2-2-2zm-2 24H6V12h20v16zm2-18H4V6h24v4z");
    			attr_dev(path2, "fill", "#f8eed5");
    			attr_dev(path2, "fill-rule", "nonzero");
    			attr_dev(path2, "stroke", "#f8eed5");
    			attr_dev(path2, "stroke-width", ".08");
    			attr_dev(path2, "stroke-linecap", "butt");
    			attr_dev(path2, "stroke-miterlimit", "2");
    			attr_dev(path2, "transform", "matrix(13.18194 0 0 13.1819 379.66 366.5)");
    			add_location(path2, file$h, 18, 906, 2549);
    			attr_dev(svg, "class", svg_class_value = /*classes*/ ctx[5] || null);
    			attr_dev(svg, "alt", svg_alt_value = /*alt*/ ctx[2] || /*title*/ ctx[3] || null);
    			attr_dev(svg, "title", svg_title_value = /*title*/ ctx[3] || /*alt*/ ctx[2] | null);
    			attr_dev(svg, "aria-hidden", /*ariaHidden*/ ctx[4]);
    			attr_dev(svg, "height", "100%");
    			attr_dev(svg, "viewBox", "0 0 1182 1182");
    			attr_dev(svg, "xmlns", "http://www.w3.org/2000/svg");
    			attr_dev(svg, "fill-rule", "evenodd");
    			attr_dev(svg, "clip-rule", "evenodd");
    			attr_dev(svg, "stroke-linecap", "round");
    			attr_dev(svg, "stroke-linejoin", "round");
    			attr_dev(svg, "stroke-miterlimit", "1.5");
    			add_location(svg, file$h, 18, 4, 1647);
    		},
    		m: function mount(target, anchor) {
    			insert_dev(target, svg, anchor);
    			append_dev(svg, path0);
    			append_dev(svg, circle0);
    			append_dev(svg, circle1);
    			append_dev(svg, path1);
    			append_dev(svg, path2);
    		},
    		p: function update(ctx, dirty) {
    			if (dirty & /*classes*/ 32 && svg_class_value !== (svg_class_value = /*classes*/ ctx[5] || null)) {
    				attr_dev(svg, "class", svg_class_value);
    			}

    			if (dirty & /*alt, title*/ 12 && svg_alt_value !== (svg_alt_value = /*alt*/ ctx[2] || /*title*/ ctx[3] || null)) {
    				attr_dev(svg, "alt", svg_alt_value);
    			}

    			if (dirty & /*title, alt*/ 12 && svg_title_value !== (svg_title_value = /*title*/ ctx[3] || /*alt*/ ctx[2] | null)) {
    				attr_dev(svg, "title", svg_title_value);
    			}

    			if (dirty & /*ariaHidden*/ 16) {
    				attr_dev(svg, "aria-hidden", /*ariaHidden*/ ctx[4]);
    			}
    		},
    		d: function destroy(detaching) {
    			if (detaching) detach_dev(svg);
    		}
    	};

    	dispatch_dev("SvelteRegisterBlock", {
    		block,
    		id: create_if_block_2$9.name,
    		type: "if",
    		source: "(18:0) {#if 'forlogsSymbol' == type && 'wb-dark' == theme}",
    		ctx
    	});

    	return block;
    }

    // (22:0) {#if 'forlogs' == type && 'wb-dark' != theme}
    function create_if_block_1$a(ctx) {
    	let svg;
    	let circle0;
    	let circle1;
    	let path0;
    	let path1;
    	let path2;
    	let g0;
    	let path3;
    	let path4;
    	let g1;
    	let path5;
    	let path6;
    	let svg_class_value;
    	let svg_alt_value;
    	let svg_title_value;

    	const block = {
    		c: function create() {
    			svg = svg_element("svg");
    			circle0 = svg_element("circle");
    			circle1 = svg_element("circle");
    			path0 = svg_element("path");
    			path1 = svg_element("path");
    			path2 = svg_element("path");
    			g0 = svg_element("g");
    			path3 = svg_element("path");
    			path4 = svg_element("path");
    			g1 = svg_element("g");
    			path5 = svg_element("path");
    			path6 = svg_element("path");
    			attr_dev(circle0, "cx", "1763.4");
    			attr_dev(circle0, "cy", "563.352");
    			attr_dev(circle0, "r", "460.653");
    			attr_dev(circle0, "fill", "#ff5259");
    			attr_dev(circle0, "transform", "translate(-1369.398 -107.054) scale(1.05116)");
    			add_location(circle0, file$h, 22, 268, 3225);
    			attr_dev(circle1, "cx", "1058.3");
    			attr_dev(circle1, "cy", "1019.56");
    			attr_dev(circle1, "r", "736.788");
    			attr_dev(circle1, "fill", "#2d3748");
    			attr_dev(circle1, "stroke", "#4b3145");
    			attr_dev(circle1, "stroke-width", "2.24");
    			attr_dev(circle1, "transform", "translate(12.77 30.943) scale(.44547)");
    			add_location(circle1, file$h, 22, 386, 3343);
    			attr_dev(path0, "d", "M1918.42 5635.64h318.66v.37c251.35 12.04 451.7 219.9 451.7 474.19v19.38c0 254.3-200.35 462.16-451.7 474.2v.37h-318.66v-968.51z");
    			attr_dev(path0, "fill", "#2d3748");
    			attr_dev(path0, "stroke", "#2d3748");
    			attr_dev(path0, "stroke-linejoin", "miter");
    			attr_dev(path0, "transform", "matrix(1 0 0 1 141.407 -5634.747)");
    			add_location(path0, file$h, 22, 534, 3491);
    			attr_dev(path1, "d", "M1557.42 5635.64v968.51H884.354c164.076-98.92 273.906-278.86 273.906-484.26 0-205.39-109.83-385.33-273.906-484.25h673.066z");
    			attr_dev(path1, "fill", "#2d3748");
    			attr_dev(path1, "stroke", "#2d3748");
    			attr_dev(path1, "stroke-linejoin", "miter");
    			attr_dev(path1, "transform", "matrix(1 0 0 1 -117.59 -5634.747)");
    			add_location(path1, file$h, 22, 774, 3731);
    			attr_dev(path2, "fill", "#2d3748");
    			attr_dev(path2, "stroke", "#2d3748");
    			attr_dev(path2, "stroke-miterlimit", "1");
    			attr_dev(path2, "d", "M1557 118.85h361v968.504h-361z");
    			attr_dev(path2, "transform", "matrix(2.05786 0 0 1 -1846.257 -117.942)");
    			add_location(path2, file$h, 22, 1010, 3967);
    			attr_dev(path3, "d", "M554.721 843.978V1181.1h-67.242v-134.94H386.388c-9.454 0-17.382-3.2-23.786-9.6-6.709-6.71-10.064-14.8-10.064-24.25V843.978h67.242v134.941h67.699V843.978h67.242z");
    			attr_dev(path3, "fill", "#ff5259");
    			attr_dev(path3, "stroke", "#ff5259");
    			attr_dev(path3, "transform", "matrix(1.13469 0 0 1.1347 781.993 -663.795)");
    			add_location(path3, file$h, 22, 1245, 4202);
    			attr_dev(path4, "d", "M689.662 1113.86V843.978H622.42V1181.1h202.183v-67.24H689.662zM1094.49 877.37v269.88c0 9.46-3.21 17.38-9.61 23.79-6.71 6.71-14.64 10.06-23.79 10.06H926.152c-9.453 0-17.382-3.35-23.786-10.06-6.709-6.41-10.063-14.33-10.063-23.79V877.37c0-9.148 3.354-17.077 10.063-23.786 6.404-6.404 14.333-9.606 23.786-9.606h134.938c9.15 0 17.08 3.202 23.79 9.606 6.4 6.709 9.61 14.638 9.61 23.786zm-134.945 236.49h67.695V911.22h-67.695v202.64zM1297.13 945.07v-33.85h-67.7v202.64h67.7v-67.7h-33.85v-67.241h101.09v168.331c0 9.46-3.2 17.38-9.61 23.79-6.71 6.71-14.64 10.06-23.78 10.06h-134.94c-9.46 0-17.39-3.35-23.79-10.06-6.71-6.41-10.07-14.33-10.07-23.79V877.37c0-9.148 3.36-17.077 10.07-23.786 6.4-6.404 14.33-9.606 23.79-9.606h134.94c9.14 0 17.07 3.202 23.78 9.606 6.41 6.709 9.61 14.638 9.61 23.786v67.7h-67.24zM1567.01 945.07v-33.85h-67.7v67.699h101.55c9.15 0 17.07 3.202 23.78 9.606 6.41 6.709 9.61 14.635 9.61 23.785v134.94c0 9.46-3.2 17.38-9.61 23.79-6.71 6.71-14.63 10.06-23.78 10.06h-134.94c-9.46 0-17.38-3.35-23.79-10.06-6.71-6.41-10.06-14.33-10.06-23.79v-67.24h67.24v33.85h67.7v-67.7h-101.09c-9.46 0-17.38-3.2-23.79-9.6-6.71-6.71-10.06-14.8-10.06-24.25V877.37c0-9.148 3.35-17.077 10.06-23.786 6.41-6.404 14.33-9.606 23.79-9.606h134.94c9.15 0 17.07 3.202 23.78 9.606 6.41 6.709 9.61 14.638 9.61 23.786v67.7h-67.24z");
    			attr_dev(path4, "fill", "#f8eed5");
    			attr_dev(path4, "stroke", "#f8eed5");
    			attr_dev(path4, "transform", "matrix(1.13469 0 0 1.1347 781.993 -663.795)");
    			add_location(path4, file$h, 22, 1505, 4462);
    			attr_dev(g0, "fill-rule", "nonzero");
    			attr_dev(g0, "stroke-width", ".88");
    			attr_dev(g0, "stroke-linecap", "butt");
    			attr_dev(g0, "stroke-miterlimit", "2");
    			add_location(g0, file$h, 22, 1159, 4116);
    			attr_dev(path5, "d", "M20 21h-8c-1.097 0-2-.903-2-2v-2c0-1.097.903-2 2-2h8c1.097 0 2 .903 2 2v2c0 1.097-.903 2-2 2zm-8-4v2h8v-2h-8z");
    			attr_dev(path5, "fill", "#f8eed5");
    			attr_dev(path5, "fill-rule", "nonzero");
    			attr_dev(path5, "stroke", "#f8eed5");
    			attr_dev(path5, "stroke-width", ".08");
    			attr_dev(path5, "stroke-linecap", "butt");
    			attr_dev(path5, "stroke-miterlimit", "2");
    			attr_dev(path5, "transform", "matrix(13.18189 0 0 13.1819 273.264 261.012)");
    			add_location(path5, file$h, 22, 2919, 5876);
    			attr_dev(path6, "d", "M28 4H4c-1.097 0-2 .903-2 2v4c0 1.097.903 2 2 2v16c0 1.097.903 2 2 2h20c1.097 0 2-.903 2-2V12c1.097 0 2-.903 2-2V6c0-1.097-.903-2-2-2zm-2 24H6V12h20v16zm2-18H4V6h24v4z");
    			attr_dev(path6, "fill", "#f8eed5");
    			attr_dev(path6, "fill-rule", "nonzero");
    			attr_dev(path6, "stroke", "#f8eed5");
    			attr_dev(path6, "stroke-width", ".08");
    			attr_dev(path6, "stroke-linecap", "butt");
    			attr_dev(path6, "stroke-miterlimit", "2");
    			attr_dev(path6, "transform", "matrix(13.18189 0 0 13.1819 273.264 261.012)");
    			add_location(path6, file$h, 22, 3212, 6169);
    			add_location(g1, file$h, 22, 2916, 5873);
    			attr_dev(svg, "class", svg_class_value = /*classes*/ ctx[5] || null);
    			attr_dev(svg, "alt", svg_alt_value = /*alt*/ ctx[2] || /*title*/ ctx[3] || null);
    			attr_dev(svg, "title", svg_title_value = /*title*/ ctx[3] || /*alt*/ ctx[2] || null);
    			attr_dev(svg, "height", "100%");
    			attr_dev(svg, "viewBox", "0 0 2832 971");
    			attr_dev(svg, "xmlns", "http://www.w3.org/2000/svg");
    			attr_dev(svg, "fill-rule", "evenodd");
    			attr_dev(svg, "clip-rule", "evenodd");
    			attr_dev(svg, "stroke-linecap", "round");
    			attr_dev(svg, "stroke-linejoin", "round");
    			attr_dev(svg, "stroke-miterlimit", "1.5");
    			add_location(svg, file$h, 22, 4, 2961);
    		},
    		m: function mount(target, anchor) {
    			insert_dev(target, svg, anchor);
    			append_dev(svg, circle0);
    			append_dev(svg, circle1);
    			append_dev(svg, path0);
    			append_dev(svg, path1);
    			append_dev(svg, path2);
    			append_dev(svg, g0);
    			append_dev(g0, path3);
    			append_dev(g0, path4);
    			append_dev(svg, g1);
    			append_dev(g1, path5);
    			append_dev(g1, path6);
    		},
    		p: function update(ctx, dirty) {
    			if (dirty & /*classes*/ 32 && svg_class_value !== (svg_class_value = /*classes*/ ctx[5] || null)) {
    				attr_dev(svg, "class", svg_class_value);
    			}

    			if (dirty & /*alt, title*/ 12 && svg_alt_value !== (svg_alt_value = /*alt*/ ctx[2] || /*title*/ ctx[3] || null)) {
    				attr_dev(svg, "alt", svg_alt_value);
    			}

    			if (dirty & /*title, alt*/ 12 && svg_title_value !== (svg_title_value = /*title*/ ctx[3] || /*alt*/ ctx[2] || null)) {
    				attr_dev(svg, "title", svg_title_value);
    			}
    		},
    		d: function destroy(detaching) {
    			if (detaching) detach_dev(svg);
    		}
    	};

    	dispatch_dev("SvelteRegisterBlock", {
    		block,
    		id: create_if_block_1$a.name,
    		type: "if",
    		source: "(22:0) {#if 'forlogs' == type && 'wb-dark' != theme}",
    		ctx
    	});

    	return block;
    }

    // (26:0) {#if 'forlogs' == type && 'wb-dark' == theme}
    function create_if_block$d(ctx) {
    	let svg;
    	let path0;
    	let circle0;
    	let circle1;
    	let path1;
    	let path2;
    	let path3;
    	let g0;
    	let path4;
    	let path5;
    	let g1;
    	let path6;
    	let path7;
    	let svg_class_value;
    	let svg_alt_value;
    	let svg_title_value;

    	const block = {
    		c: function create() {
    			svg = svg_element("svg");
    			path0 = svg_element("path");
    			circle0 = svg_element("circle");
    			circle1 = svg_element("circle");
    			path1 = svg_element("path");
    			path2 = svg_element("path");
    			path3 = svg_element("path");
    			g0 = svg_element("g");
    			path4 = svg_element("path");
    			path5 = svg_element("path");
    			g1 = svg_element("g");
    			path6 = svg_element("path");
    			path7 = svg_element("path");
    			attr_dev(path0, "fill", "none");
    			attr_dev(path0, "d", "M-.001.002H3188.97v1181.1H0z");
    			add_location(path0, file$h, 26, 269, 6853);
    			attr_dev(circle0, "cx", "1763.4");
    			attr_dev(circle0, "cy", "563.352");
    			attr_dev(circle0, "r", "460.653");
    			attr_dev(circle0, "fill", "#ff5259");
    			attr_dev(circle0, "transform", "translate(-1252.232 -1.624) scale(1.05116)");
    			add_location(circle0, file$h, 26, 321, 6905);
    			attr_dev(circle1, "cx", "1058.3");
    			attr_dev(circle1, "cy", "1019.56");
    			attr_dev(circle1, "r", "736.788");
    			attr_dev(circle1, "fill", "#2d3748");
    			attr_dev(circle1, "stroke", "#4b3145");
    			attr_dev(circle1, "stroke-width", "2.24");
    			attr_dev(circle1, "transform", "translate(129.936 136.373) scale(.44547)");
    			add_location(circle1, file$h, 26, 437, 7021);
    			attr_dev(path1, "d", "M1918.42 5635.64h318.66v.37c251.35 12.04 451.7 219.9 451.7 474.19v19.38c0 254.3-200.35 462.16-451.7 474.2v.37h-318.66v-968.51z");
    			attr_dev(path1, "fill", "#f8eed5");
    			attr_dev(path1, "stroke", "#f8eed5");
    			attr_dev(path1, "stroke-linejoin", "miter");
    			attr_dev(path1, "transform", "matrix(1 0 0 1 258.573 -5529.317)");
    			add_location(path1, file$h, 26, 588, 7172);
    			attr_dev(path2, "d", "M1557.42 5635.64v968.51H884.354c164.076-98.92 273.906-278.86 273.906-484.26 0-205.39-109.83-385.33-273.906-484.25h673.066z");
    			attr_dev(path2, "fill", "#f8eed5");
    			attr_dev(path2, "stroke", "#f8eed5");
    			attr_dev(path2, "stroke-linejoin", "miter");
    			attr_dev(path2, "transform", "matrix(1 0 0 1 -.425 -5529.317)");
    			add_location(path2, file$h, 26, 828, 7412);
    			attr_dev(path3, "fill", "#f8eed5");
    			attr_dev(path3, "stroke", "#f8eed5");
    			attr_dev(path3, "stroke-miterlimit", "1");
    			attr_dev(path3, "d", "M1557 118.85h361v968.504h-361z");
    			attr_dev(path3, "transform", "matrix(2.05786 0 0 1 -1729.09 -12.512)");
    			add_location(path3, file$h, 26, 1062, 7646);
    			attr_dev(path4, "d", "M554.721 843.978V1181.1h-67.242v-134.94H386.388c-9.454 0-17.382-3.2-23.786-9.6-6.709-6.71-10.064-14.8-10.064-24.25V843.978h67.242v134.941h67.699V843.978h67.242z");
    			attr_dev(path4, "fill", "#ff5259");
    			attr_dev(path4, "stroke", "#f8eed5");
    			attr_dev(path4, "transform", "matrix(1.13469 0 0 1.1347 899.159 -558.365)");
    			add_location(path4, file$h, 26, 1295, 7879);
    			attr_dev(path5, "d", "M689.662 1113.86V843.978H622.42V1181.1h202.183v-67.24H689.662zM1094.49 877.37v269.88c0 9.46-3.21 17.38-9.61 23.79-6.71 6.71-14.64 10.06-23.79 10.06H926.152c-9.453 0-17.382-3.35-23.786-10.06-6.709-6.41-10.063-14.33-10.063-23.79V877.37c0-9.148 3.354-17.077 10.063-23.786 6.404-6.404 14.333-9.606 23.786-9.606h134.938c9.15 0 17.08 3.202 23.79 9.606 6.4 6.709 9.61 14.638 9.61 23.786zm-134.945 236.49h67.695V911.22h-67.695v202.64zM1297.13 945.07v-33.85h-67.7v202.64h67.7v-67.7h-33.85v-67.241h101.09v168.331c0 9.46-3.2 17.38-9.61 23.79-6.71 6.71-14.64 10.06-23.78 10.06h-134.94c-9.46 0-17.39-3.35-23.79-10.06-6.71-6.41-10.07-14.33-10.07-23.79V877.37c0-9.148 3.36-17.077 10.07-23.786 6.4-6.404 14.33-9.606 23.79-9.606h134.94c9.14 0 17.07 3.202 23.78 9.606 6.41 6.709 9.61 14.638 9.61 23.786v67.7h-67.24zM1567.01 945.07v-33.85h-67.7v67.699h101.55c9.15 0 17.07 3.202 23.78 9.606 6.41 6.709 9.61 14.635 9.61 23.785v134.94c0 9.46-3.2 17.38-9.61 23.79-6.71 6.71-14.63 10.06-23.78 10.06h-134.94c-9.46 0-17.38-3.35-23.79-10.06-6.71-6.41-10.06-14.33-10.06-23.79v-67.24h67.24v33.85h67.7v-67.7h-101.09c-9.46 0-17.38-3.2-23.79-9.6-6.71-6.71-10.06-14.8-10.06-24.25V877.37c0-9.148 3.35-17.077 10.06-23.786 6.41-6.404 14.33-9.606 23.79-9.606h134.94c9.15 0 17.07 3.202 23.78 9.606 6.41 6.709 9.61 14.638 9.61 23.786v67.7h-67.24z");
    			attr_dev(path5, "fill", "#2d3748");
    			attr_dev(path5, "stroke", "#2d3748");
    			attr_dev(path5, "transform", "matrix(1.13469 0 0 1.1347 899.159 -558.365)");
    			add_location(path5, file$h, 26, 1555, 8139);
    			attr_dev(g0, "fill-rule", "nonzero");
    			attr_dev(g0, "stroke-width", ".88");
    			attr_dev(g0, "stroke-linecap", "butt");
    			attr_dev(g0, "stroke-miterlimit", "2");
    			add_location(g0, file$h, 26, 1209, 7793);
    			attr_dev(path6, "d", "M20 21h-8c-1.097 0-2-.903-2-2v-2c0-1.097.903-2 2-2h8c1.097 0 2 .903 2 2v2c0 1.097-.903 2-2 2zm-8-4v2h8v-2h-8z");
    			attr_dev(path6, "fill", "#f8eed5");
    			attr_dev(path6, "fill-rule", "nonzero");
    			attr_dev(path6, "stroke", "#f8eed5");
    			attr_dev(path6, "stroke-width", ".08");
    			attr_dev(path6, "stroke-linecap", "butt");
    			attr_dev(path6, "stroke-miterlimit", "2");
    			attr_dev(path6, "transform", "matrix(13.18189 0 0 13.1819 390.43 366.442)");
    			add_location(path6, file$h, 26, 2969, 9553);
    			attr_dev(path7, "d", "M28 4H4c-1.097 0-2 .903-2 2v4c0 1.097.903 2 2 2v16c0 1.097.903 2 2 2h20c1.097 0 2-.903 2-2V12c1.097 0 2-.903 2-2V6c0-1.097-.903-2-2-2zm-2 24H6V12h20v16zm2-18H4V6h24v4z");
    			attr_dev(path7, "fill", "#f8eed5");
    			attr_dev(path7, "fill-rule", "nonzero");
    			attr_dev(path7, "stroke", "#f8eed5");
    			attr_dev(path7, "stroke-width", ".08");
    			attr_dev(path7, "stroke-linecap", "butt");
    			attr_dev(path7, "stroke-miterlimit", "2");
    			attr_dev(path7, "transform", "matrix(13.18189 0 0 13.1819 390.43 366.442)");
    			add_location(path7, file$h, 26, 3261, 9845);
    			add_location(g1, file$h, 26, 2966, 9550);
    			attr_dev(svg, "class", svg_class_value = /*classes*/ ctx[5] || null);
    			attr_dev(svg, "alt", svg_alt_value = /*alt*/ ctx[2] || /*title*/ ctx[3] || null);
    			attr_dev(svg, "title", svg_title_value = /*title*/ ctx[3] || /*alt*/ ctx[2] || null);
    			attr_dev(svg, "height", "100%");
    			attr_dev(svg, "viewBox", "0 0 3189 1182");
    			attr_dev(svg, "xmlns", "http://www.w3.org/2000/svg");
    			attr_dev(svg, "fill-rule", "evenodd");
    			attr_dev(svg, "clip-rule", "evenodd");
    			attr_dev(svg, "stroke-linecap", "round");
    			attr_dev(svg, "stroke-linejoin", "round");
    			attr_dev(svg, "stroke-miterlimit", "1.5");
    			add_location(svg, file$h, 26, 4, 6588);
    		},
    		m: function mount(target, anchor) {
    			insert_dev(target, svg, anchor);
    			append_dev(svg, path0);
    			append_dev(svg, circle0);
    			append_dev(svg, circle1);
    			append_dev(svg, path1);
    			append_dev(svg, path2);
    			append_dev(svg, path3);
    			append_dev(svg, g0);
    			append_dev(g0, path4);
    			append_dev(g0, path5);
    			append_dev(svg, g1);
    			append_dev(g1, path6);
    			append_dev(g1, path7);
    		},
    		p: function update(ctx, dirty) {
    			if (dirty & /*classes*/ 32 && svg_class_value !== (svg_class_value = /*classes*/ ctx[5] || null)) {
    				attr_dev(svg, "class", svg_class_value);
    			}

    			if (dirty & /*alt, title*/ 12 && svg_alt_value !== (svg_alt_value = /*alt*/ ctx[2] || /*title*/ ctx[3] || null)) {
    				attr_dev(svg, "alt", svg_alt_value);
    			}

    			if (dirty & /*title, alt*/ 12 && svg_title_value !== (svg_title_value = /*title*/ ctx[3] || /*alt*/ ctx[2] || null)) {
    				attr_dev(svg, "title", svg_title_value);
    			}
    		},
    		d: function destroy(detaching) {
    			if (detaching) detach_dev(svg);
    		}
    	};

    	dispatch_dev("SvelteRegisterBlock", {
    		block,
    		id: create_if_block$d.name,
    		type: "if",
    		source: "(26:0) {#if 'forlogs' == type && 'wb-dark' == theme}",
    		ctx
    	});

    	return block;
    }

    function create_fragment$i(ctx) {
    	let t0;
    	let t1;
    	let t2;
    	let if_block3_anchor;
    	let if_block0 = 'forlogsSymbol' == /*type*/ ctx[1] && 'wb-dark' != /*theme*/ ctx[0] && create_if_block_3$4(ctx);
    	let if_block1 = 'forlogsSymbol' == /*type*/ ctx[1] && 'wb-dark' == /*theme*/ ctx[0] && create_if_block_2$9(ctx);
    	let if_block2 = 'forlogs' == /*type*/ ctx[1] && 'wb-dark' != /*theme*/ ctx[0] && create_if_block_1$a(ctx);
    	let if_block3 = 'forlogs' == /*type*/ ctx[1] && 'wb-dark' == /*theme*/ ctx[0] && create_if_block$d(ctx);

    	const block = {
    		c: function create() {
    			if (if_block0) if_block0.c();
    			t0 = space();
    			if (if_block1) if_block1.c();
    			t1 = space();
    			if (if_block2) if_block2.c();
    			t2 = space();
    			if (if_block3) if_block3.c();
    			if_block3_anchor = empty();
    		},
    		l: function claim(nodes) {
    			throw new Error("options.hydrate only works if the component was compiled with the `hydratable: true` option");
    		},
    		m: function mount(target, anchor) {
    			if (if_block0) if_block0.m(target, anchor);
    			insert_dev(target, t0, anchor);
    			if (if_block1) if_block1.m(target, anchor);
    			insert_dev(target, t1, anchor);
    			if (if_block2) if_block2.m(target, anchor);
    			insert_dev(target, t2, anchor);
    			if (if_block3) if_block3.m(target, anchor);
    			insert_dev(target, if_block3_anchor, anchor);
    		},
    		p: function update(ctx, [dirty]) {
    			if ('forlogsSymbol' == /*type*/ ctx[1] && 'wb-dark' != /*theme*/ ctx[0]) {
    				if (if_block0) {
    					if_block0.p(ctx, dirty);
    				} else {
    					if_block0 = create_if_block_3$4(ctx);
    					if_block0.c();
    					if_block0.m(t0.parentNode, t0);
    				}
    			} else if (if_block0) {
    				if_block0.d(1);
    				if_block0 = null;
    			}

    			if ('forlogsSymbol' == /*type*/ ctx[1] && 'wb-dark' == /*theme*/ ctx[0]) {
    				if (if_block1) {
    					if_block1.p(ctx, dirty);
    				} else {
    					if_block1 = create_if_block_2$9(ctx);
    					if_block1.c();
    					if_block1.m(t1.parentNode, t1);
    				}
    			} else if (if_block1) {
    				if_block1.d(1);
    				if_block1 = null;
    			}

    			if ('forlogs' == /*type*/ ctx[1] && 'wb-dark' != /*theme*/ ctx[0]) {
    				if (if_block2) {
    					if_block2.p(ctx, dirty);
    				} else {
    					if_block2 = create_if_block_1$a(ctx);
    					if_block2.c();
    					if_block2.m(t2.parentNode, t2);
    				}
    			} else if (if_block2) {
    				if_block2.d(1);
    				if_block2 = null;
    			}

    			if ('forlogs' == /*type*/ ctx[1] && 'wb-dark' == /*theme*/ ctx[0]) {
    				if (if_block3) {
    					if_block3.p(ctx, dirty);
    				} else {
    					if_block3 = create_if_block$d(ctx);
    					if_block3.c();
    					if_block3.m(if_block3_anchor.parentNode, if_block3_anchor);
    				}
    			} else if (if_block3) {
    				if_block3.d(1);
    				if_block3 = null;
    			}
    		},
    		i: noop$1,
    		o: noop$1,
    		d: function destroy(detaching) {
    			if (if_block0) if_block0.d(detaching);
    			if (detaching) detach_dev(t0);
    			if (if_block1) if_block1.d(detaching);
    			if (detaching) detach_dev(t1);
    			if (if_block2) if_block2.d(detaching);
    			if (detaching) detach_dev(t2);
    			if (if_block3) if_block3.d(detaching);
    			if (detaching) detach_dev(if_block3_anchor);
    		}
    	};

    	dispatch_dev("SvelteRegisterBlock", {
    		block,
    		id: create_fragment$i.name,
    		type: "component",
    		source: "",
    		ctx
    	});

    	return block;
    }

    function instance$i($$self, $$props, $$invalidate) {
    	let { $$slots: slots = {}, $$scope } = $$props;
    	validate_slots('Logo', slots, []);
    	let { theme = 'wb-light' } = $$props;
    	let { type = 'forlogs' } = $$props;
    	let { alt = '4LOGS logo' } = $$props;
    	let { title = '4LOGS logo' } = $$props;
    	let { ariaHidden = true } = $$props;
    	let { classes = 'wb-max-w-full' } = $$props;
    	const writable_props = ['theme', 'type', 'alt', 'title', 'ariaHidden', 'classes'];

    	Object.keys($$props).forEach(key => {
    		if (!~writable_props.indexOf(key) && key.slice(0, 2) !== '$$' && key !== 'slot') console.warn(`<Logo> was created with unknown prop '${key}'`);
    	});

    	$$self.$$set = $$props => {
    		if ('theme' in $$props) $$invalidate(0, theme = $$props.theme);
    		if ('type' in $$props) $$invalidate(1, type = $$props.type);
    		if ('alt' in $$props) $$invalidate(2, alt = $$props.alt);
    		if ('title' in $$props) $$invalidate(3, title = $$props.title);
    		if ('ariaHidden' in $$props) $$invalidate(4, ariaHidden = $$props.ariaHidden);
    		if ('classes' in $$props) $$invalidate(5, classes = $$props.classes);
    	};

    	$$self.$capture_state = () => ({
    		theme,
    		type,
    		alt,
    		title,
    		ariaHidden,
    		classes
    	});

    	$$self.$inject_state = $$props => {
    		if ('theme' in $$props) $$invalidate(0, theme = $$props.theme);
    		if ('type' in $$props) $$invalidate(1, type = $$props.type);
    		if ('alt' in $$props) $$invalidate(2, alt = $$props.alt);
    		if ('title' in $$props) $$invalidate(3, title = $$props.title);
    		if ('ariaHidden' in $$props) $$invalidate(4, ariaHidden = $$props.ariaHidden);
    		if ('classes' in $$props) $$invalidate(5, classes = $$props.classes);
    	};

    	if ($$props && "$$inject" in $$props) {
    		$$self.$inject_state($$props.$$inject);
    	}

    	return [theme, type, alt, title, ariaHidden, classes];
    }

    class Logo extends SvelteComponentDev {
    	constructor(options) {
    		super(options);

    		init(this, options, instance$i, create_fragment$i, safe_not_equal, {
    			theme: 0,
    			type: 1,
    			alt: 2,
    			title: 3,
    			ariaHidden: 4,
    			classes: 5
    		});

    		dispatch_dev("SvelteRegisterComponent", {
    			component: this,
    			tagName: "Logo",
    			options,
    			id: create_fragment$i.name
    		});
    	}

    	get theme() {
    		throw new Error("<Logo>: Props cannot be read directly from the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	set theme(value) {
    		throw new Error("<Logo>: Props cannot be set directly on the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	get type() {
    		throw new Error("<Logo>: Props cannot be read directly from the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	set type(value) {
    		throw new Error("<Logo>: Props cannot be set directly on the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	get alt() {
    		throw new Error("<Logo>: Props cannot be read directly from the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	set alt(value) {
    		throw new Error("<Logo>: Props cannot be set directly on the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	get title() {
    		throw new Error("<Logo>: Props cannot be read directly from the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	set title(value) {
    		throw new Error("<Logo>: Props cannot be set directly on the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	get ariaHidden() {
    		throw new Error("<Logo>: Props cannot be read directly from the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	set ariaHidden(value) {
    		throw new Error("<Logo>: Props cannot be set directly on the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	get classes() {
    		throw new Error("<Logo>: Props cannot be read directly from the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	set classes(value) {
    		throw new Error("<Logo>: Props cannot be set directly on the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}
    }

    /* src\app\admin\components\Logo.svelte generated by Svelte v3.41.0 */
    const file$g = "src\\app\\admin\\components\\Logo.svelte";

    function create_fragment$h(ctx) {
    	let span0;
    	let logo0;
    	let t;
    	let span1;
    	let logo1;
    	let current;

    	logo0 = new Logo({
    			props: { theme: "wb-light", type: "forlogsSymbol" },
    			$$inline: true
    		});

    	logo1 = new Logo({
    			props: { theme: "wb-light", type: "forlogs" },
    			$$inline: true
    		});

    	const block = {
    		c: function create() {
    			span0 = element("span");
    			create_component(logo0.$$.fragment);
    			t = space();
    			span1 = element("span");
    			create_component(logo1.$$.fragment);
    			attr_dev(span0, "aria-hidden", "true");
    			attr_dev(span0, "class", "wb-m-0 wb-flex-grow-0 md:wb-hidden sm:wb-inline-block wb-hidden wb-h-10");
    			add_location(span0, file$g, 5, 0, 72);
    			attr_dev(span1, "aria-hidden", "true");
    			attr_dev(span1, "class", "wb-m-0 wb-flex-grow-0 md:wb-inline-block wb-hidden wb-h-10");
    			add_location(span1, file$g, 9, 0, 237);
    		},
    		l: function claim(nodes) {
    			throw new Error("options.hydrate only works if the component was compiled with the `hydratable: true` option");
    		},
    		m: function mount(target, anchor) {
    			insert_dev(target, span0, anchor);
    			mount_component(logo0, span0, null);
    			insert_dev(target, t, anchor);
    			insert_dev(target, span1, anchor);
    			mount_component(logo1, span1, null);
    			current = true;
    		},
    		p: noop$1,
    		i: function intro(local) {
    			if (current) return;
    			transition_in(logo0.$$.fragment, local);
    			transition_in(logo1.$$.fragment, local);
    			current = true;
    		},
    		o: function outro(local) {
    			transition_out(logo0.$$.fragment, local);
    			transition_out(logo1.$$.fragment, local);
    			current = false;
    		},
    		d: function destroy(detaching) {
    			if (detaching) detach_dev(span0);
    			destroy_component(logo0);
    			if (detaching) detach_dev(t);
    			if (detaching) detach_dev(span1);
    			destroy_component(logo1);
    		}
    	};

    	dispatch_dev("SvelteRegisterBlock", {
    		block,
    		id: create_fragment$h.name,
    		type: "component",
    		source: "",
    		ctx
    	});

    	return block;
    }

    function instance$h($$self, $$props, $$invalidate) {
    	let { $$slots: slots = {}, $$scope } = $$props;
    	validate_slots('Logo', slots, []);
    	const writable_props = [];

    	Object.keys($$props).forEach(key => {
    		if (!~writable_props.indexOf(key) && key.slice(0, 2) !== '$$' && key !== 'slot') console.warn(`<Logo> was created with unknown prop '${key}'`);
    	});

    	$$self.$capture_state = () => ({ Logo });
    	return [];
    }

    class Logo_1 extends SvelteComponentDev {
    	constructor(options) {
    		super(options);
    		init(this, options, instance$h, create_fragment$h, safe_not_equal, {});

    		dispatch_dev("SvelteRegisterComponent", {
    			component: this,
    			tagName: "Logo_1",
    			options,
    			id: create_fragment$h.name
    		});
    	}
    }

    /* src\app\admin\components\Navbar.svelte generated by Svelte v3.41.0 */
    const file$f = "src\\app\\admin\\components\\Navbar.svelte";
    const get_notifications_slot_changes = dirty => ({});
    const get_notifications_slot_context = ctx => ({});
    const get_spinner_slot_changes = dirty => ({});
    const get_spinner_slot_context = ctx => ({});
    const get_toolbar_slot_changes = dirty => ({});
    const get_toolbar_slot_context = ctx => ({});

    // (42:12) {#if $appState.loggedIn == 'yes'}
    function create_if_block_2$8(ctx) {
    	let div;
    	let span0;
    	let t0;
    	let current_block_type_index;
    	let if_block0;
    	let t1;
    	let span1;
    	let t5;
    	let current;
    	const if_block_creators = [create_if_block_4$1, create_else_block$5];
    	const if_blocks = [];

    	function select_block_type(ctx, dirty) {
    		if (/*$forlogsState*/ ctx[2].viewing) return 0;
    		return 1;
    	}

    	current_block_type_index = select_block_type(ctx);
    	if_block0 = if_blocks[current_block_type_index] = if_block_creators[current_block_type_index](ctx);
    	let if_block1 = /*$forlogsState*/ ctx[2].viewing && create_if_block_3$3(ctx);

    	const block = {
    		c: function create() {
    			div = element("div");
    			span0 = element("span");
    			t0 = space();
    			if_block0.c();
    			t1 = space();
    			span1 = element("span");
    			span1.textContent = `(${/*config*/ ctx[4].path.logs})`;
    			t5 = space();
    			if (if_block1) if_block1.c();
    			attr_dev(span0, "class", "wb-mx-2 wb-hidden sm:wb-inline");
    			add_location(span0, file$f, 43, 20, 1387);
    			attr_dev(span1, "class", "wb-mx-2 wb-text-xs");
    			add_location(span1, file$f, 49, 20, 1691);
    			add_location(div, file$f, 42, 16, 1361);
    		},
    		m: function mount(target, anchor) {
    			insert_dev(target, div, anchor);
    			append_dev(div, span0);
    			append_dev(div, t0);
    			if_blocks[current_block_type_index].m(div, null);
    			append_dev(div, t1);
    			append_dev(div, span1);
    			append_dev(div, t5);
    			if (if_block1) if_block1.m(div, null);
    			current = true;
    		},
    		p: function update(ctx, dirty) {
    			let previous_block_index = current_block_type_index;
    			current_block_type_index = select_block_type(ctx);

    			if (current_block_type_index === previous_block_index) {
    				if_blocks[current_block_type_index].p(ctx, dirty);
    			} else {
    				group_outros();

    				transition_out(if_blocks[previous_block_index], 1, 1, () => {
    					if_blocks[previous_block_index] = null;
    				});

    				check_outros();
    				if_block0 = if_blocks[current_block_type_index];

    				if (!if_block0) {
    					if_block0 = if_blocks[current_block_type_index] = if_block_creators[current_block_type_index](ctx);
    					if_block0.c();
    				} else {
    					if_block0.p(ctx, dirty);
    				}

    				transition_in(if_block0, 1);
    				if_block0.m(div, t1);
    			}

    			if (/*$forlogsState*/ ctx[2].viewing) {
    				if (if_block1) {
    					if_block1.p(ctx, dirty);

    					if (dirty & /*$forlogsState*/ 4) {
    						transition_in(if_block1, 1);
    					}
    				} else {
    					if_block1 = create_if_block_3$3(ctx);
    					if_block1.c();
    					transition_in(if_block1, 1);
    					if_block1.m(div, null);
    				}
    			} else if (if_block1) {
    				group_outros();

    				transition_out(if_block1, 1, 1, () => {
    					if_block1 = null;
    				});

    				check_outros();
    			}
    		},
    		i: function intro(local) {
    			if (current) return;
    			transition_in(if_block0);
    			transition_in(if_block1);
    			current = true;
    		},
    		o: function outro(local) {
    			transition_out(if_block0);
    			transition_out(if_block1);
    			current = false;
    		},
    		d: function destroy(detaching) {
    			if (detaching) detach_dev(div);
    			if_blocks[current_block_type_index].d();
    			if (if_block1) if_block1.d();
    		}
    	};

    	dispatch_dev("SvelteRegisterBlock", {
    		block,
    		id: create_if_block_2$8.name,
    		type: "if",
    		source: "(42:12) {#if $appState.loggedIn == 'yes'}",
    		ctx
    	});

    	return block;
    }

    // (47:20) {:else}
    function create_else_block$5(ctx) {
    	let title;
    	let current;

    	title = new Title({
    			props: { title: t('main.title') },
    			$$inline: true
    		});

    	const block = {
    		c: function create() {
    			create_component(title.$$.fragment);
    		},
    		m: function mount(target, anchor) {
    			mount_component(title, target, anchor);
    			current = true;
    		},
    		p: noop$1,
    		i: function intro(local) {
    			if (current) return;
    			transition_in(title.$$.fragment, local);
    			current = true;
    		},
    		o: function outro(local) {
    			transition_out(title.$$.fragment, local);
    			current = false;
    		},
    		d: function destroy(detaching) {
    			destroy_component(title, detaching);
    		}
    	};

    	dispatch_dev("SvelteRegisterBlock", {
    		block,
    		id: create_else_block$5.name,
    		type: "else",
    		source: "(47:20) {:else}",
    		ctx
    	});

    	return block;
    }

    // (45:20) {#if $forlogsState.viewing}
    function create_if_block_4$1(ctx) {
    	let title;
    	let current;

    	title = new Title({
    			props: {
    				title: /*$forlogsState*/ ctx[2].viewing.path
    			},
    			$$inline: true
    		});

    	const block = {
    		c: function create() {
    			create_component(title.$$.fragment);
    		},
    		m: function mount(target, anchor) {
    			mount_component(title, target, anchor);
    			current = true;
    		},
    		p: function update(ctx, dirty) {
    			const title_changes = {};
    			if (dirty & /*$forlogsState*/ 4) title_changes.title = /*$forlogsState*/ ctx[2].viewing.path;
    			title.$set(title_changes);
    		},
    		i: function intro(local) {
    			if (current) return;
    			transition_in(title.$$.fragment, local);
    			current = true;
    		},
    		o: function outro(local) {
    			transition_out(title.$$.fragment, local);
    			current = false;
    		},
    		d: function destroy(detaching) {
    			destroy_component(title, detaching);
    		}
    	};

    	dispatch_dev("SvelteRegisterBlock", {
    		block,
    		id: create_if_block_4$1.name,
    		type: "if",
    		source: "(45:20) {#if $forlogsState.viewing}",
    		ctx
    	});

    	return block;
    }

    // (51:20) {#if $forlogsState.viewing}
    function create_if_block_3$3(ctx) {
    	let span0;
    	let t0;
    	let t1_value = /*$forlogsState*/ ctx[2].viewing.size + "";
    	let t1;
    	let t2;
    	let t3;
    	let span1;
    	let t4;
    	let t5_value = unixToSiteTimeZone(/*$forlogsState*/ ctx[2].viewing.mtime, /*config*/ ctx[4].language, { timeZone: /*config*/ ctx[4].tz }) + "";
    	let t5;
    	let t6;
    	let t7;
    	let button;
    	let current;

    	button = new Button({
    			props: {
    				withCloseIcon: true,
    				iconSize: "s",
    				text: t('main.close'),
    				handler: /*closeFileViewer*/ ctx[6]
    			},
    			$$inline: true
    		});

    	const block = {
    		c: function create() {
    			span0 = element("span");
    			t0 = text("(");
    			t1 = text(t1_value);
    			t2 = text(")");
    			t3 = space();
    			span1 = element("span");
    			t4 = text("(");
    			t5 = text(t5_value);
    			t6 = text(")");
    			t7 = space();
    			create_component(button.$$.fragment);
    			attr_dev(span0, "class", "wb-mx-2 wb-text-xs");
    			add_location(span0, file$f, 51, 24, 1824);
    			attr_dev(span1, "class", "wb-mx-2 wb-text-xs");
    			add_location(span1, file$f, 52, 24, 1919);
    		},
    		m: function mount(target, anchor) {
    			insert_dev(target, span0, anchor);
    			append_dev(span0, t0);
    			append_dev(span0, t1);
    			append_dev(span0, t2);
    			insert_dev(target, t3, anchor);
    			insert_dev(target, span1, anchor);
    			append_dev(span1, t4);
    			append_dev(span1, t5);
    			append_dev(span1, t6);
    			insert_dev(target, t7, anchor);
    			mount_component(button, target, anchor);
    			current = true;
    		},
    		p: function update(ctx, dirty) {
    			if ((!current || dirty & /*$forlogsState*/ 4) && t1_value !== (t1_value = /*$forlogsState*/ ctx[2].viewing.size + "")) set_data_dev(t1, t1_value);
    			if ((!current || dirty & /*$forlogsState*/ 4) && t5_value !== (t5_value = unixToSiteTimeZone(/*$forlogsState*/ ctx[2].viewing.mtime, /*config*/ ctx[4].language, { timeZone: /*config*/ ctx[4].tz }) + "")) set_data_dev(t5, t5_value);
    		},
    		i: function intro(local) {
    			if (current) return;
    			transition_in(button.$$.fragment, local);
    			current = true;
    		},
    		o: function outro(local) {
    			transition_out(button.$$.fragment, local);
    			current = false;
    		},
    		d: function destroy(detaching) {
    			if (detaching) detach_dev(span0);
    			if (detaching) detach_dev(t3);
    			if (detaching) detach_dev(span1);
    			if (detaching) detach_dev(t7);
    			destroy_component(button, detaching);
    		}
    	};

    	dispatch_dev("SvelteRegisterBlock", {
    		block,
    		id: create_if_block_3$3.name,
    		type: "if",
    		source: "(51:20) {#if $forlogsState.viewing}",
    		ctx
    	});

    	return block;
    }

    // (63:4) {#if $appState.loggedIn == 'yes'}
    function create_if_block$c(ctx) {
    	let span1;
    	let t0;
    	let t1;
    	let span0;
    	let t2;
    	let current;
    	let if_block = 'ready' == /*$appState*/ ctx[1].status && create_if_block_1$9(ctx);
    	const spinner_slot_template = /*#slots*/ ctx[8].spinner;
    	const spinner_slot = create_slot(spinner_slot_template, ctx, /*$$scope*/ ctx[7], get_spinner_slot_context);
    	const notifications_slot_template = /*#slots*/ ctx[8].notifications;
    	const notifications_slot = create_slot(notifications_slot_template, ctx, /*$$scope*/ ctx[7], get_notifications_slot_context);

    	const block = {
    		c: function create() {
    			span1 = element("span");
    			if (if_block) if_block.c();
    			t0 = space();
    			if (spinner_slot) spinner_slot.c();
    			t1 = space();
    			span0 = element("span");
    			t2 = space();
    			if (notifications_slot) notifications_slot.c();
    			attr_dev(span0, "class", "wb-mx-2 sm:wb-my-0 wb-my-1");
    			add_location(span0, file$f, 68, 12, 2646);
    			attr_dev(span1, "class", "wb-flex sm:wb-flex-row wb-flex-col wb-items-center wb-ml-4 wb-mr-2 sm:wb-ml-4");
    			add_location(span1, file$f, 63, 8, 2403);
    		},
    		m: function mount(target, anchor) {
    			insert_dev(target, span1, anchor);
    			if (if_block) if_block.m(span1, null);
    			append_dev(span1, t0);

    			if (spinner_slot) {
    				spinner_slot.m(span1, null);
    			}

    			append_dev(span1, t1);
    			append_dev(span1, span0);
    			append_dev(span1, t2);

    			if (notifications_slot) {
    				notifications_slot.m(span1, null);
    			}

    			current = true;
    		},
    		p: function update(ctx, dirty) {
    			if ('ready' == /*$appState*/ ctx[1].status) {
    				if (if_block) {
    					if_block.p(ctx, dirty);

    					if (dirty & /*$appState*/ 2) {
    						transition_in(if_block, 1);
    					}
    				} else {
    					if_block = create_if_block_1$9(ctx);
    					if_block.c();
    					transition_in(if_block, 1);
    					if_block.m(span1, t0);
    				}
    			} else if (if_block) {
    				group_outros();

    				transition_out(if_block, 1, 1, () => {
    					if_block = null;
    				});

    				check_outros();
    			}

    			if (spinner_slot) {
    				if (spinner_slot.p && (!current || dirty & /*$$scope*/ 128)) {
    					update_slot_base(
    						spinner_slot,
    						spinner_slot_template,
    						ctx,
    						/*$$scope*/ ctx[7],
    						!current
    						? get_all_dirty_from_scope(/*$$scope*/ ctx[7])
    						: get_slot_changes(spinner_slot_template, /*$$scope*/ ctx[7], dirty, get_spinner_slot_changes),
    						get_spinner_slot_context
    					);
    				}
    			}

    			if (notifications_slot) {
    				if (notifications_slot.p && (!current || dirty & /*$$scope*/ 128)) {
    					update_slot_base(
    						notifications_slot,
    						notifications_slot_template,
    						ctx,
    						/*$$scope*/ ctx[7],
    						!current
    						? get_all_dirty_from_scope(/*$$scope*/ ctx[7])
    						: get_slot_changes(notifications_slot_template, /*$$scope*/ ctx[7], dirty, get_notifications_slot_changes),
    						get_notifications_slot_context
    					);
    				}
    			}
    		},
    		i: function intro(local) {
    			if (current) return;
    			transition_in(if_block);
    			transition_in(spinner_slot, local);
    			transition_in(notifications_slot, local);
    			current = true;
    		},
    		o: function outro(local) {
    			transition_out(if_block);
    			transition_out(spinner_slot, local);
    			transition_out(notifications_slot, local);
    			current = false;
    		},
    		d: function destroy(detaching) {
    			if (detaching) detach_dev(span1);
    			if (if_block) if_block.d();
    			if (spinner_slot) spinner_slot.d(detaching);
    			if (notifications_slot) notifications_slot.d(detaching);
    		}
    	};

    	dispatch_dev("SvelteRegisterBlock", {
    		block,
    		id: create_if_block$c.name,
    		type: "if",
    		source: "(63:4) {#if $appState.loggedIn == 'yes'}",
    		ctx
    	});

    	return block;
    }

    // (65:12) {#if 'ready' == $appState.status}
    function create_if_block_1$9(ctx) {
    	let current;
    	const toolbar_slot_template = /*#slots*/ ctx[8].toolbar;
    	const toolbar_slot = create_slot(toolbar_slot_template, ctx, /*$$scope*/ ctx[7], get_toolbar_slot_context);

    	const block = {
    		c: function create() {
    			if (toolbar_slot) toolbar_slot.c();
    		},
    		m: function mount(target, anchor) {
    			if (toolbar_slot) {
    				toolbar_slot.m(target, anchor);
    			}

    			current = true;
    		},
    		p: function update(ctx, dirty) {
    			if (toolbar_slot) {
    				if (toolbar_slot.p && (!current || dirty & /*$$scope*/ 128)) {
    					update_slot_base(
    						toolbar_slot,
    						toolbar_slot_template,
    						ctx,
    						/*$$scope*/ ctx[7],
    						!current
    						? get_all_dirty_from_scope(/*$$scope*/ ctx[7])
    						: get_slot_changes(toolbar_slot_template, /*$$scope*/ ctx[7], dirty, get_toolbar_slot_changes),
    						get_toolbar_slot_context
    					);
    				}
    			}
    		},
    		i: function intro(local) {
    			if (current) return;
    			transition_in(toolbar_slot, local);
    			current = true;
    		},
    		o: function outro(local) {
    			transition_out(toolbar_slot, local);
    			current = false;
    		},
    		d: function destroy(detaching) {
    			if (toolbar_slot) toolbar_slot.d(detaching);
    		}
    	};

    	dispatch_dev("SvelteRegisterBlock", {
    		block,
    		id: create_if_block_1$9.name,
    		type: "if",
    		source: "(65:12) {#if 'ready' == $appState.status}",
    		ctx
    	});

    	return block;
    }

    function create_fragment$g(ctx) {
    	let header;
    	let section;
    	let div;
    	let logo;
    	let t0;
    	let t1;
    	let current;
    	logo = new Logo_1({ $$inline: true });
    	let if_block0 = /*$appState*/ ctx[1].loggedIn == 'yes' && create_if_block_2$8(ctx);
    	let if_block1 = /*$appState*/ ctx[1].loggedIn == 'yes' && create_if_block$c(ctx);

    	const block = {
    		c: function create() {
    			header = element("header");
    			section = element("section");
    			div = element("div");
    			create_component(logo.$$.fragment);
    			t0 = space();
    			if (if_block0) if_block0.c();
    			t1 = space();
    			if (if_block1) if_block1.c();
    			attr_dev(div, "class", "wb-flex wb-items-center");
    			add_location(div, file$f, 39, 8, 1241);
    			attr_dev(section, "class", "wb-flex wb-flex-wrap wb-flex-grow wb-justify-between wb-items-center wb-ml-2 sm:wb-m-0");
    			add_location(section, file$f, 38, 4, 1128);
    			set_style(header, "top", /*topOffset*/ ctx[0] + "px");
    			attr_dev(header, "class", "wb-sticky wb-z-50 wb-flex wb-justify-between wb-items-center md:wb-p-4 wb-px-0 wb-py-4 wb-border-b wb-border-gray-200 wb-text-gray-700 wb-bg-white");
    			add_location(header, file$f, 36, 0, 926);
    		},
    		l: function claim(nodes) {
    			throw new Error("options.hydrate only works if the component was compiled with the `hydratable: true` option");
    		},
    		m: function mount(target, anchor) {
    			insert_dev(target, header, anchor);
    			append_dev(header, section);
    			append_dev(section, div);
    			mount_component(logo, div, null);
    			append_dev(div, t0);
    			if (if_block0) if_block0.m(div, null);
    			append_dev(header, t1);
    			if (if_block1) if_block1.m(header, null);
    			current = true;
    		},
    		p: function update(ctx, [dirty]) {
    			if (/*$appState*/ ctx[1].loggedIn == 'yes') {
    				if (if_block0) {
    					if_block0.p(ctx, dirty);

    					if (dirty & /*$appState*/ 2) {
    						transition_in(if_block0, 1);
    					}
    				} else {
    					if_block0 = create_if_block_2$8(ctx);
    					if_block0.c();
    					transition_in(if_block0, 1);
    					if_block0.m(div, null);
    				}
    			} else if (if_block0) {
    				group_outros();

    				transition_out(if_block0, 1, 1, () => {
    					if_block0 = null;
    				});

    				check_outros();
    			}

    			if (/*$appState*/ ctx[1].loggedIn == 'yes') {
    				if (if_block1) {
    					if_block1.p(ctx, dirty);

    					if (dirty & /*$appState*/ 2) {
    						transition_in(if_block1, 1);
    					}
    				} else {
    					if_block1 = create_if_block$c(ctx);
    					if_block1.c();
    					transition_in(if_block1, 1);
    					if_block1.m(header, null);
    				}
    			} else if (if_block1) {
    				group_outros();

    				transition_out(if_block1, 1, 1, () => {
    					if_block1 = null;
    				});

    				check_outros();
    			}

    			if (!current || dirty & /*topOffset*/ 1) {
    				set_style(header, "top", /*topOffset*/ ctx[0] + "px");
    			}
    		},
    		i: function intro(local) {
    			if (current) return;
    			transition_in(logo.$$.fragment, local);
    			transition_in(if_block0);
    			transition_in(if_block1);
    			current = true;
    		},
    		o: function outro(local) {
    			transition_out(logo.$$.fragment, local);
    			transition_out(if_block0);
    			transition_out(if_block1);
    			current = false;
    		},
    		d: function destroy(detaching) {
    			if (detaching) detach_dev(header);
    			destroy_component(logo);
    			if (if_block0) if_block0.d();
    			if (if_block1) if_block1.d();
    		}
    	};

    	dispatch_dev("SvelteRegisterBlock", {
    		block,
    		id: create_fragment$g.name,
    		type: "component",
    		source: "",
    		ctx
    	});

    	return block;
    }

    function instance$g($$self, $$props, $$invalidate) {
    	let $appState;
    	let $forlogsState;
    	let { $$slots: slots = {}, $$scope } = $$props;
    	validate_slots('Navbar', slots, ['toolbar','spinner','notifications']);
    	let appState = getContext('appState');
    	validate_store(appState, 'appState');
    	component_subscribe($$self, appState, value => $$invalidate(1, $appState = value));
    	let config = getContext('config');
    	let forlogsState = getContext('forlogsState');
    	validate_store(forlogsState, 'forlogsState');
    	component_subscribe($$self, forlogsState, value => $$invalidate(2, $forlogsState = value));

    	const closeFileViewer = () => {
    		set_store_value(forlogsState, $forlogsState.viewing = null, $forlogsState);
    		set_store_value(appState, $appState.status = 'ready', $appState);
    	};

    	let topOffset = 0;

    	const updateTopOffset = () => {
    		if (config.platformVersion.major == 3) {
    			const topHeader = document.querySelector('nav.navbar');
    			$$invalidate(0, topOffset = topHeader ? topHeader.offsetHeight : 0);
    		}
    	};

    	onMount(updateTopOffset);
    	const writable_props = [];

    	Object.keys($$props).forEach(key => {
    		if (!~writable_props.indexOf(key) && key.slice(0, 2) !== '$$' && key !== 'slot') console.warn(`<Navbar> was created with unknown prop '${key}'`);
    	});

    	$$self.$$set = $$props => {
    		if ('$$scope' in $$props) $$invalidate(7, $$scope = $$props.$$scope);
    	};

    	$$self.$capture_state = () => ({
    		getContext,
    		onMount,
    		t,
    		unixToSiteTimeZone,
    		Title,
    		Logo: Logo_1,
    		Button,
    		appState,
    		config,
    		forlogsState,
    		closeFileViewer,
    		topOffset,
    		updateTopOffset,
    		$appState,
    		$forlogsState
    	});

    	$$self.$inject_state = $$props => {
    		if ('appState' in $$props) $$invalidate(3, appState = $$props.appState);
    		if ('config' in $$props) $$invalidate(4, config = $$props.config);
    		if ('forlogsState' in $$props) $$invalidate(5, forlogsState = $$props.forlogsState);
    		if ('topOffset' in $$props) $$invalidate(0, topOffset = $$props.topOffset);
    	};

    	if ($$props && "$$inject" in $$props) {
    		$$self.$inject_state($$props.$$inject);
    	}

    	return [
    		topOffset,
    		$appState,
    		$forlogsState,
    		appState,
    		config,
    		forlogsState,
    		closeFileViewer,
    		$$scope,
    		slots
    	];
    }

    class Navbar extends SvelteComponentDev {
    	constructor(options) {
    		super(options);
    		init(this, options, instance$g, create_fragment$g, safe_not_equal, {});

    		dispatch_dev("SvelteRegisterComponent", {
    			component: this,
    			tagName: "Navbar",
    			options,
    			id: create_fragment$g.name
    		});
    	}
    }

    /* node_modules\@weeblr\svelte\components\ClickAway.svelte generated by Svelte v3.41.0 */
    const file$e = "node_modules\\@weeblr\\svelte\\components\\ClickAway.svelte";

    function create_fragment$f(ctx) {
    	let t;
    	let div;
    	let current;
    	let mounted;
    	let dispose;
    	const default_slot_template = /*#slots*/ ctx[8].default;
    	const default_slot = create_slot(default_slot_template, ctx, /*$$scope*/ ctx[7], null);

    	const block = {
    		c: function create() {
    			t = space();
    			div = element("div");
    			if (default_slot) default_slot.c();
    			attr_dev(div, "class", /*classes*/ ctx[0]);
    			add_location(div, file$e, 51, 0, 1202);
    		},
    		l: function claim(nodes) {
    			throw new Error("options.hydrate only works if the component was compiled with the `hydratable: true` option");
    		},
    		m: function mount(target, anchor) {
    			insert_dev(target, t, anchor);
    			insert_dev(target, div, anchor);

    			if (default_slot) {
    				default_slot.m(div, null);
    			}

    			/*div_binding*/ ctx[9](div);
    			current = true;

    			if (!mounted) {
    				dispose = listen_dev(document.body, "mousedown", /*onMouseDown*/ ctx[2], false, false, false);
    				mounted = true;
    			}
    		},
    		p: function update(ctx, [dirty]) {
    			if (default_slot) {
    				if (default_slot.p && (!current || dirty & /*$$scope*/ 128)) {
    					update_slot_base(
    						default_slot,
    						default_slot_template,
    						ctx,
    						/*$$scope*/ ctx[7],
    						!current
    						? get_all_dirty_from_scope(/*$$scope*/ ctx[7])
    						: get_slot_changes(default_slot_template, /*$$scope*/ ctx[7], dirty, null),
    						null
    					);
    				}
    			}

    			if (!current || dirty & /*classes*/ 1) {
    				attr_dev(div, "class", /*classes*/ ctx[0]);
    			}
    		},
    		i: function intro(local) {
    			if (current) return;
    			transition_in(default_slot, local);
    			current = true;
    		},
    		o: function outro(local) {
    			transition_out(default_slot, local);
    			current = false;
    		},
    		d: function destroy(detaching) {
    			if (detaching) detach_dev(t);
    			if (detaching) detach_dev(div);
    			if (default_slot) default_slot.d(detaching);
    			/*div_binding*/ ctx[9](null);
    			mounted = false;
    			dispose();
    		}
    	};

    	dispatch_dev("SvelteRegisterBlock", {
    		block,
    		id: create_fragment$f.name,
    		type: "component",
    		source: "",
    		ctx
    	});

    	return block;
    }

    function instance$f($$self, $$props, $$invalidate) {
    	let { $$slots: slots = {}, $$scope } = $$props;
    	validate_slots('ClickAway', slots, ['default']);
    	let { enabled = true } = $$props;
    	let { enableOverlay } = $$props;

    	let { clickAction = () => {
    		
    	} } = $$props;

    	let { registerClicksOnNonTabbable = false } = $$props;
    	let { classes = '' } = $$props;

    	// Implementation
    	let subject;

    	const isSelfClick = target => {
    		let parent = target;

    		while (parent) {
    			if (parent === subject) {
    				return true;
    			}

    			parent = parent.parentNode;
    		}

    		if (!registerClicksOnNonTabbable && target.tabIndex < 0) {
    			// consider clicks on non-focusable elements
    			// as outside of the allowed zone. Let those
    			// clicks go through cause focus to move
    			// outside and escape the trap.
    			return false;
    		}

    		return false;
    	};

    	const onMouseDown = e => {
    		if (enabled && !isSelfClick(e.target)) {
    			e.preventDefault();
    			e.stopPropagation();
    			clickAction();
    		}
    	};

    	enableOverlay = enabled;

    	onDestroy(() => {
    		$$invalidate(3, enableOverlay = false);
    	});

    	const writable_props = [
    		'enabled',
    		'enableOverlay',
    		'clickAction',
    		'registerClicksOnNonTabbable',
    		'classes'
    	];

    	Object.keys($$props).forEach(key => {
    		if (!~writable_props.indexOf(key) && key.slice(0, 2) !== '$$' && key !== 'slot') console.warn(`<ClickAway> was created with unknown prop '${key}'`);
    	});

    	function div_binding($$value) {
    		binding_callbacks[$$value ? 'unshift' : 'push'](() => {
    			subject = $$value;
    			$$invalidate(1, subject);
    		});
    	}

    	$$self.$$set = $$props => {
    		if ('enabled' in $$props) $$invalidate(4, enabled = $$props.enabled);
    		if ('enableOverlay' in $$props) $$invalidate(3, enableOverlay = $$props.enableOverlay);
    		if ('clickAction' in $$props) $$invalidate(5, clickAction = $$props.clickAction);
    		if ('registerClicksOnNonTabbable' in $$props) $$invalidate(6, registerClicksOnNonTabbable = $$props.registerClicksOnNonTabbable);
    		if ('classes' in $$props) $$invalidate(0, classes = $$props.classes);
    		if ('$$scope' in $$props) $$invalidate(7, $$scope = $$props.$$scope);
    	};

    	$$self.$capture_state = () => ({
    		enabled,
    		enableOverlay,
    		clickAction,
    		registerClicksOnNonTabbable,
    		classes,
    		onDestroy,
    		subject,
    		isSelfClick,
    		onMouseDown
    	});

    	$$self.$inject_state = $$props => {
    		if ('enabled' in $$props) $$invalidate(4, enabled = $$props.enabled);
    		if ('enableOverlay' in $$props) $$invalidate(3, enableOverlay = $$props.enableOverlay);
    		if ('clickAction' in $$props) $$invalidate(5, clickAction = $$props.clickAction);
    		if ('registerClicksOnNonTabbable' in $$props) $$invalidate(6, registerClicksOnNonTabbable = $$props.registerClicksOnNonTabbable);
    		if ('classes' in $$props) $$invalidate(0, classes = $$props.classes);
    		if ('subject' in $$props) $$invalidate(1, subject = $$props.subject);
    	};

    	if ($$props && "$$inject" in $$props) {
    		$$self.$inject_state($$props.$$inject);
    	}

    	return [
    		classes,
    		subject,
    		onMouseDown,
    		enableOverlay,
    		enabled,
    		clickAction,
    		registerClicksOnNonTabbable,
    		$$scope,
    		slots,
    		div_binding
    	];
    }

    class ClickAway extends SvelteComponentDev {
    	constructor(options) {
    		super(options);

    		init(this, options, instance$f, create_fragment$f, safe_not_equal, {
    			enabled: 4,
    			enableOverlay: 3,
    			clickAction: 5,
    			registerClicksOnNonTabbable: 6,
    			classes: 0
    		});

    		dispatch_dev("SvelteRegisterComponent", {
    			component: this,
    			tagName: "ClickAway",
    			options,
    			id: create_fragment$f.name
    		});

    		const { ctx } = this.$$;
    		const props = options.props || {};

    		if (/*enableOverlay*/ ctx[3] === undefined && !('enableOverlay' in props)) {
    			console.warn("<ClickAway> was created without expected prop 'enableOverlay'");
    		}
    	}

    	get enabled() {
    		throw new Error("<ClickAway>: Props cannot be read directly from the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	set enabled(value) {
    		throw new Error("<ClickAway>: Props cannot be set directly on the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	get enableOverlay() {
    		throw new Error("<ClickAway>: Props cannot be read directly from the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	set enableOverlay(value) {
    		throw new Error("<ClickAway>: Props cannot be set directly on the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	get clickAction() {
    		throw new Error("<ClickAway>: Props cannot be read directly from the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	set clickAction(value) {
    		throw new Error("<ClickAway>: Props cannot be set directly on the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	get registerClicksOnNonTabbable() {
    		throw new Error("<ClickAway>: Props cannot be read directly from the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	set registerClicksOnNonTabbable(value) {
    		throw new Error("<ClickAway>: Props cannot be set directly on the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	get classes() {
    		throw new Error("<ClickAway>: Props cannot be read directly from the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	set classes(value) {
    		throw new Error("<ClickAway>: Props cannot be set directly on the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}
    }

    /* src\app\admin\components\AdminPageDialogContainer.svelte generated by Svelte v3.41.0 */
    const file$d = "src\\app\\admin\\components\\AdminPageDialogContainer.svelte";

    // (15:0) {#if pageState && $pageState.dialog}
    function create_if_block$b(ctx) {
    	let clickaway;
    	let updating_enableOverlay;
    	let current;

    	function clickaway_enableOverlay_binding(value) {
    		/*clickaway_enableOverlay_binding*/ ctx[4](value);
    	}

    	let clickaway_props = {
    		classes: "wb-relative wb-z-80",
    		$$slots: { default: [create_default_slot$4] },
    		$$scope: { ctx }
    	};

    	if (/*enableOverlay*/ ctx[0] !== void 0) {
    		clickaway_props.enableOverlay = /*enableOverlay*/ ctx[0];
    	}

    	clickaway = new ClickAway({ props: clickaway_props, $$inline: true });
    	binding_callbacks.push(() => bind(clickaway, 'enableOverlay', clickaway_enableOverlay_binding));

    	const block = {
    		c: function create() {
    			create_component(clickaway.$$.fragment);
    		},
    		m: function mount(target, anchor) {
    			mount_component(clickaway, target, anchor);
    			current = true;
    		},
    		p: function update(ctx, dirty) {
    			const clickaway_changes = {};

    			if (dirty & /*$$scope, $pageState, pageState*/ 38) {
    				clickaway_changes.$$scope = { dirty, ctx };
    			}

    			if (!updating_enableOverlay && dirty & /*enableOverlay*/ 1) {
    				updating_enableOverlay = true;
    				clickaway_changes.enableOverlay = /*enableOverlay*/ ctx[0];
    				add_flush_callback(() => updating_enableOverlay = false);
    			}

    			clickaway.$set(clickaway_changes);
    		},
    		i: function intro(local) {
    			if (current) return;
    			transition_in(clickaway.$$.fragment, local);
    			current = true;
    		},
    		o: function outro(local) {
    			transition_out(clickaway.$$.fragment, local);
    			current = false;
    		},
    		d: function destroy(detaching) {
    			destroy_component(clickaway, detaching);
    		}
    	};

    	dispatch_dev("SvelteRegisterBlock", {
    		block,
    		id: create_if_block$b.name,
    		type: "if",
    		source: "(15:0) {#if pageState && $pageState.dialog}",
    		ctx
    	});

    	return block;
    }

    // (16:4) <ClickAway bind:enableOverlay={enableOverlay} classes="wb-relative wb-z-80">
    function create_default_slot$4(ctx) {
    	let div;
    	let switch_instance;
    	let div_intro;
    	let current;
    	var switch_value = /*$pageState*/ ctx[2].dialog;

    	function switch_props(ctx) {
    		return {
    			props: { pageState: /*pageState*/ ctx[1] },
    			$$inline: true
    		};
    	}

    	if (switch_value) {
    		switch_instance = new switch_value(switch_props(ctx));
    	}

    	const block = {
    		c: function create() {
    			div = element("div");
    			if (switch_instance) create_component(switch_instance.$$.fragment);
    			attr_dev(div, "id", /*id*/ ctx[3]);
    			attr_dev(div, "class", "wb-w-full wb-m-4 wb-p-4 wb-bg-white wb-text-gray-900 wb-rounded-md wb-shadow-lg");
    			add_location(div, file$d, 16, 8, 418);
    		},
    		m: function mount(target, anchor) {
    			insert_dev(target, div, anchor);

    			if (switch_instance) {
    				mount_component(switch_instance, div, null);
    			}

    			current = true;
    		},
    		p: function update(ctx, dirty) {
    			const switch_instance_changes = {};
    			if (dirty & /*pageState*/ 2) switch_instance_changes.pageState = /*pageState*/ ctx[1];

    			if (switch_value !== (switch_value = /*$pageState*/ ctx[2].dialog)) {
    				if (switch_instance) {
    					group_outros();
    					const old_component = switch_instance;

    					transition_out(old_component.$$.fragment, 1, 0, () => {
    						destroy_component(old_component, 1);
    					});

    					check_outros();
    				}

    				if (switch_value) {
    					switch_instance = new switch_value(switch_props(ctx));
    					create_component(switch_instance.$$.fragment);
    					transition_in(switch_instance.$$.fragment, 1);
    					mount_component(switch_instance, div, null);
    				} else {
    					switch_instance = null;
    				}
    			} else if (switch_value) {
    				switch_instance.$set(switch_instance_changes);
    			}
    		},
    		i: function intro(local) {
    			if (current) return;
    			if (switch_instance) transition_in(switch_instance.$$.fragment, local);

    			if (!div_intro) {
    				add_render_callback(() => {
    					div_intro = create_in_transition(div, fly, { duration: 400, y: -100 });
    					div_intro.start();
    				});
    			}

    			current = true;
    		},
    		o: function outro(local) {
    			if (switch_instance) transition_out(switch_instance.$$.fragment, local);
    			current = false;
    		},
    		d: function destroy(detaching) {
    			if (detaching) detach_dev(div);
    			if (switch_instance) destroy_component(switch_instance);
    		}
    	};

    	dispatch_dev("SvelteRegisterBlock", {
    		block,
    		id: create_default_slot$4.name,
    		type: "slot",
    		source: "(16:4) <ClickAway bind:enableOverlay={enableOverlay} classes=\\\"wb-relative wb-z-80\\\">",
    		ctx
    	});

    	return block;
    }

    function create_fragment$e(ctx) {
    	let if_block_anchor;
    	let current;
    	let if_block = /*pageState*/ ctx[1] && /*$pageState*/ ctx[2].dialog && create_if_block$b(ctx);

    	const block = {
    		c: function create() {
    			if (if_block) if_block.c();
    			if_block_anchor = empty();
    		},
    		l: function claim(nodes) {
    			throw new Error("options.hydrate only works if the component was compiled with the `hydratable: true` option");
    		},
    		m: function mount(target, anchor) {
    			if (if_block) if_block.m(target, anchor);
    			insert_dev(target, if_block_anchor, anchor);
    			current = true;
    		},
    		p: function update(ctx, [dirty]) {
    			if (/*pageState*/ ctx[1] && /*$pageState*/ ctx[2].dialog) {
    				if (if_block) {
    					if_block.p(ctx, dirty);

    					if (dirty & /*pageState, $pageState*/ 6) {
    						transition_in(if_block, 1);
    					}
    				} else {
    					if_block = create_if_block$b(ctx);
    					if_block.c();
    					transition_in(if_block, 1);
    					if_block.m(if_block_anchor.parentNode, if_block_anchor);
    				}
    			} else if (if_block) {
    				group_outros();

    				transition_out(if_block, 1, 1, () => {
    					if_block = null;
    				});

    				check_outros();
    			}
    		},
    		i: function intro(local) {
    			if (current) return;
    			transition_in(if_block);
    			current = true;
    		},
    		o: function outro(local) {
    			transition_out(if_block);
    			current = false;
    		},
    		d: function destroy(detaching) {
    			if (if_block) if_block.d(detaching);
    			if (detaching) detach_dev(if_block_anchor);
    		}
    	};

    	dispatch_dev("SvelteRegisterBlock", {
    		block,
    		id: create_fragment$e.name,
    		type: "component",
    		source: "",
    		ctx
    	});

    	return block;
    }

    function instance$e($$self, $$props, $$invalidate) {
    	let $pageState,
    		$$unsubscribe_pageState = noop$1,
    		$$subscribe_pageState = () => ($$unsubscribe_pageState(), $$unsubscribe_pageState = subscribe(pageState, $$value => $$invalidate(2, $pageState = $$value)), pageState);

    	$$self.$$.on_destroy.push(() => $$unsubscribe_pageState());
    	let { $$slots: slots = {}, $$scope } = $$props;
    	validate_slots('AdminPageDialogContainer', slots, []);
    	let { pageState } = $$props;
    	validate_store(pageState, 'pageState');
    	$$subscribe_pageState();
    	let { enableOverlay = true } = $$props;
    	let id = v4();
    	const writable_props = ['pageState', 'enableOverlay'];

    	Object.keys($$props).forEach(key => {
    		if (!~writable_props.indexOf(key) && key.slice(0, 2) !== '$$' && key !== 'slot') console.warn(`<AdminPageDialogContainer> was created with unknown prop '${key}'`);
    	});

    	function clickaway_enableOverlay_binding(value) {
    		enableOverlay = value;
    		$$invalidate(0, enableOverlay);
    	}

    	$$self.$$set = $$props => {
    		if ('pageState' in $$props) $$subscribe_pageState($$invalidate(1, pageState = $$props.pageState));
    		if ('enableOverlay' in $$props) $$invalidate(0, enableOverlay = $$props.enableOverlay);
    	};

    	$$self.$capture_state = () => ({
    		pageState,
    		enableOverlay,
    		fly,
    		ClickAway,
    		uuid: v4,
    		id,
    		$pageState
    	});

    	$$self.$inject_state = $$props => {
    		if ('pageState' in $$props) $$subscribe_pageState($$invalidate(1, pageState = $$props.pageState));
    		if ('enableOverlay' in $$props) $$invalidate(0, enableOverlay = $$props.enableOverlay);
    		if ('id' in $$props) $$invalidate(3, id = $$props.id);
    	};

    	if ($$props && "$$inject" in $$props) {
    		$$self.$inject_state($$props.$$inject);
    	}

    	return [enableOverlay, pageState, $pageState, id, clickaway_enableOverlay_binding];
    }

    class AdminPageDialogContainer extends SvelteComponentDev {
    	constructor(options) {
    		super(options);
    		init(this, options, instance$e, create_fragment$e, safe_not_equal, { pageState: 1, enableOverlay: 0 });

    		dispatch_dev("SvelteRegisterComponent", {
    			component: this,
    			tagName: "AdminPageDialogContainer",
    			options,
    			id: create_fragment$e.name
    		});

    		const { ctx } = this.$$;
    		const props = options.props || {};

    		if (/*pageState*/ ctx[1] === undefined && !('pageState' in props)) {
    			console.warn("<AdminPageDialogContainer> was created without expected prop 'pageState'");
    		}
    	}

    	get pageState() {
    		throw new Error("<AdminPageDialogContainer>: Props cannot be read directly from the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	set pageState(value) {
    		throw new Error("<AdminPageDialogContainer>: Props cannot be set directly on the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	get enableOverlay() {
    		throw new Error("<AdminPageDialogContainer>: Props cannot be read directly from the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	set enableOverlay(value) {
    		throw new Error("<AdminPageDialogContainer>: Props cannot be set directly on the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}
    }

    /* src\app\admin\components\ConfirmationDialogContainer.svelte generated by Svelte v3.41.0 */
    const file$c = "src\\app\\admin\\components\\ConfirmationDialogContainer.svelte";

    // (15:0) <ClickAway bind:enableOverlay={enableOverlay} classes="wb-relative wb-z-80">
    function create_default_slot$3(ctx) {
    	let div;
    	let div_intro;
    	let current;
    	const default_slot_template = /*#slots*/ ctx[3].default;
    	const default_slot = create_slot(default_slot_template, ctx, /*$$scope*/ ctx[5], null);

    	const block = {
    		c: function create() {
    			div = element("div");
    			if (default_slot) default_slot.c();
    			attr_dev(div, "id", /*id*/ ctx[2]);
    			attr_dev(div, "class", /*classes*/ ctx[1]);
    			add_location(div, file$c, 15, 4, 484);
    		},
    		m: function mount(target, anchor) {
    			insert_dev(target, div, anchor);

    			if (default_slot) {
    				default_slot.m(div, null);
    			}

    			current = true;
    		},
    		p: function update(ctx, dirty) {
    			if (default_slot) {
    				if (default_slot.p && (!current || dirty & /*$$scope*/ 32)) {
    					update_slot_base(
    						default_slot,
    						default_slot_template,
    						ctx,
    						/*$$scope*/ ctx[5],
    						!current
    						? get_all_dirty_from_scope(/*$$scope*/ ctx[5])
    						: get_slot_changes(default_slot_template, /*$$scope*/ ctx[5], dirty, null),
    						null
    					);
    				}
    			}

    			if (!current || dirty & /*classes*/ 2) {
    				attr_dev(div, "class", /*classes*/ ctx[1]);
    			}
    		},
    		i: function intro(local) {
    			if (current) return;
    			transition_in(default_slot, local);

    			if (!div_intro) {
    				add_render_callback(() => {
    					div_intro = create_in_transition(div, fly, { duration: 400, y: -100 });
    					div_intro.start();
    				});
    			}

    			current = true;
    		},
    		o: function outro(local) {
    			transition_out(default_slot, local);
    			current = false;
    		},
    		d: function destroy(detaching) {
    			if (detaching) detach_dev(div);
    			if (default_slot) default_slot.d(detaching);
    		}
    	};

    	dispatch_dev("SvelteRegisterBlock", {
    		block,
    		id: create_default_slot$3.name,
    		type: "slot",
    		source: "(15:0) <ClickAway bind:enableOverlay={enableOverlay} classes=\\\"wb-relative wb-z-80\\\">",
    		ctx
    	});

    	return block;
    }

    function create_fragment$d(ctx) {
    	let clickaway;
    	let updating_enableOverlay;
    	let current;

    	function clickaway_enableOverlay_binding(value) {
    		/*clickaway_enableOverlay_binding*/ ctx[4](value);
    	}

    	let clickaway_props = {
    		classes: "wb-relative wb-z-80",
    		$$slots: { default: [create_default_slot$3] },
    		$$scope: { ctx }
    	};

    	if (/*enableOverlay*/ ctx[0] !== void 0) {
    		clickaway_props.enableOverlay = /*enableOverlay*/ ctx[0];
    	}

    	clickaway = new ClickAway({ props: clickaway_props, $$inline: true });
    	binding_callbacks.push(() => bind(clickaway, 'enableOverlay', clickaway_enableOverlay_binding));

    	const block = {
    		c: function create() {
    			create_component(clickaway.$$.fragment);
    		},
    		l: function claim(nodes) {
    			throw new Error("options.hydrate only works if the component was compiled with the `hydratable: true` option");
    		},
    		m: function mount(target, anchor) {
    			mount_component(clickaway, target, anchor);
    			current = true;
    		},
    		p: function update(ctx, [dirty]) {
    			const clickaway_changes = {};

    			if (dirty & /*$$scope, classes*/ 34) {
    				clickaway_changes.$$scope = { dirty, ctx };
    			}

    			if (!updating_enableOverlay && dirty & /*enableOverlay*/ 1) {
    				updating_enableOverlay = true;
    				clickaway_changes.enableOverlay = /*enableOverlay*/ ctx[0];
    				add_flush_callback(() => updating_enableOverlay = false);
    			}

    			clickaway.$set(clickaway_changes);
    		},
    		i: function intro(local) {
    			if (current) return;
    			transition_in(clickaway.$$.fragment, local);
    			current = true;
    		},
    		o: function outro(local) {
    			transition_out(clickaway.$$.fragment, local);
    			current = false;
    		},
    		d: function destroy(detaching) {
    			destroy_component(clickaway, detaching);
    		}
    	};

    	dispatch_dev("SvelteRegisterBlock", {
    		block,
    		id: create_fragment$d.name,
    		type: "component",
    		source: "",
    		ctx
    	});

    	return block;
    }

    function instance$d($$self, $$props, $$invalidate) {
    	let { $$slots: slots = {}, $$scope } = $$props;
    	validate_slots('ConfirmationDialogContainer', slots, ['default']);
    	let { enableOverlay = true } = $$props;
    	let { classes = 'sm:wb-max-w-md wb-max-w-sm wb-mx-auto wb-my-4 wb-p-4 wb-bg-white wb-text-gray-900 wb-rounded-md wb-shadow-lg' } = $$props;
    	let id = v4();
    	const writable_props = ['enableOverlay', 'classes'];

    	Object.keys($$props).forEach(key => {
    		if (!~writable_props.indexOf(key) && key.slice(0, 2) !== '$$' && key !== 'slot') console.warn(`<ConfirmationDialogContainer> was created with unknown prop '${key}'`);
    	});

    	function clickaway_enableOverlay_binding(value) {
    		enableOverlay = value;
    		$$invalidate(0, enableOverlay);
    	}

    	$$self.$$set = $$props => {
    		if ('enableOverlay' in $$props) $$invalidate(0, enableOverlay = $$props.enableOverlay);
    		if ('classes' in $$props) $$invalidate(1, classes = $$props.classes);
    		if ('$$scope' in $$props) $$invalidate(5, $$scope = $$props.$$scope);
    	};

    	$$self.$capture_state = () => ({
    		enableOverlay,
    		classes,
    		fly,
    		ClickAway,
    		uuid: v4,
    		id
    	});

    	$$self.$inject_state = $$props => {
    		if ('enableOverlay' in $$props) $$invalidate(0, enableOverlay = $$props.enableOverlay);
    		if ('classes' in $$props) $$invalidate(1, classes = $$props.classes);
    		if ('id' in $$props) $$invalidate(2, id = $$props.id);
    	};

    	if ($$props && "$$inject" in $$props) {
    		$$self.$inject_state($$props.$$inject);
    	}

    	return [enableOverlay, classes, id, slots, clickaway_enableOverlay_binding, $$scope];
    }

    class ConfirmationDialogContainer extends SvelteComponentDev {
    	constructor(options) {
    		super(options);
    		init(this, options, instance$d, create_fragment$d, safe_not_equal, { enableOverlay: 0, classes: 1 });

    		dispatch_dev("SvelteRegisterComponent", {
    			component: this,
    			tagName: "ConfirmationDialogContainer",
    			options,
    			id: create_fragment$d.name
    		});
    	}

    	get enableOverlay() {
    		throw new Error("<ConfirmationDialogContainer>: Props cannot be read directly from the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	set enableOverlay(value) {
    		throw new Error("<ConfirmationDialogContainer>: Props cannot be set directly on the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	get classes() {
    		throw new Error("<ConfirmationDialogContainer>: Props cannot be read directly from the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	set classes(value) {
    		throw new Error("<ConfirmationDialogContainer>: Props cannot be set directly on the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}
    }

    const subscriber_queue = [];
    /**
     * Create a `Writable` store that allows both updating and reading by subscription.
     * @param {*=}value initial value
     * @param {StartStopNotifier=}start start and stop notifications for subscriptions
     */
    function writable(value, start = noop$1) {
        let stop;
        const subscribers = new Set();
        function set(new_value) {
            if (safe_not_equal(value, new_value)) {
                value = new_value;
                if (stop) { // store is ready
                    const run_queue = !subscriber_queue.length;
                    for (const subscriber of subscribers) {
                        subscriber[1]();
                        subscriber_queue.push(subscriber, value);
                    }
                    if (run_queue) {
                        for (let i = 0; i < subscriber_queue.length; i += 2) {
                            subscriber_queue[i][0](subscriber_queue[i + 1]);
                        }
                        subscriber_queue.length = 0;
                    }
                }
            }
        }
        function update(fn) {
            set(fn(value));
        }
        function subscribe(run, invalidate = noop$1) {
            const subscriber = [run, invalidate];
            subscribers.add(subscriber);
            if (subscribers.size === 1) {
                stop = start(set) || noop$1;
            }
            run(value);
            return () => {
                subscribers.delete(subscriber);
                if (subscribers.size === 0) {
                    stop();
                    stop = null;
                }
            };
        }
        return { set, update, subscribe };
    }

    function is_date(obj) {
        return Object.prototype.toString.call(obj) === '[object Date]';
    }

    function get_interpolator(a, b) {
        if (a === b || a !== a)
            return () => a;
        const type = typeof a;
        if (type !== typeof b || Array.isArray(a) !== Array.isArray(b)) {
            throw new Error('Cannot interpolate values of different type');
        }
        if (Array.isArray(a)) {
            const arr = b.map((bi, i) => {
                return get_interpolator(a[i], bi);
            });
            return t => arr.map(fn => fn(t));
        }
        if (type === 'object') {
            if (!a || !b)
                throw new Error('Object cannot be null');
            if (is_date(a) && is_date(b)) {
                a = a.getTime();
                b = b.getTime();
                const delta = b - a;
                return t => new Date(a + t * delta);
            }
            const keys = Object.keys(b);
            const interpolators = {};
            keys.forEach(key => {
                interpolators[key] = get_interpolator(a[key], b[key]);
            });
            return t => {
                const result = {};
                keys.forEach(key => {
                    result[key] = interpolators[key](t);
                });
                return result;
            };
        }
        if (type === 'number') {
            const delta = b - a;
            return t => a + t * delta;
        }
        throw new Error(`Cannot interpolate ${type} values`);
    }
    function tweened(value, defaults = {}) {
        const store = writable(value);
        let task;
        let target_value = value;
        function set(new_value, opts) {
            if (value == null) {
                store.set(value = new_value);
                return Promise.resolve();
            }
            target_value = new_value;
            let previous_task = task;
            let started = false;
            let { delay = 0, duration = 400, easing = identity, interpolate = get_interpolator } = assign(assign({}, defaults), opts);
            if (duration === 0) {
                if (previous_task) {
                    previous_task.abort();
                    previous_task = null;
                }
                store.set(value = target_value);
                return Promise.resolve();
            }
            const start = now() + delay;
            let fn;
            task = loop(now => {
                if (now < start)
                    return true;
                if (!started) {
                    fn = interpolate(value, new_value);
                    if (typeof duration === 'function')
                        duration = duration(value, new_value);
                    started = true;
                }
                if (previous_task) {
                    previous_task.abort();
                    previous_task = null;
                }
                const elapsed = now - start;
                if (elapsed > duration) {
                    store.set(value = new_value);
                    return false;
                }
                // @ts-ignore
                store.set(value = fn(easing(elapsed / duration)));
                return true;
            });
            return task.promise;
        }
        return {
            set,
            update: (fn, opts) => set(fn(target_value, value), opts),
            subscribe: store.subscribe
        };
    }

    /* node_modules\@weeblr\svelte\components\ProgressBar.svelte generated by Svelte v3.41.0 */
    const file$b = "node_modules\\@weeblr\\svelte\\components\\ProgressBar.svelte";

    // (57:4) {#if autoclear && !stoppedAt}
    function create_if_block$a(ctx) {
    	let icon;
    	let current;

    	icon = new Icon({
    			props: {
    				listenToClicks: true,
    				icon: iconPause,
    				size: "l",
    				tabbable: true,
    				classes: "wb-mr-2"
    			},
    			$$inline: true
    		});

    	const block = {
    		c: function create() {
    			create_component(icon.$$.fragment);
    		},
    		m: function mount(target, anchor) {
    			mount_component(icon, target, anchor);
    			current = true;
    		},
    		p: noop$1,
    		i: function intro(local) {
    			if (current) return;
    			transition_in(icon.$$.fragment, local);
    			current = true;
    		},
    		o: function outro(local) {
    			transition_out(icon.$$.fragment, local);
    			current = false;
    		},
    		d: function destroy(detaching) {
    			destroy_component(icon, detaching);
    		}
    	};

    	dispatch_dev("SvelteRegisterBlock", {
    		block,
    		id: create_if_block$a.name,
    		type: "if",
    		source: "(57:4) {#if autoclear && !stoppedAt}",
    		ctx
    	});

    	return block;
    }

    function create_fragment$c(ctx) {
    	let div2;
    	let t;
    	let div1;
    	let div0;
    	let div0_class_value;
    	let div1_class_value;
    	let div2_class_value;
    	let current;
    	let mounted;
    	let dispose;
    	let if_block = /*autoclear*/ ctx[4] && !/*stoppedAt*/ ctx[6] && create_if_block$a(ctx);

    	const block = {
    		c: function create() {
    			div2 = element("div");
    			if (if_block) if_block.c();
    			t = space();
    			div1 = element("div");
    			div0 = element("div");
    			attr_dev(div0, "class", div0_class_value = "" + (/*thickness*/ ctx[3] + " wb-rounded-full " + /*color*/ ctx[0]));

    			set_style(div0, "width", /*toPercent*/ ctx[9](/*stoppedAt*/ ctx[6]
    			? /*stoppedAt*/ ctx[6]
    			: /*$progress*/ ctx[7]) + "%");

    			add_location(div0, file$b, 61, 8, 1661);
    			attr_dev(div1, "class", div1_class_value = "wb-w-full " + /*thickness*/ ctx[3] + " wb-flex wb-items-center wb-rounded-full " + /*bgColor*/ ctx[1] + " ");
    			add_location(div1, file$b, 60, 4, 1566);
    			attr_dev(div2, "class", div2_class_value = "wb-w-full wb-h-4 wb-flex wb-items-center " + /*cursorClass*/ ctx[10] + " " + /*classes*/ ctx[2]);
    			add_location(div2, file$b, 54, 0, 1286);
    		},
    		l: function claim(nodes) {
    			throw new Error("options.hydrate only works if the component was compiled with the `hydratable: true` option");
    		},
    		m: function mount(target, anchor) {
    			insert_dev(target, div2, anchor);
    			if (if_block) if_block.m(div2, null);
    			append_dev(div2, t);
    			append_dev(div2, div1);
    			append_dev(div1, div0);
    			current = true;

    			if (!mounted) {
    				dispose = listen_dev(div2, "click", /*stopProgress*/ ctx[8], false, false, false);
    				mounted = true;
    			}
    		},
    		p: function update(ctx, [dirty]) {
    			if (/*autoclear*/ ctx[4] && !/*stoppedAt*/ ctx[6]) {
    				if (if_block) {
    					if_block.p(ctx, dirty);

    					if (dirty & /*autoclear, stoppedAt*/ 80) {
    						transition_in(if_block, 1);
    					}
    				} else {
    					if_block = create_if_block$a(ctx);
    					if_block.c();
    					transition_in(if_block, 1);
    					if_block.m(div2, t);
    				}
    			} else if (if_block) {
    				group_outros();

    				transition_out(if_block, 1, 1, () => {
    					if_block = null;
    				});

    				check_outros();
    			}

    			if (!current || dirty & /*thickness, color*/ 9 && div0_class_value !== (div0_class_value = "" + (/*thickness*/ ctx[3] + " wb-rounded-full " + /*color*/ ctx[0]))) {
    				attr_dev(div0, "class", div0_class_value);
    			}

    			if (!current || dirty & /*stoppedAt, $progress*/ 192) {
    				set_style(div0, "width", /*toPercent*/ ctx[9](/*stoppedAt*/ ctx[6]
    				? /*stoppedAt*/ ctx[6]
    				: /*$progress*/ ctx[7]) + "%");
    			}

    			if (!current || dirty & /*thickness, bgColor*/ 10 && div1_class_value !== (div1_class_value = "wb-w-full " + /*thickness*/ ctx[3] + " wb-flex wb-items-center wb-rounded-full " + /*bgColor*/ ctx[1] + " ")) {
    				attr_dev(div1, "class", div1_class_value);
    			}

    			if (!current || dirty & /*classes*/ 4 && div2_class_value !== (div2_class_value = "wb-w-full wb-h-4 wb-flex wb-items-center " + /*cursorClass*/ ctx[10] + " " + /*classes*/ ctx[2])) {
    				attr_dev(div2, "class", div2_class_value);
    			}
    		},
    		i: function intro(local) {
    			if (current) return;
    			transition_in(if_block);
    			current = true;
    		},
    		o: function outro(local) {
    			transition_out(if_block);
    			current = false;
    		},
    		d: function destroy(detaching) {
    			if (detaching) detach_dev(div2);
    			if (if_block) if_block.d();
    			mounted = false;
    			dispose();
    		}
    	};

    	dispatch_dev("SvelteRegisterBlock", {
    		block,
    		id: create_fragment$c.name,
    		type: "component",
    		source: "",
    		ctx
    	});

    	return block;
    }

    function instance$c($$self, $$props, $$invalidate) {
    	let $progress,
    		$$unsubscribe_progress = noop$1,
    		$$subscribe_progress = () => ($$unsubscribe_progress(), $$unsubscribe_progress = subscribe(progress, $$value => $$invalidate(7, $progress = $$value)), progress);

    	$$self.$$.on_destroy.push(() => $$unsubscribe_progress());
    	let { $$slots: slots = {}, $$scope } = $$props;
    	validate_slots('ProgressBar', slots, []);
    	let { color = 'wb-bg-purple-200' } = $$props;
    	let { bgColor = 'wb-bg-gray-200' } = $$props;
    	let { classes = '' } = $$props;
    	let { thickness = 'wb-h-2' } = $$props;
    	let { autoclear = undefined } = $$props;
    	let { autoclearTimer = undefined } = $$props;
    	let { onCompletion = undefined } = $$props;
    	let progress;

    	const startTimer = duration => {
    		$$subscribe_progress($$invalidate(5, progress = tweened(0, { duration, easing: sineOut })));
    		progress.set(1);

    		$$invalidate(11, autoclearTimer = setTimeout(
    			() => {
    				onCompletion && onCompletion();
    			},
    			duration
    		));
    	};

    	if (autoclear > 0) {
    		startTimer(autoclear);
    	}

    	const stopProgress = () => {
    		if (!stoppedAt) {
    			clearTimeout(autoclearTimer);
    			$$invalidate(6, stoppedAt = get_store_value(progress));
    		}
    	};

    	const toPercent = value => {
    		return Math.round(100 * (1 - value));
    	};

    	let stoppedAt = undefined;
    	let cursorClass = autoclear > 0 ? 'wb-cursor-pointer' : '';

    	const writable_props = [
    		'color',
    		'bgColor',
    		'classes',
    		'thickness',
    		'autoclear',
    		'autoclearTimer',
    		'onCompletion'
    	];

    	Object.keys($$props).forEach(key => {
    		if (!~writable_props.indexOf(key) && key.slice(0, 2) !== '$$' && key !== 'slot') console.warn(`<ProgressBar> was created with unknown prop '${key}'`);
    	});

    	$$self.$$set = $$props => {
    		if ('color' in $$props) $$invalidate(0, color = $$props.color);
    		if ('bgColor' in $$props) $$invalidate(1, bgColor = $$props.bgColor);
    		if ('classes' in $$props) $$invalidate(2, classes = $$props.classes);
    		if ('thickness' in $$props) $$invalidate(3, thickness = $$props.thickness);
    		if ('autoclear' in $$props) $$invalidate(4, autoclear = $$props.autoclear);
    		if ('autoclearTimer' in $$props) $$invalidate(11, autoclearTimer = $$props.autoclearTimer);
    		if ('onCompletion' in $$props) $$invalidate(12, onCompletion = $$props.onCompletion);
    	};

    	$$self.$capture_state = () => ({
    		color,
    		bgColor,
    		classes,
    		thickness,
    		autoclear,
    		autoclearTimer,
    		onCompletion,
    		Icon,
    		iconPause,
    		get: get_store_value,
    		tweened,
    		sineOut,
    		progress,
    		startTimer,
    		stopProgress,
    		toPercent,
    		stoppedAt,
    		cursorClass,
    		$progress
    	});

    	$$self.$inject_state = $$props => {
    		if ('color' in $$props) $$invalidate(0, color = $$props.color);
    		if ('bgColor' in $$props) $$invalidate(1, bgColor = $$props.bgColor);
    		if ('classes' in $$props) $$invalidate(2, classes = $$props.classes);
    		if ('thickness' in $$props) $$invalidate(3, thickness = $$props.thickness);
    		if ('autoclear' in $$props) $$invalidate(4, autoclear = $$props.autoclear);
    		if ('autoclearTimer' in $$props) $$invalidate(11, autoclearTimer = $$props.autoclearTimer);
    		if ('onCompletion' in $$props) $$invalidate(12, onCompletion = $$props.onCompletion);
    		if ('progress' in $$props) $$subscribe_progress($$invalidate(5, progress = $$props.progress));
    		if ('stoppedAt' in $$props) $$invalidate(6, stoppedAt = $$props.stoppedAt);
    		if ('cursorClass' in $$props) $$invalidate(10, cursorClass = $$props.cursorClass);
    	};

    	if ($$props && "$$inject" in $$props) {
    		$$self.$inject_state($$props.$$inject);
    	}

    	return [
    		color,
    		bgColor,
    		classes,
    		thickness,
    		autoclear,
    		progress,
    		stoppedAt,
    		$progress,
    		stopProgress,
    		toPercent,
    		cursorClass,
    		autoclearTimer,
    		onCompletion
    	];
    }

    class ProgressBar extends SvelteComponentDev {
    	constructor(options) {
    		super(options);

    		init(this, options, instance$c, create_fragment$c, safe_not_equal, {
    			color: 0,
    			bgColor: 1,
    			classes: 2,
    			thickness: 3,
    			autoclear: 4,
    			autoclearTimer: 11,
    			onCompletion: 12
    		});

    		dispatch_dev("SvelteRegisterComponent", {
    			component: this,
    			tagName: "ProgressBar",
    			options,
    			id: create_fragment$c.name
    		});
    	}

    	get color() {
    		throw new Error("<ProgressBar>: Props cannot be read directly from the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	set color(value) {
    		throw new Error("<ProgressBar>: Props cannot be set directly on the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	get bgColor() {
    		throw new Error("<ProgressBar>: Props cannot be read directly from the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	set bgColor(value) {
    		throw new Error("<ProgressBar>: Props cannot be set directly on the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	get classes() {
    		throw new Error("<ProgressBar>: Props cannot be read directly from the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	set classes(value) {
    		throw new Error("<ProgressBar>: Props cannot be set directly on the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	get thickness() {
    		throw new Error("<ProgressBar>: Props cannot be read directly from the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	set thickness(value) {
    		throw new Error("<ProgressBar>: Props cannot be set directly on the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	get autoclear() {
    		throw new Error("<ProgressBar>: Props cannot be read directly from the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	set autoclear(value) {
    		throw new Error("<ProgressBar>: Props cannot be set directly on the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	get autoclearTimer() {
    		throw new Error("<ProgressBar>: Props cannot be read directly from the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	set autoclearTimer(value) {
    		throw new Error("<ProgressBar>: Props cannot be set directly on the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	get onCompletion() {
    		throw new Error("<ProgressBar>: Props cannot be read directly from the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	set onCompletion(value) {
    		throw new Error("<ProgressBar>: Props cannot be set directly on the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}
    }

    /**
     * Save and restore focus
     */
    let currentlyFocused;

    /**
     * Save focused element when a component is mounted
     * and restore it after destroy
     */
    const preserveMountFocus = (id = '') => {
        let focused;
        let mounted = false;
        beforeUpdate(
            () => {
                if (!mounted) {
                    focused = currentlyFocused;
                    // console.log('Component ' + id + ' mounted, active', focused)
                }
            }
        );
        onMount(
            () => {
                mounted = true;
                //   focused = currentlyFocused
                // console.log('Component ' + id + ' mounted, active', focused)
            }
        );
        onDestroy(
            () => {
                // console.log('Component ' + id + ' unmounted, preserveMountFocus:', focused, focused && focused.focus)
                mounted = false;
                focused && focused.focus && focused.focus();
            }
        );
    };

    /* node_modules\@weeblr\svelte\components\FocusTrap.svelte generated by Svelte v3.41.0 */
    const file$a = "node_modules\\@weeblr\\svelte\\components\\FocusTrap.svelte";

    function create_fragment$b(ctx) {
    	let div;
    	let current;
    	let mounted;
    	let dispose;
    	const default_slot_template = /*#slots*/ ctx[6].default;
    	const default_slot = create_slot(default_slot_template, ctx, /*$$scope*/ ctx[5], null);

    	const block = {
    		c: function create() {
    			div = element("div");
    			if (default_slot) default_slot.c();
    			attr_dev(div, "id", /*id*/ ctx[2]);
    			attr_dev(div, "class", /*classes*/ ctx[0]);
    			add_location(div, file$a, 53, 0, 1284);
    		},
    		l: function claim(nodes) {
    			throw new Error("options.hydrate only works if the component was compiled with the `hydratable: true` option");
    		},
    		m: function mount(target, anchor) {
    			insert_dev(target, div, anchor);

    			if (default_slot) {
    				default_slot.m(div, null);
    			}

    			/*div_binding*/ ctx[7](div);
    			current = true;

    			if (!mounted) {
    				dispose = listen_dev(div, "keydown", /*handler*/ ctx[3], false, false, false);
    				mounted = true;
    			}
    		},
    		p: function update(ctx, [dirty]) {
    			if (default_slot) {
    				if (default_slot.p && (!current || dirty & /*$$scope*/ 32)) {
    					update_slot_base(
    						default_slot,
    						default_slot_template,
    						ctx,
    						/*$$scope*/ ctx[5],
    						!current
    						? get_all_dirty_from_scope(/*$$scope*/ ctx[5])
    						: get_slot_changes(default_slot_template, /*$$scope*/ ctx[5], dirty, null),
    						null
    					);
    				}
    			}

    			if (!current || dirty & /*classes*/ 1) {
    				attr_dev(div, "class", /*classes*/ ctx[0]);
    			}
    		},
    		i: function intro(local) {
    			if (current) return;
    			transition_in(default_slot, local);
    			current = true;
    		},
    		o: function outro(local) {
    			transition_out(default_slot, local);
    			current = false;
    		},
    		d: function destroy(detaching) {
    			if (detaching) detach_dev(div);
    			if (default_slot) default_slot.d(detaching);
    			/*div_binding*/ ctx[7](null);
    			mounted = false;
    			dispose();
    		}
    	};

    	dispatch_dev("SvelteRegisterBlock", {
    		block,
    		id: create_fragment$b.name,
    		type: "component",
    		source: "",
    		ctx
    	});

    	return block;
    }

    function instance$b($$self, $$props, $$invalidate) {
    	let { $$slots: slots = {}, $$scope } = $$props;
    	validate_slots('FocusTrap', slots, ['default']);
    	let { enabled = true } = $$props;
    	let { classes = '' } = $$props;
    	let id = v4();
    	let subject;
    	let focusables;
    	let cancelFocus = false;

    	const handler = e => {
    		if (!enabled) {
    			return;
    		}

    		// could be done at onMount for better perf but then we would not see
    		// elements added or enabled dynamically.
    		focusables = getFocusTrapElements(subject);

    		if (!focusables.first || !focusables.last || focusables.first === focusables.last) {
    			// no focus change allowed
    			cancelFocus = true;
    		}

    		if (9 === e.keyCode) {
    			if (cancelFocus) {
    				e.preventDefault();
    				return;
    			}

    			if (e.shiftKey && document.activeElement === focusables.first) {
    				e.preventDefault();
    				focusables.last.focus();
    			} else if (!e.shiftKey && document.activeElement === focusables.last) {
    				e.preventDefault();
    				focusables.first.focus();
    			}
    		}
    	};

    	const writable_props = ['enabled', 'classes'];

    	Object.keys($$props).forEach(key => {
    		if (!~writable_props.indexOf(key) && key.slice(0, 2) !== '$$' && key !== 'slot') console.warn(`<FocusTrap> was created with unknown prop '${key}'`);
    	});

    	function div_binding($$value) {
    		binding_callbacks[$$value ? 'unshift' : 'push'](() => {
    			subject = $$value;
    			$$invalidate(1, subject);
    		});
    	}

    	$$self.$$set = $$props => {
    		if ('enabled' in $$props) $$invalidate(4, enabled = $$props.enabled);
    		if ('classes' in $$props) $$invalidate(0, classes = $$props.classes);
    		if ('$$scope' in $$props) $$invalidate(5, $$scope = $$props.$$scope);
    	};

    	$$self.$capture_state = () => ({
    		enabled,
    		classes,
    		uuidv4: v4,
    		getFocusTrapElements,
    		id,
    		subject,
    		focusables,
    		cancelFocus,
    		handler
    	});

    	$$self.$inject_state = $$props => {
    		if ('enabled' in $$props) $$invalidate(4, enabled = $$props.enabled);
    		if ('classes' in $$props) $$invalidate(0, classes = $$props.classes);
    		if ('id' in $$props) $$invalidate(2, id = $$props.id);
    		if ('subject' in $$props) $$invalidate(1, subject = $$props.subject);
    		if ('focusables' in $$props) focusables = $$props.focusables;
    		if ('cancelFocus' in $$props) cancelFocus = $$props.cancelFocus;
    	};

    	if ($$props && "$$inject" in $$props) {
    		$$self.$inject_state($$props.$$inject);
    	}

    	return [classes, subject, id, handler, enabled, $$scope, slots, div_binding];
    }

    class FocusTrap extends SvelteComponentDev {
    	constructor(options) {
    		super(options);
    		init(this, options, instance$b, create_fragment$b, safe_not_equal, { enabled: 4, classes: 0 });

    		dispatch_dev("SvelteRegisterComponent", {
    			component: this,
    			tagName: "FocusTrap",
    			options,
    			id: create_fragment$b.name
    		});
    	}

    	get enabled() {
    		throw new Error("<FocusTrap>: Props cannot be read directly from the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	set enabled(value) {
    		throw new Error("<FocusTrap>: Props cannot be set directly on the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	get classes() {
    		throw new Error("<FocusTrap>: Props cannot be read directly from the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	set classes(value) {
    		throw new Error("<FocusTrap>: Props cannot be set directly on the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}
    }

    /* node_modules\@weeblr\svelte\components\Notification.svelte generated by Svelte v3.41.0 */
    const file$9 = "node_modules\\@weeblr\\svelte\\components\\Notification.svelte";

    function get_each_context$1(ctx, list, i) {
    	const child_ctx = ctx.slice();
    	child_ctx[11] = list[i];
    	child_ctx[13] = i;
    	return child_ctx;
    }

    // (60:12) {#if notification.message}
    function create_if_block_3$2(ctx) {
    	let span;
    	let raw_value = /*notification*/ ctx[0].message + "";

    	const block = {
    		c: function create() {
    			span = element("span");
    			attr_dev(span, "class", "wb-mt-4 wb-break-all");
    			add_location(span, file$9, 60, 16, 2083);
    		},
    		m: function mount(target, anchor) {
    			insert_dev(target, span, anchor);
    			span.innerHTML = raw_value;
    		},
    		p: function update(ctx, dirty) {
    			if (dirty & /*notification*/ 1 && raw_value !== (raw_value = /*notification*/ ctx[0].message + "")) span.innerHTML = raw_value;		},
    		d: function destroy(detaching) {
    			if (detaching) detach_dev(span);
    		}
    	};

    	dispatch_dev("SvelteRegisterBlock", {
    		block,
    		id: create_if_block_3$2.name,
    		type: "if",
    		source: "(60:12) {#if notification.message}",
    		ctx
    	});

    	return block;
    }

    // (63:12) {#if notification.autoclear > 0 }
    function create_if_block_2$7(ctx) {
    	let progressbar;
    	let updating_autoclearTimer;
    	let current;

    	function progressbar_autoclearTimer_binding(value) {
    		/*progressbar_autoclearTimer_binding*/ ctx[8](value);
    	}

    	let progressbar_props = {
    		thickness: "wb-h-1",
    		onCompletion: /*func*/ ctx[7],
    		autoclear: /*notification*/ ctx[0].autoclear,
    		classes: "wb-mt-4"
    	};

    	if (/*autoclearTimer*/ ctx[2] !== void 0) {
    		progressbar_props.autoclearTimer = /*autoclearTimer*/ ctx[2];
    	}

    	progressbar = new ProgressBar({ props: progressbar_props, $$inline: true });
    	binding_callbacks.push(() => bind(progressbar, 'autoclearTimer', progressbar_autoclearTimer_binding));

    	const block = {
    		c: function create() {
    			create_component(progressbar.$$.fragment);
    		},
    		m: function mount(target, anchor) {
    			mount_component(progressbar, target, anchor);
    			current = true;
    		},
    		p: function update(ctx, dirty) {
    			const progressbar_changes = {};
    			if (dirty & /*notification*/ 1) progressbar_changes.autoclear = /*notification*/ ctx[0].autoclear;

    			if (!updating_autoclearTimer && dirty & /*autoclearTimer*/ 4) {
    				updating_autoclearTimer = true;
    				progressbar_changes.autoclearTimer = /*autoclearTimer*/ ctx[2];
    				add_flush_callback(() => updating_autoclearTimer = false);
    			}

    			progressbar.$set(progressbar_changes);
    		},
    		i: function intro(local) {
    			if (current) return;
    			transition_in(progressbar.$$.fragment, local);
    			current = true;
    		},
    		o: function outro(local) {
    			transition_out(progressbar.$$.fragment, local);
    			current = false;
    		},
    		d: function destroy(detaching) {
    			destroy_component(progressbar, detaching);
    		}
    	};

    	dispatch_dev("SvelteRegisterBlock", {
    		block,
    		id: create_if_block_2$7.name,
    		type: "if",
    		source: "(63:12) {#if notification.autoclear > 0 }",
    		ctx
    	});

    	return block;
    }

    // (73:4) {#if notification.canClear}
    function create_if_block_1$8(ctx) {
    	let div;
    	let button;
    	let icon;
    	let current;
    	let mounted;
    	let dispose;

    	icon = new Icon({
    			props: { icon: iconClose },
    			$$inline: true
    		});

    	const block = {
    		c: function create() {
    			div = element("div");
    			button = element("button");
    			create_component(icon.$$.fragment);
    			attr_dev(button, "class", "wb-inline-flex wb-text-gray-700 wb-c-focusable focus:wb-bg-gray-200 focus:wb-text-gray-900");
    			attr_dev(button, "tabindex", "0");
    			attr_dev(button, "aria-label", t('operations.dismiss'));
    			add_location(button, file$9, 74, 12, 2706);
    			attr_dev(div, "class", "wb-flex wb-flex-shrink-0 wb-w-8 wb-mx-2 wb-text-center");
    			add_location(div, file$9, 73, 8, 2606);
    		},
    		m: function mount(target, anchor) {
    			insert_dev(target, div, anchor);
    			append_dev(div, button);
    			mount_component(icon, button, null);
    			current = true;

    			if (!mounted) {
    				dispose = [
    					listen_dev(button, "click", /*clearNotification*/ ctx[4], false, false, false),
    					listen_dev(button, "keydown", /*keydown_handler*/ ctx[9], false, false, false),
    					action_destroyer(focusOnDefault.call(null, div))
    				];

    				mounted = true;
    			}
    		},
    		p: noop$1,
    		i: function intro(local) {
    			if (current) return;
    			transition_in(icon.$$.fragment, local);
    			current = true;
    		},
    		o: function outro(local) {
    			transition_out(icon.$$.fragment, local);
    			current = false;
    		},
    		d: function destroy(detaching) {
    			if (detaching) detach_dev(div);
    			destroy_component(icon);
    			mounted = false;
    			run_all(dispose);
    		}
    	};

    	dispatch_dev("SvelteRegisterBlock", {
    		block,
    		id: create_if_block_1$8.name,
    		type: "if",
    		source: "(73:4) {#if notification.canClear}",
    		ctx
    	});

    	return block;
    }

    // (85:4) {#if notification.buttons}
    function create_if_block$9(ctx) {
    	let focustrap;
    	let current;

    	focustrap = new FocusTrap({
    			props: {
    				$$slots: { default: [create_default_slot$2] },
    				$$scope: { ctx }
    			},
    			$$inline: true
    		});

    	const block = {
    		c: function create() {
    			create_component(focustrap.$$.fragment);
    		},
    		m: function mount(target, anchor) {
    			mount_component(focustrap, target, anchor);
    			current = true;
    		},
    		p: function update(ctx, dirty) {
    			const focustrap_changes = {};

    			if (dirty & /*$$scope, layout, notification*/ 16387) {
    				focustrap_changes.$$scope = { dirty, ctx };
    			}

    			focustrap.$set(focustrap_changes);
    		},
    		i: function intro(local) {
    			if (current) return;
    			transition_in(focustrap.$$.fragment, local);
    			current = true;
    		},
    		o: function outro(local) {
    			transition_out(focustrap.$$.fragment, local);
    			current = false;
    		},
    		d: function destroy(detaching) {
    			destroy_component(focustrap, detaching);
    		}
    	};

    	dispatch_dev("SvelteRegisterBlock", {
    		block,
    		id: create_if_block$9.name,
    		type: "if",
    		source: "(85:4) {#if notification.buttons}",
    		ctx
    	});

    	return block;
    }

    // (90:20) {#each notification.buttons as button,index}
    function create_each_block$1(ctx) {
    	let button;
    	let current;

    	button = new Button({
    			props: {
    				focusDefault: /*button*/ ctx[11].default ? true : false,
    				text: t(/*button*/ ctx[11].text),
    				handler: /*button*/ ctx[11].handler,
    				classes: "wb-px-4 wb-py-2 wb-m-2",
    				enabledClasses: /*button*/ ctx[11].enabledClasses
    				? /*button*/ ctx[11].enabledClasses
    				: 'wb-bg-gray-700 wb-text-gray-200 hover:wb-bg-gray-900 focus:wb-bg-gray-900 wb-c-focusable wb-cursor-pointer'
    			},
    			$$inline: true
    		});

    	const block = {
    		c: function create() {
    			create_component(button.$$.fragment);
    		},
    		m: function mount(target, anchor) {
    			mount_component(button, target, anchor);
    			current = true;
    		},
    		p: function update(ctx, dirty) {
    			const button_changes = {};
    			if (dirty & /*notification*/ 1) button_changes.focusDefault = /*button*/ ctx[11].default ? true : false;
    			if (dirty & /*notification*/ 1) button_changes.text = t(/*button*/ ctx[11].text);
    			if (dirty & /*notification*/ 1) button_changes.handler = /*button*/ ctx[11].handler;

    			if (dirty & /*notification*/ 1) button_changes.enabledClasses = /*button*/ ctx[11].enabledClasses
    			? /*button*/ ctx[11].enabledClasses
    			: 'wb-bg-gray-700 wb-text-gray-200 hover:wb-bg-gray-900 focus:wb-bg-gray-900 wb-c-focusable wb-cursor-pointer';

    			button.$set(button_changes);
    		},
    		i: function intro(local) {
    			if (current) return;
    			transition_in(button.$$.fragment, local);
    			current = true;
    		},
    		o: function outro(local) {
    			transition_out(button.$$.fragment, local);
    			current = false;
    		},
    		d: function destroy(detaching) {
    			destroy_component(button, detaching);
    		}
    	};

    	dispatch_dev("SvelteRegisterBlock", {
    		block,
    		id: create_each_block$1.name,
    		type: "each",
    		source: "(90:20) {#each notification.buttons as button,index}",
    		ctx
    	});

    	return block;
    }

    // (86:8) <FocusTrap>
    function create_default_slot$2(ctx) {
    	let div1;
    	let div0;
    	let div1_class_value;
    	let current;
    	let mounted;
    	let dispose;
    	let each_value = /*notification*/ ctx[0].buttons;
    	validate_each_argument(each_value);
    	let each_blocks = [];

    	for (let i = 0; i < each_value.length; i += 1) {
    		each_blocks[i] = create_each_block$1(get_each_context$1(ctx, each_value, i));
    	}

    	const out = i => transition_out(each_blocks[i], 1, 1, () => {
    		each_blocks[i] = null;
    	});

    	const block = {
    		c: function create() {
    			div1 = element("div");
    			div0 = element("div");

    			for (let i = 0; i < each_blocks.length; i += 1) {
    				each_blocks[i].c();
    			}

    			attr_dev(div0, "class", "wb--m-2");
    			add_location(div0, file$9, 88, 16, 3374);

    			attr_dev(div1, "class", div1_class_value = "wb-flex-shrink-0 wb-text-center " + ('horizontal' == /*layout*/ ctx[1]
    			? 'wb-w-8 wb-mx-2'
    			: 'wb-p-8'));

    			add_location(div1, file$9, 86, 12, 3221);
    		},
    		m: function mount(target, anchor) {
    			insert_dev(target, div1, anchor);
    			append_dev(div1, div0);

    			for (let i = 0; i < each_blocks.length; i += 1) {
    				each_blocks[i].m(div0, null);
    			}

    			current = true;

    			if (!mounted) {
    				dispose = action_destroyer(focusOnDefault.call(null, div1));
    				mounted = true;
    			}
    		},
    		p: function update(ctx, dirty) {
    			if (dirty & /*notification, t*/ 1) {
    				each_value = /*notification*/ ctx[0].buttons;
    				validate_each_argument(each_value);
    				let i;

    				for (i = 0; i < each_value.length; i += 1) {
    					const child_ctx = get_each_context$1(ctx, each_value, i);

    					if (each_blocks[i]) {
    						each_blocks[i].p(child_ctx, dirty);
    						transition_in(each_blocks[i], 1);
    					} else {
    						each_blocks[i] = create_each_block$1(child_ctx);
    						each_blocks[i].c();
    						transition_in(each_blocks[i], 1);
    						each_blocks[i].m(div0, null);
    					}
    				}

    				group_outros();

    				for (i = each_value.length; i < each_blocks.length; i += 1) {
    					out(i);
    				}

    				check_outros();
    			}

    			if (!current || dirty & /*layout*/ 2 && div1_class_value !== (div1_class_value = "wb-flex-shrink-0 wb-text-center " + ('horizontal' == /*layout*/ ctx[1]
    			? 'wb-w-8 wb-mx-2'
    			: 'wb-p-8'))) {
    				attr_dev(div1, "class", div1_class_value);
    			}
    		},
    		i: function intro(local) {
    			if (current) return;

    			for (let i = 0; i < each_value.length; i += 1) {
    				transition_in(each_blocks[i]);
    			}

    			current = true;
    		},
    		o: function outro(local) {
    			each_blocks = each_blocks.filter(Boolean);

    			for (let i = 0; i < each_blocks.length; i += 1) {
    				transition_out(each_blocks[i]);
    			}

    			current = false;
    		},
    		d: function destroy(detaching) {
    			if (detaching) detach_dev(div1);
    			destroy_each(each_blocks, detaching);
    			mounted = false;
    			dispose();
    		}
    	};

    	dispatch_dev("SvelteRegisterBlock", {
    		block,
    		id: create_default_slot$2.name,
    		type: "slot",
    		source: "(86:8) <FocusTrap>",
    		ctx
    	});

    	return block;
    }

    function create_fragment$a(ctx) {
    	let div3;
    	let div2;
    	let div0;
    	let t0;
    	let div1;
    	let span;
    	let raw_value = /*notification*/ ctx[0].title + "";
    	let t1;
    	let t2;
    	let t3;
    	let t4;
    	let div3_data_notification_id_value;
    	let div3_class_value;
    	let div3_role_value;
    	let current;
    	let if_block0 = /*notification*/ ctx[0].message && create_if_block_3$2(ctx);
    	let if_block1 = /*notification*/ ctx[0].autoclear > 0 && create_if_block_2$7(ctx);
    	let if_block2 = /*notification*/ ctx[0].canClear && create_if_block_1$8(ctx);
    	let if_block3 = /*notification*/ ctx[0].buttons && create_if_block$9(ctx);

    	const block = {
    		c: function create() {
    			div3 = element("div");
    			div2 = element("div");
    			div0 = element("div");
    			t0 = space();
    			div1 = element("div");
    			span = element("span");
    			t1 = space();
    			if (if_block0) if_block0.c();
    			t2 = space();
    			if (if_block1) if_block1.c();
    			t3 = space();
    			if (if_block2) if_block2.c();
    			t4 = space();
    			if (if_block3) if_block3.c();
    			attr_dev(div0, "class", "wb-w-2 wb-h-6 wb-flex-shrink-0 wb-ml-4 wb-rounded-full " + /*color*/ ctx[3]);
    			add_location(div0, file$9, 56, 8, 1757);
    			attr_dev(span, "class", "wb-text-lg " + /*titleFontWeight*/ ctx[5] + " wb-break-words");
    			add_location(span, file$9, 58, 12, 1936);
    			attr_dev(div1, "class", "wb-flex-grow wb-flex wb-pl-6 wb-pr-4 wb-flex-col wb-leading-6");
    			add_location(div1, file$9, 57, 8, 1848);
    			attr_dev(div2, "class", "wb-flex wb-items-center wb-flex-grow");
    			add_location(div2, file$9, 55, 4, 1698);
    			attr_dev(div3, "data-notification-id", div3_data_notification_id_value = /*notification*/ ctx[0].uuid);

    			attr_dev(div3, "class", div3_class_value = "wb-flex wb-justify-between wb-w-full " + ('horizontal' == /*layout*/ ctx[1]
    			? 'wb-flex-row wb-items-center'
    			: 'wb-flex-col'));

    			attr_dev(div3, "role", div3_role_value = /*notification*/ ctx[0].buttons && /*notification*/ ctx[0].buttons.length
    			? ''
    			: 'alert');

    			add_location(div3, file$9, 52, 0, 1442);
    		},
    		l: function claim(nodes) {
    			throw new Error("options.hydrate only works if the component was compiled with the `hydratable: true` option");
    		},
    		m: function mount(target, anchor) {
    			insert_dev(target, div3, anchor);
    			append_dev(div3, div2);
    			append_dev(div2, div0);
    			append_dev(div2, t0);
    			append_dev(div2, div1);
    			append_dev(div1, span);
    			span.innerHTML = raw_value;
    			append_dev(div1, t1);
    			if (if_block0) if_block0.m(div1, null);
    			append_dev(div1, t2);
    			if (if_block1) if_block1.m(div1, null);
    			append_dev(div3, t3);
    			if (if_block2) if_block2.m(div3, null);
    			append_dev(div3, t4);
    			if (if_block3) if_block3.m(div3, null);
    			current = true;
    		},
    		p: function update(ctx, [dirty]) {
    			if ((!current || dirty & /*notification*/ 1) && raw_value !== (raw_value = /*notification*/ ctx[0].title + "")) span.innerHTML = raw_value;
    			if (/*notification*/ ctx[0].message) {
    				if (if_block0) {
    					if_block0.p(ctx, dirty);
    				} else {
    					if_block0 = create_if_block_3$2(ctx);
    					if_block0.c();
    					if_block0.m(div1, t2);
    				}
    			} else if (if_block0) {
    				if_block0.d(1);
    				if_block0 = null;
    			}

    			if (/*notification*/ ctx[0].autoclear > 0) {
    				if (if_block1) {
    					if_block1.p(ctx, dirty);

    					if (dirty & /*notification*/ 1) {
    						transition_in(if_block1, 1);
    					}
    				} else {
    					if_block1 = create_if_block_2$7(ctx);
    					if_block1.c();
    					transition_in(if_block1, 1);
    					if_block1.m(div1, null);
    				}
    			} else if (if_block1) {
    				group_outros();

    				transition_out(if_block1, 1, 1, () => {
    					if_block1 = null;
    				});

    				check_outros();
    			}

    			if (/*notification*/ ctx[0].canClear) {
    				if (if_block2) {
    					if_block2.p(ctx, dirty);

    					if (dirty & /*notification*/ 1) {
    						transition_in(if_block2, 1);
    					}
    				} else {
    					if_block2 = create_if_block_1$8(ctx);
    					if_block2.c();
    					transition_in(if_block2, 1);
    					if_block2.m(div3, t4);
    				}
    			} else if (if_block2) {
    				group_outros();

    				transition_out(if_block2, 1, 1, () => {
    					if_block2 = null;
    				});

    				check_outros();
    			}

    			if (/*notification*/ ctx[0].buttons) {
    				if (if_block3) {
    					if_block3.p(ctx, dirty);

    					if (dirty & /*notification*/ 1) {
    						transition_in(if_block3, 1);
    					}
    				} else {
    					if_block3 = create_if_block$9(ctx);
    					if_block3.c();
    					transition_in(if_block3, 1);
    					if_block3.m(div3, null);
    				}
    			} else if (if_block3) {
    				group_outros();

    				transition_out(if_block3, 1, 1, () => {
    					if_block3 = null;
    				});

    				check_outros();
    			}

    			if (!current || dirty & /*notification*/ 1 && div3_data_notification_id_value !== (div3_data_notification_id_value = /*notification*/ ctx[0].uuid)) {
    				attr_dev(div3, "data-notification-id", div3_data_notification_id_value);
    			}

    			if (!current || dirty & /*layout*/ 2 && div3_class_value !== (div3_class_value = "wb-flex wb-justify-between wb-w-full " + ('horizontal' == /*layout*/ ctx[1]
    			? 'wb-flex-row wb-items-center'
    			: 'wb-flex-col'))) {
    				attr_dev(div3, "class", div3_class_value);
    			}

    			if (!current || dirty & /*notification*/ 1 && div3_role_value !== (div3_role_value = /*notification*/ ctx[0].buttons && /*notification*/ ctx[0].buttons.length
    			? ''
    			: 'alert')) {
    				attr_dev(div3, "role", div3_role_value);
    			}
    		},
    		i: function intro(local) {
    			if (current) return;
    			transition_in(if_block1);
    			transition_in(if_block2);
    			transition_in(if_block3);
    			current = true;
    		},
    		o: function outro(local) {
    			transition_out(if_block1);
    			transition_out(if_block2);
    			transition_out(if_block3);
    			current = false;
    		},
    		d: function destroy(detaching) {
    			if (detaching) detach_dev(div3);
    			if (if_block0) if_block0.d();
    			if (if_block1) if_block1.d();
    			if (if_block2) if_block2.d();
    			if (if_block3) if_block3.d();
    		}
    	};

    	dispatch_dev("SvelteRegisterBlock", {
    		block,
    		id: create_fragment$a.name,
    		type: "component",
    		source: "",
    		ctx
    	});

    	return block;
    }

    function instance$a($$self, $$props, $$invalidate) {
    	let { $$slots: slots = {}, $$scope } = $$props;
    	validate_slots('Notification', slots, []);
    	let { notification = undefined } = $$props;
    	let { layout = 'horizontal' } = $$props;
    	let { onCompletion = undefined } = $$props;
    	let autoclearTimer = undefined;

    	/**
     * Make sure to keep the full CSS classes here or else
     * postcss may remove them from the css bundle.
     *
     * @param {Object} notification
     * @returns {string}
     */
    	const getColor = notification => {
    		let color = 'wb-bg-green-500';

    		if (notification.type === '1_danger') {
    			color = 'wb-bg-red-500';
    		} else if (notification.type === '2_warning') {
    			color = 'wb-bg-yellow-600';
    		}

    		return color;
    	};

    	let color = getColor(notification);

    	const clearNotification = e => {
    		e && e.preventDefault() && e.stopPropagation();
    		clearTimeout(autoclearTimer);
    		onCompletion(notification);
    	};

    	let titleFontWeight = notification.message
    	? 'wb-font-bold'
    	: 'wb-font-semibold';

    	preserveMountFocus('notification');
    	const writable_props = ['notification', 'layout', 'onCompletion'];

    	Object.keys($$props).forEach(key => {
    		if (!~writable_props.indexOf(key) && key.slice(0, 2) !== '$$' && key !== 'slot') console.warn(`<Notification> was created with unknown prop '${key}'`);
    	});

    	const func = () => {
    		clearNotification();
    	};

    	function progressbar_autoclearTimer_binding(value) {
    		autoclearTimer = value;
    		$$invalidate(2, autoclearTimer);
    	}

    	const keydown_handler = e => notification.escapeClear && isEscapeKey(e) && clearNotification();

    	$$self.$$set = $$props => {
    		if ('notification' in $$props) $$invalidate(0, notification = $$props.notification);
    		if ('layout' in $$props) $$invalidate(1, layout = $$props.layout);
    		if ('onCompletion' in $$props) $$invalidate(6, onCompletion = $$props.onCompletion);
    	};

    	$$self.$capture_state = () => ({
    		notification,
    		layout,
    		onCompletion,
    		ProgressBar,
    		t,
    		preserveMountFocus,
    		focusOnDefault,
    		isEscapeKey,
    		Icon,
    		Button,
    		FocusTrap,
    		iconClose,
    		autoclearTimer,
    		getColor,
    		color,
    		clearNotification,
    		titleFontWeight
    	});

    	$$self.$inject_state = $$props => {
    		if ('notification' in $$props) $$invalidate(0, notification = $$props.notification);
    		if ('layout' in $$props) $$invalidate(1, layout = $$props.layout);
    		if ('onCompletion' in $$props) $$invalidate(6, onCompletion = $$props.onCompletion);
    		if ('autoclearTimer' in $$props) $$invalidate(2, autoclearTimer = $$props.autoclearTimer);
    		if ('color' in $$props) $$invalidate(3, color = $$props.color);
    		if ('titleFontWeight' in $$props) $$invalidate(5, titleFontWeight = $$props.titleFontWeight);
    	};

    	if ($$props && "$$inject" in $$props) {
    		$$self.$inject_state($$props.$$inject);
    	}

    	return [
    		notification,
    		layout,
    		autoclearTimer,
    		color,
    		clearNotification,
    		titleFontWeight,
    		onCompletion,
    		func,
    		progressbar_autoclearTimer_binding,
    		keydown_handler
    	];
    }

    class Notification extends SvelteComponentDev {
    	constructor(options) {
    		super(options);

    		init(this, options, instance$a, create_fragment$a, safe_not_equal, {
    			notification: 0,
    			layout: 1,
    			onCompletion: 6
    		});

    		dispatch_dev("SvelteRegisterComponent", {
    			component: this,
    			tagName: "Notification",
    			options,
    			id: create_fragment$a.name
    		});
    	}

    	get notification() {
    		throw new Error("<Notification>: Props cannot be read directly from the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	set notification(value) {
    		throw new Error("<Notification>: Props cannot be set directly on the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	get layout() {
    		throw new Error("<Notification>: Props cannot be read directly from the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	set layout(value) {
    		throw new Error("<Notification>: Props cannot be set directly on the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	get onCompletion() {
    		throw new Error("<Notification>: Props cannot be read directly from the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	set onCompletion(value) {
    		throw new Error("<Notification>: Props cannot be set directly on the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}
    }

    /* src\app\admin\components\AdminPageConfirmationDialogContainer.svelte generated by Svelte v3.41.0 */

    // (12:0) <ConfirmationDialogContainer {enableOverlay}>
    function create_default_slot$1(ctx) {
    	let notification;
    	let current;

    	notification = new Notification({
    			props: {
    				layout: "vertical",
    				notification: /*$pageState*/ ctx[2].confirmationDialog
    			},
    			$$inline: true
    		});

    	const block = {
    		c: function create() {
    			create_component(notification.$$.fragment);
    		},
    		m: function mount(target, anchor) {
    			mount_component(notification, target, anchor);
    			current = true;
    		},
    		p: function update(ctx, dirty) {
    			const notification_changes = {};
    			if (dirty & /*$pageState*/ 4) notification_changes.notification = /*$pageState*/ ctx[2].confirmationDialog;
    			notification.$set(notification_changes);
    		},
    		i: function intro(local) {
    			if (current) return;
    			transition_in(notification.$$.fragment, local);
    			current = true;
    		},
    		o: function outro(local) {
    			transition_out(notification.$$.fragment, local);
    			current = false;
    		},
    		d: function destroy(detaching) {
    			destroy_component(notification, detaching);
    		}
    	};

    	dispatch_dev("SvelteRegisterBlock", {
    		block,
    		id: create_default_slot$1.name,
    		type: "slot",
    		source: "(12:0) <ConfirmationDialogContainer {enableOverlay}>",
    		ctx
    	});

    	return block;
    }

    function create_fragment$9(ctx) {
    	let confirmationdialogcontainer;
    	let current;

    	confirmationdialogcontainer = new ConfirmationDialogContainer({
    			props: {
    				enableOverlay: /*enableOverlay*/ ctx[1],
    				$$slots: { default: [create_default_slot$1] },
    				$$scope: { ctx }
    			},
    			$$inline: true
    		});

    	const block = {
    		c: function create() {
    			create_component(confirmationdialogcontainer.$$.fragment);
    		},
    		l: function claim(nodes) {
    			throw new Error("options.hydrate only works if the component was compiled with the `hydratable: true` option");
    		},
    		m: function mount(target, anchor) {
    			mount_component(confirmationdialogcontainer, target, anchor);
    			current = true;
    		},
    		p: function update(ctx, [dirty]) {
    			const confirmationdialogcontainer_changes = {};
    			if (dirty & /*enableOverlay*/ 2) confirmationdialogcontainer_changes.enableOverlay = /*enableOverlay*/ ctx[1];

    			if (dirty & /*$$scope, $pageState*/ 12) {
    				confirmationdialogcontainer_changes.$$scope = { dirty, ctx };
    			}

    			confirmationdialogcontainer.$set(confirmationdialogcontainer_changes);
    		},
    		i: function intro(local) {
    			if (current) return;
    			transition_in(confirmationdialogcontainer.$$.fragment, local);
    			current = true;
    		},
    		o: function outro(local) {
    			transition_out(confirmationdialogcontainer.$$.fragment, local);
    			current = false;
    		},
    		d: function destroy(detaching) {
    			destroy_component(confirmationdialogcontainer, detaching);
    		}
    	};

    	dispatch_dev("SvelteRegisterBlock", {
    		block,
    		id: create_fragment$9.name,
    		type: "component",
    		source: "",
    		ctx
    	});

    	return block;
    }

    function instance$9($$self, $$props, $$invalidate) {
    	let $pageState,
    		$$unsubscribe_pageState = noop$1,
    		$$subscribe_pageState = () => ($$unsubscribe_pageState(), $$unsubscribe_pageState = subscribe(pageState, $$value => $$invalidate(2, $pageState = $$value)), pageState);

    	$$self.$$.on_destroy.push(() => $$unsubscribe_pageState());
    	let { $$slots: slots = {}, $$scope } = $$props;
    	validate_slots('AdminPageConfirmationDialogContainer', slots, []);
    	let { pageState } = $$props;
    	validate_store(pageState, 'pageState');
    	$$subscribe_pageState();
    	let { enableOverlay = true } = $$props;
    	const writable_props = ['pageState', 'enableOverlay'];

    	Object.keys($$props).forEach(key => {
    		if (!~writable_props.indexOf(key) && key.slice(0, 2) !== '$$' && key !== 'slot') console.warn(`<AdminPageConfirmationDialogContainer> was created with unknown prop '${key}'`);
    	});

    	$$self.$$set = $$props => {
    		if ('pageState' in $$props) $$subscribe_pageState($$invalidate(0, pageState = $$props.pageState));
    		if ('enableOverlay' in $$props) $$invalidate(1, enableOverlay = $$props.enableOverlay);
    	};

    	$$self.$capture_state = () => ({
    		pageState,
    		enableOverlay,
    		ConfirmationDialogContainer,
    		Notification,
    		$pageState
    	});

    	$$self.$inject_state = $$props => {
    		if ('pageState' in $$props) $$subscribe_pageState($$invalidate(0, pageState = $$props.pageState));
    		if ('enableOverlay' in $$props) $$invalidate(1, enableOverlay = $$props.enableOverlay);
    	};

    	if ($$props && "$$inject" in $$props) {
    		$$self.$inject_state($$props.$$inject);
    	}

    	return [pageState, enableOverlay, $pageState];
    }

    class AdminPageConfirmationDialogContainer extends SvelteComponentDev {
    	constructor(options) {
    		super(options);
    		init(this, options, instance$9, create_fragment$9, safe_not_equal, { pageState: 0, enableOverlay: 1 });

    		dispatch_dev("SvelteRegisterComponent", {
    			component: this,
    			tagName: "AdminPageConfirmationDialogContainer",
    			options,
    			id: create_fragment$9.name
    		});

    		const { ctx } = this.$$;
    		const props = options.props || {};

    		if (/*pageState*/ ctx[0] === undefined && !('pageState' in props)) {
    			console.warn("<AdminPageConfirmationDialogContainer> was created without expected prop 'pageState'");
    		}
    	}

    	get pageState() {
    		throw new Error("<AdminPageConfirmationDialogContainer>: Props cannot be read directly from the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	set pageState(value) {
    		throw new Error("<AdminPageConfirmationDialogContainer>: Props cannot be set directly on the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	get enableOverlay() {
    		throw new Error("<AdminPageConfirmationDialogContainer>: Props cannot be read directly from the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	set enableOverlay(value) {
    		throw new Error("<AdminPageConfirmationDialogContainer>: Props cannot be set directly on the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}
    }

    /* src\app\admin\components\AdminPage.svelte generated by Svelte v3.41.0 */
    const file$8 = "src\\app\\admin\\components\\AdminPage.svelte";
    const get_ready_slot_changes = dirty => ({});
    const get_ready_slot_context = ctx => ({});
    const get_error_slot_changes = dirty => ({});
    const get_error_slot_context = ctx => ({});
    const get_loading_slot_changes = dirty => ({});
    const get_loading_slot_context = ctx => ({});
    const get_custom_content_slot_changes = dirty => ({});
    const get_custom_content_slot_context = ctx => ({});

    // (24:4) {#if $appState.loggedIn == 'yes'}
    function create_if_block_2$6(ctx) {
    	let t0;
    	let t1;
    	let t2;
    	let current;
    	let if_block0 = /*$pageState*/ ctx[2].confirmationDialog && create_if_block_5(ctx);
    	let if_block1 = /*$pageState*/ ctx[2].dialog && create_if_block_4(ctx);
    	let if_block2 = /*$pageState*/ ctx[2].alert && create_if_block_3$1(ctx);
    	const custom_content_slot_template = /*#slots*/ ctx[6].custom_content;
    	const custom_content_slot = create_slot(custom_content_slot_template, ctx, /*$$scope*/ ctx[5], get_custom_content_slot_context);

    	const block = {
    		c: function create() {
    			if (if_block0) if_block0.c();
    			t0 = space();
    			if (if_block1) if_block1.c();
    			t1 = space();
    			if (if_block2) if_block2.c();
    			t2 = space();
    			if (custom_content_slot) custom_content_slot.c();
    		},
    		m: function mount(target, anchor) {
    			if (if_block0) if_block0.m(target, anchor);
    			insert_dev(target, t0, anchor);
    			if (if_block1) if_block1.m(target, anchor);
    			insert_dev(target, t1, anchor);
    			if (if_block2) if_block2.m(target, anchor);
    			insert_dev(target, t2, anchor);

    			if (custom_content_slot) {
    				custom_content_slot.m(target, anchor);
    			}

    			current = true;
    		},
    		p: function update(ctx, dirty) {
    			if (/*$pageState*/ ctx[2].confirmationDialog) {
    				if (if_block0) {
    					if_block0.p(ctx, dirty);

    					if (dirty & /*$pageState*/ 4) {
    						transition_in(if_block0, 1);
    					}
    				} else {
    					if_block0 = create_if_block_5(ctx);
    					if_block0.c();
    					transition_in(if_block0, 1);
    					if_block0.m(t0.parentNode, t0);
    				}
    			} else if (if_block0) {
    				group_outros();

    				transition_out(if_block0, 1, 1, () => {
    					if_block0 = null;
    				});

    				check_outros();
    			}

    			if (/*$pageState*/ ctx[2].dialog) {
    				if (if_block1) {
    					if_block1.p(ctx, dirty);

    					if (dirty & /*$pageState*/ 4) {
    						transition_in(if_block1, 1);
    					}
    				} else {
    					if_block1 = create_if_block_4(ctx);
    					if_block1.c();
    					transition_in(if_block1, 1);
    					if_block1.m(t1.parentNode, t1);
    				}
    			} else if (if_block1) {
    				group_outros();

    				transition_out(if_block1, 1, 1, () => {
    					if_block1 = null;
    				});

    				check_outros();
    			}

    			if (/*$pageState*/ ctx[2].alert) {
    				if (if_block2) {
    					if_block2.p(ctx, dirty);

    					if (dirty & /*$pageState*/ 4) {
    						transition_in(if_block2, 1);
    					}
    				} else {
    					if_block2 = create_if_block_3$1(ctx);
    					if_block2.c();
    					transition_in(if_block2, 1);
    					if_block2.m(t2.parentNode, t2);
    				}
    			} else if (if_block2) {
    				group_outros();

    				transition_out(if_block2, 1, 1, () => {
    					if_block2 = null;
    				});

    				check_outros();
    			}

    			if (custom_content_slot) {
    				if (custom_content_slot.p && (!current || dirty & /*$$scope*/ 32)) {
    					update_slot_base(
    						custom_content_slot,
    						custom_content_slot_template,
    						ctx,
    						/*$$scope*/ ctx[5],
    						!current
    						? get_all_dirty_from_scope(/*$$scope*/ ctx[5])
    						: get_slot_changes(custom_content_slot_template, /*$$scope*/ ctx[5], dirty, get_custom_content_slot_changes),
    						get_custom_content_slot_context
    					);
    				}
    			}
    		},
    		i: function intro(local) {
    			if (current) return;
    			transition_in(if_block0);
    			transition_in(if_block1);
    			transition_in(if_block2);
    			transition_in(custom_content_slot, local);
    			current = true;
    		},
    		o: function outro(local) {
    			transition_out(if_block0);
    			transition_out(if_block1);
    			transition_out(if_block2);
    			transition_out(custom_content_slot, local);
    			current = false;
    		},
    		d: function destroy(detaching) {
    			if (if_block0) if_block0.d(detaching);
    			if (detaching) detach_dev(t0);
    			if (if_block1) if_block1.d(detaching);
    			if (detaching) detach_dev(t1);
    			if (if_block2) if_block2.d(detaching);
    			if (detaching) detach_dev(t2);
    			if (custom_content_slot) custom_content_slot.d(detaching);
    		}
    	};

    	dispatch_dev("SvelteRegisterBlock", {
    		block,
    		id: create_if_block_2$6.name,
    		type: "if",
    		source: "(24:4) {#if $appState.loggedIn == 'yes'}",
    		ctx
    	});

    	return block;
    }

    // (26:8) {#if $pageState.confirmationDialog}
    function create_if_block_5(ctx) {
    	let adminpageconfirmationdialogcontainer;
    	let updating_enableOverlay;
    	let current;

    	function adminpageconfirmationdialogcontainer_enableOverlay_binding(value) {
    		/*adminpageconfirmationdialogcontainer_enableOverlay_binding*/ ctx[7](value);
    	}

    	let adminpageconfirmationdialogcontainer_props = { pageState: /*pageState*/ ctx[0] };

    	if (/*$appState*/ ctx[1].trapClicks !== void 0) {
    		adminpageconfirmationdialogcontainer_props.enableOverlay = /*$appState*/ ctx[1].trapClicks;
    	}

    	adminpageconfirmationdialogcontainer = new AdminPageConfirmationDialogContainer({
    			props: adminpageconfirmationdialogcontainer_props,
    			$$inline: true
    		});

    	binding_callbacks.push(() => bind(adminpageconfirmationdialogcontainer, 'enableOverlay', adminpageconfirmationdialogcontainer_enableOverlay_binding));

    	const block = {
    		c: function create() {
    			create_component(adminpageconfirmationdialogcontainer.$$.fragment);
    		},
    		m: function mount(target, anchor) {
    			mount_component(adminpageconfirmationdialogcontainer, target, anchor);
    			current = true;
    		},
    		p: function update(ctx, dirty) {
    			const adminpageconfirmationdialogcontainer_changes = {};
    			if (dirty & /*pageState*/ 1) adminpageconfirmationdialogcontainer_changes.pageState = /*pageState*/ ctx[0];

    			if (!updating_enableOverlay && dirty & /*$appState*/ 2) {
    				updating_enableOverlay = true;
    				adminpageconfirmationdialogcontainer_changes.enableOverlay = /*$appState*/ ctx[1].trapClicks;
    				add_flush_callback(() => updating_enableOverlay = false);
    			}

    			adminpageconfirmationdialogcontainer.$set(adminpageconfirmationdialogcontainer_changes);
    		},
    		i: function intro(local) {
    			if (current) return;
    			transition_in(adminpageconfirmationdialogcontainer.$$.fragment, local);
    			current = true;
    		},
    		o: function outro(local) {
    			transition_out(adminpageconfirmationdialogcontainer.$$.fragment, local);
    			current = false;
    		},
    		d: function destroy(detaching) {
    			destroy_component(adminpageconfirmationdialogcontainer, detaching);
    		}
    	};

    	dispatch_dev("SvelteRegisterBlock", {
    		block,
    		id: create_if_block_5.name,
    		type: "if",
    		source: "(26:8) {#if $pageState.confirmationDialog}",
    		ctx
    	});

    	return block;
    }

    // (32:8) {#if $pageState.dialog}
    function create_if_block_4(ctx) {
    	let adminpagedialogcontainer;
    	let updating_enableOverlay;
    	let current;

    	function adminpagedialogcontainer_enableOverlay_binding(value) {
    		/*adminpagedialogcontainer_enableOverlay_binding*/ ctx[8](value);
    	}

    	let adminpagedialogcontainer_props = { pageState: /*pageState*/ ctx[0] };

    	if (/*$appState*/ ctx[1].trapClicks !== void 0) {
    		adminpagedialogcontainer_props.enableOverlay = /*$appState*/ ctx[1].trapClicks;
    	}

    	adminpagedialogcontainer = new AdminPageDialogContainer({
    			props: adminpagedialogcontainer_props,
    			$$inline: true
    		});

    	binding_callbacks.push(() => bind(adminpagedialogcontainer, 'enableOverlay', adminpagedialogcontainer_enableOverlay_binding));

    	const block = {
    		c: function create() {
    			create_component(adminpagedialogcontainer.$$.fragment);
    		},
    		m: function mount(target, anchor) {
    			mount_component(adminpagedialogcontainer, target, anchor);
    			current = true;
    		},
    		p: function update(ctx, dirty) {
    			const adminpagedialogcontainer_changes = {};
    			if (dirty & /*pageState*/ 1) adminpagedialogcontainer_changes.pageState = /*pageState*/ ctx[0];

    			if (!updating_enableOverlay && dirty & /*$appState*/ 2) {
    				updating_enableOverlay = true;
    				adminpagedialogcontainer_changes.enableOverlay = /*$appState*/ ctx[1].trapClicks;
    				add_flush_callback(() => updating_enableOverlay = false);
    			}

    			adminpagedialogcontainer.$set(adminpagedialogcontainer_changes);
    		},
    		i: function intro(local) {
    			if (current) return;
    			transition_in(adminpagedialogcontainer.$$.fragment, local);
    			current = true;
    		},
    		o: function outro(local) {
    			transition_out(adminpagedialogcontainer.$$.fragment, local);
    			current = false;
    		},
    		d: function destroy(detaching) {
    			destroy_component(adminpagedialogcontainer, detaching);
    		}
    	};

    	dispatch_dev("SvelteRegisterBlock", {
    		block,
    		id: create_if_block_4.name,
    		type: "if",
    		source: "(32:8) {#if $pageState.dialog}",
    		ctx
    	});

    	return block;
    }

    // (38:8) {#if $pageState.alert}
    function create_if_block_3$1(ctx) {
    	let switch_instance;
    	let switch_instance_anchor;
    	let current;
    	var switch_value = /*$pageState*/ ctx[2].alert;

    	function switch_props(ctx) {
    		return { $$inline: true };
    	}

    	if (switch_value) {
    		switch_instance = new switch_value(switch_props());
    	}

    	const block = {
    		c: function create() {
    			if (switch_instance) create_component(switch_instance.$$.fragment);
    			switch_instance_anchor = empty();
    		},
    		m: function mount(target, anchor) {
    			if (switch_instance) {
    				mount_component(switch_instance, target, anchor);
    			}

    			insert_dev(target, switch_instance_anchor, anchor);
    			current = true;
    		},
    		p: function update(ctx, dirty) {
    			if (switch_value !== (switch_value = /*$pageState*/ ctx[2].alert)) {
    				if (switch_instance) {
    					group_outros();
    					const old_component = switch_instance;

    					transition_out(old_component.$$.fragment, 1, 0, () => {
    						destroy_component(old_component, 1);
    					});

    					check_outros();
    				}

    				if (switch_value) {
    					switch_instance = new switch_value(switch_props());
    					create_component(switch_instance.$$.fragment);
    					transition_in(switch_instance.$$.fragment, 1);
    					mount_component(switch_instance, switch_instance_anchor.parentNode, switch_instance_anchor);
    				} else {
    					switch_instance = null;
    				}
    			}
    		},
    		i: function intro(local) {
    			if (current) return;
    			if (switch_instance) transition_in(switch_instance.$$.fragment, local);
    			current = true;
    		},
    		o: function outro(local) {
    			if (switch_instance) transition_out(switch_instance.$$.fragment, local);
    			current = false;
    		},
    		d: function destroy(detaching) {
    			if (detaching) detach_dev(switch_instance_anchor);
    			if (switch_instance) destroy_component(switch_instance, detaching);
    		}
    	};

    	dispatch_dev("SvelteRegisterBlock", {
    		block,
    		id: create_if_block_3$1.name,
    		type: "if",
    		source: "(38:8) {#if $pageState.alert}",
    		ctx
    	});

    	return block;
    }

    // (58:12) {:else}
    function create_else_block$4(ctx) {
    	let current;
    	const ready_slot_template = /*#slots*/ ctx[6].ready;
    	const ready_slot = create_slot(ready_slot_template, ctx, /*$$scope*/ ctx[5], get_ready_slot_context);
    	const ready_slot_or_fallback = ready_slot || fallback_block_2(ctx);

    	const block = {
    		c: function create() {
    			if (ready_slot_or_fallback) ready_slot_or_fallback.c();
    		},
    		m: function mount(target, anchor) {
    			if (ready_slot_or_fallback) {
    				ready_slot_or_fallback.m(target, anchor);
    			}

    			current = true;
    		},
    		p: function update(ctx, dirty) {
    			if (ready_slot) {
    				if (ready_slot.p && (!current || dirty & /*$$scope*/ 32)) {
    					update_slot_base(
    						ready_slot,
    						ready_slot_template,
    						ctx,
    						/*$$scope*/ ctx[5],
    						!current
    						? get_all_dirty_from_scope(/*$$scope*/ ctx[5])
    						: get_slot_changes(ready_slot_template, /*$$scope*/ ctx[5], dirty, get_ready_slot_changes),
    						get_ready_slot_context
    					);
    				}
    			}
    		},
    		i: function intro(local) {
    			if (current) return;
    			transition_in(ready_slot_or_fallback, local);
    			current = true;
    		},
    		o: function outro(local) {
    			transition_out(ready_slot_or_fallback, local);
    			current = false;
    		},
    		d: function destroy(detaching) {
    			if (ready_slot_or_fallback) ready_slot_or_fallback.d(detaching);
    		}
    	};

    	dispatch_dev("SvelteRegisterBlock", {
    		block,
    		id: create_else_block$4.name,
    		type: "else",
    		source: "(58:12) {:else}",
    		ctx
    	});

    	return block;
    }

    // (52:50) 
    function create_if_block_1$7(ctx) {
    	let div;
    	let current;
    	const error_slot_template = /*#slots*/ ctx[6].error;
    	const error_slot = create_slot(error_slot_template, ctx, /*$$scope*/ ctx[5], get_error_slot_context);
    	const error_slot_or_fallback = error_slot || fallback_block_1(ctx);

    	const block = {
    		c: function create() {
    			div = element("div");
    			if (error_slot_or_fallback) error_slot_or_fallback.c();
    			attr_dev(div, "class", "wb-bg-gray-200 wb-p-4 wb-text-lg wb-font-semibold wb-text-always-red");
    			add_location(div, file$8, 52, 16, 1951);
    		},
    		m: function mount(target, anchor) {
    			insert_dev(target, div, anchor);

    			if (error_slot_or_fallback) {
    				error_slot_or_fallback.m(div, null);
    			}

    			current = true;
    		},
    		p: function update(ctx, dirty) {
    			if (error_slot) {
    				if (error_slot.p && (!current || dirty & /*$$scope*/ 32)) {
    					update_slot_base(
    						error_slot,
    						error_slot_template,
    						ctx,
    						/*$$scope*/ ctx[5],
    						!current
    						? get_all_dirty_from_scope(/*$$scope*/ ctx[5])
    						: get_slot_changes(error_slot_template, /*$$scope*/ ctx[5], dirty, get_error_slot_changes),
    						get_error_slot_context
    					);
    				}
    			}
    		},
    		i: function intro(local) {
    			if (current) return;
    			transition_in(error_slot_or_fallback, local);
    			current = true;
    		},
    		o: function outro(local) {
    			transition_out(error_slot_or_fallback, local);
    			current = false;
    		},
    		d: function destroy(detaching) {
    			if (detaching) detach_dev(div);
    			if (error_slot_or_fallback) error_slot_or_fallback.d(detaching);
    		}
    	};

    	dispatch_dev("SvelteRegisterBlock", {
    		block,
    		id: create_if_block_1$7.name,
    		type: "if",
    		source: "(52:50) ",
    		ctx
    	});

    	return block;
    }

    // (48:12) {#if 'loading' == $appState.status}
    function create_if_block$8(ctx) {
    	let current;
    	const loading_slot_template = /*#slots*/ ctx[6].loading;
    	const loading_slot = create_slot(loading_slot_template, ctx, /*$$scope*/ ctx[5], get_loading_slot_context);
    	const loading_slot_or_fallback = loading_slot || fallback_block(ctx);

    	const block = {
    		c: function create() {
    			if (loading_slot_or_fallback) loading_slot_or_fallback.c();
    		},
    		m: function mount(target, anchor) {
    			if (loading_slot_or_fallback) {
    				loading_slot_or_fallback.m(target, anchor);
    			}

    			current = true;
    		},
    		p: function update(ctx, dirty) {
    			if (loading_slot) {
    				if (loading_slot.p && (!current || dirty & /*$$scope*/ 32)) {
    					update_slot_base(
    						loading_slot,
    						loading_slot_template,
    						ctx,
    						/*$$scope*/ ctx[5],
    						!current
    						? get_all_dirty_from_scope(/*$$scope*/ ctx[5])
    						: get_slot_changes(loading_slot_template, /*$$scope*/ ctx[5], dirty, get_loading_slot_changes),
    						get_loading_slot_context
    					);
    				}
    			}
    		},
    		i: function intro(local) {
    			if (current) return;
    			transition_in(loading_slot_or_fallback, local);
    			current = true;
    		},
    		o: function outro(local) {
    			transition_out(loading_slot_or_fallback, local);
    			current = false;
    		},
    		d: function destroy(detaching) {
    			if (loading_slot_or_fallback) loading_slot_or_fallback.d(detaching);
    		}
    	};

    	dispatch_dev("SvelteRegisterBlock", {
    		block,
    		id: create_if_block$8.name,
    		type: "if",
    		source: "(48:12) {#if 'loading' == $appState.status}",
    		ctx
    	});

    	return block;
    }

    // (59:35)                      
    function fallback_block_2(ctx) {
    	let div;
    	let span;
    	let logo;
    	let current;
    	logo = new Logo({ $$inline: true });

    	const block = {
    		c: function create() {
    			div = element("div");
    			span = element("span");
    			create_component(logo.$$.fragment);
    			attr_dev(span, "aria-hidden", "true");
    			attr_dev(span, "class", "wb-h-24 wb-opacity-50");
    			add_location(span, file$8, 61, 24, 2474);
    			attr_dev(div, "role", "status");
    			attr_dev(div, "alt", "Page is loading");
    			attr_dev(div, "class", "wb-flex-grow wb-w-full wb-flex wb-justify-center wb-items-center");
    			add_location(div, file$8, 59, 20, 2310);
    		},
    		m: function mount(target, anchor) {
    			insert_dev(target, div, anchor);
    			append_dev(div, span);
    			mount_component(logo, span, null);
    			current = true;
    		},
    		i: function intro(local) {
    			if (current) return;
    			transition_in(logo.$$.fragment, local);
    			current = true;
    		},
    		o: function outro(local) {
    			transition_out(logo.$$.fragment, local);
    			current = false;
    		},
    		d: function destroy(detaching) {
    			if (detaching) detach_dev(div);
    			destroy_component(logo);
    		}
    	};

    	dispatch_dev("SvelteRegisterBlock", {
    		block,
    		id: fallback_block_2.name,
    		type: "fallback",
    		source: "(59:35)                      ",
    		ctx
    	});

    	return block;
    }

    // (54:39)                          
    function fallback_block_1(ctx) {
    	let statusmessage;
    	let current;

    	statusmessage = new StatusMessage({
    			props: {
    				status: "error",
    				text: t('backendComm.errorLoadingDataFromServer')
    			},
    			$$inline: true
    		});

    	const block = {
    		c: function create() {
    			create_component(statusmessage.$$.fragment);
    		},
    		m: function mount(target, anchor) {
    			mount_component(statusmessage, target, anchor);
    			current = true;
    		},
    		p: noop$1,
    		i: function intro(local) {
    			if (current) return;
    			transition_in(statusmessage.$$.fragment, local);
    			current = true;
    		},
    		o: function outro(local) {
    			transition_out(statusmessage.$$.fragment, local);
    			current = false;
    		},
    		d: function destroy(detaching) {
    			destroy_component(statusmessage, detaching);
    		}
    	};

    	dispatch_dev("SvelteRegisterBlock", {
    		block,
    		id: fallback_block_1.name,
    		type: "fallback",
    		source: "(54:39)                          ",
    		ctx
    	});

    	return block;
    }

    // (49:37)                      
    function fallback_block(ctx) {
    	let contentloading;
    	let current;

    	contentloading = new ContentLoading({
    			props: { linesCount: "8", classes: "wb-p-4" },
    			$$inline: true
    		});

    	const block = {
    		c: function create() {
    			create_component(contentloading.$$.fragment);
    		},
    		m: function mount(target, anchor) {
    			mount_component(contentloading, target, anchor);
    			current = true;
    		},
    		p: noop$1,
    		i: function intro(local) {
    			if (current) return;
    			transition_in(contentloading.$$.fragment, local);
    			current = true;
    		},
    		o: function outro(local) {
    			transition_out(contentloading.$$.fragment, local);
    			current = false;
    		},
    		d: function destroy(detaching) {
    			destroy_component(contentloading, detaching);
    		}
    	};

    	dispatch_dev("SvelteRegisterBlock", {
    		block,
    		id: fallback_block.name,
    		type: "fallback",
    		source: "(49:37)                      ",
    		ctx
    	});

    	return block;
    }

    function create_fragment$8(ctx) {
    	let section;
    	let t_1;
    	let div1;
    	let div0;
    	let current_block_type_index;
    	let if_block1;
    	let section_class_value;
    	let section_intro;
    	let current;
    	let if_block0 = /*$appState*/ ctx[1].loggedIn == 'yes' && create_if_block_2$6(ctx);
    	const if_block_creators = [create_if_block$8, create_if_block_1$7, create_else_block$4];
    	const if_blocks = [];

    	function select_block_type(ctx, dirty) {
    		if ('loading' == /*$appState*/ ctx[1].status) return 0;
    		if ('error' == /*$appState*/ ctx[1].status) return 1;
    		return 2;
    	}

    	current_block_type_index = select_block_type(ctx);
    	if_block1 = if_blocks[current_block_type_index] = if_block_creators[current_block_type_index](ctx);

    	const block = {
    		c: function create() {
    			section = element("section");
    			if (if_block0) if_block0.c();
    			t_1 = space();
    			div1 = element("div");
    			div0 = element("div");
    			if_block1.c();
    			attr_dev(div0, "class", "wb-flex wb-flex-grow-0 wb-w-full wb-px-0");
    			add_location(div0, file$8, 46, 8, 1649);
    			attr_dev(div1, "class", "wb-flex wb-flex-grow wb-overflow-hidden wb-relative");
    			add_location(div1, file$8, 45, 4, 1575);

    			attr_dev(section, "class", section_class_value = "wb-relative wb-flex wb-flex-col " + (/*$appState*/ ctx[1].helpOpen && /*$appState*/ ctx[1].currentPage && /*$appState*/ ctx[1].currentPage.helpId
    			? 'wb-overflow-y-auto'
    			: 'wb-overflow-hidden') + " wb-w-full wb-text-gray-700 wb-bg-gray-200 wb-fill-current");

    			add_location(section, file$8, 20, 0, 692);
    		},
    		l: function claim(nodes) {
    			throw new Error("options.hydrate only works if the component was compiled with the `hydratable: true` option");
    		},
    		m: function mount(target, anchor) {
    			insert_dev(target, section, anchor);
    			if (if_block0) if_block0.m(section, null);
    			append_dev(section, t_1);
    			append_dev(section, div1);
    			append_dev(div1, div0);
    			if_blocks[current_block_type_index].m(div0, null);
    			current = true;
    		},
    		p: function update(new_ctx, [dirty]) {
    			ctx = new_ctx;

    			if (/*$appState*/ ctx[1].loggedIn == 'yes') {
    				if (if_block0) {
    					if_block0.p(ctx, dirty);

    					if (dirty & /*$appState*/ 2) {
    						transition_in(if_block0, 1);
    					}
    				} else {
    					if_block0 = create_if_block_2$6(ctx);
    					if_block0.c();
    					transition_in(if_block0, 1);
    					if_block0.m(section, t_1);
    				}
    			} else if (if_block0) {
    				group_outros();

    				transition_out(if_block0, 1, 1, () => {
    					if_block0 = null;
    				});

    				check_outros();
    			}

    			let previous_block_index = current_block_type_index;
    			current_block_type_index = select_block_type(ctx);

    			if (current_block_type_index === previous_block_index) {
    				if_blocks[current_block_type_index].p(ctx, dirty);
    			} else {
    				group_outros();

    				transition_out(if_blocks[previous_block_index], 1, 1, () => {
    					if_blocks[previous_block_index] = null;
    				});

    				check_outros();
    				if_block1 = if_blocks[current_block_type_index];

    				if (!if_block1) {
    					if_block1 = if_blocks[current_block_type_index] = if_block_creators[current_block_type_index](ctx);
    					if_block1.c();
    				} else {
    					if_block1.p(ctx, dirty);
    				}

    				transition_in(if_block1, 1);
    				if_block1.m(div0, null);
    			}

    			if (!current || dirty & /*$appState*/ 2 && section_class_value !== (section_class_value = "wb-relative wb-flex wb-flex-col " + (/*$appState*/ ctx[1].helpOpen && /*$appState*/ ctx[1].currentPage && /*$appState*/ ctx[1].currentPage.helpId
    			? 'wb-overflow-y-auto'
    			: 'wb-overflow-hidden') + " wb-w-full wb-text-gray-700 wb-bg-gray-200 wb-fill-current")) {
    				attr_dev(section, "class", section_class_value);
    			}
    		},
    		i: function intro(local) {
    			if (current) return;
    			transition_in(if_block0);
    			transition_in(if_block1);

    			if (!section_intro) {
    				add_render_callback(() => {
    					section_intro = create_in_transition(section, fade, {
    						duration: /*features*/ ctx[4].get('ui.fadeInDuration')
    					});

    					section_intro.start();
    				});
    			}

    			current = true;
    		},
    		o: function outro(local) {
    			transition_out(if_block0);
    			transition_out(if_block1);
    			current = false;
    		},
    		d: function destroy(detaching) {
    			if (detaching) detach_dev(section);
    			if (if_block0) if_block0.d();
    			if_blocks[current_block_type_index].d();
    		}
    	};

    	dispatch_dev("SvelteRegisterBlock", {
    		block,
    		id: create_fragment$8.name,
    		type: "component",
    		source: "",
    		ctx
    	});

    	return block;
    }

    function instance$8($$self, $$props, $$invalidate) {
    	let $appState;

    	let $pageState,
    		$$unsubscribe_pageState = noop$1,
    		$$subscribe_pageState = () => ($$unsubscribe_pageState(), $$unsubscribe_pageState = subscribe(pageState, $$value => $$invalidate(2, $pageState = $$value)), pageState);

    	$$self.$$.on_destroy.push(() => $$unsubscribe_pageState());
    	let { $$slots: slots = {}, $$scope } = $$props;
    	validate_slots('AdminPage', slots, ['custom_content','loading','error','ready']);
    	let { pageState } = $$props;
    	validate_store(pageState, 'pageState');
    	$$subscribe_pageState();
    	let appState = getContext('appState');
    	validate_store(appState, 'appState');
    	component_subscribe($$self, appState, value => $$invalidate(1, $appState = value));
    	let features = getContext('features');
    	const writable_props = ['pageState'];

    	Object.keys($$props).forEach(key => {
    		if (!~writable_props.indexOf(key) && key.slice(0, 2) !== '$$' && key !== 'slot') console.warn(`<AdminPage> was created with unknown prop '${key}'`);
    	});

    	function adminpageconfirmationdialogcontainer_enableOverlay_binding(value) {
    		if ($$self.$$.not_equal($appState.trapClicks, value)) {
    			$appState.trapClicks = value;
    			appState.set($appState);
    		}
    	}

    	function adminpagedialogcontainer_enableOverlay_binding(value) {
    		if ($$self.$$.not_equal($appState.trapClicks, value)) {
    			$appState.trapClicks = value;
    			appState.set($appState);
    		}
    	}

    	$$self.$$set = $$props => {
    		if ('pageState' in $$props) $$subscribe_pageState($$invalidate(0, pageState = $$props.pageState));
    		if ('$$scope' in $$props) $$invalidate(5, $$scope = $$props.$$scope);
    	};

    	$$self.$capture_state = () => ({
    		pageState,
    		fade,
    		t,
    		AdminPageDialogContainer,
    		AdminPageConfirmationDialogContainer,
    		Logo,
    		StatusMessage,
    		ContentLoading,
    		getContext,
    		appState,
    		features,
    		$appState,
    		$pageState
    	});

    	$$self.$inject_state = $$props => {
    		if ('pageState' in $$props) $$subscribe_pageState($$invalidate(0, pageState = $$props.pageState));
    		if ('appState' in $$props) $$invalidate(3, appState = $$props.appState);
    		if ('features' in $$props) $$invalidate(4, features = $$props.features);
    	};

    	if ($$props && "$$inject" in $$props) {
    		$$self.$inject_state($$props.$$inject);
    	}

    	return [
    		pageState,
    		$appState,
    		$pageState,
    		appState,
    		features,
    		$$scope,
    		slots,
    		adminpageconfirmationdialogcontainer_enableOverlay_binding,
    		adminpagedialogcontainer_enableOverlay_binding
    	];
    }

    class AdminPage extends SvelteComponentDev {
    	constructor(options) {
    		super(options);
    		init(this, options, instance$8, create_fragment$8, safe_not_equal, { pageState: 0 });

    		dispatch_dev("SvelteRegisterComponent", {
    			component: this,
    			tagName: "AdminPage",
    			options,
    			id: create_fragment$8.name
    		});

    		const { ctx } = this.$$;
    		const props = options.props || {};

    		if (/*pageState*/ ctx[0] === undefined && !('pageState' in props)) {
    			console.warn("<AdminPage> was created without expected prop 'pageState'");
    		}
    	}

    	get pageState() {
    		throw new Error("<AdminPage>: Props cannot be read directly from the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	set pageState(value) {
    		throw new Error("<AdminPage>: Props cannot be set directly on the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}
    }

    /* src\app\admin\components\Loggedout.svelte generated by Svelte v3.41.0 */
    const file$7 = "src\\app\\admin\\components\\Loggedout.svelte";

    // (25:53) 
    function create_if_block_1$6(ctx) {
    	let h2;
    	let t1;
    	let img;
    	let img_src_value;
    	let t2;
    	let p;

    	const block = {
    		c: function create() {
    			h2 = element("h2");
    			h2.textContent = `${t('ui.connectionRecovered')}`;
    			t1 = space();
    			img = element("img");
    			t2 = space();
    			p = element("p");
    			p.textContent = `${t('ui.connectionLostResume')}`;
    			attr_dev(h2, "class", "wb-m-5 wb-text-lg wb-font-semibold");
    			add_location(h2, file$7, 25, 16, 1029);
    			attr_dev(img, "class", "wb-m-5 wb-max-w-full");
    			if (!src_url_equal(img.src, img_src_value = "https://cdn.weeblr.net/dist/weeblr/img/4seo/undraw_Login_re_4vu2.svg")) attr_dev(img, "src", img_src_value);
    			attr_dev(img, "width", "350");
    			attr_dev(img, "alt", "");
    			add_location(img, file$7, 26, 16, 1127);
    			attr_dev(p, "class", "wb-mx-5 wb-max-w-lg wb-text-left");
    			add_location(p, file$7, 30, 16, 1335);
    		},
    		m: function mount(target, anchor) {
    			insert_dev(target, h2, anchor);
    			insert_dev(target, t1, anchor);
    			insert_dev(target, img, anchor);
    			insert_dev(target, t2, anchor);
    			insert_dev(target, p, anchor);
    		},
    		p: noop$1,
    		d: function destroy(detaching) {
    			if (detaching) detach_dev(h2);
    			if (detaching) detach_dev(t1);
    			if (detaching) detach_dev(img);
    			if (detaching) detach_dev(t2);
    			if (detaching) detach_dev(p);
    		}
    	};

    	dispatch_dev("SvelteRegisterBlock", {
    		block,
    		id: create_if_block_1$6.name,
    		type: "if",
    		source: "(25:53) ",
    		ctx
    	});

    	return block;
    }

    // (17:12) {#if $appState.loggedIn == 'no'}
    function create_if_block$7(ctx) {
    	let h2;
    	let t1;
    	let img;
    	let img_src_value;
    	let t2;
    	let p0;
    	let t4;
    	let p1;

    	const block = {
    		c: function create() {
    			h2 = element("h2");
    			h2.textContent = `${t('ui.connectionLost')}`;
    			t1 = space();
    			img = element("img");
    			t2 = space();
    			p0 = element("p");
    			p0.textContent = `${t('ui.connectionLostDetails')}`;
    			t4 = space();
    			p1 = element("p");
    			p1.textContent = `${t('ui.connectionLostDetailsPrompt')}`;
    			attr_dev(h2, "class", "wb-m-5 wb-text-lg wb-font-semibold");
    			add_location(h2, file$7, 17, 16, 470);
    			attr_dev(img, "class", "wb-m-5 wb-max-w-full");
    			if (!src_url_equal(img.src, img_src_value = "https://cdn.weeblr.net/dist/weeblr/img/4seo/undraw_secure_login_pdn4-2.svg")) attr_dev(img, "src", img_src_value);
    			attr_dev(img, "width", "350");
    			attr_dev(img, "alt", "");
    			add_location(img, file$7, 18, 16, 563);
    			attr_dev(p0, "class", "wb-mx-5 wb-max-w-lg wb-text-left");
    			add_location(p0, file$7, 22, 16, 777);
    			attr_dev(p1, "class", "wb-mx-5 wb-max-w-lg wb-text-left");
    			add_location(p1, file$7, 23, 16, 873);
    		},
    		m: function mount(target, anchor) {
    			insert_dev(target, h2, anchor);
    			insert_dev(target, t1, anchor);
    			insert_dev(target, img, anchor);
    			insert_dev(target, t2, anchor);
    			insert_dev(target, p0, anchor);
    			insert_dev(target, t4, anchor);
    			insert_dev(target, p1, anchor);
    		},
    		p: noop$1,
    		d: function destroy(detaching) {
    			if (detaching) detach_dev(h2);
    			if (detaching) detach_dev(t1);
    			if (detaching) detach_dev(img);
    			if (detaching) detach_dev(t2);
    			if (detaching) detach_dev(p0);
    			if (detaching) detach_dev(t4);
    			if (detaching) detach_dev(p1);
    		}
    	};

    	dispatch_dev("SvelteRegisterBlock", {
    		block,
    		id: create_if_block$7.name,
    		type: "if",
    		source: "(17:12) {#if $appState.loggedIn == 'no'}",
    		ctx
    	});

    	return block;
    }

    // (15:4) 
    function create_ready_slot$1(ctx) {
    	let div1;
    	let div0;
    	let t0;
    	let p;
    	let a;
    	let span;
    	let mounted;
    	let dispose;

    	function select_block_type(ctx, dirty) {
    		if (/*$appState*/ ctx[0].loggedIn == 'no') return create_if_block$7;
    		if (/*$appState*/ ctx[0].loggedIn == 'resume') return create_if_block_1$6;
    	}

    	let current_block_type = select_block_type(ctx);
    	let if_block = current_block_type && current_block_type(ctx);

    	const block = {
    		c: function create() {
    			div1 = element("div");
    			div0 = element("div");
    			if (if_block) if_block.c();
    			t0 = space();
    			p = element("p");
    			a = element("a");
    			span = element("span");
    			span.textContent = `${t('ui.goToLoginPageOrRefresh')}`;
    			attr_dev(span, "class", "wb-px-4 wb-py-2 wb-bg-transparent wb-text-gray-700 wb-underline wb-rounded hover:wb-text-gray-200 hover:wb-bg-gray-900 focus:wb-bg-gray-200");
    			add_location(span, file$7, 36, 16, 1676);
    			attr_dev(a, "class", "wb-appearance-none");
    			attr_dev(a, "href", "#0");
    			add_location(a, file$7, 33, 16, 1506);
    			attr_dev(p, "class", "wb-m-5 wb-max-w-lg wb-text-center");
    			add_location(p, file$7, 32, 12, 1444);
    			attr_dev(div0, "class", "wb-flex wb-flex-col wb-p-4 wb-max-w-lg wb-text-center");
    			add_location(div0, file$7, 15, 8, 341);
    			attr_dev(div1, "slot", "ready");
    			attr_dev(div1, "class", "wb-p-4 wb-w-full wb-bg-white");
    			add_location(div1, file$7, 14, 4, 277);
    		},
    		m: function mount(target, anchor) {
    			insert_dev(target, div1, anchor);
    			append_dev(div1, div0);
    			if (if_block) if_block.m(div0, null);
    			append_dev(div0, t0);
    			append_dev(div0, p);
    			append_dev(p, a);
    			append_dev(a, span);

    			if (!mounted) {
    				dispose = listen_dev(a, "click", stop_propagation(prevent_default(/*click_handler*/ ctx[2])), false, true, true);
    				mounted = true;
    			}
    		},
    		p: function update(ctx, dirty) {
    			if (current_block_type === (current_block_type = select_block_type(ctx)) && if_block) {
    				if_block.p(ctx, dirty);
    			} else {
    				if (if_block) if_block.d(1);
    				if_block = current_block_type && current_block_type(ctx);

    				if (if_block) {
    					if_block.c();
    					if_block.m(div0, t0);
    				}
    			}
    		},
    		d: function destroy(detaching) {
    			if (detaching) detach_dev(div1);

    			if (if_block) {
    				if_block.d();
    			}

    			mounted = false;
    			dispose();
    		}
    	};

    	dispatch_dev("SvelteRegisterBlock", {
    		block,
    		id: create_ready_slot$1.name,
    		type: "slot",
    		source: "(15:4) ",
    		ctx
    	});

    	return block;
    }

    function create_fragment$7(ctx) {
    	let adminpage;
    	let current;

    	adminpage = new AdminPage({
    			props: {
    				status: 'ready',
    				$$slots: { ready: [create_ready_slot$1] },
    				$$scope: { ctx }
    			},
    			$$inline: true
    		});

    	const block = {
    		c: function create() {
    			create_component(adminpage.$$.fragment);
    		},
    		l: function claim(nodes) {
    			throw new Error("options.hydrate only works if the component was compiled with the `hydratable: true` option");
    		},
    		m: function mount(target, anchor) {
    			mount_component(adminpage, target, anchor);
    			current = true;
    		},
    		p: function update(ctx, [dirty]) {
    			const adminpage_changes = {};

    			if (dirty & /*$$scope, $appState*/ 9) {
    				adminpage_changes.$$scope = { dirty, ctx };
    			}

    			adminpage.$set(adminpage_changes);
    		},
    		i: function intro(local) {
    			if (current) return;
    			transition_in(adminpage.$$.fragment, local);
    			current = true;
    		},
    		o: function outro(local) {
    			transition_out(adminpage.$$.fragment, local);
    			current = false;
    		},
    		d: function destroy(detaching) {
    			destroy_component(adminpage, detaching);
    		}
    	};

    	dispatch_dev("SvelteRegisterBlock", {
    		block,
    		id: create_fragment$7.name,
    		type: "component",
    		source: "",
    		ctx
    	});

    	return block;
    }

    function instance$7($$self, $$props, $$invalidate) {
    	let $appState;
    	let { $$slots: slots = {}, $$scope } = $$props;
    	validate_slots('Loggedout', slots, []);
    	let appState = getContext('appState');
    	validate_store(appState, 'appState');
    	component_subscribe($$self, appState, value => $$invalidate(0, $appState = value));
    	const writable_props = [];

    	Object.keys($$props).forEach(key => {
    		if (!~writable_props.indexOf(key) && key.slice(0, 2) !== '$$' && key !== 'slot') console.warn(`<Loggedout> was created with unknown prop '${key}'`);
    	});

    	const click_handler = () => window.location.reload();

    	$$self.$capture_state = () => ({
    		AdminPage,
    		t,
    		getContext,
    		appState,
    		$appState
    	});

    	$$self.$inject_state = $$props => {
    		if ('appState' in $$props) $$invalidate(1, appState = $$props.appState);
    	};

    	if ($$props && "$$inject" in $$props) {
    		$$self.$inject_state($$props.$$inject);
    	}

    	return [$appState, appState, click_handler];
    }

    class Loggedout extends SvelteComponentDev {
    	constructor(options) {
    		super(options);
    		init(this, options, instance$7, create_fragment$7, safe_not_equal, {});

    		dispatch_dev("SvelteRegisterComponent", {
    			component: this,
    			tagName: "Loggedout",
    			options,
    			id: create_fragment$7.name
    		});
    	}
    }

    /* node_modules\@weeblr\svelte\components\NotificationList.svelte generated by Svelte v3.41.0 */
    const file$6 = "node_modules\\@weeblr\\svelte\\components\\NotificationList.svelte";

    function get_each_context(ctx, list, i) {
    	const child_ctx = ctx.slice();
    	child_ctx[5] = list[i];
    	return child_ctx;
    }

    function get_each_context_1(ctx, list, i) {
    	const child_ctx = ctx.slice();
    	child_ctx[5] = list[i];
    	child_ctx[9] = i;
    	return child_ctx;
    }

    // (17:0) {#if "permanent" == type && store && expanded}
    function create_if_block_1$5(ctx) {
    	let div;
    	let each_blocks = [];
    	let each_1_lookup = new Map();
    	let current;
    	let each_value_1 = /*$store*/ ctx[3].store;
    	validate_each_argument(each_value_1);
    	const get_key = ctx => /*notification*/ ctx[5].uuid;
    	validate_each_keys(ctx, each_value_1, get_each_context_1, get_key);

    	for (let i = 0; i < each_value_1.length; i += 1) {
    		let child_ctx = get_each_context_1(ctx, each_value_1, i);
    		let key = get_key(child_ctx);
    		each_1_lookup.set(key, each_blocks[i] = create_each_block_1(key, child_ctx));
    	}

    	const block = {
    		c: function create() {
    			div = element("div");

    			for (let i = 0; i < each_blocks.length; i += 1) {
    				each_blocks[i].c();
    			}

    			attr_dev(div, "id", "wbnotifications");
    			attr_dev(div, "class", "wb-z-90 wb-absolute wb-right-0 wb-mt-2 wb-mr-2 md:wb-w-sm wb-w-84 wb-max-w-sm md:wb-max-w-md wb-overflow-y-auto wb-bg-gray-200 wb-text-gray-900 wb-border wb-border-gray-300 wb-rounded-md wb-shadow-lg");
    			add_location(div, file$6, 17, 4, 452);
    		},
    		m: function mount(target, anchor) {
    			insert_dev(target, div, anchor);

    			for (let i = 0; i < each_blocks.length; i += 1) {
    				each_blocks[i].m(div, null);
    			}

    			current = true;
    		},
    		p: function update(ctx, dirty) {
    			if (dirty & /*$store, completionHandler*/ 24) {
    				each_value_1 = /*$store*/ ctx[3].store;
    				validate_each_argument(each_value_1);
    				group_outros();
    				validate_each_keys(ctx, each_value_1, get_each_context_1, get_key);
    				each_blocks = update_keyed_each(each_blocks, dirty, get_key, 1, ctx, each_value_1, each_1_lookup, div, outro_and_destroy_block, create_each_block_1, null, get_each_context_1);
    				check_outros();
    			}
    		},
    		i: function intro(local) {
    			if (current) return;

    			for (let i = 0; i < each_value_1.length; i += 1) {
    				transition_in(each_blocks[i]);
    			}

    			current = true;
    		},
    		o: function outro(local) {
    			for (let i = 0; i < each_blocks.length; i += 1) {
    				transition_out(each_blocks[i]);
    			}

    			current = false;
    		},
    		d: function destroy(detaching) {
    			if (detaching) detach_dev(div);

    			for (let i = 0; i < each_blocks.length; i += 1) {
    				each_blocks[i].d();
    			}
    		}
    	};

    	dispatch_dev("SvelteRegisterBlock", {
    		block,
    		id: create_if_block_1$5.name,
    		type: "if",
    		source: "(17:0) {#if \\\"permanent\\\" == type && store && expanded}",
    		ctx
    	});

    	return block;
    }

    // (26:12) {#if index < $store.store.length - 1 && $store.store.length > 1 }
    function create_if_block_2$5(ctx) {
    	let div;

    	const block = {
    		c: function create() {
    			div = element("div");
    			attr_dev(div, "class", "wb-border-t wb-border-gray-300");
    			add_location(div, file$6, 26, 16, 1166);
    		},
    		m: function mount(target, anchor) {
    			insert_dev(target, div, anchor);
    		},
    		d: function destroy(detaching) {
    			if (detaching) detach_dev(div);
    		}
    	};

    	dispatch_dev("SvelteRegisterBlock", {
    		block,
    		id: create_if_block_2$5.name,
    		type: "if",
    		source: "(26:12) {#if index < $store.store.length - 1 && $store.store.length > 1 }",
    		ctx
    	});

    	return block;
    }

    // (20:8) {#each $store.store as notification,index (notification.uuid)}
    function create_each_block_1(key_1, ctx) {
    	let div;
    	let notification;
    	let div_outro;
    	let t;
    	let if_block_anchor;
    	let current;
    	let mounted;
    	let dispose;

    	notification = new Notification({
    			props: {
    				notification: /*notification*/ ctx[5],
    				onCompletion: /*completionHandler*/ ctx[4]
    			},
    			$$inline: true
    		});

    	let if_block = /*index*/ ctx[9] < /*$store*/ ctx[3].store.length - 1 && /*$store*/ ctx[3].store.length > 1 && create_if_block_2$5(ctx);

    	const block = {
    		key: key_1,
    		first: null,
    		c: function create() {
    			div = element("div");
    			create_component(notification.$$.fragment);
    			t = space();
    			if (if_block) if_block.c();
    			if_block_anchor = empty();
    			attr_dev(div, "class", "wb-py-4 wb-flex wb-items-center wb-justify-between hover:wb-bg-white hover:wb-text-gray-900");
    			add_location(div, file$6, 20, 12, 779);
    			this.first = div;
    		},
    		m: function mount(target, anchor) {
    			insert_dev(target, div, anchor);
    			mount_component(notification, div, null);
    			insert_dev(target, t, anchor);
    			if (if_block) if_block.m(target, anchor);
    			insert_dev(target, if_block_anchor, anchor);
    			current = true;

    			if (!mounted) {
    				dispose = action_destroyer(focusOnDefault.call(null, div));
    				mounted = true;
    			}
    		},
    		p: function update(new_ctx, dirty) {
    			ctx = new_ctx;
    			const notification_changes = {};
    			if (dirty & /*$store*/ 8) notification_changes.notification = /*notification*/ ctx[5];
    			notification.$set(notification_changes);

    			if (/*index*/ ctx[9] < /*$store*/ ctx[3].store.length - 1 && /*$store*/ ctx[3].store.length > 1) {
    				if (if_block) ; else {
    					if_block = create_if_block_2$5(ctx);
    					if_block.c();
    					if_block.m(if_block_anchor.parentNode, if_block_anchor);
    				}
    			} else if (if_block) {
    				if_block.d(1);
    				if_block = null;
    			}
    		},
    		i: function intro(local) {
    			if (current) return;
    			transition_in(notification.$$.fragment, local);
    			if (div_outro) div_outro.end(1);
    			current = true;
    		},
    		o: function outro(local) {
    			transition_out(notification.$$.fragment, local);

    			if (local) {
    				div_outro = create_out_transition(div, slide, { duration: 400 });
    			}

    			current = false;
    		},
    		d: function destroy(detaching) {
    			if (detaching) detach_dev(div);
    			destroy_component(notification);
    			if (detaching && div_outro) div_outro.end();
    			if (detaching) detach_dev(t);
    			if (if_block) if_block.d(detaching);
    			if (detaching) detach_dev(if_block_anchor);
    			mounted = false;
    			dispose();
    		}
    	};

    	dispatch_dev("SvelteRegisterBlock", {
    		block,
    		id: create_each_block_1.name,
    		type: "each",
    		source: "(20:8) {#each $store.store as notification,index (notification.uuid)}",
    		ctx
    	});

    	return block;
    }

    // (33:0) {#if "oneoff" == type && store && $store.total > 0 }
    function create_if_block$6(ctx) {
    	let div;
    	let each_blocks = [];
    	let each_1_lookup = new Map();
    	let current;
    	let each_value = /*$store*/ ctx[3].store;
    	validate_each_argument(each_value);
    	const get_key = ctx => /*notification*/ ctx[5].uuid;
    	validate_each_keys(ctx, each_value, get_each_context, get_key);

    	for (let i = 0; i < each_value.length; i += 1) {
    		let child_ctx = get_each_context(ctx, each_value, i);
    		let key = get_key(child_ctx);
    		each_1_lookup.set(key, each_blocks[i] = create_each_block(key, child_ctx));
    	}

    	const block = {
    		c: function create() {
    			div = element("div");

    			for (let i = 0; i < each_blocks.length; i += 1) {
    				each_blocks[i].c();
    			}

    			attr_dev(div, "id", "wbsinglenotifications");
    			attr_dev(div, "class", "wb-z-90 wb-absolute wb-self-center wb-mt-4 wb-w-full wb-max-w-sm md:wb-max-w-md lg:wb-max-w-lg wb-overflow-y-auto wb-bg-transparent");
    			add_location(div, file$6, 33, 4, 1326);
    		},
    		m: function mount(target, anchor) {
    			insert_dev(target, div, anchor);

    			for (let i = 0; i < each_blocks.length; i += 1) {
    				each_blocks[i].m(div, null);
    			}

    			current = true;
    		},
    		p: function update(ctx, dirty) {
    			if (dirty & /*$store, completionHandler*/ 24) {
    				each_value = /*$store*/ ctx[3].store;
    				validate_each_argument(each_value);
    				group_outros();
    				validate_each_keys(ctx, each_value, get_each_context, get_key);
    				each_blocks = update_keyed_each(each_blocks, dirty, get_key, 1, ctx, each_value, each_1_lookup, div, outro_and_destroy_block, create_each_block, null, get_each_context);
    				check_outros();
    			}
    		},
    		i: function intro(local) {
    			if (current) return;

    			for (let i = 0; i < each_value.length; i += 1) {
    				transition_in(each_blocks[i]);
    			}

    			current = true;
    		},
    		o: function outro(local) {
    			for (let i = 0; i < each_blocks.length; i += 1) {
    				transition_out(each_blocks[i]);
    			}

    			current = false;
    		},
    		d: function destroy(detaching) {
    			if (detaching) detach_dev(div);

    			for (let i = 0; i < each_blocks.length; i += 1) {
    				each_blocks[i].d();
    			}
    		}
    	};

    	dispatch_dev("SvelteRegisterBlock", {
    		block,
    		id: create_if_block$6.name,
    		type: "if",
    		source: "(33:0) {#if \\\"oneoff\\\" == type && store && $store.total > 0 }",
    		ctx
    	});

    	return block;
    }

    // (36:8) {#each $store.store as notification (notification.uuid)}
    function create_each_block(key_1, ctx) {
    	let div;
    	let notification;
    	let t;
    	let div_intro;
    	let div_outro;
    	let current;

    	notification = new Notification({
    			props: {
    				notification: /*notification*/ ctx[5],
    				onCompletion: /*completionHandler*/ ctx[4]
    			},
    			$$inline: true
    		});

    	const block = {
    		key: key_1,
    		first: null,
    		c: function create() {
    			div = element("div");
    			create_component(notification.$$.fragment);
    			t = space();
    			attr_dev(div, "class", "wb-mb-4 wb-py-4 wb-flex wb-flex-col wb-bg-gray-200 wb-text-gray-900 wb-border wb-border-gray-300 wb-rounded-md wb-shadow-lg hover:wb-bg-white hover:wb-text-gray-900");
    			add_location(div, file$6, 36, 12, 1585);
    			this.first = div;
    		},
    		m: function mount(target, anchor) {
    			insert_dev(target, div, anchor);
    			mount_component(notification, div, null);
    			append_dev(div, t);
    			current = true;
    		},
    		p: function update(new_ctx, dirty) {
    			ctx = new_ctx;
    			const notification_changes = {};
    			if (dirty & /*$store*/ 8) notification_changes.notification = /*notification*/ ctx[5];
    			notification.$set(notification_changes);
    		},
    		i: function intro(local) {
    			if (current) return;
    			transition_in(notification.$$.fragment, local);

    			add_render_callback(() => {
    				if (div_outro) div_outro.end(1);
    				div_intro = create_in_transition(div, fly, { duration: 400, y: -100 });
    				div_intro.start();
    			});

    			current = true;
    		},
    		o: function outro(local) {
    			transition_out(notification.$$.fragment, local);
    			if (div_intro) div_intro.invalidate();
    			div_outro = create_out_transition(div, slide, { duration: 400 });
    			current = false;
    		},
    		d: function destroy(detaching) {
    			if (detaching) detach_dev(div);
    			destroy_component(notification);
    			if (detaching && div_outro) div_outro.end();
    		}
    	};

    	dispatch_dev("SvelteRegisterBlock", {
    		block,
    		id: create_each_block.name,
    		type: "each",
    		source: "(36:8) {#each $store.store as notification (notification.uuid)}",
    		ctx
    	});

    	return block;
    }

    function create_fragment$6(ctx) {
    	let t;
    	let if_block1_anchor;
    	let current;
    	let if_block0 = "permanent" == /*type*/ ctx[1] && /*store*/ ctx[0] && /*expanded*/ ctx[2] && create_if_block_1$5(ctx);
    	let if_block1 = "oneoff" == /*type*/ ctx[1] && /*store*/ ctx[0] && /*$store*/ ctx[3].total > 0 && create_if_block$6(ctx);

    	const block = {
    		c: function create() {
    			if (if_block0) if_block0.c();
    			t = space();
    			if (if_block1) if_block1.c();
    			if_block1_anchor = empty();
    		},
    		l: function claim(nodes) {
    			throw new Error("options.hydrate only works if the component was compiled with the `hydratable: true` option");
    		},
    		m: function mount(target, anchor) {
    			if (if_block0) if_block0.m(target, anchor);
    			insert_dev(target, t, anchor);
    			if (if_block1) if_block1.m(target, anchor);
    			insert_dev(target, if_block1_anchor, anchor);
    			current = true;
    		},
    		p: function update(ctx, [dirty]) {
    			if ("permanent" == /*type*/ ctx[1] && /*store*/ ctx[0] && /*expanded*/ ctx[2]) {
    				if (if_block0) {
    					if_block0.p(ctx, dirty);

    					if (dirty & /*type, store, expanded*/ 7) {
    						transition_in(if_block0, 1);
    					}
    				} else {
    					if_block0 = create_if_block_1$5(ctx);
    					if_block0.c();
    					transition_in(if_block0, 1);
    					if_block0.m(t.parentNode, t);
    				}
    			} else if (if_block0) {
    				group_outros();

    				transition_out(if_block0, 1, 1, () => {
    					if_block0 = null;
    				});

    				check_outros();
    			}

    			if ("oneoff" == /*type*/ ctx[1] && /*store*/ ctx[0] && /*$store*/ ctx[3].total > 0) {
    				if (if_block1) {
    					if_block1.p(ctx, dirty);

    					if (dirty & /*type, store, $store*/ 11) {
    						transition_in(if_block1, 1);
    					}
    				} else {
    					if_block1 = create_if_block$6(ctx);
    					if_block1.c();
    					transition_in(if_block1, 1);
    					if_block1.m(if_block1_anchor.parentNode, if_block1_anchor);
    				}
    			} else if (if_block1) {
    				group_outros();

    				transition_out(if_block1, 1, 1, () => {
    					if_block1 = null;
    				});

    				check_outros();
    			}
    		},
    		i: function intro(local) {
    			if (current) return;
    			transition_in(if_block0);
    			transition_in(if_block1);
    			current = true;
    		},
    		o: function outro(local) {
    			transition_out(if_block0);
    			transition_out(if_block1);
    			current = false;
    		},
    		d: function destroy(detaching) {
    			if (if_block0) if_block0.d(detaching);
    			if (detaching) detach_dev(t);
    			if (if_block1) if_block1.d(detaching);
    			if (detaching) detach_dev(if_block1_anchor);
    		}
    	};

    	dispatch_dev("SvelteRegisterBlock", {
    		block,
    		id: create_fragment$6.name,
    		type: "component",
    		source: "",
    		ctx
    	});

    	return block;
    }

    function instance$6($$self, $$props, $$invalidate) {
    	let $store,
    		$$unsubscribe_store = noop$1,
    		$$subscribe_store = () => ($$unsubscribe_store(), $$unsubscribe_store = subscribe(store, $$value => $$invalidate(3, $store = $$value)), store);

    	$$self.$$.on_destroy.push(() => $$unsubscribe_store());
    	let { $$slots: slots = {}, $$scope } = $$props;
    	validate_slots('NotificationList', slots, []);
    	let { store = undefined } = $$props;
    	validate_store(store, 'store');
    	$$subscribe_store();
    	let { type = undefined } = $$props;
    	let { expanded = false } = $$props;

    	// Implementation
    	const completionHandler = notification => store.dismiss(notification.uuid);

    	const writable_props = ['store', 'type', 'expanded'];

    	Object.keys($$props).forEach(key => {
    		if (!~writable_props.indexOf(key) && key.slice(0, 2) !== '$$' && key !== 'slot') console.warn(`<NotificationList> was created with unknown prop '${key}'`);
    	});

    	$$self.$$set = $$props => {
    		if ('store' in $$props) $$subscribe_store($$invalidate(0, store = $$props.store));
    		if ('type' in $$props) $$invalidate(1, type = $$props.type);
    		if ('expanded' in $$props) $$invalidate(2, expanded = $$props.expanded);
    	};

    	$$self.$capture_state = () => ({
    		store,
    		type,
    		expanded,
    		Notification,
    		fly,
    		slide,
    		focusOnDefault,
    		completionHandler,
    		$store
    	});

    	$$self.$inject_state = $$props => {
    		if ('store' in $$props) $$subscribe_store($$invalidate(0, store = $$props.store));
    		if ('type' in $$props) $$invalidate(1, type = $$props.type);
    		if ('expanded' in $$props) $$invalidate(2, expanded = $$props.expanded);
    	};

    	if ($$props && "$$inject" in $$props) {
    		$$self.$inject_state($$props.$$inject);
    	}

    	return [store, type, expanded, $store, completionHandler];
    }

    class NotificationList extends SvelteComponentDev {
    	constructor(options) {
    		super(options);
    		init(this, options, instance$6, create_fragment$6, safe_not_equal, { store: 0, type: 1, expanded: 2 });

    		dispatch_dev("SvelteRegisterComponent", {
    			component: this,
    			tagName: "NotificationList",
    			options,
    			id: create_fragment$6.name
    		});
    	}

    	get store() {
    		throw new Error("<NotificationList>: Props cannot be read directly from the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	set store(value) {
    		throw new Error("<NotificationList>: Props cannot be set directly on the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	get type() {
    		throw new Error("<NotificationList>: Props cannot be read directly from the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	set type(value) {
    		throw new Error("<NotificationList>: Props cannot be set directly on the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	get expanded() {
    		throw new Error("<NotificationList>: Props cannot be read directly from the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	set expanded(value) {
    		throw new Error("<NotificationList>: Props cannot be set directly on the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}
    }

    /**
     * Notification record:
     * {
     *     canClear: Boolean
     *     title: string
     *     message: string
     *     link: string
     *     uuid: string
     * }
     *
     */

    let defaults$3 = {
        total: 0,
        store: [],
    };

    const addNotification$1 = (state, type, title, message, canClear, link) => {
        let notification = {
            type: type,
            canClear: !!canClear,
            escapeClear: false,
            title: title,
            message: message,
            link: link,
            uuid: 's' + v4()
        };
        state['store'].push(notification);
        state['store'].sort(
            propAlphaSort('type')()
        );

        state.total = state['store'].length;

        return {state, notification}
    };

    const removeNotification$1 = (state, uuid) => {
        if (!state || !state['store']) {
            return false
        }
        state['store'] = state['store'].filter(
            notification => {
                if (notification.uuid == uuid) {
                    return false
                }

                return true
            }
        );

        state.total = state['store'].length;

        return state
    };

    const hashNotification = (parts) => {
        return parts.join()
    };

    const hasNotification = (state, type, title, msg) => {
        if (!state || !state['store']) {
            return false
        }

        let newNotificationHash = hashNotification([type, title, msg]);
        let matchingNotification = state['store'].find(
            notification =>
                hashNotification(
                    [
                        notification.type,
                        notification.title,
                        notification.message
                    ]
                ) == newNotificationHash
        );
        return matchingNotification
    };

    const doUpdate = (update, type, once, title, msg, canClear, link) => {
        let notification;
        update((state) => {
            if (once && hasNotification(state, type, title, msg)) {
                return state
            }

            let r = addNotification$1(state, type, title, msg, canClear, link);
            notification = r.notification;
            return r.state
        });

        return notification
            ? notification.uuid
            : false
    };

    const createStore$4 = () => {
        const {subscribe, set, update} = writable(defaults$3);
        return {
            subscribe: subscribe,
            addError: (title, msg, canClear, link) => {
                return doUpdate(update, '1_danger', false, title, msg, canClear, link)
            },
            addWarning: (title, msg, canClear, link) => {
                return doUpdate(update, '2_warning', false, title, msg, canClear, link)
            },
            addInfo: (title, msg, canClear, link) => {
                return doUpdate(update, '3_info', false, title, msg, canClear, link)
            },
            addErrorOnce: (title, msg, canClear, link) => {
                return doUpdate(update, '1_danger', true, title, msg, canClear, link)
            },
            addWarningOnce: (title, msg, canClear, link) => {
                return doUpdate(update, '2_warning', true, title, msg, canClear, link)
            },
            addInfoOnce: (title, msg, canClear, link) => {
                return doUpdate(update, '3_info', true, title, msg, canClear, link)
            },
            dismiss: (uuid) => update((state) => {
                return removeNotification$1(state, uuid)
            }),
            reset: () => set(defaults$3)
        }

    };
    const appNotifications = createStore$4();

    /* src\app\admin\components\NotificationIcon.svelte generated by Svelte v3.41.0 */
    const file$5 = "src\\app\\admin\\components\\NotificationIcon.svelte";

    // (58:12) {#if $appNotifications.total > 0}
    function create_if_block_2$4(ctx) {
    	let span;
    	let t_value = /*$appNotifications*/ ctx[1].total + "";
    	let t;

    	const block = {
    		c: function create() {
    			span = element("span");
    			t = text(t_value);
    			attr_dev(span, "class", "wb-mx-1 wb-pl-1 wb-leading-normal wb-font-bold");
    			add_location(span, file$5, 58, 16, 1851);
    		},
    		m: function mount(target, anchor) {
    			insert_dev(target, span, anchor);
    			append_dev(span, t);
    		},
    		p: function update(ctx, dirty) {
    			if (dirty & /*$appNotifications*/ 2 && t_value !== (t_value = /*$appNotifications*/ ctx[1].total + "")) set_data_dev(t, t_value);
    		},
    		d: function destroy(detaching) {
    			if (detaching) detach_dev(span);
    		}
    	};

    	dispatch_dev("SvelteRegisterBlock", {
    		block,
    		id: create_if_block_2$4.name,
    		type: "if",
    		source: "(58:12) {#if $appNotifications.total > 0}",
    		ctx
    	});

    	return block;
    }

    // (66:12) {:else}
    function create_else_block$3(ctx) {
    	let icon;
    	let current;

    	icon = new Icon({
    			props: {
    				listenToClicks: true,
    				tabbable: /*$appNotifications*/ ctx[1].total > 0,
    				icon: iconNotification,
    				size: "xl",
    				classes: "wb-mx-1 wb-pr-1",
    				ariaExpanded: false,
    				ariaOwns: "wbnotifications"
    			},
    			$$inline: true
    		});

    	const block = {
    		c: function create() {
    			create_component(icon.$$.fragment);
    		},
    		m: function mount(target, anchor) {
    			mount_component(icon, target, anchor);
    			current = true;
    		},
    		p: function update(ctx, dirty) {
    			const icon_changes = {};
    			if (dirty & /*$appNotifications*/ 2) icon_changes.tabbable = /*$appNotifications*/ ctx[1].total > 0;
    			icon.$set(icon_changes);
    		},
    		i: function intro(local) {
    			if (current) return;
    			transition_in(icon.$$.fragment, local);
    			current = true;
    		},
    		o: function outro(local) {
    			transition_out(icon.$$.fragment, local);
    			current = false;
    		},
    		d: function destroy(detaching) {
    			destroy_component(icon, detaching);
    		}
    	};

    	dispatch_dev("SvelteRegisterBlock", {
    		block,
    		id: create_else_block$3.name,
    		type: "else",
    		source: "(66:12) {:else}",
    		ctx
    	});

    	return block;
    }

    // (61:12) {#if expanded}
    function create_if_block_1$4(ctx) {
    	let icon;
    	let current;

    	icon = new Icon({
    			props: {
    				listenToClicks: true,
    				tabbable: /*$appNotifications*/ ctx[1].total > 0,
    				icon: iconClose,
    				size: "xl",
    				classes: "wb-mx-1 wb-pr-1",
    				ariaExpanded: true,
    				ariaOwns: "wbnotifications"
    			},
    			$$inline: true
    		});

    	const block = {
    		c: function create() {
    			create_component(icon.$$.fragment);
    		},
    		m: function mount(target, anchor) {
    			mount_component(icon, target, anchor);
    			current = true;
    		},
    		p: function update(ctx, dirty) {
    			const icon_changes = {};
    			if (dirty & /*$appNotifications*/ 2) icon_changes.tabbable = /*$appNotifications*/ ctx[1].total > 0;
    			icon.$set(icon_changes);
    		},
    		i: function intro(local) {
    			if (current) return;
    			transition_in(icon.$$.fragment, local);
    			current = true;
    		},
    		o: function outro(local) {
    			transition_out(icon.$$.fragment, local);
    			current = false;
    		},
    		d: function destroy(detaching) {
    			destroy_component(icon, detaching);
    		}
    	};

    	dispatch_dev("SvelteRegisterBlock", {
    		block,
    		id: create_if_block_1$4.name,
    		type: "if",
    		source: "(61:12) {#if expanded}",
    		ctx
    	});

    	return block;
    }

    // (74:8) {#if expanded}
    function create_if_block$5(ctx) {
    	let clickaway;
    	let current;

    	clickaway = new ClickAway({
    			props: {
    				clickAction: /*func*/ ctx[5],
    				registerClicksOnNonTabbable: true,
    				$$slots: { default: [create_default_slot_1] },
    				$$scope: { ctx }
    			},
    			$$inline: true
    		});

    	const block = {
    		c: function create() {
    			create_component(clickaway.$$.fragment);
    		},
    		m: function mount(target, anchor) {
    			mount_component(clickaway, target, anchor);
    			current = true;
    		},
    		p: function update(ctx, dirty) {
    			const clickaway_changes = {};
    			if (dirty & /*expanded*/ 1) clickaway_changes.clickAction = /*func*/ ctx[5];

    			if (dirty & /*$$scope, expanded*/ 129) {
    				clickaway_changes.$$scope = { dirty, ctx };
    			}

    			clickaway.$set(clickaway_changes);
    		},
    		i: function intro(local) {
    			if (current) return;
    			transition_in(clickaway.$$.fragment, local);
    			current = true;
    		},
    		o: function outro(local) {
    			transition_out(clickaway.$$.fragment, local);
    			current = false;
    		},
    		d: function destroy(detaching) {
    			destroy_component(clickaway, detaching);
    		}
    	};

    	dispatch_dev("SvelteRegisterBlock", {
    		block,
    		id: create_if_block$5.name,
    		type: "if",
    		source: "(74:8) {#if expanded}",
    		ctx
    	});

    	return block;
    }

    // (75:12) <ClickAway clickAction="{() => {expanded = false}}" registerClicksOnNonTabbable="{true}">
    function create_default_slot_1(ctx) {
    	let notificationlist;
    	let current;

    	notificationlist = new NotificationList({
    			props: {
    				type: "permanent",
    				store: appNotifications,
    				expanded: /*expanded*/ ctx[0]
    			},
    			$$inline: true
    		});

    	const block = {
    		c: function create() {
    			create_component(notificationlist.$$.fragment);
    		},
    		m: function mount(target, anchor) {
    			mount_component(notificationlist, target, anchor);
    			current = true;
    		},
    		p: function update(ctx, dirty) {
    			const notificationlist_changes = {};
    			if (dirty & /*expanded*/ 1) notificationlist_changes.expanded = /*expanded*/ ctx[0];
    			notificationlist.$set(notificationlist_changes);
    		},
    		i: function intro(local) {
    			if (current) return;
    			transition_in(notificationlist.$$.fragment, local);
    			current = true;
    		},
    		o: function outro(local) {
    			transition_out(notificationlist.$$.fragment, local);
    			current = false;
    		},
    		d: function destroy(detaching) {
    			destroy_component(notificationlist, detaching);
    		}
    	};

    	dispatch_dev("SvelteRegisterBlock", {
    		block,
    		id: create_default_slot_1.name,
    		type: "slot",
    		source: "(75:12) <ClickAway clickAction=\\\"{() => {expanded = false}}\\\" registerClicksOnNonTabbable=\\\"{true}\\\">",
    		ctx
    	});

    	return block;
    }

    // (53:0) <FocusTrap enabled="{expanded}" classes="wb-align-middle">
    function create_default_slot(ctx) {
    	let div1;
    	let div0;
    	let t0;
    	let current_block_type_index;
    	let if_block1;
    	let div0_class_value;
    	let t1;
    	let current;
    	let mounted;
    	let dispose;
    	let if_block0 = /*$appNotifications*/ ctx[1].total > 0 && create_if_block_2$4(ctx);
    	const if_block_creators = [create_if_block_1$4, create_else_block$3];
    	const if_blocks = [];

    	function select_block_type(ctx, dirty) {
    		if (/*expanded*/ ctx[0]) return 0;
    		return 1;
    	}

    	current_block_type_index = select_block_type(ctx);
    	if_block1 = if_blocks[current_block_type_index] = if_block_creators[current_block_type_index](ctx);
    	let if_block2 = /*expanded*/ ctx[0] && create_if_block$5(ctx);

    	const block = {
    		c: function create() {
    			div1 = element("div");
    			div0 = element("div");
    			if (if_block0) if_block0.c();
    			t0 = space();
    			if_block1.c();
    			t1 = space();
    			if (if_block2) if_block2.c();

    			attr_dev(div0, "class", div0_class_value = "wb-flex wb-items-center " + (/*animate*/ ctx[2] ? 'wb-animate-pulse' : '') + " " + (/*$appNotifications*/ ctx[1].total > 0
    			? 'wb-cursor-pointer wb-bg-accent wb-text-always-dark wb-c-focusable'
    			: 'wb-text-gray-200') + " wb-rounded-full");

    			add_location(div0, file$5, 54, 8, 1510);
    			attr_dev(div1, "class", "wb-relative wb-inline-block wb-m-0 wb-align-middle");
    			add_location(div1, file$5, 53, 4, 1437);
    		},
    		m: function mount(target, anchor) {
    			insert_dev(target, div1, anchor);
    			append_dev(div1, div0);
    			if (if_block0) if_block0.m(div0, null);
    			append_dev(div0, t0);
    			if_blocks[current_block_type_index].m(div0, null);
    			append_dev(div1, t1);
    			if (if_block2) if_block2.m(div1, null);
    			current = true;

    			if (!mounted) {
    				dispose = [
    					listen_dev(div0, "click", /*onClick*/ ctx[3], false, false, false),
    					listen_dev(div0, "keydown", /*onKeydown*/ ctx[4], false, false, false)
    				];

    				mounted = true;
    			}
    		},
    		p: function update(ctx, dirty) {
    			if (/*$appNotifications*/ ctx[1].total > 0) {
    				if (if_block0) {
    					if_block0.p(ctx, dirty);
    				} else {
    					if_block0 = create_if_block_2$4(ctx);
    					if_block0.c();
    					if_block0.m(div0, t0);
    				}
    			} else if (if_block0) {
    				if_block0.d(1);
    				if_block0 = null;
    			}

    			let previous_block_index = current_block_type_index;
    			current_block_type_index = select_block_type(ctx);

    			if (current_block_type_index === previous_block_index) {
    				if_blocks[current_block_type_index].p(ctx, dirty);
    			} else {
    				group_outros();

    				transition_out(if_blocks[previous_block_index], 1, 1, () => {
    					if_blocks[previous_block_index] = null;
    				});

    				check_outros();
    				if_block1 = if_blocks[current_block_type_index];

    				if (!if_block1) {
    					if_block1 = if_blocks[current_block_type_index] = if_block_creators[current_block_type_index](ctx);
    					if_block1.c();
    				} else {
    					if_block1.p(ctx, dirty);
    				}

    				transition_in(if_block1, 1);
    				if_block1.m(div0, null);
    			}

    			if (!current || dirty & /*animate, $appNotifications*/ 6 && div0_class_value !== (div0_class_value = "wb-flex wb-items-center " + (/*animate*/ ctx[2] ? 'wb-animate-pulse' : '') + " " + (/*$appNotifications*/ ctx[1].total > 0
    			? 'wb-cursor-pointer wb-bg-accent wb-text-always-dark wb-c-focusable'
    			: 'wb-text-gray-200') + " wb-rounded-full")) {
    				attr_dev(div0, "class", div0_class_value);
    			}

    			if (/*expanded*/ ctx[0]) {
    				if (if_block2) {
    					if_block2.p(ctx, dirty);

    					if (dirty & /*expanded*/ 1) {
    						transition_in(if_block2, 1);
    					}
    				} else {
    					if_block2 = create_if_block$5(ctx);
    					if_block2.c();
    					transition_in(if_block2, 1);
    					if_block2.m(div1, null);
    				}
    			} else if (if_block2) {
    				group_outros();

    				transition_out(if_block2, 1, 1, () => {
    					if_block2 = null;
    				});

    				check_outros();
    			}
    		},
    		i: function intro(local) {
    			if (current) return;
    			transition_in(if_block1);
    			transition_in(if_block2);
    			current = true;
    		},
    		o: function outro(local) {
    			transition_out(if_block1);
    			transition_out(if_block2);
    			current = false;
    		},
    		d: function destroy(detaching) {
    			if (detaching) detach_dev(div1);
    			if (if_block0) if_block0.d();
    			if_blocks[current_block_type_index].d();
    			if (if_block2) if_block2.d();
    			mounted = false;
    			run_all(dispose);
    		}
    	};

    	dispatch_dev("SvelteRegisterBlock", {
    		block,
    		id: create_default_slot.name,
    		type: "slot",
    		source: "(53:0) <FocusTrap enabled=\\\"{expanded}\\\" classes=\\\"wb-align-middle\\\">",
    		ctx
    	});

    	return block;
    }

    function create_fragment$5(ctx) {
    	let focustrap;
    	let current;

    	focustrap = new FocusTrap({
    			props: {
    				enabled: /*expanded*/ ctx[0],
    				classes: "wb-align-middle",
    				$$slots: { default: [create_default_slot] },
    				$$scope: { ctx }
    			},
    			$$inline: true
    		});

    	const block = {
    		c: function create() {
    			create_component(focustrap.$$.fragment);
    		},
    		l: function claim(nodes) {
    			throw new Error("options.hydrate only works if the component was compiled with the `hydratable: true` option");
    		},
    		m: function mount(target, anchor) {
    			mount_component(focustrap, target, anchor);
    			current = true;
    		},
    		p: function update(ctx, [dirty]) {
    			const focustrap_changes = {};
    			if (dirty & /*expanded*/ 1) focustrap_changes.enabled = /*expanded*/ ctx[0];

    			if (dirty & /*$$scope, expanded, animate, $appNotifications*/ 135) {
    				focustrap_changes.$$scope = { dirty, ctx };
    			}

    			focustrap.$set(focustrap_changes);
    		},
    		i: function intro(local) {
    			if (current) return;
    			transition_in(focustrap.$$.fragment, local);
    			current = true;
    		},
    		o: function outro(local) {
    			transition_out(focustrap.$$.fragment, local);
    			current = false;
    		},
    		d: function destroy(detaching) {
    			destroy_component(focustrap, detaching);
    		}
    	};

    	dispatch_dev("SvelteRegisterBlock", {
    		block,
    		id: create_fragment$5.name,
    		type: "component",
    		source: "",
    		ctx
    	});

    	return block;
    }

    function instance$5($$self, $$props, $$invalidate) {
    	let $appNotifications;
    	validate_store(appNotifications, 'appNotifications');
    	component_subscribe($$self, appNotifications, $$value => $$invalidate(1, $appNotifications = $$value));
    	let { $$slots: slots = {}, $$scope } = $$props;
    	validate_slots('NotificationIcon', slots, []);
    	let expanded = false;

    	const onClick = e => {
    		e.preventDefault();
    		e.stopPropagation();
    		$$invalidate(0, expanded = $appNotifications.total > 0 ? !expanded : expanded);
    	};

    	const onKeydown = e => {
    		if (expanded && isEscapeKey(e)) {
    			e.preventDefault();
    			e.stopPropagation();
    			$$invalidate(0, expanded = false);
    		}
    	};

    	let animate;

    	const triggerAnimation = () => {
    		if (!$appNotifications.total || expanded) {
    			$$invalidate(2, animate = false);
    		} else {
    			$$invalidate(2, animate = !animate);
    		}

    		setTimeout(triggerAnimation, animate ? 3000 : 6000);
    	};

    	onMount(triggerAnimation);
    	const writable_props = [];

    	Object.keys($$props).forEach(key => {
    		if (!~writable_props.indexOf(key) && key.slice(0, 2) !== '$$' && key !== 'slot') console.warn(`<NotificationIcon> was created with unknown prop '${key}'`);
    	});

    	const func = () => {
    		$$invalidate(0, expanded = false);
    	};

    	$$self.$capture_state = () => ({
    		NotificationList,
    		appNotifications,
    		Icon,
    		ClickAway,
    		iconNotification,
    		iconClose,
    		FocusTrap,
    		isEscapeKey,
    		onMount,
    		expanded,
    		onClick,
    		onKeydown,
    		animate,
    		triggerAnimation,
    		$appNotifications
    	});

    	$$self.$inject_state = $$props => {
    		if ('expanded' in $$props) $$invalidate(0, expanded = $$props.expanded);
    		if ('animate' in $$props) $$invalidate(2, animate = $$props.animate);
    	};

    	if ($$props && "$$inject" in $$props) {
    		$$self.$inject_state($$props.$$inject);
    	}

    	$$self.$$.update = () => {
    		if ($$self.$$.dirty & /*$appNotifications, expanded*/ 3) {
    			$$invalidate(0, expanded = $appNotifications.total > 0 ? expanded : false);
    		}
    	};

    	return [expanded, $appNotifications, animate, onClick, onKeydown, func];
    }

    class NotificationIcon extends SvelteComponentDev {
    	constructor(options) {
    		super(options);
    		init(this, options, instance$5, create_fragment$5, safe_not_equal, {});

    		dispatch_dev("SvelteRegisterComponent", {
    			component: this,
    			tagName: "NotificationIcon",
    			options,
    			id: create_fragment$5.name
    		});
    	}
    }

    /* node_modules\@weeblr\svelte\components\Spinner.svelte generated by Svelte v3.41.0 */
    const file$4 = "node_modules\\@weeblr\\svelte\\components\\Spinner.svelte";

    // (34:0) {:else }
    function create_else_block$2(ctx) {
    	let div;

    	const block = {
    		c: function create() {
    			div = element("div");
    			attr_dev(div, "class", "wb-mx-2 wb-my-0 wb-h-6 wb-w-6 wb-leading-normal");
    			add_location(div, file$4, 34, 4, 1041);
    		},
    		m: function mount(target, anchor) {
    			insert_dev(target, div, anchor);
    		},
    		p: noop$1,
    		i: noop$1,
    		o: noop$1,
    		d: function destroy(detaching) {
    			if (detaching) detach_dev(div);
    		}
    	};

    	dispatch_dev("SvelteRegisterBlock", {
    		block,
    		id: create_else_block$2.name,
    		type: "else",
    		source: "(34:0) {:else }",
    		ctx
    	});

    	return block;
    }

    // (28:15) 
    function create_if_block_2$3(ctx) {
    	let div;
    	let icon;
    	let div_class_value;
    	let current;

    	icon = new Icon({
    			props: { icon: iconCheckmarkOutline },
    			$$inline: true
    		});

    	const block = {
    		c: function create() {
    			div = element("div");
    			create_component(icon.$$.fragment);
    			attr_dev(div, "role", "status");
    			attr_dev(div, "class", div_class_value = "wb-flex wb-items-center wb-mx-2 wb-my-0 wb-h-6 wb-w-6 wb-leading-normal " + /*textColor*/ ctx[3] + " svelte-fo37kr");
    			attr_dev(div, "aria-label", t('operations.done'));
    			add_location(div, file$4, 28, 4, 805);
    		},
    		m: function mount(target, anchor) {
    			insert_dev(target, div, anchor);
    			mount_component(icon, div, null);
    			current = true;
    		},
    		p: function update(ctx, dirty) {
    			if (!current || dirty & /*textColor*/ 8 && div_class_value !== (div_class_value = "wb-flex wb-items-center wb-mx-2 wb-my-0 wb-h-6 wb-w-6 wb-leading-normal " + /*textColor*/ ctx[3] + " svelte-fo37kr")) {
    				attr_dev(div, "class", div_class_value);
    			}
    		},
    		i: function intro(local) {
    			if (current) return;
    			transition_in(icon.$$.fragment, local);
    			current = true;
    		},
    		o: function outro(local) {
    			transition_out(icon.$$.fragment, local);
    			current = false;
    		},
    		d: function destroy(detaching) {
    			if (detaching) detach_dev(div);
    			destroy_component(icon);
    		}
    	};

    	dispatch_dev("SvelteRegisterBlock", {
    		block,
    		id: create_if_block_2$3.name,
    		type: "if",
    		source: "(28:15) ",
    		ctx
    	});

    	return block;
    }

    // (22:18) 
    function create_if_block_1$3(ctx) {
    	let div;
    	let icon;
    	let div_class_value;
    	let current;

    	icon = new Icon({
    			props: { icon: iconInProgress },
    			$$inline: true
    		});

    	const block = {
    		c: function create() {
    			div = element("div");
    			create_component(icon.$$.fragment);
    			attr_dev(div, "role", "status");
    			attr_dev(div, "class", div_class_value = "wb-flex wb-items-center wb-mx-2 wb-my-0 wb-h-6 wb-w-6 wb-leading-normal " + /*textColor*/ ctx[3] + " svelte-fo37kr");
    			attr_dev(div, "aria-label", t('operations.pending'));
    			add_location(div, file$4, 22, 4, 565);
    		},
    		m: function mount(target, anchor) {
    			insert_dev(target, div, anchor);
    			mount_component(icon, div, null);
    			current = true;
    		},
    		p: function update(ctx, dirty) {
    			if (!current || dirty & /*textColor*/ 8 && div_class_value !== (div_class_value = "wb-flex wb-items-center wb-mx-2 wb-my-0 wb-h-6 wb-w-6 wb-leading-normal " + /*textColor*/ ctx[3] + " svelte-fo37kr")) {
    				attr_dev(div, "class", div_class_value);
    			}
    		},
    		i: function intro(local) {
    			if (current) return;
    			transition_in(icon.$$.fragment, local);
    			current = true;
    		},
    		o: function outro(local) {
    			transition_out(icon.$$.fragment, local);
    			current = false;
    		},
    		d: function destroy(detaching) {
    			if (detaching) detach_dev(div);
    			destroy_component(icon);
    		}
    	};

    	dispatch_dev("SvelteRegisterBlock", {
    		block,
    		id: create_if_block_1$3.name,
    		type: "if",
    		source: "(22:18) ",
    		ctx
    	});

    	return block;
    }

    // (17:0) {#if busy}
    function create_if_block$4(ctx) {
    	let div;

    	const block = {
    		c: function create() {
    			div = element("div");
    			attr_dev(div, "role", "status");
    			attr_dev(div, "class", "wbl-spinner wb-mx-2 wb-my-0 wb-h-6 wb-w-6 wb-leading-normal wb-border-2 wb-border-solid svelte-fo37kr");
    			attr_dev(div, "aria-label", t('operations.inProgress'));
    			add_location(div, file$4, 17, 4, 360);
    		},
    		m: function mount(target, anchor) {
    			insert_dev(target, div, anchor);
    		},
    		p: noop$1,
    		i: noop$1,
    		o: noop$1,
    		d: function destroy(detaching) {
    			if (detaching) detach_dev(div);
    		}
    	};

    	dispatch_dev("SvelteRegisterBlock", {
    		block,
    		id: create_if_block$4.name,
    		type: "if",
    		source: "(17:0) {#if busy}",
    		ctx
    	});

    	return block;
    }

    function create_fragment$4(ctx) {
    	let current_block_type_index;
    	let if_block;
    	let if_block_anchor;
    	let current;
    	const if_block_creators = [create_if_block$4, create_if_block_1$3, create_if_block_2$3, create_else_block$2];
    	const if_blocks = [];

    	function select_block_type(ctx, dirty) {
    		if (/*busy*/ ctx[0]) return 0;
    		if (/*pending*/ ctx[1]) return 1;
    		if (/*done*/ ctx[2]) return 2;
    		return 3;
    	}

    	current_block_type_index = select_block_type(ctx);
    	if_block = if_blocks[current_block_type_index] = if_block_creators[current_block_type_index](ctx);

    	const block = {
    		c: function create() {
    			if_block.c();
    			if_block_anchor = empty();
    		},
    		l: function claim(nodes) {
    			throw new Error("options.hydrate only works if the component was compiled with the `hydratable: true` option");
    		},
    		m: function mount(target, anchor) {
    			if_blocks[current_block_type_index].m(target, anchor);
    			insert_dev(target, if_block_anchor, anchor);
    			current = true;
    		},
    		p: function update(ctx, [dirty]) {
    			let previous_block_index = current_block_type_index;
    			current_block_type_index = select_block_type(ctx);

    			if (current_block_type_index === previous_block_index) {
    				if_blocks[current_block_type_index].p(ctx, dirty);
    			} else {
    				group_outros();

    				transition_out(if_blocks[previous_block_index], 1, 1, () => {
    					if_blocks[previous_block_index] = null;
    				});

    				check_outros();
    				if_block = if_blocks[current_block_type_index];

    				if (!if_block) {
    					if_block = if_blocks[current_block_type_index] = if_block_creators[current_block_type_index](ctx);
    					if_block.c();
    				} else {
    					if_block.p(ctx, dirty);
    				}

    				transition_in(if_block, 1);
    				if_block.m(if_block_anchor.parentNode, if_block_anchor);
    			}
    		},
    		i: function intro(local) {
    			if (current) return;
    			transition_in(if_block);
    			current = true;
    		},
    		o: function outro(local) {
    			transition_out(if_block);
    			current = false;
    		},
    		d: function destroy(detaching) {
    			if_blocks[current_block_type_index].d(detaching);
    			if (detaching) detach_dev(if_block_anchor);
    		}
    	};

    	dispatch_dev("SvelteRegisterBlock", {
    		block,
    		id: create_fragment$4.name,
    		type: "component",
    		source: "",
    		ctx
    	});

    	return block;
    }

    function instance$4($$self, $$props, $$invalidate) {
    	let { $$slots: slots = {}, $$scope } = $$props;
    	validate_slots('Spinner', slots, []);
    	let { busy = false } = $$props;
    	let { pending = false } = $$props;
    	let { done = false } = $$props;
    	let { textColor = 'wb-text-gray-900' } = $$props;
    	const writable_props = ['busy', 'pending', 'done', 'textColor'];

    	Object.keys($$props).forEach(key => {
    		if (!~writable_props.indexOf(key) && key.slice(0, 2) !== '$$' && key !== 'slot') console.warn(`<Spinner> was created with unknown prop '${key}'`);
    	});

    	$$self.$$set = $$props => {
    		if ('busy' in $$props) $$invalidate(0, busy = $$props.busy);
    		if ('pending' in $$props) $$invalidate(1, pending = $$props.pending);
    		if ('done' in $$props) $$invalidate(2, done = $$props.done);
    		if ('textColor' in $$props) $$invalidate(3, textColor = $$props.textColor);
    	};

    	$$self.$capture_state = () => ({
    		busy,
    		pending,
    		done,
    		textColor,
    		iconCheckmarkOutline,
    		iconInProgress,
    		t,
    		Icon
    	});

    	$$self.$inject_state = $$props => {
    		if ('busy' in $$props) $$invalidate(0, busy = $$props.busy);
    		if ('pending' in $$props) $$invalidate(1, pending = $$props.pending);
    		if ('done' in $$props) $$invalidate(2, done = $$props.done);
    		if ('textColor' in $$props) $$invalidate(3, textColor = $$props.textColor);
    	};

    	if ($$props && "$$inject" in $$props) {
    		$$self.$inject_state($$props.$$inject);
    	}

    	return [busy, pending, done, textColor];
    }

    class Spinner extends SvelteComponentDev {
    	constructor(options) {
    		super(options);

    		init(this, options, instance$4, create_fragment$4, safe_not_equal, {
    			busy: 0,
    			pending: 1,
    			done: 2,
    			textColor: 3
    		});

    		dispatch_dev("SvelteRegisterComponent", {
    			component: this,
    			tagName: "Spinner",
    			options,
    			id: create_fragment$4.name
    		});
    	}

    	get busy() {
    		throw new Error("<Spinner>: Props cannot be read directly from the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	set busy(value) {
    		throw new Error("<Spinner>: Props cannot be set directly on the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	get pending() {
    		throw new Error("<Spinner>: Props cannot be read directly from the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	set pending(value) {
    		throw new Error("<Spinner>: Props cannot be set directly on the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	get done() {
    		throw new Error("<Spinner>: Props cannot be read directly from the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	set done(value) {
    		throw new Error("<Spinner>: Props cannot be set directly on the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	get textColor() {
    		throw new Error("<Spinner>: Props cannot be read directly from the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	set textColor(value) {
    		throw new Error("<Spinner>: Props cannot be set directly on the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}
    }

    /* node_modules\@weeblr\svelte\components\forms\Required.svelte generated by Svelte v3.41.0 */
    const file$3 = "node_modules\\@weeblr\\svelte\\components\\forms\\Required.svelte";

    // (10:0) {#if required}
    function create_if_block$3(ctx) {
    	let span;

    	const block = {
    		c: function create() {
    			span = element("span");
    			span.textContent = `(${t('base.required')})`;
    			attr_dev(span, "class", "wb-inline-block wb-text-sm wb-ml-4 wb-text-red-500");
    			add_location(span, file$3, 10, 4, 152);
    		},
    		m: function mount(target, anchor) {
    			insert_dev(target, span, anchor);
    		},
    		p: noop$1,
    		d: function destroy(detaching) {
    			if (detaching) detach_dev(span);
    		}
    	};

    	dispatch_dev("SvelteRegisterBlock", {
    		block,
    		id: create_if_block$3.name,
    		type: "if",
    		source: "(10:0) {#if required}",
    		ctx
    	});

    	return block;
    }

    function create_fragment$3(ctx) {
    	let if_block_anchor;
    	let if_block = /*required*/ ctx[0] && create_if_block$3(ctx);

    	const block = {
    		c: function create() {
    			if (if_block) if_block.c();
    			if_block_anchor = empty();
    		},
    		l: function claim(nodes) {
    			throw new Error("options.hydrate only works if the component was compiled with the `hydratable: true` option");
    		},
    		m: function mount(target, anchor) {
    			if (if_block) if_block.m(target, anchor);
    			insert_dev(target, if_block_anchor, anchor);
    		},
    		p: function update(ctx, [dirty]) {
    			if (/*required*/ ctx[0]) {
    				if (if_block) {
    					if_block.p(ctx, dirty);
    				} else {
    					if_block = create_if_block$3(ctx);
    					if_block.c();
    					if_block.m(if_block_anchor.parentNode, if_block_anchor);
    				}
    			} else if (if_block) {
    				if_block.d(1);
    				if_block = null;
    			}
    		},
    		i: noop$1,
    		o: noop$1,
    		d: function destroy(detaching) {
    			if (if_block) if_block.d(detaching);
    			if (detaching) detach_dev(if_block_anchor);
    		}
    	};

    	dispatch_dev("SvelteRegisterBlock", {
    		block,
    		id: create_fragment$3.name,
    		type: "component",
    		source: "",
    		ctx
    	});

    	return block;
    }

    function instance$3($$self, $$props, $$invalidate) {
    	let { $$slots: slots = {}, $$scope } = $$props;
    	validate_slots('Required', slots, []);
    	let { required = false } = $$props;
    	const writable_props = ['required'];

    	Object.keys($$props).forEach(key => {
    		if (!~writable_props.indexOf(key) && key.slice(0, 2) !== '$$' && key !== 'slot') console.warn(`<Required> was created with unknown prop '${key}'`);
    	});

    	$$self.$$set = $$props => {
    		if ('required' in $$props) $$invalidate(0, required = $$props.required);
    	};

    	$$self.$capture_state = () => ({ required, t });

    	$$self.$inject_state = $$props => {
    		if ('required' in $$props) $$invalidate(0, required = $$props.required);
    	};

    	if ($$props && "$$inject" in $$props) {
    		$$self.$inject_state($$props.$$inject);
    	}

    	return [required];
    }

    class Required extends SvelteComponentDev {
    	constructor(options) {
    		super(options);
    		init(this, options, instance$3, create_fragment$3, safe_not_equal, { required: 0 });

    		dispatch_dev("SvelteRegisterComponent", {
    			component: this,
    			tagName: "Required",
    			options,
    			id: create_fragment$3.name
    		});
    	}

    	get required() {
    		throw new Error("<Required>: Props cannot be read directly from the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	set required(value) {
    		throw new Error("<Required>: Props cannot be set directly on the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}
    }

    /* node_modules\@weeblr\svelte\components\forms\Input.svelte generated by Svelte v3.41.0 */
    const file$2 = "node_modules\\@weeblr\\svelte\\components\\forms\\Input.svelte";
    const get_after_slot_changes = dirty => ({});
    const get_after_slot_context = ctx => ({});
    const get_afterInput_slot_changes = dirty => ({});
    const get_afterInput_slot_context = ctx => ({});
    const get_oppositeInput_slot_changes = dirty => ({});
    const get_oppositeInput_slot_context = ctx => ({});
    const get_afterLabel_slot_changes = dirty => ({});
    const get_afterLabel_slot_context = ctx => ({});
    const get_oppositeLabel_slot_changes = dirty => ({});
    const get_oppositeLabel_slot_context = ctx => ({});
    const get_before_slot_changes = dirty => ({});
    const get_before_slot_context = ctx => ({});

    // (74:0) {#if label}
    function create_if_block_3(ctx) {
    	let span;
    	let label_1;
    	let t0;
    	let t1;
    	let required_1;
    	let label_1_class_value;
    	let t2;
    	let current;

    	required_1 = new Required({
    			props: { required: /*required*/ ctx[12] },
    			$$inline: true
    		});

    	const oppositeLabel_slot_template = /*#slots*/ ctx[34].oppositeLabel;
    	const oppositeLabel_slot = create_slot(oppositeLabel_slot_template, ctx, /*$$scope*/ ctx[33], get_oppositeLabel_slot_context);

    	const block = {
    		c: function create() {
    			span = element("span");
    			label_1 = element("label");
    			t0 = text(/*label*/ ctx[4]);
    			t1 = space();
    			create_component(required_1.$$.fragment);
    			t2 = space();
    			if (oppositeLabel_slot) oppositeLabel_slot.c();
    			attr_dev(label_1, "for", /*id*/ ctx[0]);

    			attr_dev(label_1, "class", label_1_class_value = "wb-text-base wb-font-semibold wb-tracking-wide " + (/*errors*/ ctx[27]
    			? 'wb-text-red-500'
    			: 'wb-text-gray-700') + " wb-bg-transparent");

    			add_location(label_1, file$2, 75, 8, 2099);
    			attr_dev(span, "class", "wb-flex wb-justify-between wb-items-center wb-max-w-full wb-mb-4");
    			add_location(span, file$2, 74, 4, 2011);
    		},
    		m: function mount(target, anchor) {
    			insert_dev(target, span, anchor);
    			append_dev(span, label_1);
    			append_dev(label_1, t0);
    			append_dev(label_1, t1);
    			mount_component(required_1, label_1, null);
    			append_dev(span, t2);

    			if (oppositeLabel_slot) {
    				oppositeLabel_slot.m(span, null);
    			}

    			current = true;
    		},
    		p: function update(ctx, dirty) {
    			if (!current || dirty[0] & /*label*/ 16) set_data_dev(t0, /*label*/ ctx[4]);
    			const required_1_changes = {};
    			if (dirty[0] & /*required*/ 4096) required_1_changes.required = /*required*/ ctx[12];
    			required_1.$set(required_1_changes);

    			if (!current || dirty[0] & /*id*/ 1) {
    				attr_dev(label_1, "for", /*id*/ ctx[0]);
    			}

    			if (!current || dirty[0] & /*errors*/ 134217728 && label_1_class_value !== (label_1_class_value = "wb-text-base wb-font-semibold wb-tracking-wide " + (/*errors*/ ctx[27]
    			? 'wb-text-red-500'
    			: 'wb-text-gray-700') + " wb-bg-transparent")) {
    				attr_dev(label_1, "class", label_1_class_value);
    			}

    			if (oppositeLabel_slot) {
    				if (oppositeLabel_slot.p && (!current || dirty[1] & /*$$scope*/ 4)) {
    					update_slot_base(
    						oppositeLabel_slot,
    						oppositeLabel_slot_template,
    						ctx,
    						/*$$scope*/ ctx[33],
    						!current
    						? get_all_dirty_from_scope(/*$$scope*/ ctx[33])
    						: get_slot_changes(oppositeLabel_slot_template, /*$$scope*/ ctx[33], dirty, get_oppositeLabel_slot_changes),
    						get_oppositeLabel_slot_context
    					);
    				}
    			}
    		},
    		i: function intro(local) {
    			if (current) return;
    			transition_in(required_1.$$.fragment, local);
    			transition_in(oppositeLabel_slot, local);
    			current = true;
    		},
    		o: function outro(local) {
    			transition_out(required_1.$$.fragment, local);
    			transition_out(oppositeLabel_slot, local);
    			current = false;
    		},
    		d: function destroy(detaching) {
    			if (detaching) detach_dev(span);
    			destroy_component(required_1);
    			if (oppositeLabel_slot) oppositeLabel_slot.d(detaching);
    		}
    	};

    	dispatch_dev("SvelteRegisterBlock", {
    		block,
    		id: create_if_block_3.name,
    		type: "if",
    		source: "(74:0) {#if label}",
    		ctx
    	});

    	return block;
    }

    // (109:8) {#if withClearIcon}
    function create_if_block_2$2(ctx) {
    	let div;
    	let icon;
    	let current;
    	let mounted;
    	let dispose;

    	icon = new Icon({
    			props: {
    				icon: iconClose,
    				size: "l",
    				tabbable: true,
    				listenToClicks: "true"
    			},
    			$$inline: true
    		});

    	const block = {
    		c: function create() {
    			div = element("div");
    			create_component(icon.$$.fragment);
    			attr_dev(div, "class", "wb-absolute wb-inset-y-0 wb-right-0 wb-flex-shrink-0 wb-flex wb-items-center sm:wb-px-2 wb-pl-2 wb-cursor-pointer");
    			attr_dev(div, "aria-hidden", "false");
    			add_location(div, file$2, 109, 8, 3906);
    		},
    		m: function mount(target, anchor) {
    			insert_dev(target, div, anchor);
    			mount_component(icon, div, null);
    			current = true;

    			if (!mounted) {
    				dispose = listen_dev(div, "click", stop_propagation(prevent_default(/*click_handler*/ ctx[35])), false, true, true);
    				mounted = true;
    			}
    		},
    		p: noop$1,
    		i: function intro(local) {
    			if (current) return;
    			transition_in(icon.$$.fragment, local);
    			current = true;
    		},
    		o: function outro(local) {
    			transition_out(icon.$$.fragment, local);
    			current = false;
    		},
    		d: function destroy(detaching) {
    			if (detaching) detach_dev(div);
    			destroy_component(icon);
    			mounted = false;
    			dispose();
    		}
    	};

    	dispatch_dev("SvelteRegisterBlock", {
    		block,
    		id: create_if_block_2$2.name,
    		type: "if",
    		source: "(109:8) {#if withClearIcon}",
    		ctx
    	});

    	return block;
    }

    // (126:24) 
    function create_if_block_1$2(ctx) {
    	let div;
    	let icon;
    	let current;

    	icon = new Icon({
    			props: {
    				icon: /*withIconAfter*/ ctx[23],
    				size: "l"
    			},
    			$$inline: true
    		});

    	const block = {
    		c: function create() {
    			div = element("div");
    			create_component(icon.$$.fragment);
    			attr_dev(div, "class", "sm:wb-absolute wb-inset-y-0 wb-right-0 wb-flex-shrink-0 wb-flex wb-items-center wb-px-2 wb-cursor-pointer");
    			add_location(div, file$2, 126, 4, 4771);
    		},
    		m: function mount(target, anchor) {
    			insert_dev(target, div, anchor);
    			mount_component(icon, div, null);
    			current = true;
    		},
    		p: function update(ctx, dirty) {
    			const icon_changes = {};
    			if (dirty[0] & /*withIconAfter*/ 8388608) icon_changes.icon = /*withIconAfter*/ ctx[23];
    			icon.$set(icon_changes);
    		},
    		i: function intro(local) {
    			if (current) return;
    			transition_in(icon.$$.fragment, local);
    			current = true;
    		},
    		o: function outro(local) {
    			transition_out(icon.$$.fragment, local);
    			current = false;
    		},
    		d: function destroy(detaching) {
    			if (detaching) detach_dev(div);
    			destroy_component(icon);
    		}
    	};

    	dispatch_dev("SvelteRegisterBlock", {
    		block,
    		id: create_if_block_1$2.name,
    		type: "if",
    		source: "(126:24) ",
    		ctx
    	});

    	return block;
    }

    // (120:0) {#if withIconAfter && onIconClick}
    function create_if_block$2(ctx) {
    	let div;
    	let icon;
    	let current;
    	let mounted;
    	let dispose;

    	icon = new Icon({
    			props: {
    				icon: /*withIconAfter*/ ctx[23],
    				size: "l",
    				tabbable: true,
    				listenToClicks: "true"
    			},
    			$$inline: true
    		});

    	const block = {
    		c: function create() {
    			div = element("div");
    			create_component(icon.$$.fragment);
    			attr_dev(div, "class", "sm:wb-absolute wb-inset-y-0 wb-right-0 wb-flex-shrink-0 wb-flex wb-items-center wb-px-2 wb-cursor-pointer");
    			attr_dev(div, "aria-hidden", "false");
    			add_location(div, file$2, 120, 4, 4410);
    		},
    		m: function mount(target, anchor) {
    			insert_dev(target, div, anchor);
    			mount_component(icon, div, null);
    			current = true;

    			if (!mounted) {
    				dispose = listen_dev(div, "click", stop_propagation(prevent_default(/*click_handler_1*/ ctx[36])), false, true, true);
    				mounted = true;
    			}
    		},
    		p: function update(ctx, dirty) {
    			const icon_changes = {};
    			if (dirty[0] & /*withIconAfter*/ 8388608) icon_changes.icon = /*withIconAfter*/ ctx[23];
    			icon.$set(icon_changes);
    		},
    		i: function intro(local) {
    			if (current) return;
    			transition_in(icon.$$.fragment, local);
    			current = true;
    		},
    		o: function outro(local) {
    			transition_out(icon.$$.fragment, local);
    			current = false;
    		},
    		d: function destroy(detaching) {
    			if (detaching) detach_dev(div);
    			destroy_component(icon);
    			mounted = false;
    			dispose();
    		}
    	};

    	dispatch_dev("SvelteRegisterBlock", {
    		block,
    		id: create_if_block$2.name,
    		type: "if",
    		source: "(120:0) {#if withIconAfter && onIconClick}",
    		ctx
    	});

    	return block;
    }

    function create_fragment$2(ctx) {
    	let t0;
    	let t1;
    	let t2;
    	let span;
    	let div;
    	let input;
    	let input_class_value;
    	let input_title_value;
    	let input_aria_invalid_value;
    	let input_aria_label_value;
    	let input_aria_describedby_value;
    	let input_aria_owns_value;
    	let input_aria_autocomplete_value;
    	let input_aria_expanded_value;
    	let input_aria_haspopup_value;
    	let t3;
    	let t4;
    	let t5;
    	let t6;
    	let current_block_type_index;
    	let if_block2;
    	let t7;
    	let current;
    	let mounted;
    	let dispose;
    	const before_slot_template = /*#slots*/ ctx[34].before;
    	const before_slot = create_slot(before_slot_template, ctx, /*$$scope*/ ctx[33], get_before_slot_context);
    	let if_block0 = /*label*/ ctx[4] && create_if_block_3(ctx);
    	const afterLabel_slot_template = /*#slots*/ ctx[34].afterLabel;
    	const afterLabel_slot = create_slot(afterLabel_slot_template, ctx, /*$$scope*/ ctx[33], get_afterLabel_slot_context);
    	let if_block1 = /*withClearIcon*/ ctx[22] && create_if_block_2$2(ctx);
    	const oppositeInput_slot_template = /*#slots*/ ctx[34].oppositeInput;
    	const oppositeInput_slot = create_slot(oppositeInput_slot_template, ctx, /*$$scope*/ ctx[33], get_oppositeInput_slot_context);
    	const afterInput_slot_template = /*#slots*/ ctx[34].afterInput;
    	const afterInput_slot = create_slot(afterInput_slot_template, ctx, /*$$scope*/ ctx[33], get_afterInput_slot_context);
    	const if_block_creators = [create_if_block$2, create_if_block_1$2];
    	const if_blocks = [];

    	function select_block_type(ctx, dirty) {
    		if (/*withIconAfter*/ ctx[23] && /*onIconClick*/ ctx[24]) return 0;
    		if (/*withIconAfter*/ ctx[23]) return 1;
    		return -1;
    	}

    	if (~(current_block_type_index = select_block_type(ctx))) {
    		if_block2 = if_blocks[current_block_type_index] = if_block_creators[current_block_type_index](ctx);
    	}

    	const after_slot_template = /*#slots*/ ctx[34].after;
    	const after_slot = create_slot(after_slot_template, ctx, /*$$scope*/ ctx[33], get_after_slot_context);

    	const block = {
    		c: function create() {
    			if (before_slot) before_slot.c();
    			t0 = space();
    			if (if_block0) if_block0.c();
    			t1 = space();
    			if (afterLabel_slot) afterLabel_slot.c();
    			t2 = space();
    			span = element("span");
    			div = element("div");
    			input = element("input");
    			t3 = space();
    			if (if_block1) if_block1.c();
    			t4 = space();
    			if (oppositeInput_slot) oppositeInput_slot.c();
    			t5 = space();
    			if (afterInput_slot) afterInput_slot.c();
    			t6 = space();
    			if (if_block2) if_block2.c();
    			t7 = space();
    			if (after_slot) after_slot.c();
    			attr_dev(input, "id", /*id*/ ctx[0]);
    			attr_dev(input, "autocomplete", /*autocomplete*/ ctx[15]);
    			attr_dev(input, "spellcheck", /*spellcheck*/ ctx[16]);
    			attr_dev(input, "autocapitalize", /*autocapitalize*/ ctx[17]);
    			input.value = /*value*/ ctx[1];

    			attr_dev(input, "class", input_class_value = "wb-appearance-none wb-relative wb-block " + /*widthClass*/ ctx[28] + " wb-h-auto wb-max-w-full wb-bg-gray-200 wb-text-gray-900 wb-border wb-border-gray-200 wb-rounded wb-py-2 wb-pl-" + /*px*/ ctx[14] + " " + (/*withIconAfter*/ ctx[23] || /*withClearIcon*/ ctx[22]
    			? 'wb-pr-8'
    			: 'wb-pr-' + /*px*/ ctx[14]) + " wb-m-0 wb-leading-normal " + (/*errors*/ ctx[27]
    			? 'wb-c-focusable-error'
    			: 'wb-c-focusable') + " " + (/*disabled*/ ctx[11] ? 'wb-opacity-50' : '') + " " + (/*withShadow*/ ctx[25] ? 'wb-shadow-md' : '') + " " + /*classes*/ ctx[26]);

    			attr_dev(input, "type", /*type*/ ctx[2]);
    			attr_dev(input, "title", input_title_value = /*title*/ ctx[3] || /*label*/ ctx[4] || /*ariaLabel*/ ctx[5] || null);
    			attr_dev(input, "aria-invalid", input_aria_invalid_value = /*errors*/ ctx[27] ? 'true' : 'false');
    			attr_dev(input, "aria-label", input_aria_label_value = /*ariaLabel*/ ctx[5] || /*label*/ ctx[4] || /*title*/ ctx[3] || null);
    			attr_dev(input, "aria-describedby", input_aria_describedby_value = /*ariaDescribedBy*/ ctx[6] || null);
    			attr_dev(input, "aria-owns", input_aria_owns_value = /*ariaOwns*/ ctx[7] || null);
    			attr_dev(input, "aria-autocomplete", input_aria_autocomplete_value = /*ariaAutocomplete*/ ctx[8] || null);

    			attr_dev(input, "aria-expanded", input_aria_expanded_value = typeof /*ariaExpanded*/ ctx[9] == 'undefined'
    			? null
    			: /*ariaExpanded*/ ctx[9]);

    			attr_dev(input, "aria-haspopup", input_aria_haspopup_value = typeof /*ariaHasPopup*/ ctx[10] == 'undefined'
    			? null
    			: /*ariaHasPopup*/ ctx[10]);

    			input.disabled = /*disabled*/ ctx[11];
    			attr_dev(input, "data-focusdefault", /*focusDefault*/ ctx[13]);
    			add_location(input, file$2, 86, 8, 2534);
    			attr_dev(div, "class", "wb-relative wb-max-w-full");
    			add_location(div, file$2, 85, 4, 2486);
    			attr_dev(span, "class", "wb-flex wb-items-center wb-max-w-full wb-flex-wrap wb-gap-y-4");
    			add_location(span, file$2, 84, 0, 2405);
    		},
    		l: function claim(nodes) {
    			throw new Error("options.hydrate only works if the component was compiled with the `hydratable: true` option");
    		},
    		m: function mount(target, anchor) {
    			if (before_slot) {
    				before_slot.m(target, anchor);
    			}

    			insert_dev(target, t0, anchor);
    			if (if_block0) if_block0.m(target, anchor);
    			insert_dev(target, t1, anchor);

    			if (afterLabel_slot) {
    				afterLabel_slot.m(target, anchor);
    			}

    			insert_dev(target, t2, anchor);
    			insert_dev(target, span, anchor);
    			append_dev(span, div);
    			append_dev(div, input);
    			append_dev(div, t3);
    			if (if_block1) if_block1.m(div, null);
    			append_dev(span, t4);

    			if (oppositeInput_slot) {
    				oppositeInput_slot.m(span, null);
    			}

    			insert_dev(target, t5, anchor);

    			if (afterInput_slot) {
    				afterInput_slot.m(target, anchor);
    			}

    			insert_dev(target, t6, anchor);

    			if (~current_block_type_index) {
    				if_blocks[current_block_type_index].m(target, anchor);
    			}

    			insert_dev(target, t7, anchor);

    			if (after_slot) {
    				after_slot.m(target, anchor);
    			}

    			current = true;

    			if (!mounted) {
    				dispose = [
    					listen_dev(
    						input,
    						"focus",
    						function () {
    							if (is_function(/*onInputFocus*/ ctx[18])) /*onInputFocus*/ ctx[18].apply(this, arguments);
    						},
    						false,
    						false,
    						false
    					),
    					listen_dev(
    						input,
    						"blur",
    						function () {
    							if (is_function(/*onInputBlur*/ ctx[19])) /*onInputBlur*/ ctx[19].apply(this, arguments);
    						},
    						false,
    						false,
    						false
    					),
    					listen_dev(
    						input,
    						"click",
    						function () {
    							if (is_function(/*onInputClick*/ ctx[20])) /*onInputClick*/ ctx[20].apply(this, arguments);
    						},
    						false,
    						false,
    						false
    					),
    					listen_dev(
    						input,
    						"keydown",
    						function () {
    							if (is_function(/*onInputKeydown*/ ctx[21])) /*onInputKeydown*/ ctx[21].apply(this, arguments);
    						},
    						false,
    						false,
    						false
    					),
    					listen_dev(input, "keyup", /*onInputKeyup*/ ctx[29], false, false, false)
    				];

    				mounted = true;
    			}
    		},
    		p: function update(new_ctx, dirty) {
    			ctx = new_ctx;

    			if (before_slot) {
    				if (before_slot.p && (!current || dirty[1] & /*$$scope*/ 4)) {
    					update_slot_base(
    						before_slot,
    						before_slot_template,
    						ctx,
    						/*$$scope*/ ctx[33],
    						!current
    						? get_all_dirty_from_scope(/*$$scope*/ ctx[33])
    						: get_slot_changes(before_slot_template, /*$$scope*/ ctx[33], dirty, get_before_slot_changes),
    						get_before_slot_context
    					);
    				}
    			}

    			if (/*label*/ ctx[4]) {
    				if (if_block0) {
    					if_block0.p(ctx, dirty);

    					if (dirty[0] & /*label*/ 16) {
    						transition_in(if_block0, 1);
    					}
    				} else {
    					if_block0 = create_if_block_3(ctx);
    					if_block0.c();
    					transition_in(if_block0, 1);
    					if_block0.m(t1.parentNode, t1);
    				}
    			} else if (if_block0) {
    				group_outros();

    				transition_out(if_block0, 1, 1, () => {
    					if_block0 = null;
    				});

    				check_outros();
    			}

    			if (afterLabel_slot) {
    				if (afterLabel_slot.p && (!current || dirty[1] & /*$$scope*/ 4)) {
    					update_slot_base(
    						afterLabel_slot,
    						afterLabel_slot_template,
    						ctx,
    						/*$$scope*/ ctx[33],
    						!current
    						? get_all_dirty_from_scope(/*$$scope*/ ctx[33])
    						: get_slot_changes(afterLabel_slot_template, /*$$scope*/ ctx[33], dirty, get_afterLabel_slot_changes),
    						get_afterLabel_slot_context
    					);
    				}
    			}

    			if (!current || dirty[0] & /*id*/ 1) {
    				attr_dev(input, "id", /*id*/ ctx[0]);
    			}

    			if (!current || dirty[0] & /*autocomplete*/ 32768) {
    				attr_dev(input, "autocomplete", /*autocomplete*/ ctx[15]);
    			}

    			if (!current || dirty[0] & /*spellcheck*/ 65536) {
    				attr_dev(input, "spellcheck", /*spellcheck*/ ctx[16]);
    			}

    			if (!current || dirty[0] & /*autocapitalize*/ 131072) {
    				attr_dev(input, "autocapitalize", /*autocapitalize*/ ctx[17]);
    			}

    			if (!current || dirty[0] & /*value*/ 2 && input.value !== /*value*/ ctx[1]) {
    				prop_dev(input, "value", /*value*/ ctx[1]);
    			}

    			if (!current || dirty[0] & /*px, withIconAfter, withClearIcon, errors, disabled, withShadow, classes*/ 247482368 && input_class_value !== (input_class_value = "wb-appearance-none wb-relative wb-block " + /*widthClass*/ ctx[28] + " wb-h-auto wb-max-w-full wb-bg-gray-200 wb-text-gray-900 wb-border wb-border-gray-200 wb-rounded wb-py-2 wb-pl-" + /*px*/ ctx[14] + " " + (/*withIconAfter*/ ctx[23] || /*withClearIcon*/ ctx[22]
    			? 'wb-pr-8'
    			: 'wb-pr-' + /*px*/ ctx[14]) + " wb-m-0 wb-leading-normal " + (/*errors*/ ctx[27]
    			? 'wb-c-focusable-error'
    			: 'wb-c-focusable') + " " + (/*disabled*/ ctx[11] ? 'wb-opacity-50' : '') + " " + (/*withShadow*/ ctx[25] ? 'wb-shadow-md' : '') + " " + /*classes*/ ctx[26])) {
    				attr_dev(input, "class", input_class_value);
    			}

    			if (!current || dirty[0] & /*type*/ 4) {
    				attr_dev(input, "type", /*type*/ ctx[2]);
    			}

    			if (!current || dirty[0] & /*title, label, ariaLabel*/ 56 && input_title_value !== (input_title_value = /*title*/ ctx[3] || /*label*/ ctx[4] || /*ariaLabel*/ ctx[5] || null)) {
    				attr_dev(input, "title", input_title_value);
    			}

    			if (!current || dirty[0] & /*errors*/ 134217728 && input_aria_invalid_value !== (input_aria_invalid_value = /*errors*/ ctx[27] ? 'true' : 'false')) {
    				attr_dev(input, "aria-invalid", input_aria_invalid_value);
    			}

    			if (!current || dirty[0] & /*ariaLabel, label, title*/ 56 && input_aria_label_value !== (input_aria_label_value = /*ariaLabel*/ ctx[5] || /*label*/ ctx[4] || /*title*/ ctx[3] || null)) {
    				attr_dev(input, "aria-label", input_aria_label_value);
    			}

    			if (!current || dirty[0] & /*ariaDescribedBy*/ 64 && input_aria_describedby_value !== (input_aria_describedby_value = /*ariaDescribedBy*/ ctx[6] || null)) {
    				attr_dev(input, "aria-describedby", input_aria_describedby_value);
    			}

    			if (!current || dirty[0] & /*ariaOwns*/ 128 && input_aria_owns_value !== (input_aria_owns_value = /*ariaOwns*/ ctx[7] || null)) {
    				attr_dev(input, "aria-owns", input_aria_owns_value);
    			}

    			if (!current || dirty[0] & /*ariaAutocomplete*/ 256 && input_aria_autocomplete_value !== (input_aria_autocomplete_value = /*ariaAutocomplete*/ ctx[8] || null)) {
    				attr_dev(input, "aria-autocomplete", input_aria_autocomplete_value);
    			}

    			if (!current || dirty[0] & /*ariaExpanded*/ 512 && input_aria_expanded_value !== (input_aria_expanded_value = typeof /*ariaExpanded*/ ctx[9] == 'undefined'
    			? null
    			: /*ariaExpanded*/ ctx[9])) {
    				attr_dev(input, "aria-expanded", input_aria_expanded_value);
    			}

    			if (!current || dirty[0] & /*ariaHasPopup*/ 1024 && input_aria_haspopup_value !== (input_aria_haspopup_value = typeof /*ariaHasPopup*/ ctx[10] == 'undefined'
    			? null
    			: /*ariaHasPopup*/ ctx[10])) {
    				attr_dev(input, "aria-haspopup", input_aria_haspopup_value);
    			}

    			if (!current || dirty[0] & /*disabled*/ 2048) {
    				prop_dev(input, "disabled", /*disabled*/ ctx[11]);
    			}

    			if (!current || dirty[0] & /*focusDefault*/ 8192) {
    				attr_dev(input, "data-focusdefault", /*focusDefault*/ ctx[13]);
    			}

    			if (/*withClearIcon*/ ctx[22]) {
    				if (if_block1) {
    					if_block1.p(ctx, dirty);

    					if (dirty[0] & /*withClearIcon*/ 4194304) {
    						transition_in(if_block1, 1);
    					}
    				} else {
    					if_block1 = create_if_block_2$2(ctx);
    					if_block1.c();
    					transition_in(if_block1, 1);
    					if_block1.m(div, null);
    				}
    			} else if (if_block1) {
    				group_outros();

    				transition_out(if_block1, 1, 1, () => {
    					if_block1 = null;
    				});

    				check_outros();
    			}

    			if (oppositeInput_slot) {
    				if (oppositeInput_slot.p && (!current || dirty[1] & /*$$scope*/ 4)) {
    					update_slot_base(
    						oppositeInput_slot,
    						oppositeInput_slot_template,
    						ctx,
    						/*$$scope*/ ctx[33],
    						!current
    						? get_all_dirty_from_scope(/*$$scope*/ ctx[33])
    						: get_slot_changes(oppositeInput_slot_template, /*$$scope*/ ctx[33], dirty, get_oppositeInput_slot_changes),
    						get_oppositeInput_slot_context
    					);
    				}
    			}

    			if (afterInput_slot) {
    				if (afterInput_slot.p && (!current || dirty[1] & /*$$scope*/ 4)) {
    					update_slot_base(
    						afterInput_slot,
    						afterInput_slot_template,
    						ctx,
    						/*$$scope*/ ctx[33],
    						!current
    						? get_all_dirty_from_scope(/*$$scope*/ ctx[33])
    						: get_slot_changes(afterInput_slot_template, /*$$scope*/ ctx[33], dirty, get_afterInput_slot_changes),
    						get_afterInput_slot_context
    					);
    				}
    			}

    			let previous_block_index = current_block_type_index;
    			current_block_type_index = select_block_type(ctx);

    			if (current_block_type_index === previous_block_index) {
    				if (~current_block_type_index) {
    					if_blocks[current_block_type_index].p(ctx, dirty);
    				}
    			} else {
    				if (if_block2) {
    					group_outros();

    					transition_out(if_blocks[previous_block_index], 1, 1, () => {
    						if_blocks[previous_block_index] = null;
    					});

    					check_outros();
    				}

    				if (~current_block_type_index) {
    					if_block2 = if_blocks[current_block_type_index];

    					if (!if_block2) {
    						if_block2 = if_blocks[current_block_type_index] = if_block_creators[current_block_type_index](ctx);
    						if_block2.c();
    					} else {
    						if_block2.p(ctx, dirty);
    					}

    					transition_in(if_block2, 1);
    					if_block2.m(t7.parentNode, t7);
    				} else {
    					if_block2 = null;
    				}
    			}

    			if (after_slot) {
    				if (after_slot.p && (!current || dirty[1] & /*$$scope*/ 4)) {
    					update_slot_base(
    						after_slot,
    						after_slot_template,
    						ctx,
    						/*$$scope*/ ctx[33],
    						!current
    						? get_all_dirty_from_scope(/*$$scope*/ ctx[33])
    						: get_slot_changes(after_slot_template, /*$$scope*/ ctx[33], dirty, get_after_slot_changes),
    						get_after_slot_context
    					);
    				}
    			}
    		},
    		i: function intro(local) {
    			if (current) return;
    			transition_in(before_slot, local);
    			transition_in(if_block0);
    			transition_in(afterLabel_slot, local);
    			transition_in(if_block1);
    			transition_in(oppositeInput_slot, local);
    			transition_in(afterInput_slot, local);
    			transition_in(if_block2);
    			transition_in(after_slot, local);
    			current = true;
    		},
    		o: function outro(local) {
    			transition_out(before_slot, local);
    			transition_out(if_block0);
    			transition_out(afterLabel_slot, local);
    			transition_out(if_block1);
    			transition_out(oppositeInput_slot, local);
    			transition_out(afterInput_slot, local);
    			transition_out(if_block2);
    			transition_out(after_slot, local);
    			current = false;
    		},
    		d: function destroy(detaching) {
    			if (before_slot) before_slot.d(detaching);
    			if (detaching) detach_dev(t0);
    			if (if_block0) if_block0.d(detaching);
    			if (detaching) detach_dev(t1);
    			if (afterLabel_slot) afterLabel_slot.d(detaching);
    			if (detaching) detach_dev(t2);
    			if (detaching) detach_dev(span);
    			if (if_block1) if_block1.d();
    			if (oppositeInput_slot) oppositeInput_slot.d(detaching);
    			if (detaching) detach_dev(t5);
    			if (afterInput_slot) afterInput_slot.d(detaching);
    			if (detaching) detach_dev(t6);

    			if (~current_block_type_index) {
    				if_blocks[current_block_type_index].d(detaching);
    			}

    			if (detaching) detach_dev(t7);
    			if (after_slot) after_slot.d(detaching);
    			mounted = false;
    			run_all(dispose);
    		}
    	};

    	dispatch_dev("SvelteRegisterBlock", {
    		block,
    		id: create_fragment$2.name,
    		type: "component",
    		source: "",
    		ctx
    	});

    	return block;
    }

    function instance$2($$self, $$props, $$invalidate) {
    	let { $$slots: slots = {}, $$scope } = $$props;
    	validate_slots('Input', slots, ['before','oppositeLabel','afterLabel','oppositeInput','afterInput','after']);
    	let { type = 'text' } = $$props;
    	let { id = '' } = $$props;
    	let { title = '' } = $$props;
    	let { label = '' } = $$props;
    	let { ariaLabel = '' } = $$props;
    	let { ariaDescribedBy = '' } = $$props;
    	let { ariaOwns = '' } = $$props;
    	let { ariaAutocomplete = '' } = $$props;
    	let { ariaExpanded = undefined } = $$props;
    	let { ariaHasPopup = undefined } = $$props;
    	let { value } = $$props;
    	let { disabled = false } = $$props;
    	let { required = false } = $$props;
    	let { focusDefault = false } = $$props;
    	let { size = 'sm' } = $$props;
    	let { px = 4 } = $$props;
    	let { autocomplete = 'off' } = $$props;
    	let { spellcheck = "false" } = $$props;
    	let { autocapitalize = "off" } = $$props;
    	let { onInputFocus = noop } = $$props;
    	let { onInputBlur = noop } = $$props;
    	let { onInputClick = noop } = $$props;
    	let { onInputKeydown = noop } = $$props;
    	let { onInputChange = noop } = $$props;
    	let { withClearIcon = true } = $$props;
    	let { withIconAfter = undefined } = $$props;
    	let { onIconClick = undefined } = $$props;
    	let { withShadow = getContext('features').isTruthy('ui.withShadow', false) } = $$props;
    	let { debounce: debounce$1 = 0 } = $$props;
    	let { classes = '' } = $$props;
    	let { errors = undefined } = $$props;
    	id = id || v4();
    	let widthClass = 'wb-w-' + size;

    	const updateValue = newValue => {
    		$$invalidate(1, value = newValue);
    		onInputChange(value);
    	};

    	const onInputKeyup = event => {
    		if (value == event.target.value) {
    			return;
    		}

    		if (debounce$1) {
    			debounce(
    				() => {
    					updateValue(event.target.value);
    				},
    				debounce$1
    			)();
    		} else {
    			updateValue(event.target.value);
    		}
    	};

    	const writable_props = [
    		'type',
    		'id',
    		'title',
    		'label',
    		'ariaLabel',
    		'ariaDescribedBy',
    		'ariaOwns',
    		'ariaAutocomplete',
    		'ariaExpanded',
    		'ariaHasPopup',
    		'value',
    		'disabled',
    		'required',
    		'focusDefault',
    		'size',
    		'px',
    		'autocomplete',
    		'spellcheck',
    		'autocapitalize',
    		'onInputFocus',
    		'onInputBlur',
    		'onInputClick',
    		'onInputKeydown',
    		'onInputChange',
    		'withClearIcon',
    		'withIconAfter',
    		'onIconClick',
    		'withShadow',
    		'debounce',
    		'classes',
    		'errors'
    	];

    	Object.keys($$props).forEach(key => {
    		if (!~writable_props.indexOf(key) && key.slice(0, 2) !== '$$' && key !== 'slot') console.warn(`<Input> was created with unknown prop '${key}'`);
    	});

    	const click_handler = () => {
    		$$invalidate(1, value = '');
    		document.getElementById(id).focus();
    	};

    	const click_handler_1 = e => {
    		onIconClick(e, 'after');
    	};

    	$$self.$$set = $$props => {
    		if ('type' in $$props) $$invalidate(2, type = $$props.type);
    		if ('id' in $$props) $$invalidate(0, id = $$props.id);
    		if ('title' in $$props) $$invalidate(3, title = $$props.title);
    		if ('label' in $$props) $$invalidate(4, label = $$props.label);
    		if ('ariaLabel' in $$props) $$invalidate(5, ariaLabel = $$props.ariaLabel);
    		if ('ariaDescribedBy' in $$props) $$invalidate(6, ariaDescribedBy = $$props.ariaDescribedBy);
    		if ('ariaOwns' in $$props) $$invalidate(7, ariaOwns = $$props.ariaOwns);
    		if ('ariaAutocomplete' in $$props) $$invalidate(8, ariaAutocomplete = $$props.ariaAutocomplete);
    		if ('ariaExpanded' in $$props) $$invalidate(9, ariaExpanded = $$props.ariaExpanded);
    		if ('ariaHasPopup' in $$props) $$invalidate(10, ariaHasPopup = $$props.ariaHasPopup);
    		if ('value' in $$props) $$invalidate(1, value = $$props.value);
    		if ('disabled' in $$props) $$invalidate(11, disabled = $$props.disabled);
    		if ('required' in $$props) $$invalidate(12, required = $$props.required);
    		if ('focusDefault' in $$props) $$invalidate(13, focusDefault = $$props.focusDefault);
    		if ('size' in $$props) $$invalidate(30, size = $$props.size);
    		if ('px' in $$props) $$invalidate(14, px = $$props.px);
    		if ('autocomplete' in $$props) $$invalidate(15, autocomplete = $$props.autocomplete);
    		if ('spellcheck' in $$props) $$invalidate(16, spellcheck = $$props.spellcheck);
    		if ('autocapitalize' in $$props) $$invalidate(17, autocapitalize = $$props.autocapitalize);
    		if ('onInputFocus' in $$props) $$invalidate(18, onInputFocus = $$props.onInputFocus);
    		if ('onInputBlur' in $$props) $$invalidate(19, onInputBlur = $$props.onInputBlur);
    		if ('onInputClick' in $$props) $$invalidate(20, onInputClick = $$props.onInputClick);
    		if ('onInputKeydown' in $$props) $$invalidate(21, onInputKeydown = $$props.onInputKeydown);
    		if ('onInputChange' in $$props) $$invalidate(31, onInputChange = $$props.onInputChange);
    		if ('withClearIcon' in $$props) $$invalidate(22, withClearIcon = $$props.withClearIcon);
    		if ('withIconAfter' in $$props) $$invalidate(23, withIconAfter = $$props.withIconAfter);
    		if ('onIconClick' in $$props) $$invalidate(24, onIconClick = $$props.onIconClick);
    		if ('withShadow' in $$props) $$invalidate(25, withShadow = $$props.withShadow);
    		if ('debounce' in $$props) $$invalidate(32, debounce$1 = $$props.debounce);
    		if ('classes' in $$props) $$invalidate(26, classes = $$props.classes);
    		if ('errors' in $$props) $$invalidate(27, errors = $$props.errors);
    		if ('$$scope' in $$props) $$invalidate(33, $$scope = $$props.$$scope);
    	};

    	$$self.$capture_state = () => ({
    		noop,
    		getContext,
    		type,
    		id,
    		title,
    		label,
    		ariaLabel,
    		ariaDescribedBy,
    		ariaOwns,
    		ariaAutocomplete,
    		ariaExpanded,
    		ariaHasPopup,
    		value,
    		disabled,
    		required,
    		focusDefault,
    		size,
    		px,
    		autocomplete,
    		spellcheck,
    		autocapitalize,
    		onInputFocus,
    		onInputBlur,
    		onInputClick,
    		onInputKeydown,
    		onInputChange,
    		withClearIcon,
    		withIconAfter,
    		onIconClick,
    		withShadow,
    		debounce: debounce$1,
    		classes,
    		errors,
    		debounceFunction: debounce,
    		Icon,
    		Required,
    		iconClose,
    		uuidv4: v4,
    		widthClass,
    		updateValue,
    		onInputKeyup
    	});

    	$$self.$inject_state = $$props => {
    		if ('type' in $$props) $$invalidate(2, type = $$props.type);
    		if ('id' in $$props) $$invalidate(0, id = $$props.id);
    		if ('title' in $$props) $$invalidate(3, title = $$props.title);
    		if ('label' in $$props) $$invalidate(4, label = $$props.label);
    		if ('ariaLabel' in $$props) $$invalidate(5, ariaLabel = $$props.ariaLabel);
    		if ('ariaDescribedBy' in $$props) $$invalidate(6, ariaDescribedBy = $$props.ariaDescribedBy);
    		if ('ariaOwns' in $$props) $$invalidate(7, ariaOwns = $$props.ariaOwns);
    		if ('ariaAutocomplete' in $$props) $$invalidate(8, ariaAutocomplete = $$props.ariaAutocomplete);
    		if ('ariaExpanded' in $$props) $$invalidate(9, ariaExpanded = $$props.ariaExpanded);
    		if ('ariaHasPopup' in $$props) $$invalidate(10, ariaHasPopup = $$props.ariaHasPopup);
    		if ('value' in $$props) $$invalidate(1, value = $$props.value);
    		if ('disabled' in $$props) $$invalidate(11, disabled = $$props.disabled);
    		if ('required' in $$props) $$invalidate(12, required = $$props.required);
    		if ('focusDefault' in $$props) $$invalidate(13, focusDefault = $$props.focusDefault);
    		if ('size' in $$props) $$invalidate(30, size = $$props.size);
    		if ('px' in $$props) $$invalidate(14, px = $$props.px);
    		if ('autocomplete' in $$props) $$invalidate(15, autocomplete = $$props.autocomplete);
    		if ('spellcheck' in $$props) $$invalidate(16, spellcheck = $$props.spellcheck);
    		if ('autocapitalize' in $$props) $$invalidate(17, autocapitalize = $$props.autocapitalize);
    		if ('onInputFocus' in $$props) $$invalidate(18, onInputFocus = $$props.onInputFocus);
    		if ('onInputBlur' in $$props) $$invalidate(19, onInputBlur = $$props.onInputBlur);
    		if ('onInputClick' in $$props) $$invalidate(20, onInputClick = $$props.onInputClick);
    		if ('onInputKeydown' in $$props) $$invalidate(21, onInputKeydown = $$props.onInputKeydown);
    		if ('onInputChange' in $$props) $$invalidate(31, onInputChange = $$props.onInputChange);
    		if ('withClearIcon' in $$props) $$invalidate(22, withClearIcon = $$props.withClearIcon);
    		if ('withIconAfter' in $$props) $$invalidate(23, withIconAfter = $$props.withIconAfter);
    		if ('onIconClick' in $$props) $$invalidate(24, onIconClick = $$props.onIconClick);
    		if ('withShadow' in $$props) $$invalidate(25, withShadow = $$props.withShadow);
    		if ('debounce' in $$props) $$invalidate(32, debounce$1 = $$props.debounce);
    		if ('classes' in $$props) $$invalidate(26, classes = $$props.classes);
    		if ('errors' in $$props) $$invalidate(27, errors = $$props.errors);
    		if ('widthClass' in $$props) $$invalidate(28, widthClass = $$props.widthClass);
    	};

    	if ($$props && "$$inject" in $$props) {
    		$$self.$inject_state($$props.$$inject);
    	}

    	return [
    		id,
    		value,
    		type,
    		title,
    		label,
    		ariaLabel,
    		ariaDescribedBy,
    		ariaOwns,
    		ariaAutocomplete,
    		ariaExpanded,
    		ariaHasPopup,
    		disabled,
    		required,
    		focusDefault,
    		px,
    		autocomplete,
    		spellcheck,
    		autocapitalize,
    		onInputFocus,
    		onInputBlur,
    		onInputClick,
    		onInputKeydown,
    		withClearIcon,
    		withIconAfter,
    		onIconClick,
    		withShadow,
    		classes,
    		errors,
    		widthClass,
    		onInputKeyup,
    		size,
    		onInputChange,
    		debounce$1,
    		$$scope,
    		slots,
    		click_handler,
    		click_handler_1
    	];
    }

    class Input extends SvelteComponentDev {
    	constructor(options) {
    		super(options);

    		init(
    			this,
    			options,
    			instance$2,
    			create_fragment$2,
    			safe_not_equal,
    			{
    				type: 2,
    				id: 0,
    				title: 3,
    				label: 4,
    				ariaLabel: 5,
    				ariaDescribedBy: 6,
    				ariaOwns: 7,
    				ariaAutocomplete: 8,
    				ariaExpanded: 9,
    				ariaHasPopup: 10,
    				value: 1,
    				disabled: 11,
    				required: 12,
    				focusDefault: 13,
    				size: 30,
    				px: 14,
    				autocomplete: 15,
    				spellcheck: 16,
    				autocapitalize: 17,
    				onInputFocus: 18,
    				onInputBlur: 19,
    				onInputClick: 20,
    				onInputKeydown: 21,
    				onInputChange: 31,
    				withClearIcon: 22,
    				withIconAfter: 23,
    				onIconClick: 24,
    				withShadow: 25,
    				debounce: 32,
    				classes: 26,
    				errors: 27
    			},
    			null,
    			[-1, -1]
    		);

    		dispatch_dev("SvelteRegisterComponent", {
    			component: this,
    			tagName: "Input",
    			options,
    			id: create_fragment$2.name
    		});

    		const { ctx } = this.$$;
    		const props = options.props || {};

    		if (/*value*/ ctx[1] === undefined && !('value' in props)) {
    			console.warn("<Input> was created without expected prop 'value'");
    		}
    	}

    	get type() {
    		throw new Error("<Input>: Props cannot be read directly from the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	set type(value) {
    		throw new Error("<Input>: Props cannot be set directly on the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	get id() {
    		throw new Error("<Input>: Props cannot be read directly from the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	set id(value) {
    		throw new Error("<Input>: Props cannot be set directly on the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	get title() {
    		throw new Error("<Input>: Props cannot be read directly from the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	set title(value) {
    		throw new Error("<Input>: Props cannot be set directly on the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	get label() {
    		throw new Error("<Input>: Props cannot be read directly from the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	set label(value) {
    		throw new Error("<Input>: Props cannot be set directly on the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	get ariaLabel() {
    		throw new Error("<Input>: Props cannot be read directly from the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	set ariaLabel(value) {
    		throw new Error("<Input>: Props cannot be set directly on the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	get ariaDescribedBy() {
    		throw new Error("<Input>: Props cannot be read directly from the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	set ariaDescribedBy(value) {
    		throw new Error("<Input>: Props cannot be set directly on the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	get ariaOwns() {
    		throw new Error("<Input>: Props cannot be read directly from the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	set ariaOwns(value) {
    		throw new Error("<Input>: Props cannot be set directly on the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	get ariaAutocomplete() {
    		throw new Error("<Input>: Props cannot be read directly from the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	set ariaAutocomplete(value) {
    		throw new Error("<Input>: Props cannot be set directly on the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	get ariaExpanded() {
    		throw new Error("<Input>: Props cannot be read directly from the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	set ariaExpanded(value) {
    		throw new Error("<Input>: Props cannot be set directly on the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	get ariaHasPopup() {
    		throw new Error("<Input>: Props cannot be read directly from the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	set ariaHasPopup(value) {
    		throw new Error("<Input>: Props cannot be set directly on the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	get value() {
    		throw new Error("<Input>: Props cannot be read directly from the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	set value(value) {
    		throw new Error("<Input>: Props cannot be set directly on the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	get disabled() {
    		throw new Error("<Input>: Props cannot be read directly from the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	set disabled(value) {
    		throw new Error("<Input>: Props cannot be set directly on the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	get required() {
    		throw new Error("<Input>: Props cannot be read directly from the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	set required(value) {
    		throw new Error("<Input>: Props cannot be set directly on the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	get focusDefault() {
    		throw new Error("<Input>: Props cannot be read directly from the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	set focusDefault(value) {
    		throw new Error("<Input>: Props cannot be set directly on the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	get size() {
    		throw new Error("<Input>: Props cannot be read directly from the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	set size(value) {
    		throw new Error("<Input>: Props cannot be set directly on the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	get px() {
    		throw new Error("<Input>: Props cannot be read directly from the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	set px(value) {
    		throw new Error("<Input>: Props cannot be set directly on the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	get autocomplete() {
    		throw new Error("<Input>: Props cannot be read directly from the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	set autocomplete(value) {
    		throw new Error("<Input>: Props cannot be set directly on the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	get spellcheck() {
    		throw new Error("<Input>: Props cannot be read directly from the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	set spellcheck(value) {
    		throw new Error("<Input>: Props cannot be set directly on the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	get autocapitalize() {
    		throw new Error("<Input>: Props cannot be read directly from the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	set autocapitalize(value) {
    		throw new Error("<Input>: Props cannot be set directly on the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	get onInputFocus() {
    		throw new Error("<Input>: Props cannot be read directly from the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	set onInputFocus(value) {
    		throw new Error("<Input>: Props cannot be set directly on the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	get onInputBlur() {
    		throw new Error("<Input>: Props cannot be read directly from the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	set onInputBlur(value) {
    		throw new Error("<Input>: Props cannot be set directly on the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	get onInputClick() {
    		throw new Error("<Input>: Props cannot be read directly from the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	set onInputClick(value) {
    		throw new Error("<Input>: Props cannot be set directly on the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	get onInputKeydown() {
    		throw new Error("<Input>: Props cannot be read directly from the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	set onInputKeydown(value) {
    		throw new Error("<Input>: Props cannot be set directly on the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	get onInputChange() {
    		throw new Error("<Input>: Props cannot be read directly from the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	set onInputChange(value) {
    		throw new Error("<Input>: Props cannot be set directly on the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	get withClearIcon() {
    		throw new Error("<Input>: Props cannot be read directly from the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	set withClearIcon(value) {
    		throw new Error("<Input>: Props cannot be set directly on the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	get withIconAfter() {
    		throw new Error("<Input>: Props cannot be read directly from the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	set withIconAfter(value) {
    		throw new Error("<Input>: Props cannot be set directly on the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	get onIconClick() {
    		throw new Error("<Input>: Props cannot be read directly from the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	set onIconClick(value) {
    		throw new Error("<Input>: Props cannot be set directly on the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	get withShadow() {
    		throw new Error("<Input>: Props cannot be read directly from the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	set withShadow(value) {
    		throw new Error("<Input>: Props cannot be set directly on the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	get debounce() {
    		throw new Error("<Input>: Props cannot be read directly from the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	set debounce(value) {
    		throw new Error("<Input>: Props cannot be set directly on the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	get classes() {
    		throw new Error("<Input>: Props cannot be read directly from the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	set classes(value) {
    		throw new Error("<Input>: Props cannot be set directly on the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	get errors() {
    		throw new Error("<Input>: Props cannot be read directly from the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}

    	set errors(value) {
    		throw new Error("<Input>: Props cannot be set directly on the component instance unless compiling with 'accessors: true' or '<svelte:options accessors/>'");
    	}
    }

    /* src\app\admin\components\App.svelte generated by Svelte v3.41.0 */

    const { console: console_1 } = globals;
    const file$1 = "src\\app\\admin\\components\\App.svelte";

    // (152:16) {#if !$forlogsState.viewing}
    function create_if_block_2$1(ctx) {
    	let span;
    	let t2;
    	let input;
    	let updating_value;
    	let t3;
    	let button0;
    	let t4;
    	let button1;
    	let current;

    	function input_value_binding(value) {
    		/*input_value_binding*/ ctx[9](value);
    	}

    	let input_props = {
    		ariaLabel: t('main.search'),
    		size: "xs",
    		debounce: 400,
    		withClearIcon: true,
    		onInputChange: /*search*/ ctx[8]
    	};

    	if (/*$searchFiles*/ ctx[0] !== void 0) {
    		input_props.value = /*$searchFiles*/ ctx[0];
    	}

    	input = new Input({ props: input_props, $$inline: true });
    	binding_callbacks.push(() => bind(input, 'value', input_value_binding));

    	button0 = new Button({
    			props: {
    				withIcon: iconRenew,
    				title: t('main.refreshFilesList'),
    				handler: /*func*/ ctx[10]
    			},
    			$$inline: true
    		});

    	button1 = new Button({
    			props: {
    				withCloseIcon: true,
    				iconSize: "s",
    				text: t('main.closeAndBack'),
    				handler: /*func_1*/ ctx[11]
    			},
    			$$inline: true
    		});

    	const block = {
    		c: function create() {
    			span = element("span");
    			span.textContent = `${t('main.search')}:`;
    			t2 = space();
    			create_component(input.$$.fragment);
    			t3 = space();
    			create_component(button0.$$.fragment);
    			t4 = space();
    			create_component(button1.$$.fragment);
    			attr_dev(span, "class", "wb-text-sm");
    			add_location(span, file$1, 152, 21, 5595);
    		},
    		m: function mount(target, anchor) {
    			insert_dev(target, span, anchor);
    			insert_dev(target, t2, anchor);
    			mount_component(input, target, anchor);
    			insert_dev(target, t3, anchor);
    			mount_component(button0, target, anchor);
    			insert_dev(target, t4, anchor);
    			mount_component(button1, target, anchor);
    			current = true;
    		},
    		p: function update(ctx, dirty) {
    			const input_changes = {};

    			if (!updating_value && dirty & /*$searchFiles*/ 1) {
    				updating_value = true;
    				input_changes.value = /*$searchFiles*/ ctx[0];
    				add_flush_callback(() => updating_value = false);
    			}

    			input.$set(input_changes);
    			const button0_changes = {};
    			if (dirty & /*$forlogsState*/ 2) button0_changes.handler = /*func*/ ctx[10];
    			button0.$set(button0_changes);
    		},
    		i: function intro(local) {
    			if (current) return;
    			transition_in(input.$$.fragment, local);
    			transition_in(button0.$$.fragment, local);
    			transition_in(button1.$$.fragment, local);
    			current = true;
    		},
    		o: function outro(local) {
    			transition_out(input.$$.fragment, local);
    			transition_out(button0.$$.fragment, local);
    			transition_out(button1.$$.fragment, local);
    			current = false;
    		},
    		d: function destroy(detaching) {
    			if (detaching) detach_dev(span);
    			if (detaching) detach_dev(t2);
    			destroy_component(input, detaching);
    			if (detaching) detach_dev(t3);
    			destroy_component(button0, detaching);
    			if (detaching) detach_dev(t4);
    			destroy_component(button1, detaching);
    		}
    	};

    	dispatch_dev("SvelteRegisterBlock", {
    		block,
    		id: create_if_block_2$1.name,
    		type: "if",
    		source: "(152:16) {#if !$forlogsState.viewing}",
    		ctx
    	});

    	return block;
    }

    // (149:4) 
    function create_toolbar_slot(ctx) {
    	let span;
    	let section;
    	let current;
    	let if_block = !/*$forlogsState*/ ctx[1].viewing && create_if_block_2$1(ctx);

    	const block = {
    		c: function create() {
    			span = element("span");
    			section = element("section");
    			if (if_block) if_block.c();
    			attr_dev(section, "class", "wb-flex wb-items-center wb-space-x-4");
    			add_location(section, file$1, 150, 12, 5474);
    			attr_dev(span, "slot", "toolbar");
    			add_location(span, file$1, 148, 4, 5439);
    		},
    		m: function mount(target, anchor) {
    			insert_dev(target, span, anchor);
    			append_dev(span, section);
    			if (if_block) if_block.m(section, null);
    			current = true;
    		},
    		p: function update(ctx, dirty) {
    			if (!/*$forlogsState*/ ctx[1].viewing) {
    				if (if_block) {
    					if_block.p(ctx, dirty);

    					if (dirty & /*$forlogsState*/ 2) {
    						transition_in(if_block, 1);
    					}
    				} else {
    					if_block = create_if_block_2$1(ctx);
    					if_block.c();
    					transition_in(if_block, 1);
    					if_block.m(section, null);
    				}
    			} else if (if_block) {
    				group_outros();

    				transition_out(if_block, 1, 1, () => {
    					if_block = null;
    				});

    				check_outros();
    			}
    		},
    		i: function intro(local) {
    			if (current) return;
    			transition_in(if_block);
    			current = true;
    		},
    		o: function outro(local) {
    			transition_out(if_block);
    			current = false;
    		},
    		d: function destroy(detaching) {
    			if (detaching) detach_dev(span);
    			if (if_block) if_block.d();
    		}
    	};

    	dispatch_dev("SvelteRegisterBlock", {
    		block,
    		id: create_toolbar_slot.name,
    		type: "slot",
    		source: "(149:4) ",
    		ctx
    	});

    	return block;
    }

    // (168:4) 
    function create_notifications_slot(ctx) {
    	let span;
    	let notificationicon;
    	let current;
    	notificationicon = new NotificationIcon({ $$inline: true });

    	const block = {
    		c: function create() {
    			span = element("span");
    			create_component(notificationicon.$$.fragment);
    			attr_dev(span, "slot", "notifications");
    			add_location(span, file$1, 167, 4, 6416);
    		},
    		m: function mount(target, anchor) {
    			insert_dev(target, span, anchor);
    			mount_component(notificationicon, span, null);
    			current = true;
    		},
    		i: function intro(local) {
    			if (current) return;
    			transition_in(notificationicon.$$.fragment, local);
    			current = true;
    		},
    		o: function outro(local) {
    			transition_out(notificationicon.$$.fragment, local);
    			current = false;
    		},
    		d: function destroy(detaching) {
    			if (detaching) detach_dev(span);
    			destroy_component(notificationicon);
    		}
    	};

    	dispatch_dev("SvelteRegisterBlock", {
    		block,
    		id: create_notifications_slot.name,
    		type: "slot",
    		source: "(168:4) ",
    		ctx
    	});

    	return block;
    }

    // (171:4) 
    function create_spinner_slot(ctx) {
    	let span;
    	let spinner;
    	let current;

    	spinner = new Spinner({
    			props: {
    				busy: /*$busyState*/ ctx[3].isBusy,
    				done: !/*$busyState*/ ctx[3].isBusy
    			},
    			$$inline: true
    		});

    	const block = {
    		c: function create() {
    			span = element("span");
    			create_component(spinner.$$.fragment);
    			attr_dev(span, "slot", "spinner");
    			add_location(span, file$1, 170, 4, 6480);
    		},
    		m: function mount(target, anchor) {
    			insert_dev(target, span, anchor);
    			mount_component(spinner, span, null);
    			current = true;
    		},
    		p: function update(ctx, dirty) {
    			const spinner_changes = {};
    			if (dirty & /*$busyState*/ 8) spinner_changes.busy = /*$busyState*/ ctx[3].isBusy;
    			if (dirty & /*$busyState*/ 8) spinner_changes.done = !/*$busyState*/ ctx[3].isBusy;
    			spinner.$set(spinner_changes);
    		},
    		i: function intro(local) {
    			if (current) return;
    			transition_in(spinner.$$.fragment, local);
    			current = true;
    		},
    		o: function outro(local) {
    			transition_out(spinner.$$.fragment, local);
    			current = false;
    		},
    		d: function destroy(detaching) {
    			if (detaching) detach_dev(span);
    			destroy_component(spinner);
    		}
    	};

    	dispatch_dev("SvelteRegisterBlock", {
    		block,
    		id: create_spinner_slot.name,
    		type: "slot",
    		source: "(171:4) ",
    		ctx
    	});

    	return block;
    }

    // (189:4) {:else}
    function create_else_block_1(ctx) {
    	let loggedout;
    	let current;
    	loggedout = new Loggedout({ $$inline: true });

    	const block = {
    		c: function create() {
    			create_component(loggedout.$$.fragment);
    		},
    		m: function mount(target, anchor) {
    			mount_component(loggedout, target, anchor);
    			current = true;
    		},
    		p: noop$1,
    		i: function intro(local) {
    			if (current) return;
    			transition_in(loggedout.$$.fragment, local);
    			current = true;
    		},
    		o: function outro(local) {
    			transition_out(loggedout.$$.fragment, local);
    			current = false;
    		},
    		d: function destroy(detaching) {
    			destroy_component(loggedout, detaching);
    		}
    	};

    	dispatch_dev("SvelteRegisterBlock", {
    		block,
    		id: create_else_block_1.name,
    		type: "else",
    		source: "(189:4) {:else}",
    		ctx
    	});

    	return block;
    }

    // (178:4) {#if $appState.loggedIn == 'yes'}
    function create_if_block$1(ctx) {
    	let adminpage;
    	let current;

    	adminpage = new AdminPage({
    			props: {
    				pageState: /*forlogsState*/ ctx[6],
    				$$slots: { ready: [create_ready_slot] },
    				$$scope: { ctx }
    			},
    			$$inline: true
    		});

    	const block = {
    		c: function create() {
    			create_component(adminpage.$$.fragment);
    		},
    		m: function mount(target, anchor) {
    			mount_component(adminpage, target, anchor);
    			current = true;
    		},
    		p: function update(ctx, dirty) {
    			const adminpage_changes = {};

    			if (dirty & /*$$scope, $forlogsState*/ 131074) {
    				adminpage_changes.$$scope = { dirty, ctx };
    			}

    			adminpage.$set(adminpage_changes);
    		},
    		i: function intro(local) {
    			if (current) return;
    			transition_in(adminpage.$$.fragment, local);
    			current = true;
    		},
    		o: function outro(local) {
    			transition_out(adminpage.$$.fragment, local);
    			current = false;
    		},
    		d: function destroy(detaching) {
    			destroy_component(adminpage, detaching);
    		}
    	};

    	dispatch_dev("SvelteRegisterBlock", {
    		block,
    		id: create_if_block$1.name,
    		type: "if",
    		source: "(178:4) {#if $appState.loggedIn == 'yes'}",
    		ctx
    	});

    	return block;
    }

    // (184:16) {:else}
    function create_else_block$1(ctx) {
    	let list;
    	let current;
    	list = new List({ $$inline: true });

    	const block = {
    		c: function create() {
    			create_component(list.$$.fragment);
    		},
    		m: function mount(target, anchor) {
    			mount_component(list, target, anchor);
    			current = true;
    		},
    		i: function intro(local) {
    			if (current) return;
    			transition_in(list.$$.fragment, local);
    			current = true;
    		},
    		o: function outro(local) {
    			transition_out(list.$$.fragment, local);
    			current = false;
    		},
    		d: function destroy(detaching) {
    			destroy_component(list, detaching);
    		}
    	};

    	dispatch_dev("SvelteRegisterBlock", {
    		block,
    		id: create_else_block$1.name,
    		type: "else",
    		source: "(184:16) {:else}",
    		ctx
    	});

    	return block;
    }

    // (182:16) {#if $forlogsState.viewing}
    function create_if_block_1$1(ctx) {
    	let view;
    	let current;
    	view = new View({ $$inline: true });

    	const block = {
    		c: function create() {
    			create_component(view.$$.fragment);
    		},
    		m: function mount(target, anchor) {
    			mount_component(view, target, anchor);
    			current = true;
    		},
    		i: function intro(local) {
    			if (current) return;
    			transition_in(view.$$.fragment, local);
    			current = true;
    		},
    		o: function outro(local) {
    			transition_out(view.$$.fragment, local);
    			current = false;
    		},
    		d: function destroy(detaching) {
    			destroy_component(view, detaching);
    		}
    	};

    	dispatch_dev("SvelteRegisterBlock", {
    		block,
    		id: create_if_block_1$1.name,
    		type: "if",
    		source: "(182:16) {#if $forlogsState.viewing}",
    		ctx
    	});

    	return block;
    }

    // (180:12) 
    function create_ready_slot(ctx) {
    	let div;
    	let current_block_type_index;
    	let if_block;
    	let current;
    	const if_block_creators = [create_if_block_1$1, create_else_block$1];
    	const if_blocks = [];

    	function select_block_type_1(ctx, dirty) {
    		if (/*$forlogsState*/ ctx[1].viewing) return 0;
    		return 1;
    	}

    	current_block_type_index = select_block_type_1(ctx);
    	if_block = if_blocks[current_block_type_index] = if_block_creators[current_block_type_index](ctx);

    	const block = {
    		c: function create() {
    			div = element("div");
    			if_block.c();
    			attr_dev(div, "slot", "ready");
    			attr_dev(div, "class", "wb-w-full wb-bg-white wb-text-gray-900");
    			add_location(div, file$1, 179, 12, 6759);
    		},
    		m: function mount(target, anchor) {
    			insert_dev(target, div, anchor);
    			if_blocks[current_block_type_index].m(div, null);
    			current = true;
    		},
    		p: function update(ctx, dirty) {
    			let previous_block_index = current_block_type_index;
    			current_block_type_index = select_block_type_1(ctx);

    			if (current_block_type_index !== previous_block_index) {
    				group_outros();

    				transition_out(if_blocks[previous_block_index], 1, 1, () => {
    					if_blocks[previous_block_index] = null;
    				});

    				check_outros();
    				if_block = if_blocks[current_block_type_index];

    				if (!if_block) {
    					if_block = if_blocks[current_block_type_index] = if_block_creators[current_block_type_index](ctx);
    					if_block.c();
    				}

    				transition_in(if_block, 1);
    				if_block.m(div, null);
    			}
    		},
    		i: function intro(local) {
    			if (current) return;
    			transition_in(if_block);
    			current = true;
    		},
    		o: function outro(local) {
    			transition_out(if_block);
    			current = false;
    		},
    		d: function destroy(detaching) {
    			if (detaching) detach_dev(div);
    			if_blocks[current_block_type_index].d();
    		}
    	};

    	dispatch_dev("SvelteRegisterBlock", {
    		block,
    		id: create_ready_slot.name,
    		type: "slot",
    		source: "(180:12) ",
    		ctx
    	});

    	return block;
    }

    function create_fragment$1(ctx) {
    	let navbar;
    	let t_1;
    	let section;
    	let current_block_type_index;
    	let if_block;
    	let current;

    	navbar = new Navbar({
    			props: {
    				$$slots: {
    					spinner: [create_spinner_slot],
    					notifications: [create_notifications_slot],
    					toolbar: [create_toolbar_slot]
    				},
    				$$scope: { ctx }
    			},
    			$$inline: true
    		});

    	const if_block_creators = [create_if_block$1, create_else_block_1];
    	const if_blocks = [];

    	function select_block_type(ctx, dirty) {
    		if (/*$appState*/ ctx[2].loggedIn == 'yes') return 0;
    		return 1;
    	}

    	current_block_type_index = select_block_type(ctx);
    	if_block = if_blocks[current_block_type_index] = if_block_creators[current_block_type_index](ctx);

    	const block = {
    		c: function create() {
    			create_component(navbar.$$.fragment);
    			t_1 = space();
    			section = element("section");
    			if_block.c();
    			attr_dev(section, "class", "wb-overflow-y-auto wb-p-0 wb-flex wb-flex-row");
    			add_location(section, file$1, 175, 0, 6590);
    		},
    		l: function claim(nodes) {
    			throw new Error("options.hydrate only works if the component was compiled with the `hydratable: true` option");
    		},
    		m: function mount(target, anchor) {
    			mount_component(navbar, target, anchor);
    			insert_dev(target, t_1, anchor);
    			insert_dev(target, section, anchor);
    			if_blocks[current_block_type_index].m(section, null);
    			current = true;
    		},
    		p: function update(ctx, [dirty]) {
    			const navbar_changes = {};

    			if (dirty & /*$$scope, $busyState, $forlogsState, $searchFiles*/ 131083) {
    				navbar_changes.$$scope = { dirty, ctx };
    			}

    			navbar.$set(navbar_changes);
    			let previous_block_index = current_block_type_index;
    			current_block_type_index = select_block_type(ctx);

    			if (current_block_type_index === previous_block_index) {
    				if_blocks[current_block_type_index].p(ctx, dirty);
    			} else {
    				group_outros();

    				transition_out(if_blocks[previous_block_index], 1, 1, () => {
    					if_blocks[previous_block_index] = null;
    				});

    				check_outros();
    				if_block = if_blocks[current_block_type_index];

    				if (!if_block) {
    					if_block = if_blocks[current_block_type_index] = if_block_creators[current_block_type_index](ctx);
    					if_block.c();
    				} else {
    					if_block.p(ctx, dirty);
    				}

    				transition_in(if_block, 1);
    				if_block.m(section, null);
    			}
    		},
    		i: function intro(local) {
    			if (current) return;
    			transition_in(navbar.$$.fragment, local);
    			transition_in(if_block);
    			current = true;
    		},
    		o: function outro(local) {
    			transition_out(navbar.$$.fragment, local);
    			transition_out(if_block);
    			current = false;
    		},
    		d: function destroy(detaching) {
    			destroy_component(navbar, detaching);
    			if (detaching) detach_dev(t_1);
    			if (detaching) detach_dev(section);
    			if_blocks[current_block_type_index].d();
    		}
    	};

    	dispatch_dev("SvelteRegisterBlock", {
    		block,
    		id: create_fragment$1.name,
    		type: "component",
    		source: "",
    		ctx
    	});

    	return block;
    }

    function instance$1($$self, $$props, $$invalidate) {
    	let $searchFiles;
    	let $forlogsState;
    	let $appState;
    	let $busyState;
    	let { $$slots: slots = {}, $$scope } = $$props;
    	validate_slots('App', slots, []);
    	let appState = getContext('appState');
    	validate_store(appState, 'appState');
    	component_subscribe($$self, appState, value => $$invalidate(2, $appState = value));
    	let busyState = getContext('busyState');
    	validate_store(busyState, 'busyState');
    	component_subscribe($$self, busyState, value => $$invalidate(3, $busyState = value));
    	let oneoffNotifications = getContext('oneoffNotifications');
    	let forlogsState = getContext('forlogsState');
    	validate_store(forlogsState, 'forlogsState');
    	component_subscribe($$self, forlogsState, value => $$invalidate(1, $forlogsState = value));
    	let config = getContext('config');
    	let searchFiles = getContext('searchFiles');
    	validate_store(searchFiles, 'searchFiles');
    	component_subscribe($$self, searchFiles, value => $$invalidate(0, $searchFiles = value));
    	let loadingFiles = false;

    	const loadFiles = () => {
    		if (loadingFiles) {
    			return;
    		}

    		loadingFiles = true;
    		set_store_value(appState, $appState.status = 'loading', $appState);
    		busyState.startSpinner('files');

    		filesModel.fetchList($forlogsState.search).then(json => {
    			busyState.stopSpinner('files');

    			if (json.failed || json.status && json.status == 'error') {
    				set_store_value(appState, $appState.status = 'error', $appState);
    				oneoffNotifications.addError(t(json.message), t('backendComm.errorViewDetailsInConsole'), true);
    				console.error('4LOGS: ', json.message, ': ', json.details);
    			} else {
    				set_store_value(forlogsState, $forlogsState.files = json.data.files, $forlogsState);
    				let folders = [];

    				for (const folder in json.data.files) {
    					if (json.data.files.hasOwnProperty(folder)) {
    						folders.push(folder);
    						json.data.files[folder] = json.data.files[folder].sort((a, b) => a.mtime > b.mtime ? -1 : 1);
    					}
    				}

    				set_store_value(forlogsState, $forlogsState.folders = folders, $forlogsState);
    				json.data.files = search($searchFiles, json.data.files);

    				for (const folder in json.data.files) {
    					if (json.data.files.hasOwnProperty(folder)) {
    						json.data.files[folder] = json.data.files[folder].sort((a, b) => a.mtime > b.mtime ? -1 : 1);
    					}
    				}

    				filterFiles($searchFiles);
    				set_store_value(forlogsState, $forlogsState.file = '', $forlogsState);
    				set_store_value(forlogsState, $forlogsState.total = json.meta.total, $forlogsState);
    				set_store_value(forlogsState, $forlogsState.errorsCount = 0, $forlogsState);
    				set_store_value(forlogsState, $forlogsState.selected = [], $forlogsState);
    				set_store_value(forlogsState, $forlogsState.viewing = null, $forlogsState);
    				set_store_value(forlogsState, $forlogsState.editing = null, $forlogsState);
    				set_store_value(forlogsState, $forlogsState.dialog = null, $forlogsState);
    				set_store_value(forlogsState, $forlogsState.confirmationDialog = null, $forlogsState);
    				set_store_value(forlogsState, $forlogsState.alert = false, $forlogsState);
    				set_store_value(forlogsState, $forlogsState.nextAction = '', $forlogsState);
    				set_store_value(appState, $appState.status = 'ready', $appState);
    			}

    			loadingFiles = false;
    		});
    	};

    	onMount(loadFiles);

    	const search = (searchTerm, files) => {
    		files = files || $forlogsState.files;

    		if (!searchTerm) {
    			return files;
    		}

    		let filteredFiles = {};

    		if (searchTerm) {
    			let lcSearchTerm = searchTerm.toLowerCase();

    			for (const folder in files) {
    				if (files.hasOwnProperty(folder)) {
    					// Mootools protection
    					let f = files[folder].filter(file => file.path.toLowerCase().includes(lcSearchTerm));

    					if (f.length) {
    						filteredFiles[folder] = f;
    					}
    				}
    			}
    		}

    		return filteredFiles;
    	};

    	const filterFiles = searchTerm => {
    		if (!searchTerm) {
    			set_store_value(forlogsState, $forlogsState.filteredFiles = $forlogsState.files, $forlogsState);
    			set_store_value(forlogsState, $forlogsState.filteredFolders = $forlogsState.folders, $forlogsState);
    			return;
    		}

    		let filteredFiles = search(searchTerm, $forlogsState.files);
    		let folders = [];

    		for (const folder in filteredFiles) {
    			if (filteredFiles.hasOwnProperty(folder)) {
    				folders.push(folder);
    				filteredFiles[folder] = filteredFiles[folder].sort((a, b) => a.mtime > b.mtime ? -1 : 1);
    			}
    		}

    		set_store_value(forlogsState, $forlogsState.filteredFiles = filteredFiles, $forlogsState);
    		set_store_value(forlogsState, $forlogsState.filteredFolders = folders, $forlogsState);
    	};

    	const writable_props = [];

    	Object.keys($$props).forEach(key => {
    		if (!~writable_props.indexOf(key) && key.slice(0, 2) !== '$$' && key !== 'slot') console_1.warn(`<App> was created with unknown prop '${key}'`);
    	});

    	function input_value_binding(value) {
    		$searchFiles = value;
    		searchFiles.set($searchFiles);
    	}

    	const func = () => set_store_value(forlogsState, $forlogsState.nextAction = 'refresh', $forlogsState);
    	const func_1 = () => Joomla.submitbutton('plugin.cancel');

    	$$self.$capture_state = () => ({
    		getContext,
    		onMount,
    		t,
    		iconRenew,
    		filesModel,
    		List,
    		View,
    		Navbar,
    		AdminPage,
    		Loggedout,
    		NotificationIcon,
    		Spinner,
    		Button,
    		Input,
    		appState,
    		busyState,
    		oneoffNotifications,
    		forlogsState,
    		config,
    		searchFiles,
    		loadingFiles,
    		loadFiles,
    		search,
    		filterFiles,
    		$searchFiles,
    		$forlogsState,
    		$appState,
    		$busyState
    	});

    	$$self.$inject_state = $$props => {
    		if ('appState' in $$props) $$invalidate(4, appState = $$props.appState);
    		if ('busyState' in $$props) $$invalidate(5, busyState = $$props.busyState);
    		if ('oneoffNotifications' in $$props) oneoffNotifications = $$props.oneoffNotifications;
    		if ('forlogsState' in $$props) $$invalidate(6, forlogsState = $$props.forlogsState);
    		if ('config' in $$props) config = $$props.config;
    		if ('searchFiles' in $$props) $$invalidate(7, searchFiles = $$props.searchFiles);
    		if ('loadingFiles' in $$props) loadingFiles = $$props.loadingFiles;
    	};

    	if ($$props && "$$inject" in $$props) {
    		$$self.$inject_state($$props.$$inject);
    	}

    	$$self.$$.update = () => {
    		if ($$self.$$.dirty & /*$forlogsState*/ 2) {
    			// trigger a new fetch after a toolbar action
    			'refresh' == $forlogsState.nextAction
    			? loadFiles()
    			: null;
    		}

    		if ($$self.$$.dirty & /*$searchFiles*/ 1) {
    			filterFiles($searchFiles);
    		}
    	};

    	return [
    		$searchFiles,
    		$forlogsState,
    		$appState,
    		$busyState,
    		appState,
    		busyState,
    		forlogsState,
    		searchFiles,
    		search,
    		input_value_binding,
    		func,
    		func_1
    	];
    }

    class App extends SvelteComponentDev {
    	constructor(options) {
    		super(options);
    		init(this, options, instance$1, create_fragment$1, safe_not_equal, {});

    		dispatch_dev("SvelteRegisterComponent", {
    			component: this,
    			tagName: "App",
    			options,
    			id: create_fragment$1.name
    		});
    	}
    }

    let defaults$2 = {
        status: 'ready',
        trapClicks: false,
        loggedIn: 'yes',
    };

    const appState = writable(defaults$2);

    const defaults$1 = {
        running: [],
        isBusy: false
    };

    const createStore$3 = () => {
        const {subscribe, set, update} = writable(defaults$1);
        return {
            subscribe,
            startSpinner: (id) => update((state) => {
                state.running.push(id || 's' + v4());
                state.isBusy = state.running.length > 0;
                return state
            }),
            stopSpinner: (id) => update((state) => {
                state.running.pop();
                state.isBusy = state.running.length > 0;
                return state
            }),
            reset: () => set(defaults$1)
        }

    };
    const busyState$1 = createStore$3();

    /**
     * 4LOGS
     *
     * @author       Yannick Gaultier - Weeblr llc
     * @copyright    Copyright Weeblr llc - 2021
     * @package      4LOGS
     * @license      GNU General Public License version 3; see LICENSE.md
     * @version      1.1.2.217
     * @date        2021-08-20
     */

    /***********************************************************************************************************************
     * Local storage utilities
     **********************************************************************************************************************/

    const createStorage = (storageType, storageScope) => {

        const storage = 'session' == storageType ? window.sessionStorage : window.localStorage;
        const scope = storageScope;

        /**
         * Retrieve an item stored as JSON under {itemKey}
         * from localstorage and unserialize it.
         *
         * @param string itemKey
         * @returns {any}
         */
        const get = (itemKey) => {
            let item = storage.getItem(scope + itemKey);
            try {
                item = JSON.parse(item);
            } catch (e) {
                console.error(e);
                item = null;
            }
            return item
        };

        /**
         * Store some data as JSON under itemKey in localstorage.
         *
         * @param string itemKey
         * @param miwed data
         */
        const set = (itemKey, data) => {
            storage.setItem(scope + itemKey, JSON.stringify(data));
        };

        /**
         * Clear an item from localstorage or ALL items if no
         * itemKey provided.
         *
         * @param string|null itemKey
         */
        const clear = (itemKey) => {
            if (itemKey) {
                storage.removeItem(scope + itemKey);
            } else {
                storage.clear();
            }
        };

        return {
            storage: storage,
            scope: scope,
            set: set,
            get: get,
            clear: clear
        }
    };

    /**
     * Create/returns a scoped local storage object.
     *
     * @param string scope
     * @returns {*}
     */
    const getLocalStorage = (scope) => createStorage('local', scope);

    /**
     * Create/returns a scoped session storage object.
     * @param string scope
     * @returns {*}
     */
    const getSessionStorage = (scope) => createStorage('session', scope);

    // do not edit .js files directly - edit src/index.jst



    var fastDeepEqual = function equal(a, b) {
      if (a === b) return true;

      if (a && b && typeof a == 'object' && typeof b == 'object') {
        if (a.constructor !== b.constructor) return false;

        var length, i, keys;
        if (Array.isArray(a)) {
          length = a.length;
          if (length != b.length) return false;
          for (i = length; i-- !== 0;)
            if (!equal(a[i], b[i])) return false;
          return true;
        }



        if (a.constructor === RegExp) return a.source === b.source && a.flags === b.flags;
        if (a.valueOf !== Object.prototype.valueOf) return a.valueOf() === b.valueOf();
        if (a.toString !== Object.prototype.toString) return a.toString() === b.toString();

        keys = Object.keys(a);
        length = keys.length;
        if (length !== Object.keys(b).length) return false;

        for (i = length; i-- !== 0;)
          if (!Object.prototype.hasOwnProperty.call(b, keys[i])) return false;

        for (i = length; i-- !== 0;) {
          var key = keys[i];

          if (!equal(a[key], b[key])) return false;
        }

        return true;
      }

      // true if both NaN, false otherwise
      return a!==a && b!==b;
    };

    /**
     * Notification record:
     * {
     *     canClear: Boolean
     *     title: string
     *     message: string
     *     autoclear: 300
     *     uuid: string
     * }
     *
     */

    let defaults = {
        total: 0,
        store: [],
    };

    const addNotification = (state, type, title, message, canClear, autoclear) => {
        state['store'].push(
            {
                type: type,
                canClear: !!canClear,
                escapeClear: true,
                title: title,
                message: message,
                autoclear: autoclear,
                uuid: 's' + v4()
            }
        );

        state.total = state['store'].length;

        return state
    };

    const removeNotification = (state, uuid) => {
        state['store'] = state['store'].filter(
            notification => notification.uuid != uuid
        );

        state.total = state['store'].length;

        return state
    };

    const createStore$2 = () => {
        const {subscribe, set, update} = writable(defaults);
        return {
            subscribe: subscribe,
            /**
             * Add a oneoff error notification.
             *
             * @param {string} title
             * @param {string} msg
             * @param {boolean} canClear
             * @param {int} autoclear
             */
            addError: (title, msg, canClear = true, autoclear = 0) => update((state) => {
                return addNotification(state, '1_danger', title, msg, canClear, autoclear)
            }),
            /**
             * Add a oneoff error notification.
             *
             * @param {string} title
             * @param {string} msg
             * @param {boolean} canClear
             * @param {int} autoclear
             */
            addWarning: (title, msg, canClear = true, autoclear = 0) => update((state) => {
                return addNotification(state, '2_warning', title, msg, canClear, autoclear)
            }),
            /**
             * Add a oneoff error notification.
             *
             * @param {string} title
             * @param {string} msg
             * @param {boolean} canClear
             * @param {int} autoclear
             */
            addInfo: (title, msg, canClear = true, autoclear = false) => update((state) => {
                return addNotification(state, '3_info', title, msg, canClear, autoclear)
            }),
            /**
             * Dismiss a specific oneoff notification.
             *
             * @param {string} uuid
             */
            dismiss: (uuid) => update((state) => {
                return removeNotification(state, uuid)
            }),
            /**
             * Remove all oneoff notifications.
             */
            reset: () => set(defaults)
        }

    };
    const oneoffNotifications = createStore$2();

    // Implementation

    const storageBackends = ['local', 'session', 'remote'];
    const storageBackendsWithKey = ['local', 'session'];

    const createStore$1 = (defaultData, storeOptions) => {

        /**
         * Declaration section
         */

        const onStatusChange = (newStatus) => {
            try {
                if ('ready' == newStatus || 'error' == newStatus) {
                    options.busyState.stopSpinner();
                    options.isAppBlocker && options.appState.loadingComplete();
                } else if ('loading' == newStatus) {
                    options.isAppBlocker && options.appState.addLoader();
                    options.busyState.startSpinner();
                }
            } catch (e) {
                console.error(e);
                oneoffNotifications.addError(
                    e.message,
                    true,
                    ''
                );
            }
        };

        let options = {
            ...{
                storeId: '',
                defaultData: defaultData,
                isAppBlocker: false,
                debounce: 0,
                autoload: 'disabled', // disabled || global
                autosave: 'disabled', // disabled || global || per_key
                autosaveExclude: [],
                storage: {
                    type: 'local', // local | session | remote
                    key: '', // must be provided, on a per user basis
                },
                validateState: newState => ({isValid: true, errors: []}),
                onStatusChange: onStatusChange

            },
            ...storeOptions
        };

        if(!options.busyState || !options.appState) {
            throw new Error('Missing busyState or appState when creating store ' + options.storeId)
        }

        // memory storage for the store
        let previousState = options.autoload == 'disabled' ? defaultData : undefined;
        let currentState = options.autoload == 'disabled' ? defaultData : undefined;
        let initialState = currentState;

        // Storage for the subscriptions to the store, either global or per key
        const subscriptions = new Map();
        const subscriptionsByKey = new Map();
        const subscriptionstoStatus = new Map();
        const subscriptionstoErrors = new Map();
        const subscriptionstoChange = new Map();

        /**
         * Implementation section
         */

        /**
         * Flag set to false until autoloaded data is available and the store can be used.
         *
         */
        let status = 'loading';

        /**
         * Storage for errors. Errors are simple messages, property name matches
         * store item name.

         */
        let errors = {};

        /**
         * Options validation, if autosave, needs storage with proper type and key.
         *
         * @param options
         * @returns {*}
         */
        const validateOptions = (options) => {
            let err = false;
            if (
                (
                    options.autosave != 'disabled'
                    ||
                    options.autoload != 'disabled'
                )
                &&
                (
                    !options.storage
                    ||
                    !options.storage.type
                    ||
                    !storageBackends.includes(options.storage.type)
                )

            ) {
                options.autoload = 'disabled';
                options.autosave = 'disabled';
                err = true;
            }

            // local and session storage requires a key to identify this instance storage.
            // Remote storage handles that themselves.
            if (
                (
                    options.autosave != 'disabled'
                    ||
                    options.autoload != 'disabled'
                )
                &&
                storageBackendsWithKey.includes(options.storage.type)
                &&
                !options.storage.key
            ) {
                err = true;
            }

            if (err) {
                console.error(err);
                throw new Error(
                    'Invalid storage options for store ' + JSON.stringify(options, null, 2)
                )
            }

            return err
        };

        const debounce = (fn, delay) => {
            let timer;
            return function () {
                const that = this;
                const args = arguments;
                clearTimeout(timer);
                timer = setTimeout(
                    () => {
                        fn.apply(that, args);
                    }
                    , delay);
            };
        };

        /**
         * Reset the initial state, typically after saving to backend,
         * to be able to detect new changes.
         *
         * No side-effect, just reset initial state.
         *
         * @param {*} newInitialState
         */
        const resetInitialState = (newInitialState) => {
            initialState = newInitialState;
        };

        /**
         * Set the entire store. Value must be an object.
         *
         * @param {Object} value
         */
        const set = (newState) => {
            let validation = options.validateState && options.validateState(newState);
            if (validation && validation.isValid && validation.hasOwnProperty('newState')) {
                newState = validation.newState;
            }

            if (
                validation
                &&
                'ready' == status
                &&
                (
                    'per_key' == options.autosave
                    ||
                    'disabled' == options.autosave
                )
            ) {
                updateErrors(validation.errors);
                if (!validation.isValid) {
                    return
                }
            }

            // Store previous value to serve as reference in determining
            // modified keys, but do not change the currentStae yet. That's
            // done after the (optional) debouncing, in case multiple values
            // are modified quickly
            previousState = currentState;

            // Debounce notifications and storage
            if ('loading' != status && options.debounce > 0) {
                debouncedNotifyAfterSet(newState);
            } else {
                notifyAfterSet(newState);
            }
        };

        /**
         * Notify all subscribers to state or to individual keys
         * and perform storing.
         *
         */
        const notifyAfterSet = (newState) => {
            currentState = newState;
            // push to subscribers
            subscriptions.size > 0 && subscriptions.forEach((unusedValue, subFn) => {
                // important! subscriber should not be allowed to modify current value directly
                subFn({...{}, ...currentState});
            });

            // and persist entire state if configured.
            // (we know it's valid, that's the 1st thing we tested in set()
            if ('ready' == status && options.autosave == 'global') {
                let stored = putToStorage(options, currentState);
                if (isPromise(stored)) {
                    stored.then(() => {
                        console.error('Weeblr: storing global configuration');
                    }).catch(() => {
                        console.error('Weeblr: storing global configuration');
                    });
                }
            }

            // trigger per-key workflow
            notifyByKeySubscribers();
        };

        const debouncedNotifyAfterSet = debounce(notifyAfterSet, options.debounce);

        /**
         * Identify which keys have changed between 2 states.
         *
         */
        const findModifiedItemsKeys = (previous, current) => {
            let modifiedKeys = [];
            for (let key in current) {
                if (!current.hasOwnProperty(key)) {
                    // prevent standard object props pollution
                    continue
                }
                let modified = false;
                if (previous && previous.hasOwnProperty(key)) {
                    modified = !fastDeepEqual.equal(current[key], previous[key]);
                } else {
                    // this is a new key
                    modified = true;
                }
                if (modified) {
                    modifiedKeys.push(key);
                }
            }

            return modifiedKeys
        };

        /**
         * Notify subscribers for specific keys of the store data.
         */
        const notifyByKeySubscribers = () => {

            // leave if no subscription per key
            if (
                !subscriptionsByKey.size
                &&
                !subscriptionstoChange.size
                &&
                'per_key' != options.autosave
            ) {
                // leave if nobody is subscribed to either a specific key value change
                // or any change or if we don't save individual keys content
                // ie: we don't need to know which specific key(s) have changed
                return
            }

            // run a diff to identify which keys have changed
            let modifiedItemsKeys = findModifiedItemsKeys(previousState, currentState);

            // notify change subscribers. "Change" means compared to initial state
            if (modifiedItemsKeys.length && subscriptionstoChange.size) {
                let changesFromInitialState = findModifiedItemsKeys(initialState, currentState);

                // notify even if no difference from initial state
                // as we may have come BACK to initial state
                subscriptionstoChange.forEach(
                    (unusedValue, subFn) => {
                        subFn(
                            changesFromInitialState,
                        );
                    }
                );
            }

            // push to per-key subscribers if any
            for (let index = 0; index < modifiedItemsKeys.length; index++) {

                // maybe store this key
                if (
                    'loading' != status
                    &&
                    options.autosave == 'per_key'
                    &&
                    !options.autosaveExclude.includes(modifiedItemsKeys[index])
                    &&
                    (
                        // only save if value has been modified
                        !previousState.hasOwnProperty(modifiedItemsKeys[index])
                        ||
                        currentState[modifiedItemsKeys[index]] !== previousState[modifiedItemsKeys[index]]
                    )
                ) {
                    busyState.startSpinner();
                    let stored = putToStorage(options, currentState, modifiedItemsKeys[index]);
                    if (isPromise(stored)) {
                        stored.then((storedValue) => {
                            busyState.stopSpinner();
                            if (
                                storedValue.hasOwnProperty(modifiedItemsKeys[index])
                                &&
                                storedValue[modifiedItemsKeys[index]] !== currentState[modifiedItemsKeys[index]]
                            ) {
                                // the server updated the value, it did not keep the one sent. Update state.
                                previousState[modifiedItemsKeys[index]] = storedValue[modifiedItemsKeys[index]];
                                set(
                                    {
                                        ...currentState,
                                        ...storedValue
                                    }
                                );
                            }

                        }).catch((error) => {
                            busyState.stopSpinner();
                            oneoffNotifications.addError(
                                t('configuration.cannotSaveTitle'),
                                `${t('configuration.cannotSaveDetails')}<p class="wb-text-xs wb-mt-2">Details: ${error.message}</p>`,
                                true
                            );
                            updateStatus('error');
                        });
                    }
                }

                // maybe notify subscribers
                const subsForThisKey = subscriptionsByKey.get(modifiedItemsKeys[index]);
                if (!subsForThisKey || !subsForThisKey.size) {
                    continue
                }
                subsForThisKey.forEach(
                    (unusedValue, subFn) => {
                        let key = modifiedItemsKeys[index];
                        let prev = previousState && previousState.hasOwnProperty(key) ? previousState[key] : null;
                        subFn(
                            currentState[key],
                            prev
                        );
                    }
                );
            }
        };

        /**
         * Implements Svelte store update.
         *
         * @param fn
         */
        const update = fn => {
            set(fn(currentState));
        };

        /**
         * Subscribe to the entire store, per Svelte stores contract.
         *
         */
        const subscribe = (subscriptionFn) => {

            subscriptions.set(subscriptionFn, true);
            // important! subscriber should not be allowed to modify current value directly
            subscriptionFn({...{}, ...currentState});

            return () => {
                if (!subscriptions.has(subscriptionFn)) {
                    return
                }
                subscriptions.delete(subscriptionFn);
            }
        };

        /**
         * Only sets a specific key of the store.
         *
         * @param {string} key
         * @param {*} value
         */
        const setByKey = (key, value) => {
            let newState = {...{}, ...currentState};
            newState[key] = value;
            set(newState);
        };

        /**
         * Subscribe to the changes of a specific key of the store only.
         *
         * @param {string} key
         */
        const subscribeToKey = (key, subscriptionFn) => {
            if (!currentState || !currentState.hasOwnProperty(key)) {
                throw new Error('Cannot subscribe to key ' + key + ', it does not exist yet.')
            }

            if (!subscriptionsByKey.has(key)) {
                subscriptionsByKey.set(key, new Map());
            }

            subscriptionsByKey.get(key).set(subscriptionFn, true);
            let prev = previousState && previousState.hasOwnProperty(key) ? previousState[key] : null;
            subscriptionFn(currentState[key], prev);

            return () => {
                if (!subscriptionsByKey.has(key)) {
                    return
                }
                let subs = subscriptionsByKey.get(key);
                if (subs.has(subscriptionFn)) {
                    subs.delete(subscriptionFn);
                }
            }
        };

        /**
         * Subscribe to any actual change of the store state.
         *
         */
        const subscribeToChange = (subscriptionFn) => {

            subscriptionstoChange.set(subscriptionFn, true);
            // important! subscriber should not be allowed to modify current value directly
            subscriptionFn([]); // no changes at first

            return () => {
                if (!subscriptionstoChange.has(subscriptionFn)) {
                    return
                }
                subscriptionstoChange.delete(subscriptionFn);
            }
        };

        /**
         * Subscribe to the changes of the store readyness.
         *
         */
        const subscribeToStatus = (subscriptionFn) => {

            subscriptionstoStatus.set(subscriptionFn, true);
            // important! subscriber should not be allowed to modify current value directly
            subscriptionFn(status);

            return () => {
                if (!subscriptionstoStatus.has(subscriptionFn)) {
                    return
                }
                subscriptionstoStatus.delete(subscriptionFn);
            }
        };

        /**
         * Getter for the store status.
         *
         * @returns {string}
         */
        const getStatus = () => status;

        /**
         * Wether all autoloaded data is available and the store can be used.
         *
         * @returns {boolean}
         */
        const updateStatus = (newStatus) => {

            if (newStatus !== status) {
                status = newStatus;
                options.onStatusChange && options.onStatusChange(status);
                notifyStatusSubscribers();
            }
        };

        /**
         * Notify subscribers for the store ready state.
         */
        const notifyStatusSubscribers = () => {

            if (!subscriptionstoStatus.size) {
                return
            }

            subscriptionstoStatus.forEach(
                (unusedValue, subFn) => {
                    subFn(status);
                }
            );
        };

        /**
         * Subscribe to the changes of the store errors list
         *
         */
        const subscribeToErrors = (subscriptionFn) => {

            subscriptionstoErrors.set(subscriptionFn, true);
            // important! subscriber should not be allowed to modify current value directly
            subscriptionFn(errors);

            return () => {
                if (!subscriptionstoErrors.has(subscriptionFn)) {
                    return
                }
                subscriptionstoErrors.delete(subscriptionFn);
            }
        };

        /**
         * Getter for the store errors
         *
         * @returns {}
         */
        const getErrors = () => errors;

        /**
         * Wether all autoloaded data is available and the store can be used.
         *
         * @returns {boolean}
         */
        const updateErrors = (newErrors) => {

            if (newErrors !== errors) {
                errors = newErrors;
                notifyErrorsSubscribers();
            }
        };

        /**
         * Notify subscribers for the store errors list.
         */
        const notifyErrorsSubscribers = () => {

            if (!subscriptionstoErrors.size) {
                return
            }

            subscriptionstoErrors.forEach(
                (unusedValue, subFn) => {
                    subFn(errors);
                }
            );
        };

        /**
         * If enabled, get a storage backend handler.
         */
        var setupStorage = () => {
            // console.error('Base setupStorage putToStorage')
        };

        /**
         * Load previously stored data, if any.
         */
        var autoloadFromStorage = () => {
            // console.error('Base setupStorage putToStorage')
        };

        /**
         * Store just updated data, either the entire state or just a single key.
         */
        var putToStorage = () => {
            // console.error('Base Store putToStorage')
        };

        const initStore = (overrides) => {
            validateOptions(options);
            if (overrides) {
                // apply overrides
                if (overrides['setupStorage']) {
                    setupStorage = overrides['setupStorage'];
                }
                if (overrides['autoloadFromStorage']) {
                    autoloadFromStorage = overrides['autoloadFromStorage'];
                }
                if (overrides['putToStorage']) {
                    putToStorage = overrides['putToStorage'];
                }

                // run init
                setupStorage(options);
                let loadedState = autoloadFromStorage(options, currentState);
                if (isPromise(loadedState)) {
                    updateStatus('loading');
                    loadedState.then(
                        (updatedState) => {
                            if (updatedState instanceof Error) {
                                // Autoloading failed
                                let message = 'Error autoloading data for store ' + options.storeId + ' - ' + updatedState.toString();
                                console.error('4SEO: ' + message, updatedState);
                                oneoffNotifications.addError(
                                    t('backendComm.errorLoadingDataFromServer'),
                                    message,
                                    true
                                );
                                updateStatus('error');
                            } else {
                                set(updatedState);
                                resetInitialState(updatedState);
                                updateStatus('ready');
                            }
                        }
                    ).catch(
                        (error) => {
                            let message = error.toString() + ' (Init storage, initial state)';
                            console.error(' 4SEO: ', message, error);
                            oneoffNotifications.addError(
                                t('backendComm.errorLoadingDataFromServer'),
                                message,
                                true
                            );
                            previousState = {};
                            currentState = defaultData;
                            resetInitialState(defaultData);
                            updateStatus('error');
                        }
                    );
                } else {
                    let validation = options.validateState(loadedState);
                    if (!validation.isValid) {
                        oneoffNotifications.addError(
                            'Server provided invalid data',
                            'Data retrieved from this browser for store ' + options.storeId + ' is invalid, we cannot use it. Resetting display options to defaults.',
                            true
                        );
                        console.error('4SEO: Invalid initial state for store ', options.storeId, validation.errors);
                        previousState = {};
                        currentState = defaultData;
                    } else {
                        resetInitialState(loadedState);
                        previousState = loadedState;
                        currentState = loadedState;
                    }
                    updateStatus('ready');
                }
            } else {
                // no overrides to load content from a source,
                // we're ready to go with only default values
                updateStatus('ready');
            }
        };

        // debug tool
        const dump = () => {
            return JSON.stringify(
                currentState,
                null,
                ' '
            )
        };

        /**
         * Informs app that store is loading, maybe.
         *
         * @param {string} newStatus
         */


        /**
         * Run section
         */

        options.onStatusChange && options.onStatusChange(status);

        return {
            // Public interface
            getStatus: getStatus,
            updateStatus: updateStatus,
            subscribeToStatus: subscribeToStatus,
            subscribeToChange: subscribeToChange,
            getErrors: getErrors,
            updateErrors: updateErrors,
            subscribeToErrors: subscribeToErrors,
            validateState: options.validateState,
            set: set,
            resetInitialState: resetInitialState,
            update: update,
            setByKey: setByKey,
            subscribe: subscribe,
            subscribeToKey: subscribeToKey,
            initStore: initStore,
            dump: dump
        }
    };

    const createStore = (defaultData, storeOptions) => {

        /**
         * Declaration section
         */
        let storage;
        let sharedStorage;

        const tabId = sessionStorage.forseoTabId
            ? sessionStorage.forseoTabId
            : sessionStorage.forseoTabId = v4();

        /**
         * If enabled, get a storage backend handler.
         */
        const setupStorage = (options) => {
            if (
                options.autosave == 'disabled'
                &&
                options.autoload == 'disabled'
            ) {
                return
            }

            let storageScope = 'weeblr_' + options.storeId + '_' + options.storage.key + '_' + tabId;
            let sharedStorageScope = 'weeblr_' + options.storeId + '_' + options.storage.key;

            if (options.storage.type == 'local') {
                storage = getSessionStorage(storageScope);
                sharedStorage = getLocalStorage(sharedStorageScope);
            } else if (options.storage.type == 'session') {
                storage = getSessionStorage(storageScope);
            }
        };

        /**
         * Load previously stored data, if any.
         */
        const autoloadFromStorage = (options, currentState) => {
            if (options.autoload == 'disabled') {
                return currentState
            }

            if (options.autoload == 'global') {
                let readData = storage.get('_data');
                if (null === readData && sharedStorage) {
                    readData = sharedStorage.get('_data');
                }

                return null === readData
                    ? options.defaultData
                    : {...options.defaultData, ...readData}
            }
        };

        /**
         * Store just updated data, either the entire state or just a single key.
         *
         * Applies exclusion list (ie do not autosave keys from the options.autosaveExclude array
         * but only when autosave is set to "per_key". When autosaving globally, all keys are loaded.
         *
         * Attention: this works for local storage. For storing to an API, need to only store the single key data
         */
        const putToStorage = (options, currentState, key) => {
            if (key && options.autosaveExclude.includes(key)) {
                return
            }

            let stateToStore = currentState;
            if (!key && options.autosaveExclude && options.autosaveExclude.length) {
                stateToStore = removeObjectProperty(
                    stateToStore,
                    options.autosaveExclude
                );
            }

            storage.set(
                '_data',
                stateToStore
            );
            sharedStorage && sharedStorage.set(
                '_data',
                stateToStore
            );
        };

        /**
         * Run section
         */
        let backend = createStore$1(
            defaultData,
            storeOptions,
        );
        let overrides = {
            setupStorage: setupStorage,
            autoloadFromStorage: autoloadFromStorage,
            putToStorage: putToStorage,
        };
        backend.initStore(overrides);

        return {
            set: backend.set,
            resetInitialState: backend.resetInitialState,
            update: backend.update,
            setByKey: backend.setByKey,
            subscribe: backend.subscribe,
            subscribeToKey: backend.subscribeToKey,
            subscribeToStatus: backend.subscribeToStatus,
            subscribeToChange: backend.subscribeToChange,
            dump: backend.dump
        }
    };

    // Utilities

    // only persist information when we have a user id
    const createCustomStore = (busyState, appState) => {

        let storageOptions = config.user.id
            ? {
                busyState,
                appState,

                storeId: 'forlogs.main',
                isAppBlocker: false,
                debounce: 0,
                autosave: 'global',
                autoload: 'global',
                storage: {
                    type: 'local',
                    key: config.user.id
                },
                autosaveExclude: [
                    'total',
                    'errorsCount',
                    'selected',
                    'allSelected',
                    'nextAction',
                    'folders',
                    'filteredFolders',
                    'files',
                    'filteredFiles',
                    'viewing',
                    'editing',
                    'dialog',
                    'confirmationDialog',
                    'alert'
                ],
                validateState: (newState) => {
                    let filteredState = newState;
                    filteredState.search = newState.search
                        ? newState.search.replace(/[`'"<>]/gi, '')
                        : newState.search;
                    return {
                        newState: filteredState,
                        isValid: true,
                        errors: []
                    }
                }
            }
            : {};

        let store = createStore(
            {
                perPage: 10,
                currentPage: 1,
                search: '',
                orderBy: '',
                status: 'ready',
                filterFolder: '',
                filterSort: '',
                folders: [],
                filteredFolders: [],
                files: [],
                filteredFiles: [],
                total: 0,
                errorsCount: 0,
                selected: [],
                allSelected: false,
                viewing: '',
                editing: '',
                dialog: null,
                confirmationDialog: null,
                alert: false,
                nextAction: '',
            },
            storageOptions
        );

        return store
    };

    let store;
    const forlogsStateFactory = (busyState, appState) => store
        ? store
        : store = createCustomStore(busyState, appState);

    const searchFiles = writable('');

    /* src\app\admin\Forlogs.svelte generated by Svelte v3.41.0 */
    const file = "src\\app\\admin\\Forlogs.svelte";

    // (38:4) {#if $appState.trapClicks}
    function create_if_block_2(ctx) {
    	let div;

    	const block = {
    		c: function create() {
    			div = element("div");
    			attr_dev(div, "id", "wb-clickaway");
    			attr_dev(div, "class", "wb-absolute wb-z-30 wb-h-full wb-w-full wb-m-0 wb-p-0 wb-bg-gray-200 wb-opacity-60 wb-border-0 wb-cursor-default");
    			add_location(div, file, 38, 8, 1740);
    		},
    		m: function mount(target, anchor) {
    			insert_dev(target, div, anchor);
    		},
    		d: function destroy(detaching) {
    			if (detaching) detach_dev(div);
    		}
    	};

    	dispatch_dev("SvelteRegisterBlock", {
    		block,
    		id: create_if_block_2.name,
    		type: "if",
    		source: "(38:4) {#if $appState.trapClicks}",
    		ctx
    	});

    	return block;
    }

    // (42:4) {#if $appState.loggedIn == 'yes'}
    function create_if_block_1(ctx) {
    	let notificationlist;
    	let current;

    	notificationlist = new NotificationList({
    			props: {
    				type: "oneoff",
    				store: oneoffNotifications
    			},
    			$$inline: true
    		});

    	const block = {
    		c: function create() {
    			create_component(notificationlist.$$.fragment);
    		},
    		m: function mount(target, anchor) {
    			mount_component(notificationlist, target, anchor);
    			current = true;
    		},
    		p: noop$1,
    		i: function intro(local) {
    			if (current) return;
    			transition_in(notificationlist.$$.fragment, local);
    			current = true;
    		},
    		o: function outro(local) {
    			transition_out(notificationlist.$$.fragment, local);
    			current = false;
    		},
    		d: function destroy(detaching) {
    			destroy_component(notificationlist, detaching);
    		}
    	};

    	dispatch_dev("SvelteRegisterBlock", {
    		block,
    		id: create_if_block_1.name,
    		type: "if",
    		source: "(42:4) {#if $appState.loggedIn == 'yes'}",
    		ctx
    	});

    	return block;
    }

    // (49:4) {:else}
    function create_else_block(ctx) {
    	let section;
    	let app;
    	let current;
    	app = new App({ $$inline: true });

    	const block = {
    		c: function create() {
    			section = element("section");
    			create_component(app.$$.fragment);
    			attr_dev(section, "class", "wb-light wb-accent-blue sm:wb-pb-0 wb-pb-24 wb-text-base");
    			add_location(section, file, 49, 8, 2193);
    		},
    		m: function mount(target, anchor) {
    			insert_dev(target, section, anchor);
    			mount_component(app, section, null);
    			current = true;
    		},
    		i: function intro(local) {
    			if (current) return;
    			transition_in(app.$$.fragment, local);
    			current = true;
    		},
    		o: function outro(local) {
    			transition_out(app.$$.fragment, local);
    			current = false;
    		},
    		d: function destroy(detaching) {
    			if (detaching) detach_dev(section);
    			destroy_component(app);
    		}
    	};

    	dispatch_dev("SvelteRegisterBlock", {
    		block,
    		id: create_else_block.name,
    		type: "else",
    		source: "(49:4) {:else}",
    		ctx
    	});

    	return block;
    }

    // (45:4) {#if config.platformVersion.major == 3}
    function create_if_block(ctx) {
    	let main;
    	let app;
    	let current;
    	app = new App({ $$inline: true });

    	const block = {
    		c: function create() {
    			main = element("main");
    			create_component(app.$$.fragment);
    			attr_dev(main, "class", "wb-light wb-accent-blue wb-text-base");
    			add_location(main, file, 45, 8, 2086);
    		},
    		m: function mount(target, anchor) {
    			insert_dev(target, main, anchor);
    			mount_component(app, main, null);
    			current = true;
    		},
    		i: function intro(local) {
    			if (current) return;
    			transition_in(app.$$.fragment, local);
    			current = true;
    		},
    		o: function outro(local) {
    			transition_out(app.$$.fragment, local);
    			current = false;
    		},
    		d: function destroy(detaching) {
    			if (detaching) detach_dev(main);
    			destroy_component(app);
    		}
    	};

    	dispatch_dev("SvelteRegisterBlock", {
    		block,
    		id: create_if_block.name,
    		type: "if",
    		source: "(45:4) {#if config.platformVersion.major == 3}",
    		ctx
    	});

    	return block;
    }

    function create_fragment(ctx) {
    	let div;
    	let t0;
    	let t1;
    	let current_block_type_index;
    	let if_block2;
    	let current;
    	let if_block0 = /*$appState*/ ctx[0].trapClicks && create_if_block_2(ctx);
    	let if_block1 = /*$appState*/ ctx[0].loggedIn == 'yes' && create_if_block_1(ctx);
    	const if_block_creators = [create_if_block, create_else_block];
    	const if_blocks = [];

    	function select_block_type(ctx, dirty) {
    		if (config.platformVersion.major == 3) return 0;
    		return 1;
    	}

    	current_block_type_index = select_block_type();
    	if_block2 = if_blocks[current_block_type_index] = if_block_creators[current_block_type_index](ctx);

    	const block = {
    		c: function create() {
    			div = element("div");
    			if (if_block0) if_block0.c();
    			t0 = space();
    			if (if_block1) if_block1.c();
    			t1 = space();
    			if_block2.c();

    			attr_dev(div, "class", "wb-relative wb-flex wb-flex-col wb-h-full wb-w-full " + (config.platformVersion.major == 4
    			? 'wb-bg-white wb-p-4'
    			: ''));

    			add_location(div, file, 36, 0, 1574);
    		},
    		l: function claim(nodes) {
    			throw new Error("options.hydrate only works if the component was compiled with the `hydratable: true` option");
    		},
    		m: function mount(target, anchor) {
    			insert_dev(target, div, anchor);
    			if (if_block0) if_block0.m(div, null);
    			append_dev(div, t0);
    			if (if_block1) if_block1.m(div, null);
    			append_dev(div, t1);
    			if_blocks[current_block_type_index].m(div, null);
    			current = true;
    		},
    		p: function update(ctx, [dirty]) {
    			if (/*$appState*/ ctx[0].trapClicks) {
    				if (if_block0) ; else {
    					if_block0 = create_if_block_2(ctx);
    					if_block0.c();
    					if_block0.m(div, t0);
    				}
    			} else if (if_block0) {
    				if_block0.d(1);
    				if_block0 = null;
    			}

    			if (/*$appState*/ ctx[0].loggedIn == 'yes') {
    				if (if_block1) {
    					if_block1.p(ctx, dirty);

    					if (dirty & /*$appState*/ 1) {
    						transition_in(if_block1, 1);
    					}
    				} else {
    					if_block1 = create_if_block_1(ctx);
    					if_block1.c();
    					transition_in(if_block1, 1);
    					if_block1.m(div, t1);
    				}
    			} else if (if_block1) {
    				group_outros();

    				transition_out(if_block1, 1, 1, () => {
    					if_block1 = null;
    				});

    				check_outros();
    			}
    		},
    		i: function intro(local) {
    			if (current) return;
    			transition_in(if_block1);
    			transition_in(if_block2);
    			current = true;
    		},
    		o: function outro(local) {
    			transition_out(if_block1);
    			transition_out(if_block2);
    			current = false;
    		},
    		d: function destroy(detaching) {
    			if (detaching) detach_dev(div);
    			if (if_block0) if_block0.d();
    			if (if_block1) if_block1.d();
    			if_blocks[current_block_type_index].d();
    		}
    	};

    	dispatch_dev("SvelteRegisterBlock", {
    		block,
    		id: create_fragment.name,
    		type: "component",
    		source: "",
    		ctx
    	});

    	return block;
    }

    function instance($$self, $$props, $$invalidate) {
    	let $appState;
    	validate_store(appState, 'appState');
    	component_subscribe($$self, appState, $$value => $$invalidate(0, $appState = $$value));
    	let { $$slots: slots = {}, $$scope } = $$props;
    	validate_slots('Forlogs', slots, []);
    	const forlogsState = forlogsStateFactory(busyState$1, appState);

    	// ------------------------------------------------------------------------------- /
    	// Make state available to subcomponents
    	// ------------------------------------------------------------------------------- \
    	setContext('appState', appState);

    	setContext('busyState', busyState$1);
    	setContext('forlogsState', forlogsState);
    	setContext('oneoffNotifications', oneoffNotifications);
    	setContext('config', config);
    	setContext('features', setFeatures());
    	setContext('searchFiles', searchFiles);
    	const writable_props = [];

    	Object.keys($$props).forEach(key => {
    		if (!~writable_props.indexOf(key) && key.slice(0, 2) !== '$$' && key !== 'slot') console.warn(`<Forlogs> was created with unknown prop '${key}'`);
    	});

    	$$self.$capture_state = () => ({
    		App,
    		NotificationList,
    		config,
    		setFeatures,
    		appState,
    		busyState: busyState$1,
    		forlogsStateFactory,
    		oneoffNotifications,
    		searchFiles,
    		setContext,
    		forlogsState,
    		$appState
    	});

    	return [$appState];
    }

    class Forlogs extends SvelteComponentDev {
    	constructor(options) {
    		super(options);
    		init(this, options, instance, create_fragment, safe_not_equal, {});

    		dispatch_dev("SvelteRegisterComponent", {
    			component: this,
    			tagName: "Forlogs",
    			options,
    			id: create_fragment.name
    		});
    	}
    }

    const errorImages = [
        'https://cdn.weeblr.net/dist/weeblr/img/4seo/undraw_fixing_bugs_w7gi.svg',
        'https://cdn.weeblr.net/dist/weeblr/img/4seo/undraw_bug_fixing_oc7a.svg'
    ];

    /**
     * Detect supported browsers.
     *
     * @returns {boolean}
     */
    const isSupportedBrowser = () => {
        if (window.document.documentMode) {
            // IE11
            return false
        }

        if (window.navigator.userAgent.match(/MSIE|Trident/) !== null) {
            // older IE
            return false
        }

        return true
    };

    /**
     * Builds a rawh HTML message in case of unsupported browser.
     * @returns {string}
     */
    const getUnsupportedBrowserBanner = () => {
        return `
<h2 class="mt-5 ml-5 mb-3 font-weight-bold" style="font-size: 1.2rem;">Your browser is not supported :(</h2>
<img class="m-5 mw-100" src="https://cdn.weeblr.net/dist/weeblr/img/4seo/undraw_warning_cyit.svg" width="350" alt="">
<p class="ml-5" style="max-width: 50rem;">You appear to be using a browser that's too old or does not have some features required for running this app. Please switch to using a more modern browser and try again. You can check technical requirements for this app <a href="${config.urls.helpRequirements}" target="_blank" rel="noreferrer noopener">on this page</a></p>
<p class="m-5" style="max-width: 50rem;">If you think this is wrong and the app should work in this browser, please contact <a href="https://weeblr.com" target="_blank" rel="noreferrer noopener">Weeblr</a> with details on the browser you're using. Sorry about the trouble!</p>
`
    };

    /**
     * Select one image from the list.
     *
     * @returns {string}
     */
    const pickErrorImage = () => {
        return errorImages[Math.floor(Math.random() * errorImages.length)]
    };

    /**
     * Install global error handlers.
     */
    const forLOGSGlobalErrorHandler = (e) => {
        window.removeEventListener('error', forLOGSGlobalErrorHandler);
        window.removeEventListener('unhandledrejection', forLOGSGlobalErrorHandler);
        let reason = e.reason || e.error;
        let errorText = reason && reason.stack
            ? reason.stack
            : e.reason || e.message || e;
        let details = '';
        if (e.filename && e.filename.endsWith(config.bundleFile)) {
            details = 'File: ' + e.filename + '\nLine: ' + e.lineno + '\ncolno: ' + e.colno + '\n\n';
        } else if (!errorText.includes('4LOGS')) {
            // not sure where it came from
            console.error('Looks like a javascript error happened on this page. It\s not coming from 4LOGS but likely from another script running on the page. You may want to contact the developers of the extension the file belongs to. 4LOGS will keep working now ');
            return false
        }
        details += errorText + '\n\n';

        let rendered = `
<div class="wb-flex wb-flex-col wb-p-8 wb-max-w-full wb-w-xl wb-mx-auto">
<h2 class="wb-font-semibold wb-text-2xl wb-text-center">Something went wrong :(</h2>
<img class="wb-self-center wb-my-16" src="${pickErrorImage()}" width="350" alt="">
<p>An error occured and this app cannot continue. Please contact <a href="https://weeblr.com" target="_blank" rel="noreferrer noopener">Weeblr</a> with the error message below. Sorry about the trouble!</p>
`;
        rendered += `<pre class="wb-mt-4" style="max-width: 50rem;">${details}</pre>`;

        rendered += '</div>';

        document.getElementById('forlogs_app').innerHTML = rendered;
        return true
    };

    window.addEventListener('error', forLOGSGlobalErrorHandler);
    window.addEventListener('unhandledrejection', forLOGSGlobalErrorHandler);

    /**
     * Initialize language support.
     */
    setLang(config.language);
    setObject(
        window.forLogsConfig.language,
        window.forLogsLanguageStrings
    );

    /**
     * Add custom styles to platform for better viewing.
     */
    const displayNoneElement = (elSelector) => {
        let el = document.getElementById(elSelector);
        if (el) {
            el.style.display = 'none';
        }
    };
    if (config.platformVersion.major == 3) {
        let el = document.querySelector('.subhead-collapse');
        if (el) {
            el.style.display = 'none';
        }
        displayNoneElement('content');
    }
    if (config.platformVersion.major == 4) {
        displayNoneElement('subhead');
        displayNoneElement('subhead-container');
        displayNoneElement('content');
    }
    document.body.style.overflowY = 'scroll';
    const target = document.querySelector('.container-fluid.container-main');
    target.setAttribute('id', 'forlogs_app');

    /**
     * And instantiate the app.
     */
    const app = () => {
        if (isSupportedBrowser()) {
            return new Forlogs({
                target: target
            })
        } else {
            target.innerHTML = getUnsupportedBrowserBanner();
        }
    };

    var forlogs = app();

    return forlogs;

}());
