/**
 * OS Property Locator Map - Core Functions
 * OpenStreetMap Integration with Leaflet
 */

(function() {
    'use strict';
    
    // Global variables
    var locatorMap = null;
    var mapInstance = null;
    var markers = [];
    var markerClusterGroup = null;
    var drawnItems = null;
    var drawControl = null;
    var searchCircle = null;
    var currentLocation = null;
    var config = {};
    var activeFilters = {};
    const LANG = Joomla.getOptions('com_osproperty.locator.lang');
    /**
     * Initialize Locator Map
     */
    window.OspropertyLocatorMapInit = function(userConfig) {
        config = userConfig;
        
        // Initialize map
        initMap();
        
        // Load initial properties
        if (config.properties && config.properties.length > 0) {
            displayProperties(config.properties);
        }
        
        // Setup event listeners
        setupEventListeners();
        
        // Initialize radius slider
        initRadiusSlider();
        
        // Auto-search if search type is set
        if (config.searchType === 'radius') {
            performRadiusSearch(config.center[0], config.center[1], config.defaultRadius);
        }
        
        console.log('Locator Map initialized successfully');
    };
    
    /**
     * Initialize Leaflet Map
     */
    function initMap() {
        // Create map instance
        mapInstance = L.map(config.mapId, {
            center: config.center,
            zoom: config.zoom,
            zoomControl: true,
            scrollWheelZoom: true
        });
        
        // Add OpenStreetMap tile layer
        L.tileLayer('https://{s}.tile.openstreetmap.org/{z}/{x}/{y}.png', {
            attribution: '&copy; <a href="https://www.openstreetmap.org/copyright">OpenStreetMap</a> contributors',
            maxZoom: 19
        }).addTo(mapInstance);
        
        // Initialize marker cluster group
        if (config.enableClustering) {
            markerClusterGroup = L.markerClusterGroup({
                maxClusterRadius: 50,
                spiderfyOnMaxZoom: true,
                showCoverageOnHover: false,
                zoomToBoundsOnClick: true,
                iconCreateFunction: function(cluster) {
                    var count = cluster.getChildCount();
                    var className = 'marker-cluster-';
                    
                    if (count < 10) {
                        className += 'small';
                    } else if (count < 50) {
                        className += 'medium';
                    } else {
                        className += 'large';
                    }
                    
                    return L.divIcon({
                        html: '<div><span>' + count + '</span></div>',
                        className: 'marker-cluster ' + className,
                        iconSize: L.point(40, 40)
                    });
                }
            });
            mapInstance.addLayer(markerClusterGroup);
        }
        
        // Initialize drawing tools
        if (config.enableDraw) {
            initDrawingTools();
        }
        
        // Add fullscreen control
        if (L.control.fullscreen) {
            mapInstance.addControl(new L.Control.Fullscreen({
                position: 'topright'
            }));
        }
    }
    
    /**
     * Initialize Drawing Tools
     */
    function initDrawingTools() {
        drawnItems = new L.FeatureGroup();
        mapInstance.addLayer(drawnItems);
        
        drawControl = new L.Control.Draw({
            position: 'topright',
            draw: {
                polyline: false,
                marker: false,
                circlemarker: false,
                circle: {
                    shapeOptions: {
                        color: '#007bff',
                        fillOpacity: 0.2
                    }
                },
                rectangle: {
                    shapeOptions: {
                        color: '#007bff',
                        fillOpacity: 0.2
                    }
                },
                polygon: {
                    shapeOptions: {
                        color: '#007bff',
                        fillOpacity: 0.2
                    }
                }
            },
            edit: {
                featureGroup: drawnItems,
                remove: true
            }
        });
        
        mapInstance.addControl(drawControl);
        
        // Handle draw events
        mapInstance.on(L.Draw.Event.CREATED, function(e) {
            var layer = e.layer;
            drawnItems.addLayer(layer);
            handleDrawComplete(e.layerType, layer);
        });
        
        mapInstance.on(L.Draw.Event.DELETED, function(e) {
            clearSearchResults();
        });
    }
    
    /**
     * Setup Event Listeners
     */
    function setupEventListeners() {
        // Address search
        var btnSearchAddress = document.getElementById('btn-search-address');
        if (btnSearchAddress) {
            btnSearchAddress.addEventListener('click', handleAddressSearch);
        }
        
        var searchAddressInput = document.getElementById('search_address');
        if (searchAddressInput) {
            searchAddressInput.addEventListener('keypress', function(e) {
                if (e.key === 'Enter') {
                    e.preventDefault();
                    handleAddressSearch();
                }
            });
        }
        
        // Near me button
        var btnNearMe = document.getElementById('btn-find-near-me');
        if (btnNearMe) {
            btnNearMe.addEventListener('click', handleNearMeSearch);
        }
        
        // Drawing tools buttons
        document.getElementById('btn-draw-circle')?.addEventListener('click', function() {
            new L.Draw.Circle(mapInstance, drawControl.options.draw.circle).enable();
        });
        
        document.getElementById('btn-draw-rectangle')?.addEventListener('click', function() {
            new L.Draw.Rectangle(mapInstance, drawControl.options.draw.rectangle).enable();
        });
        
        document.getElementById('btn-draw-polygon')?.addEventListener('click', function() {
            new L.Draw.Polygon(mapInstance, drawControl.options.draw.polygon).enable();
        });
        
        document.getElementById('btn-clear-drawings')?.addEventListener('click', function() {
            drawnItems.clearLayers();
            clearSearchResults();
        });
        
        // Apply filters
        var btnApplyFilters = document.getElementById('btn-apply-filters');
        if (btnApplyFilters) {
            btnApplyFilters.addEventListener('click', handleApplyFilters);
        }
        
        // Search in view
        var btnSearchInView = document.getElementById('btn-search-in-view');
        if (btnSearchInView) {
            btnSearchInView.addEventListener('click', handleSearchInView);
        }
        
        // Sort properties
        var sortSelect = document.getElementById('sort-properties');
        if (sortSelect) {
            sortSelect.addEventListener('change', handleSortChange);
        }
        
        // Property list item click
        document.addEventListener('click', function(e) {
            var propertyItem = e.target.closest('.property-list-item');
            if (propertyItem) {
                handlePropertyItemClick(propertyItem);
            }
        });
    }
    
    /**
     * Initialize Radius Slider
     */
    function initRadiusSlider() {
        var sliderElement = document.getElementById('radius-slider');
        if (!sliderElement || typeof noUiSlider === 'undefined') {
            return;
        }
        
        noUiSlider.create(sliderElement, {
            start: [config.defaultRadius],
            connect: [true, false],
            range: {
                'min': 1,
                'max': 50
            },
            step: 1,
            tooltips: false
        });
        
        sliderElement.noUiSlider.on('update', function(values, handle) {
            var radius = Math.round(values[handle]);
            document.getElementById('radius-display').textContent = radius;
            document.getElementById('search_radius').value = radius;
        });
    }
    
    /**
     * Handle Address Search
     */
    function handleAddressSearch() {
        var address = document.getElementById('search_address').value.trim();
        
        if (!address) {
            showAlert(LANG.OS_PLEASE_ENTER_ADDRESS, 'warning');
            return;
        }
        
        showLoading(true);
        
        // Geocode address
        ajaxRequest('ajax_locator_geocode_address', {address: address}, function(response) {
            showLoading(false);
            
            if (response.success) {
                var lat = parseFloat(response.lat);
                var lon = parseFloat(response.lon);
                var radius = parseFloat(document.getElementById('search_radius').value);
                
                // Update hidden fields
                document.getElementById('search_lat').value = lat;
                document.getElementById('search_long').value = lon;
                document.getElementById('search_type').value = 'radius';
                
                // Center map
                mapInstance.setView([lat, lon], 13);
                
                // Perform radius search
                performRadiusSearch(lat, lon, radius);
                
                showAlert(LANG.OS_LOCATION_FOUND + ': ' + response.display_name, 'success');
            } else {
                showAlert(response.message || LANG.OS_ADDRESS_NOT_FOUND , 'danger');
            }
        });
    }
    
    /**
     * Handle Near Me Search
     */
    function handleNearMeSearch() {
        if (!navigator.geolocation) {
            showAlert(LANG.OS_GEOLOCATION_NOT_SUPPORTED, 'danger');
            return;
        }
        
        showLoading(true);
        
        navigator.geolocation.getCurrentPosition(
            function(position) {
                var lat = position.coords.latitude;
                var lon = position.coords.longitude;
                var radius = parseFloat(document.getElementById('search_radius').value);
                
                currentLocation = [lat, lon];
                
                // Update hidden fields
                document.getElementById('search_lat').value = lat;
                document.getElementById('search_long').value = lon;
                document.getElementById('search_type').value = 'radius';
                
                // Center map
                mapInstance.setView([lat, lon], 13);

                config.keepUserLocation = true;
                
                // Perform radius search
                performRadiusSearch(lat, lon, radius);
                
                showLoading(false);
                showAlert(LANG.OS_LOCATION_DETECTED, 'success');
            },
            function(error) {
                showLoading(false);
                showAlert(LANG.OS_UNABLE_TO_RETRIEVE_LOCATION + ': ' + error.message, 'danger');
            }
        );
    }
    
    /**
     * Perform Radius Search
     */
    /**
     * Perform Radius Search (Updated with Smart Fallback)
     */
    function performRadiusSearch(lat, lon, radius) {
        showLoading(true);
        
        // Get current filters
        var filters = getCurrentFilters();
        filters.lat = lat;
        filters.lon = lon;
        filters.radius = radius;
        
        ajaxRequest('ajax_locator_search_radius', filters, function(response) {
            showLoading(false);
            
            if (response.success && response.count > 0) {
                // Properties found - display normally
                displayProperties(response.properties);
                drawSearchCircle(lat, lon, radius);
                updatePropertiesCount(response.count);
                showAlert(response.count + ' properties found within ' + radius + ' km', 'success');
            } else {
                // No properties found - Smart fallback
                handleNoPropertiesFound(lat, lon, radius);
            }
        });

    }

    /**
     * Handle No Properties Found - Smart Fallback
     */
    function handleNoPropertiesFound(userLat, userLon, searchRadius) {
        // Keep map centered on user location
        mapInstance.setView([userLat, userLon], 13);
        
        // Draw search circle to show search area
        drawSearchCircle(userLat, userLon, searchRadius);
        
        // Add user location marker
        addUserLocationMarker(userLat, userLon);
        
        // Clear properties list
        showNoPropertiesMessage();
        updatePropertiesCount(0);
        
        // Find nearest property
        showLoading(true);
        ajaxRequest('ajax_locator_find_nearest', {lat: userLat, lon: userLon}, function(response) {
            showLoading(false);
            
            if (response.success && response.nearest) {
                var nearest = response.nearest;
                var distanceKm = nearest.distance;
                
                // Determine action based on distance
                if (distanceKm > 1000) {
                    // Very far - show smart dialog
                    showSmartFallbackDialog(userLat, userLon, nearest, searchRadius);
                } else if (distanceKm > 100) {
                    // Far - offer progressive search
                    showProgressiveSearchDialog(userLat, userLon, searchRadius, distanceKm);
                } else {
                    // Medium distance - auto expand
                    showAlert('No properties within ' + searchRadius + ' km. Nearest property is ' + 
                            distanceKm + ' km away. Expanding search...', 'info');
                    performProgressiveSearch(userLat, userLon);
                }
            } else {
                // No properties at all in database
                showAlert(LANG.OS_NO_PROPERTIES_FOUND_IN_SYSTEM, 'warning');
            }
        });
    }

    /**
     * Show Smart Fallback Dialog (for very far distances)
     */
    function showSmartFallbackDialog(userLat, userLon, nearest, searchRadius) {
        // Lấy các hằng ngôn ngữ
        var LANG = Joomla.getOptions('com_osproperty.locator.lang') || {};

        var distanceKm = Math.round(nearest.distance);

        // Sử dụng hằng ngôn ngữ, thay thế %d bằng dữ liệu động
        var noPropertiesText = LANG.OS_NO_PROPERTIES_WITHIN_RADIUS
            ? LANG.OS_NO_PROPERTIES_WITHIN_RADIUS.replace('%d', searchRadius)
            : 'No properties found within ' + searchRadius + ' km radius';

        var nearestPropertyText = LANG.OS_NEAREST_PROPERTY_DISTANCE
            ? LANG.OS_NEAREST_PROPERTY_DISTANCE.replace('%d', distanceKm)
            : 'Nearest property is ' + distanceKm.toLocaleString() + ' km away';

        var viewAllQuestion = LANG.OS_VIEW_ALL_PROPERTIES_QUESTION || 'Would you like to view all available properties?';
        var stayHereText = LANG.OS_STAY_AT_MY_LOCATION || 'Stay at My Location';
        var goToPropertiesText = LANG.OS_GO_TO_PROPERTIES || 'View All Properties';

        var dialogHtml =
            '<div class="modal fade" id="smartFallbackModal" tabindex="-1" role="dialog">' +
                '<div class="modal-dialog modal-dialog-centered" role="document">' +
                    '<div class="modal-content">' +
                        '<div class="modal-header bg-warning text-white">' +
                            '<h5 class="modal-title">' +
                                '<i class="fa fa-exclamation-triangle"></i> ' + noPropertiesText +
                            '</h5>' +
                            '<button type="button" class="close text-white" data-dismiss="modal">' +
                                '<span>&times;</span>' +
                            '</button>' +
                        '</div>' +
                        '<div class="modal-body text-center">' +
                            '<p class="lead">' + noPropertiesText + '</p>' +
                            '<p>' + nearestPropertyText + '</p>' +
                            '<hr>' +
                            '<p class="text-muted">' + viewAllQuestion + '</p>' +
                        '</div>' +
                        '<div class="modal-footer justify-content-center">' +
                            '<button type="button" class="btn btn-secondary" data-dismiss="modal">' +
                                '<i class="fa fa-map-marker"></i> ' + stayHereText +
                            '</button>' +
                            '<button type="button" class="btn btn-primary" id="btn-go-to-properties">' +
                                '<i class="fa fa-home"></i> ' + goToPropertiesText +
                            '</button>' +
                        '</div>' +
                    '</div>' +
                '</div>' +
            '</div>';

        // Remove existing modal if any
        var existingModal = document.getElementById('smartFallbackModal');
        if (existingModal) {
            existingModal.remove();
        }

        // Add modal to body
        document.body.insertAdjacentHTML('beforeend', dialogHtml);

        // Show modal
        if (typeof $ !== 'undefined' && $.fn.modal) {
            $('#smartFallbackModal').modal('show');

            // Handle "Go to Properties" button
            document.getElementById('btn-go-to-properties').addEventListener('click', function() {
                $('#smartFallbackModal').modal('hide');
                goToPropertiesCenter();
            });
        }
    }


    /**
     * Show Progressive Search Dialog (for medium distances)
     */
    function showProgressiveSearchDialog(userLat, userLon, currentRadius, nearestDistance) {
        // Lấy các hằng ngôn ngữ từ Joomla options
        var LANG = Joomla.getOptions('com_osproperty.locator.lang') || {};

        // Sử dụng hằng ngôn ngữ, thay thế %d bằng dữ liệu động
        var noPropertiesText = LANG.OS_NO_PROPERTIES_WITHIN_RADIUS
            ? LANG.OS_NO_PROPERTIES_WITHIN_RADIUS.replace('%d', currentRadius)
            : 'No properties found within ' + currentRadius + ' km.';

        var nearestPropertyText = LANG.OS_NEAREST_PROPERTY_DISTANCE
            ? LANG.OS_NEAREST_PROPERTY_DISTANCE.replace('%d', Math.round(nearestDistance))
            : 'Nearest property is ' + Math.round(nearestDistance) + ' km away.';

        var expandSearchQuestion = LANG.OS_EXPAND_SEARCH_QUESTION || 'Would you like to automatically expand the search radius?';
        var noStayHereText = LANG.OS_NO_STAY_HERE || 'No, Stay Here';
        var expandSearchText = LANG.OS_EXPAND_SEARCH || 'Yes, Expand Search';

        var dialogHtml =
            '<div class="modal fade" id="progressiveSearchModal" tabindex="-1" role="dialog">' +
                '<div class="modal-dialog modal-dialog-centered" role="document">' +
                    '<div class="modal-content">' +
                        '<div class="modal-header bg-info text-white">' +
                            '<h5 class="modal-title">' +
                                '<i class="fa fa-search-plus"></i> ' + expandSearchQuestion +
                            '</h5>' +
                            '<button type="button" class="close text-white" data-dismiss="modal">' +
                                '<span>&times;</span>' +
                            '</button>' +
                        '</div>' +
                        '<div class="modal-body text-center">' +
                            '<p>' + noPropertiesText + '</p>' +
                            '<p>' + nearestPropertyText + '</p>' +
                            '<hr>' +
                            '<p class="text-muted">' + expandSearchQuestion + '</p>' +
                        '</div>' +
                        '<div class="modal-footer justify-content-center">' +
                            '<button type="button" class="btn btn-secondary" data-dismiss="modal">' +
                                '<i class="fa fa-times"></i> ' + noStayHereText +
                            '</button>' +
                            '<button type="button" class="btn btn-success" id="btn-expand-search">' +
                                '<i class="fa fa-expand"></i> ' + expandSearchText +
                            '</button>' +
                        '</div>' +
                    '</div>' +
                '</div>' +
            '</div>';

        // Remove existing modal if any
        var existingModal = document.getElementById('progressiveSearchModal');
        if (existingModal) {
            existingModal.remove();
        }

        // Add modal to body
        document.body.insertAdjacentHTML('beforeend', dialogHtml);

        // Show modal
        if (typeof $ !== 'undefined' && $.fn.modal) {
            $('#progressiveSearchModal').modal('show');

            // Handle "Expand Search" button
            document.getElementById('btn-expand-search').addEventListener('click', function() {
                $('#progressiveSearchModal').modal('hide');
                performProgressiveSearch(userLat, userLon);
            });
        }
    }


    /**
     * Perform Progressive Radius Search
     */
    function performProgressiveSearch(lat, lon) {
        showLoading(true);

        // Lấy các hằng ngôn ngữ
        var LANG = Joomla.getOptions('com_osproperty.locator.lang') || {};

        // Get current filters
        var filters = getCurrentFilters();
        filters.lat = lat;
        filters.lon = lon;
        filters.max_radius = 100; // Max 100km

        ajaxRequest('ajax_locator_progressive_search', filters, function(response) {
            showLoading(false);

            if (response.success && response.count > 0) {
                // Properties found at expanded radius
                displayProperties(response.properties);
                drawSearchCircle(lat, lon, response.radius_used);
                updatePropertiesCount(response.count);

                // Sử dụng hằng ngôn ngữ, thay thế %d
                var foundText = LANG.OS_PROPERTIES_FOUND_EXPANDED
                    ? LANG.OS_PROPERTIES_FOUND_EXPANDED
                        .replace('%d', response.count)
                        .replace('%d', response.radius_used)
                    : (response.count + ' properties found by expanding search to ' +
                    response.radius_used + ' km radius');
                showAlert(foundText, 'success');
            } else {
                // Still no properties even at max radius
                var noFoundText = LANG.OS_NO_PROPERTIES_FOUND_MAX_RADIUS
                    ? LANG.OS_NO_PROPERTIES_FOUND_MAX_RADIUS.replace('%d', response.radius_checked)
                    : ('No properties found within ' + response.radius_checked + ' km. Try viewing all properties instead.');
                showAlert(noFoundText, 'warning');

                // Offer to go to properties center
                setTimeout(function() {
                    var confirmText = LANG.OS_VIEW_ALL_PROPERTIES_QUESTION || 'Would you like to view all available properties?';
                    if (confirm(confirmText)) {
                        goToPropertiesCenter();
                    }
                }, 2000);
            }
        });
    }


    /**
     * Go to Properties Center (show all properties)
     */
    function goToPropertiesCenter() {
        showLoading(true);
        
        ajaxRequest('ajax_locator_get_center', {}, function(response) {
            if (response.success && response.center) {
                var center = response.center;
                
                // Move map to properties center
                mapInstance.setView([center.lat, center.lon], 10);
                
                // Load all properties in view
                handleSearchInView();
                
                showAlert(
                    'Showing ' + center.total_properties + ' available properties', 
                    'info'
                );
            } else {
                showAlert('Unable to load properties', 'danger');
            }
            showLoading(false);
        });
    }

    /**
     * Add User Location Marker
     */
    function addUserLocationMarker(lat, lon) {
        var LANG = Joomla.getOptions('com_osproperty.locator.lang') || {};

        var userIcon = L.divIcon({
            html: '<div style="background-color: #007bff; width: 20px; height: 20px; border-radius: 50%; border: 3px solid white; box-shadow: 0 2px 8px rgba(0,0,0,0.3);"></div>',
            className: 'user-location-marker',
            iconSize: [20, 20],
            iconAnchor: [10, 10]
        });

        var marker = L.marker([lat, lon], {
            icon: userIcon,
            title: LANG.OS_YOUR_LOCATION || 'Your Location'
        }).addTo(mapInstance);

        marker.bindPopup(
            '<div style="text-align: center; padding: 10px;">' +
            '<strong>' + (LANG.OS_YOUR_LOCATION || 'Your Location') + '</strong><br>' +
            '<small>' +
                (LANG.OS_LAT || 'Lat') + ': ' + lat.toFixed(6) + '<br>' +
                (LANG.OS_LON || 'Lon') + ': ' + lon.toFixed(6) +
            '</small>' +
            '</div>'
        );
    }


    /**
     * Handle Search in View
     */
    function handleSearchInView() {
        var LANG = Joomla.getOptions('com_osproperty.locator.lang') || {};
        var bounds = mapInstance.getBounds();

        showLoading(true);

        // Get current filters
        var filters = getCurrentFilters();
        filters.north = bounds.getNorth();
        filters.south = bounds.getSouth();
        filters.east = bounds.getEast();
        filters.west = bounds.getWest();

        ajaxRequest('ajax_locator_search_bounds', filters, function(response) {
            showLoading(false);

            if (response.success) {
                displayProperties(response.properties);
                updatePropertiesCount(response.count);

                // Format message with count
                var foundMsg = (LANG.OS_PROPERTIES_FOUND || '%d properties found in this area')
                    .replace('%d', response.count);

                showAlert(foundMsg, 'success');
            } else {
                showAlert(response.message || (LANG.OS_SEARCH_FAILED || 'Search failed'), 'danger');
            }
        });
    }

    
    /**
     * Handle Draw Complete
     */
    function handleDrawComplete(type, layer) {
        showLoading(true);
        
        var coordinates = [];
        
        if (type === 'circle') {
            // Convert circle to polygon approximation
            var center = layer.getLatLng();
            var radius = layer.getRadius() / 1000; // Convert to km
            performRadiusSearch(center.lat, center.lng, radius);
            return;
        } else if (type === 'rectangle') {
            var bounds = layer.getBounds();
            coordinates = [
                [bounds.getNorth(), bounds.getWest()],
                [bounds.getNorth(), bounds.getEast()],
                [bounds.getSouth(), bounds.getEast()],
                [bounds.getSouth(), bounds.getWest()]
            ];
        } else if (type === 'polygon') {
            var latlngs = layer.getLatLngs()[0];
            latlngs.forEach(function(latlng) {
                coordinates.push([latlng.lat, latlng.lng]);
            });
        }
        
        // Get current filters
        var filters = getCurrentFilters();
        filters.coordinates = JSON.stringify(coordinates);
        
        ajaxRequest('ajax_locator_search_polygon', filters, function(response) {
            showLoading(false);
            
            if (response.success) {
                displayProperties(response.properties);
                updatePropertiesCount(response.count);
                showAlert(response.count + ' properties found in drawn area', 'success');
            } else {
                showAlert(response.message || 'Search failed', 'danger');
            }
        });
    }
    
    /**
     * Handle Apply Filters
     */
    function handleApplyFilters() {
        // Get current search type
        var searchType = document.getElementById('search_type').value;
        
        if (searchType === 'radius') {
            var lat = parseFloat(document.getElementById('search_lat').value);
            var lon = parseFloat(document.getElementById('search_long').value);
            var radius = parseFloat(document.getElementById('search_radius').value);
            performRadiusSearch(lat, lon, radius);
        } else {
            handleSearchInView();
        }
    }
    
    /**
     * Handle Sort Change
     */
    function handleSortChange(e) {
        var sortBy = e.target.value;
        var propertiesList = document.getElementById('properties-list');
        var items = Array.from(propertiesList.querySelectorAll('.property-list-item'));
        
        items.sort(function(a, b) {
            switch(sortBy) {
                case 'distance':
                    var distA = parseFloat(a.querySelector('.property-item-distance')?.textContent) || 999999;
                    var distB = parseFloat(b.querySelector('.property-item-distance')?.textContent) || 999999;
                    return distA - distB;
                    
                case 'price_asc':
                    var priceA = parseFloat(a.dataset.price) || 0;
                    var priceB = parseFloat(b.dataset.price) || 0;
                    return priceA - priceB;
                    
                case 'price_desc':
                    var priceA = parseFloat(a.dataset.price) || 0;
                    var priceB = parseFloat(b.dataset.price) || 0;
                    return priceB - priceA;
                    
                case 'newest':
                    var dateA = new Date(a.dataset.date || 0);
                    var dateB = new Date(b.dataset.date || 0);
                    return dateB - dateA;
                    
                default:
                    return 0;
            }
        });
        
        // Re-append sorted items
        propertiesList.innerHTML = '';
        items.forEach(function(item) {
            propertiesList.appendChild(item);
        });
    }
    
    /**
     * Handle Property List Item Click
     */
    function handlePropertyItemClick(item) {
        var propertyId = parseInt(item.dataset.propertyId);
        var lat = parseFloat(item.dataset.lat);
        var lng = parseFloat(item.dataset.lng);
        
        // Remove active class from all items
        document.querySelectorAll('.property-list-item').forEach(function(el) {
            el.classList.remove('active');
        });
        
        // Add active class to clicked item
        item.classList.add('active');
        
        // Center map on property
        mapInstance.setView([lat, lng], 16);
        
        // Find and open marker popup
        markers.forEach(function(marker) {
            if (marker.propertyId === propertyId) {
                marker.openPopup();
            }
        });
        
        // Scroll item into view
        item.scrollIntoView({ behavior: 'smooth', block: 'nearest' });
    }
    
    /**
     * Display Properties on Map and List
     */
    function displayProperties(properties) {
        // Clear existing markers
        clearMarkers();
        
        if (!properties || properties.length === 0) {
            showNoPropertiesMessage();
            return;
        }
        
        // Add markers to map
        properties.forEach(function(property) {
            addPropertyMarker(property);
        });
        
        // Update list
        updatePropertiesList(properties);
        
        // Fit map bounds to show all markers
        // ONLY if this is NOT from a user location search
        if (markers.length > 0 && !config.keepUserLocation) {
            var group = new L.featureGroup(markers);
            mapInstance.fitBounds(group.getBounds().pad(0.1));
        }
        // Reset flag
        config.keepUserLocation = false;



    }
    
    /**
     * Add Property Marker to Map
     */
    function addPropertyMarker(property) {
        // Create custom icon
        var iconHtml = '<div class="custom-marker' + 
                       (property.featured ? ' marker-featured' : '') + 
                       '" data-property-id="' + property.id + '">' +
                       '<i class="fa fa-home"></i>' +
                       '</div>';
        
        var customIcon = L.divIcon({
            html: iconHtml,
            className: 'custom-marker-wrapper',
            iconSize: [30, 30],
            iconAnchor: [15, 15],
            popupAnchor: [0, -15]
        });
        
        // Create marker
        var marker = L.marker([property.lat, property.lng], {
            icon: customIcon,
            title: property.title
        });
        
        // Store property ID
        marker.propertyId = property.id;
        
        // Create popup content
        var popupContent = createPopupContent(property);
        marker.bindPopup(popupContent, {
            maxWidth: 320,
            minWidth: 280
        });
        
        // Add click event
        marker.on('click', function() {
            highlightPropertyInList(property.id);
        });
        
        // Add to cluster group or directly to map
        if (config.enableClustering && markerClusterGroup) {
            markerClusterGroup.addLayer(marker);
        } else {
            marker.addTo(mapInstance);
        }
        
        markers.push(marker);
    }
    
    /**
     * Create Popup Content for Property
     */
    function createPopupContent(property) {
        var LANG = Joomla.getOptions('com_osproperty.locator.lang') || {};
        var html = '<div class="osproperty-map-popup">';
        
        // Image
        html += '<div class="popup-image">';
        html += '<img src="' + escapeHtml(property.image) + '" alt="' + escapeHtml(property.title) + '">';
        if (property.featured) {
            html += '<span class="badge badge-featured">' + (LANG.OS_FEATURED || 'Featured') + '</span>';
        }
        html += '</div>';
        
        // Content
        html += '<div class="popup-content">';
        
        // Title
        html += '<h4 class="popup-title">';
        html += '<a href="' + escapeHtml(property.url) + '" target="_blank">';
        html += escapeHtml(property.title);
        html += '</a>';
        html += '</h4>';
        
        // Price
        html += '<div class="popup-price">';
        html += '<strong>' + property.price + '</strong>';
        html += '</div>';
        
        // Address
        html += '<div class="popup-address">';
        html += '<i class="fa fa-map-marker"></i> ';
        html += escapeHtml(property.address);
        html += '</div>';
        
        // Meta
        html += '<div class="popup-meta">';
        if (property.bedrooms > 0) {
            html += '<span class="meta-item">';
            html += '<i class="fa fa-bed"></i> ' + property.bedrooms + ' ' + (LANG.OS_BEDS || 'Beds');
            html += '</span>';
        }
        if (property.bathrooms > 0) {
            html += '<span class="meta-item">';
            html += '<i class="fa fa-bath"></i> ' + property.bathrooms + ' ' + (LANG.OS_BATHS || 'Baths');
            html += '</span>';
        }
        if (property.area > 0) {
            html += '<span class="meta-item">';
            html += '<i class="fa fa-arrows-alt"></i> ' + property.area + ' ' + property.area_unit;
            html += '</span>';
        }
        html += '</div>';
        
        // Distance
        if (property.distance !== null && property.distance !== undefined) {
            html += '<div class="popup-distance">';
            html += '<i class="fa fa-location-arrow"></i> ';
            html += property.distance + ' ' + (LANG.OS_AWAY || 'km away');
            html += '</div>';
        }
        
        // Actions
        html += '<div class="popup-actions">';
        html += '<a href="' + escapeHtml(property.url) + '" class="btn btn-primary btn-sm" target="_blank">';
        html += '<i class="fa fa-eye"></i> ' + (LANG.OS_VIEW_DETAILS || 'View Details');
        html += '</a>';
        html += '</div>';
        
        html += '</div>'; // .popup-content
        html += '</div>'; // .osproperty-map-popup
        
        return html;
    }

    
    /**
     * Update Properties List in Sidebar
     */
    function updatePropertiesList(properties) {
        var LANG = Joomla.getOptions('com_osproperty.locator.lang') || {};
        var listContainer = document.getElementById('properties-list');
        
        if (!properties || properties.length === 0) {
            listContainer.innerHTML = 
                '<div class="no-properties-message text-center py-5">' +
                    '<i class="fa fa-search fa-3x text-muted mb-3"></i>' +
                    '<p>' + (LANG.OS_NO_PROPERTIES_FOUND || 'No properties found') + '</p>' +
                    '<p class="text-muted">' + (LANG.OS_TRY_ADJUST_SEARCH || 'Try adjusting your search criteria') + '</p>' +
                '</div>';
            return;
        }
        
        var html = '';
        properties.forEach(function(property) {
            html += createPropertyListItem(property);
        });
        
        listContainer.innerHTML = html;
    }

    
    /**
     * Create Property List Item HTML
     */
    function createPropertyListItem(property) {
        var html = '<div class="property-list-item" ' +
                   'data-property-id="' + property.id + '" ' +
                   'data-lat="' + property.lat + '" ' +
                   'data-lng="' + property.lng + '" ' +
                   'data-price="' + property.price_raw + '">';
        
        // Image
        html += '<div class="property-item-image">';
        html += '<img src="' + escapeHtml(property.image) + '" alt="' + escapeHtml(property.title) + '">';
        if (property.featured) {
            html += '<span class="badge badge-warning badge-featured">Featured</span>';
        }
        html += '</div>';
        
        // Content
        html += '<div class="property-item-content">';
        
        // Title
        html += '<h5 class="property-item-title">';
        html += '<a href="' + escapeHtml(property.url) + '">';
        html += escapeHtml(property.title);
        html += '</a>';
        html += '</h5>';
        
        // Price
        html += '<div class="property-item-price">';
        html += '<strong>' + property.price + '</strong>';
        html += '</div>';
        
        // Address
        html += '<div class="property-item-address">';
        html += '<i class="fa fa-map-marker"></i> ';
        var address = property.address.length > 50 ? property.address.substring(0, 50) + '...' : property.address;
        html += escapeHtml(address);
        html += '</div>';
        
        // Meta
        html += '<div class="property-item-meta">';
        if (property.bedrooms > 0) {
            html += '<span><i class="fa fa-bed"></i> ' + property.bedrooms + '</span>';
        }
        if (property.bathrooms > 0) {
            html += '<span><i class="fa fa-bath"></i> ' + property.bathrooms + '</span>';
        }
        if (property.area > 0) {
            html += '<span><i class="fa fa-arrows-alt"></i> ' + property.area + ' ' + property.area_unit + '</span>';
        }
        html += '</div>';
        
        // Distance
        if (property.distance !== null && property.distance !== undefined) {
            html += '<div class="property-item-distance">';
            html += '<i class="fa fa-location-arrow"></i> ';
            html += property.distance + ' km away';
            html += '</div>';
        }
        
        html += '</div>'; // .property-item-content
        html += '</div>'; // .property-list-item
        
        return html;
    }
    
    /**
     * Highlight Property in List
     */
    function highlightPropertyInList(propertyId) {
        // Remove active class from all items
        document.querySelectorAll('.property-list-item').forEach(function(el) {
            el.classList.remove('active');
        });
        
        // Add active class to matching item
        var item = document.querySelector('.property-list-item[data-property-id="' + propertyId + '"]');
        if (item) {
            item.classList.add('active');
            item.scrollIntoView({ behavior: 'smooth', block: 'nearest' });
        }
    }
    
    /**
     * Draw Search Circle on Map
     */
    function drawSearchCircle(lat, lon, radius) {
        // Remove existing circle
        if (searchCircle) {
            mapInstance.removeLayer(searchCircle);
        }
        
        // Draw new circle
        searchCircle = L.circle([lat, lon], {
            color: '#007bff',
            fillColor: '#007bff',
            fillOpacity: 0.1,
            radius: radius * 1000 // Convert km to meters
        }).addTo(mapInstance);
        
        // Add center marker
        L.marker([lat, lon], {
            icon: L.divIcon({
                html: '<div style="background-color: #dc3545; width: 20px; height: 20px; border-radius: 50%; border: 3px solid white;"></div>',
                className: 'search-center-marker',
                iconSize: [20, 20],
                iconAnchor: [10, 10]
            })
        }).addTo(mapInstance);
    }
    
    /**
     * Clear Markers from Map
     */
    function clearMarkers() {
        if (config.enableClustering && markerClusterGroup) {
            markerClusterGroup.clearLayers();
        } else {
            markers.forEach(function(marker) {
                mapInstance.removeLayer(marker);
            });
        }
        markers = [];
    }
    
    /**
     * Clear Search Results
     */
    function clearSearchResults() {
        clearMarkers();
        
        if (searchCircle) {
            mapInstance.removeLayer(searchCircle);
            searchCircle = null;
        }
        
        document.getElementById('properties-list').innerHTML = '';
        updatePropertiesCount(0);
    }
    
    /**
     * Get Current Filters
     */
    function getCurrentFilters() {
        return {
            type_id: document.getElementById('filter_type_id')?.value || '',
            category_id: document.getElementById('filter_category_id')?.value || '',
            price_min: document.getElementById('filter_price_min')?.value || '',
            price_max: document.getElementById('filter_price_max')?.value || '',
            bedrooms: document.getElementById('filter_bedrooms')?.value || '',
            bathrooms: document.getElementById('filter_bathrooms')?.value || ''
        };
    }
    
    /**
     * Update Properties Count Badge
     */
    function updatePropertiesCount(count) {
        var badge = document.getElementById('properties-count-badge');
        if (badge) {
            badge.textContent = count + ' Properties';
        }
    }
    
    /**
     * Show/Hide Loading Overlay
     */
    function showLoading(show) {
        var overlay = document.getElementById('map-loading-overlay');
        if (overlay) {
            overlay.style.display = show ? 'flex' : 'none';
        }
    }
    
    /**
     * Show No Properties Message
     */
    function showNoPropertiesMessage() {
        var LANG = Joomla.getOptions('com_osproperty.locator.lang') || {};

        var listContainer = document.getElementById('properties-list');
        listContainer.innerHTML = '<div class="no-properties-message text-center py-5">' +
            '<i class="fa fa-search fa-3x text-muted mb-3"></i>' +
            '<p>' + (LANG.OS_NO_PROPERTIES_FOUND || 'No properties found') + '</p>' +
            '<p class="text-muted">' + (LANG.OS_TRY_ADJUST_SEARCH || 'Try adjusting your search criteria') + '</p>' +
            '</div>';
        updatePropertiesCount(0);
    }

    
    /**
     * Show Alert Message
     */
    function showAlert(message, type) {
        var alertHtml = '<div class="alert alert-' + type + ' alert-dismissible fade show alert-floating" role="alert">' +
                        message +
                        '<button type="button" class="close" data-dismiss="alert" aria-label="Close">' +
                        '<span aria-hidden="true">&times;</span>' +
                        '</button>' +
                        '</div>';
        
        var alertContainer = document.createElement('div');
        alertContainer.innerHTML = alertHtml;
        document.body.appendChild(alertContainer);
        
        // Auto remove after 5 seconds
        setTimeout(function() {
            alertContainer.remove();
        }, 5000);
    }
    
    /**
     * AJAX Request Helper
     */
    function ajaxRequest(task, data, callback) {
        data.task = task;
        data.format = 'raw';

        var url = config.ajaxUrl;
        var params = new URLSearchParams(data).toString();
        var fullUrl = url + '&' + params;

        // Show the URL for direct checking
        console.log("AJAX URL:", fullUrl);

        fetch(fullUrl, {
            method: 'GET',
            headers: {
                'Content-Type': 'application/json'
            }
        })
        .then(function(response) {
            return response.json();
        })
        .then(function(data) {
            callback(data);
        })
        .catch(function(error) {
            console.error('AJAX Error:', error);
            showAlert('An error occurred. Please try again.', 'danger');
            showLoading(false);
        });
    }

    
    /**
     * Escape HTML
     */
    function escapeHtml(text) {
        var map = {
            '&': '&amp;',
            '<': '&lt;',
            '>': '&gt;',
            '"': '&quot;',
            "'": '&#039;'
        };
        return text.replace(/[&<>"']/g, function(m) { return map[m]; });
    }
    
    // Export for global access
    window.OspropertyLocatorMap = {
        getMapInstance: function() { return mapInstance; },
        getMarkers: function() { return markers; },
        clearMarkers: clearMarkers,
        displayProperties: displayProperties,
        performRadiusSearch: performRadiusSearch
    };
    
})();
