<?php
/**
 * @package     OSL
 * @subpackage  View
 *
 * @copyright   Copyright (C) 2016 Ossolution Team, Inc. All rights reserved.
 * @license     GNU General Public License version 2 or later; see LICENSE
 */

namespace OSL\View;

use Joomla\CMS\HTML\HTMLHelper;
use Joomla\CMS\Language\Text;
use Joomla\CMS\Toolbar\ToolbarHelper;
use OSL\Model\AdminModel;
use OSL\Model\State;
use OSL\Utils\Helper;
use OSL\Utils\Html as HtmlUtils;

/**
 * Class ItemView
 *
 * Joomla CMS Item View Class. This class is used to display details information of an item
 * or display form allow add/editing items
 *
 * @property AdminModel $model
 *
 */
class ItemView extends HtmlView
{
	use FormView;

	/**
	 * The model state.
	 *
	 * @var State
	 */
	protected $state;

	/**
	 * The record which is being added/edited
	 *
	 * @var Object
	 */
	protected $item;

	/**
	 * The array which keeps list of "list" options which will be displayed on the form
	 *
	 * @var array $lists
	 */
	protected $lists = [];

	/**
	 * Active languages use on the site
	 *
	 * @var array
	 */
	protected $languages = [];

	/**
	 * Method to prepare all the data for the view before it is displayed
	 */
	protected function beforeRender()
	{
		$this->state = $this->model->getState();
		$this->item  = $this->model->getData();

		if (property_exists($this->item, 'published'))
		{
			$this->lists['published'] = HtmlUtils::getBooleanInput('published', $this->item->published);
		}

		if (property_exists($this->item, 'access'))
		{
			$this->lists['access'] = HTMLHelper::_('access.level', 'access', $this->item->access, 'class="form-select"', false);
		}

		if (property_exists($this->item, 'language'))
		{
			$this->lists['language'] = HTMLHelper::_(
				'select.genericlist',
				HTMLHelper::_('contentlanguage.existing', true, true),
				'language',
				'class="form-select"',
				'value',
				'text',
				$this->item->language
			);
		}

		$this->languages = Helper::getLanguages();

		if ($this->isAdminView)
		{
			$this->addToolbar();
		}
	}

	/**
	 * Add toolbar buttons for add/edit item form
	 */
	protected function addToolbar()
	{
		$helperClass = $this->container->componentNamespace . '\\Site\\Helper\\Helper';

		if (is_callable($helperClass . '::getActions'))
		{
			$canDo = call_user_func([$helperClass, 'getActions'], $this->name, $this->state);
		}
		else
		{
			$canDo = call_user_func(['\\OSL\Utils\\Helper', 'getActions'], $this->container->option);
		}

		if ($this->item->id)
		{
			$toolbarTitle = $this->container->languagePrefix . '_' . $this->name . '_EDIT';
		}
		else
		{
			$toolbarTitle = $this->container->languagePrefix . '_' . $this->name . '_NEW';
		}

		ToolbarHelper::title(Text::_(strtoupper($toolbarTitle)));

		if (($canDo->get('core.edit') || ($canDo->get('core.create'))) && !in_array('save', $this->hideButtons))
		{
			ToolbarHelper::apply('apply', 'JTOOLBAR_APPLY');
			ToolbarHelper::save('save', 'JTOOLBAR_SAVE');
		}

		if ($canDo->get('core.create') && !in_array('save2new', $this->hideButtons))
		{
			ToolbarHelper::custom('save2new', 'save-new.png', 'save-new_f2.png', 'JTOOLBAR_SAVE_AND_NEW', false);
		}

		if ($this->item->id && $canDo->get('core.create') && !in_array('save2copy', $this->hideButtons))
		{
			ToolbarHelper::save2copy('save2copy');
		}

		if ($this->item->id)
		{
			ToolbarHelper::cancel('cancel', 'JTOOLBAR_CLOSE');
		}
		else
		{
			ToolbarHelper::cancel('cancel', 'JTOOLBAR_CANCEL');
		}
	}
}