<?php
/**
 * @package     OSL
 * @subpackage  View
 *
 * @copyright   Copyright (C) 2016 Ossolution Team, Inc. All rights reserved.
 * @license     GNU General Public License version 2 or later; see LICENSE
 */

namespace OSL\View;

use Joomla\CMS\Document\HtmlDocument;
use Joomla\CMS\Factory;
use Joomla\CMS\Uri\Uri;
use Joomla\Filesystem\Path;
use Joomla\Registry\Registry;
use OSL\Container\Container;
use OSL\Input\Input;
use RuntimeException;

/**
 * Class HtmlView
 *
 * @property-read HtmlDocument $document
 * @property-read Input        $input
 *
 */
class HtmlView extends AbstractView
{
	/**
	 * The view layout.
	 *
	 * @var string
	 */
	protected $layout = 'default';

	/**
	 * The paths queue.
	 *
	 * @var array
	 */
	protected $paths = [];

	/**
	 * ID of the active menu item, use as default Itemid for links in the view
	 *
	 * @var int
	 */
	public $Itemid;

	/**
	 * This is a front-end or back-end view.
	 * We need this field to determine whether we need to addToolbar or build the filter
	 *
	 * @var boolean
	 */
	protected $isAdminView = false;

	/**
	 * Options to allow hide default toolbar buttons from backend view
	 *
	 * @var array
	 */
	protected $hideButtons = [];

	/**
	 * Html document object, used to process meta data for menu item in frontend
	 *
	 * @var HtmlDocument
	 */
	protected $document = null;

	/**
	 * Relative base URI
	 *
	 * @var string
	 */
	protected $relativeRootUri = null;

	/**
	 * Absolute base URI
	 *
	 * @var string
	 */
	protected $rootUri = null;

	/**
	 * Method to instantiate the view.
	 *
	 * @param   array  $config  A named configuration array for object construction
	 *
	 */
	public function __construct(Container $container, $config = [])
	{
		parent::__construct($container, $config);

		if (isset($config['layout']))
		{
			$this->layout = $config['layout'];
		}

		if (isset($config['paths']))
		{
			$this->paths = $config['paths'];
		}

		if (isset($config['is_admin_view']))
		{
			$this->isAdminView = $config['is_admin_view'];
		}
		else
		{
			$this->isAdminView = $this->container->app->isClient('administrator');
		}

		if (isset($config['hide_buttons']))
		{
			$this->hideButtons = $config['hide_buttons'];
		}

		$this->Itemid = $this->input->getInt('Itemid', 0);

		// Uris used for link to resources like css, js, images
		$this->relativeRootUri = Uri::root(true) . '/';
		$this->rootUri         = Uri::root();
		$this->document        = Factory::getApplication()->getDocument();
	}

	/**
	 * Method to display the view
	 */
	public function display()
	{
		echo $this->render();
	}

	/**
	 * Magic toString method that is a proxy for the render method.
	 *
	 * @return string
	 */
	public function __toString()
	{
		return $this->render();
	}

	/**
	 * Method to escape output.
	 *
	 * @param   string  $output  The output to escape.
	 *
	 * @return string The escaped output.
	 */
	public function escape($output)
	{
		return htmlspecialchars((string) $output, ENT_COMPAT, 'UTF-8');
	}

	/**
	 * Method to get the view layout.
	 *
	 * @return string The layout name.
	 */
	public function getLayout()
	{
		return $this->layout;
	}

	/**
	 * Method to get the layout path.
	 *
	 * @param   string  $layout  The layout name.
	 *
	 * @return mixed The layout file name if found, false otherwise.
	 */
	public function getPath($layout)
	{
		// Try to find the layout file with the following priority order: Device type, Joomla version, Default Layout
		$filesToFind = [$layout];

		foreach ($filesToFind as $fileLayout)
		{
			$file = Path::clean($fileLayout . '.php');
			$path = Path::find($this->paths, $file);

			if ($path)
			{
				break;
			}
		}

		return $path;
	}

	/**
	 * Method to get the view paths.
	 *
	 * @return array The paths queue.
	 *
	 */
	public function getPaths()
	{
		return $this->paths;
	}

	/**
	 * Method to render the view.
	 *
	 * @return string The rendered view.
	 *
	 * @throws RuntimeException
	 */
	public function render()
	{
		$this->beforeRender();

		// Get the layout path.
		$path = $this->getPath($this->getLayout());

		// Check if the layout path was found.
		if (!$path)
		{
			throw new RuntimeException('Layout Path Not Found');
		}

		// Start an output buffer.
		ob_start();

		// Load the layout.
		include $path;

		// Get the layout contents.
		return ob_get_clean();
	}

	/**
	 * Load sub-template for the current layout
	 *
	 * @param   string  $template
	 *
	 * @return string The output of sub-layout
	 * @throws RuntimeException
	 *
	 */
	public function loadTemplate($template, $data = [])
	{
		// Get the layout path.
		$path = $this->getPath($this->getLayout() . '_' . $template);

		// Check if the layout path was found.
		if (!$path)
		{
			throw new RuntimeException('Layout Path Not Found');
		}

		extract($data);

		// Start an output buffer.
		ob_start();

		// Load the layout.
		include $path;

		// Get the layout contents.
		return ob_get_clean();
	}

	/**
	 * Include sub-template for the current layout
	 *
	 * @param   string  $template
	 *
	 * @return void
	 * @throws RuntimeException
	 *
	 */
	public function includeTemplate($template, $data = []): void
	{
		// Get the layout path.
		$path = $this->getPath($this->getLayout() . '_' . $template);

		// Check if the layout path was found.
		if (!$path)
		{
			throw new RuntimeException('Layout Path Not Found');
		}

		extract($data);

		// Load the layout.
		include $path;
	}

	/**
	 * Load sub-template for the current layout
	 *
	 * @param   string  $layout
	 *
	 * @return string The output of sub-layout
	 * @throws RuntimeException
	 *
	 */
	public function loadCommonLayout($layout, $data = [])
	{
		// Get the layout path.
		$path = $this->getPath($layout);

		// Check if the layout path was found.
		if (!$path)
		{
			throw new RuntimeException('Layout Path Not Found');
		}

		extract($data);

		// Start an output buffer.
		ob_start();

		// Load the layout.
		include $path;

		// Get the layout contents.
		return ob_get_clean();
	}

	/**
	 * Method to set the view layout.
	 *
	 * @param   string  $layout  The layout name.
	 *
	 * @return HtmlView Method supports chaining.
	 */
	public function setLayout($layout)
	{
		$this->layout = $layout;

		return $this;
	}

	/**
	 * Method to set the view paths.
	 *
	 * @param   array  $paths  The paths queue.
	 *
	 * @return HtmlView Method supports chaining.
	 *
	 */
	public function setPaths($paths)
	{
		$this->paths = $paths;

		return $this;
	}

	/**
	 * Magic get method. Handles magic properties:
	 * $this->document  mapped to $this->container->document
	 *
	 * @param   string  $name  The property to fetch
	 *
	 * @return  mixed|null
	 */
	public function __get($name)
	{
		if ($name === 'document')
		{
			return $this->document;
		}

		$magicProperties = [
			'input',
		];

		if (in_array($name, $magicProperties))
		{
			return $this->container->get($name);
		}
	}

	/**
	 * Child class can use this method to get additional data needed for the view before it is rendered
	 */
	protected function beforeRender()
	{
	}

	/**
	 * Set document meta data
	 *
	 * @param   Registry  $params
	 *
	 * @return void
	 */
	protected function setDocumentMetadata($params)
	{
		$app              = Factory::getApplication();
		$document         = $app->getDocument();
		$siteNamePosition = $app->get('sitename_pagetitles');
		$siteName         = $app->get('sitename');

		if ($pageTitle = $params->get('page_title'))
		{
			if ($siteNamePosition == 0)
			{
				$document->setTitle($pageTitle);
			}
			elseif ($siteNamePosition == 1)
			{
				$document->setTitle($siteName . ' - ' . $pageTitle);
			}
			else
			{
				$document->setTitle($pageTitle . ' - ' . $siteName);
			}
		}

		if ($params->get('menu-meta_keywords'))
		{
			$document->setMetaData('keywords', $params->get('menu-meta_keywords'));
		}

		if ($params->get('menu-meta_description'))
		{
			$document->setDescription($params->get('menu-meta_description'));
		}

		if ($params->get('robots'))
		{
			$document->setMetaData('robots', $params->get('robots'));
		}
	}
}