<?php
/**
 * @package     OSL
 * @subpackage  Input
 *
 * @copyright   Copyright (C) 2016 Ossolution Team, Inc. All rights reserved.
 * @license     GNU General Public License version 2 or later; see LICENSE
 */

namespace OSL\Input;

use Joomla\CMS\Filter\InputFilter;
use Joomla\Input\Input as BaseInput;
use ReflectionClass;

/**
 * Extends JInput class to allow getting raw data from Input object
 *
 * @property    Input $get
 * @property    Input $post
 */
class Input extends BaseInput
{
	const INPUT_ALLOWRAW = 2;

	const INPUT_ALLOWHTML = 4;

	/**
	 * Constructor.
	 *
	 * @param   array  $source   Source data (Optional, default is $_REQUEST)
	 * @param   array  $options  Array of configuration parameters (Optional)
	 *
	 */
	public function __construct($source = null, array $options = [])
	{
		if ($source instanceof BaseInput)
		{
			$reflection = new ReflectionClass($source);
			$property   = $reflection->getProperty('data');

			if (version_compare(PHP_VERSION, '8.1.0', '<')) {
				$property->setAccessible(true);
			}

			$source = $property->getValue($source);
		}

		if (!isset($options['filter']))
		{
			$options['filter'] = InputFilter::getInstance([], [], 1, 1);
		}

		parent::__construct($source, $options);
	}

	/**
	 *
	 * Get data from the input
	 *
	 * @param   int  $mask
	 *
	 * @return array
	 */
	public function getData($mask = Input::INPUT_ALLOWHTML)
	{
		if ($mask & 2)
		{
			return $this->data;
		}

		return $this->filter->clean($this->data, '');
	}

	/**
	 * Set data for the input object. This is usually called when you get data, modify it, and then set it back
	 *
	 * @param $data
	 */
	public function setData(array $data)
	{
		$this->data = $data;
	}

	/**
	 * Check to see if a variable is available in the input or not
	 *
	 * @param   string  $name  the variable name
	 *
	 * @return boolean
	 */
	public function has($name)
	{
		return $this->exists($name);
	}

	/**
	 * Remove a variable from input
	 *
	 * @param $name
	 */
	public function remove($name)
	{
		unset($this->data[$name]);
	}

	/**
	 * Magic method to get an input object
	 *
	 * @param   mixed  $name  Name of the input object to retrieve.
	 *
	 * @return  Input  The request input object
	 */
	public function __get($name)
	{
		$allowedGlobals = ['REQUEST', 'GET', 'POST', 'SERVER', 'ENV'];

		$superGlobal = '_' . strtoupper($name);

		if (!isset($this->inputs[$name])
			&& \in_array(strtoupper($name), $allowedGlobals, true)
			&& isset($GLOBALS[$superGlobal]))
		{
			$this->inputs[$name] = new Input($GLOBALS[$superGlobal], $this->options);
		}

		return parent::__get($name);
	}
}