<?php
/*------------------------------------------------------------------------
# googlemap.lib.php - Ossolution Property
# ------------------------------------------------------------------------
# author    Dang Thuc Dam
# copyright Copyright (C) 2025 joomdonation.com. All Rights Reserved.
# @license - http://www.gnu.org/licenses/gpl-2.0.html GNU/GPL
# Websites: http://www.joomdonation.com
# Technical Support:  Forum - http://www.joomdonation.com/forum.html
*/
// No direct access.
defined('_JEXEC') or die;
use Joomla\CMS\Language\Text;
use Joomla\CMS\HTML\HTMLHelper;
use Joomla\CMS\Factory;
use Joomla\CMS\Router\Route;
use Joomla\CMS\Uri\Uri;
use Joomla\CMS\Http\HttpFactory;

class HelperOspropertyOpenStreetMap
{
    static private $url = "https://maps.google.com/maps/api/geocode/json";
	public static function loadGoogleScript($suffix = false)
	{
		global $configClass;
		OSPHelper::loadGoogleJS($suffix);
	}

	/**
	 * Get Lat Long address
	 *
	 * @param unknown_type $address
	 * @return unknown
	 */
	static function getLatlongAdd($address)
	{
		$address = urlencode($address);
		$http    = HttpFactory::getHttp();
		$base_url = "https://nominatim.openstreetmap.org/search?q=".$address."&format=json&addressdetails=1";
		
		$fileContents = $http->get($base_url)->body;
		//echo $fileContents;die();
		
		
		$data =  json_decode($fileContents, true);

		if (isset($data[0])) 
		{
			// Lấy latitude và longitude từ kết quả
			$latitude		= $data[0]['lat'];
			$longitude		= $data[0]['lon'];

			return array($latitude,$longitude);
		}
		//die();
        //return array($output[0]['lat'], $output[0]['lon']);
	}

    /**
     * Find Address
     *
     * @param unknown_type $option
     * @param unknown_type $row
     * @return unknown
     */
    public static function findAddress($option,$row,$address,$type)
    {
        if($type == 0)
        {
            $db = Factory::getContainer()->get('db');
            $address = "";
            if($row->address != "")
            {
                $address .= $row->address;
            }
            if($row->city != "")
            {
                $address .= " ".$row->city;
            }
            if($row->state != "")
            {
                $db->setQuery("Select state_name from #__osrs_states where id = '$row->state'");
                $state_name = $db->loadResult();
                $address .= " ".$state_name;
                $db->setQuery("Select country_id from #__osrs_states where id = '$row->state'");
                $country_id = $db->loadResult();
                $db->setQuery("Select country_name from #__osrs_countries where id = '$country_id'");
                $country_name = $db->loadResult();
                $address .= " ".$country_name;
            }
            $address = trim($address);
        }

        $return = self::getLatlongAdd($address);
        $return[2] = "OK";
        return $return;
    }
	
	/**
	 * Check curl existing
	 *
	 * @return unknown
	 */
	public static function _iscurlinstalled() {
		if  (in_array  ('curl', get_loaded_extensions())) {
			return true;
		}
		else{
			return false;
		}
	}
	
	/**
	 * Load Google Map
	 *
	 * @param unknown_type $geocode
	 */
	static function loadGoogleMap($geocode,$mapdiv,$maptype){
		global $configClass;
		switch ($maptype){
			case "agent":
			case "company":
				$icon = "workoffice.png";
			break;
			default:
				$icon = "house.png";
			break;
		}
		if(count($geocode) == 1){
			$lat = $geocode[0]->lat;
			$long = $geocode[0]->long;
		}else{
			$lat = $geocode[0]->lat;
			$long = $geocode[0]->long;
		}
		if (!isset($configClass['goole_map_overlay'])) $configClass['goole_map_overlay'] = 'ROADMAP';

        OSPHelper::loadGoogleJS('');
		?>
		<script type="text/javascript">
		function initialize() {
		  var myOptions = {
		  zoom: <?php echo $configClass['goole_map_resolution']?>,
center: new google.maps.LatLng(<?php echo $lat?>,<?php echo $long?>),mapTypeId: google.maps.MapTypeId.<?php echo  $configClass['goole_map_overlay'];?>};
		  var map = new google.maps.Map(document.getElementById("<?php echo $mapdiv?>"),myOptions);
		  setMarkers(map, addressArray);
		}
		
		<?php
		$address = "[";
		for($i=0;$i<count($geocode);$i++){
			$geo = $geocode[$i];
			$address .= '["'.$geo->text.'",'.$geo->lat.','.$geo->long.','.$i.'],';
		}
		$address = substr($address,0,strlen($address)-1);
		$address .= "]";
		?>
		
		var addressArray = <?php echo $address?>;
		
		function setMarkers(map, locations)
        {
             for (var i = 0; i < locations.length; i++) {
                  var j = i + 1;
                  var imagelink = '<?php echo Uri::root()?>media/com_osproperty/assets/images/mapicon/i' + j + '.png';
                  var image = new google.maps.MarkerImage(imagelink,new google.maps.Size(36, 31),new google.maps.Point(0,0),new google.maps.Point(0, 31));
                  var add = locations[i];
                  var myLatLng = new google.maps.LatLng(add[1], add[2]);
                  var marker = new google.maps.Marker({
                       position: myLatLng,
                       map: map,
                       icon: image,
                       title: add[0],
                       zIndex: add[3]
                  });
             }
		}
		</script>
		<?php
	}
	
	/**
	 * load Google Map in Edit Property
	 */
	static function loadGMapinEditProperty($geocode,$div_name,$lat_div,$long_div)
    {
		global $mainframe,$configClass;
        $document           = Factory::getApplication()->getDocument()
            ->addScript(Uri::root() . 'media/com_osproperty/assets/js/leaflet/leaflet.js');
        $document->getWebAssetManager()->registerAndUseStyle('com_osproperty.leaflet.css',Uri::root() . 'media/com_osproperty/assets/js/leaflet/leaflet.css');
		$icon = "house.png";
		?>

		<script type="text/javascript">
        jQuery(document).ready(function(){
            var map = L.map('map').setView([<?php echo $geocode[0]->lat?>, <?php echo $geocode[0]->long;?>], <?php echo $configClass['goole_map_resolution']?>);

            L.tileLayer('https://{s}.tile.openstreetmap.org/{z}/{x}/{y}.png', {
                attribution: 'Map data &copy; <a href="https://www.openstreetmap.org/">OpenStreetMap</a> contributors, <a href="https://creativecommons.org/licenses/by-sa/2.0/">CC-BY-SA</a>, Imagery © <a href="https://www.mapbox.com/">Mapbox</a>',
                maxZoom: 18,
                id: 'mapbox.streets',
                zoom: <?php echo $configClass['goole_map_resolution']?>,
            }).addTo(map);
            var marker = L.marker([<?php echo $geocode[0]->lat?>, <?php echo $geocode[0]->long;?>], {draggable: true}).addTo(map);
            marker.on('dragend', function(event){
                var marker = event.target;
                var location = marker.getLatLng();
                var lat = location.lat;
                var lon = location.lng;
                document.getElementById("<?php echo $lat_div?>").value = lat.toFixed(5);
                document.getElementById("<?php echo $long_div?>").value = lon.toFixed(5);
            });
        });

		</script>
		<?php
	}
	
	/**
	 * Load Locator Map
	 *
	 * @param unknown_type $rows
	 * @param unknown_type $mapdiv
	 */
	static function loadLocatorMap($rows,$mapdiv,$zoomlevel,$search_lat,$search_long)
    {
		global $mainframe,$configClass;
		
		$icon = "house.png";
		if(($search_lat != "") or ($search_long != "")){
			$lat = $search_lat;
			$long = $search_long;
		}else{
			if(count($rows) == 1){
				$lat = $rows[0]->lat;
				$long = $rows[0]->long;
			}else{
				$lat = $rows[0]->lat;
				$long = $rows[0]->long;
			}
		}
		if (!isset($configClass['goole_map_overlay'])) $configClass['goole_map_overlay'] = 'ROADMAP';
        OSPHelper::loadGoogleJS('');
		?>
		<script type="text/javascript">
		  var infowindow;
		  var map;
		  var bounds;
		  var markers = [];
		  var markerIndex=0;
		  var markerArray = [];
		  var bounds = new google.maps.LatLngBounds();
		  function initMap() {
			  var myOptions = {
			  zoom: <?php echo $zoomlevel;?>,
			  center: new google.maps.LatLng(<?php echo $lat?>,<?php echo $long?>),
			  mapTypeId: google.maps.MapTypeId.<?php echo $configClass['goole_map_overlay'];?>,
			  mapTypeControl: true,
              navigationControl: true,
              streetViewControl: true,
              zoomControl: true,
	          zoomControlOptions: {
	            style: google.maps.ZoomControlStyle.SMALL
	          }
			  };
			  var map = new google.maps.Map(document.getElementById("<?php echo $mapdiv?>"),myOptions);
			  //setMarkers(map, addressArray);
			  var infoWindow = new google.maps.InfoWindow();
	 		  var markerBounds = new google.maps.LatLngBounds();
	 
	 		  function makeMarker(options){
				   var pushPin = new google.maps.Marker({map:map});
				   pushPin.setOptions(options);
				   google.maps.event.addListener(pushPin, 'click', function(){
					     infoWindow.setOptions(options);
					     infoWindow.open(map, pushPin);
				   });
				   markerArray.push(pushPin);
				   return pushPin;
			 }

			 google.maps.event.addListener(map, 'click', function(){
			   infoWindow.close();
			 });
			 
			 <?php
			 for($i=0;$i<count($rows);$i++){
			 	$row = $rows[$i];
			 	if($mapdiv == "map_canvas"){
			 	?>
			 	 makeMarker({
				   position: new google.maps.LatLng(<?php echo $row->lat?>,<?php echo $row->long?>),
				   title: "<?php echo htmlspecialchars($row->title);?>",
				   content: "<?php echo htmlspecialchars($row->content)?>",
				   icon:new google.maps.MarkerImage('<?php echo Uri::root()?>media/com_osproperty/assets/images/<?php echo $icon?>')
				 });
			 	<?php
			 	}else{
			 	?>
			 	 makeMarker({
				   position: new google.maps.LatLng(<?php echo $row->lat?>,<?php echo $row->long?>),
				   title: "<?php echo htmlspecialchars($row->title)?>",
				   content: "<?php echo htmlspecialchars($row->content)?>"
				 });
				 bounds.extend(new google.maps.LatLng(<?php echo $row->lat?>,<?php echo $row->long?>));
			 	<?php
			 	}
			 }
			 ?>
			 map.fitBounds(bounds);
			 var listener = google.maps.event.addListener(map, "idle", function() { 
			    if (map.getZoom() > 16) map.setZoom(16); 
			    google.maps.event.removeListener(listener); 
			 });
		  }
		  
		  window.onload=function(){
			if(window.initMap) initMap();
		  }
		  window.onunload=function(){
		    if(typeof(GUnload)!="undefined") GUnload();
		  }
		  
		  function openMarker(i){
			   google.maps.event.trigger(markerArray[i],'click');
		  };
		</script>
		<?php
	}
	

    /**
     * This function is used to load Google Map at top of listing page
     * @param $rows
     * @param int $state_id
     * @param int $city_id
     * @return bool
     */
    static function loadMapInListing($rows,$state_id = 0,$city_id = 0)
    {
        global $mainframe,$configClass,$bootstrapHelper;
        $rootUri            = Uri::root();
        $document           = Factory::getApplication()->getDocument()
            ->addScript($rootUri . 'media/com_osproperty/assets/js/leaflet/leaflet.js')
            ->addScript($rootUri . 'media/com_osproperty/assets/js/leaflet/leaflet.markercluster.js');
        $document->getWebAssetManager()->registerAndUseStyle('com_osproperty.leaflet.css',$rootUri . 'media/com_osproperty/assets/js/leaflet/leaflet.css');
        $document->getWebAssetManager()->registerAndUseStyle('com_osproperty.MarkerCluster.Default.css',$rootUri . 'media/com_osproperty/assets/js/leaflet/MarkerCluster.Default.css');
        $document->getWebAssetManager()->registerAndUseStyle('com_osproperty.MarkerCluster.Default.ie.css',$rootUri . 'media/com_osproperty/assets/js/leaflet/MarkerCluster.Default.ie.css');
        $document->getWebAssetManager()->registerAndUseStyle('com_osproperty.MarkerCluster.css',$rootUri . 'media/com_osproperty/assets/js/leaflet/MarkerCluster.css');

        //find default lat/long addresses
        $show_map           = 0;
        $city_name          = "";
        $state_name         = "";
        $db                 = Factory::getContainer()->get('db');
        if($state_id > 0)
        {
            $db->setQuery("Select state_name from #__osrs_states where id = '$state_id'");
            $state_name     = $db->loadResult();

            $db->setQuery("Select country_id from #__osrs_states where id = '$state_id'");
            $country_id     = $db->loadResult();

            $db->setQuery("Select country_name from #__osrs_countries where id = '$country_id'");
            $country_name   = $db->loadResult();
        }
        if($city_id > 0)
        {
            $db->setQuery("Select city from #__osrs_cities where id = '$city_id'");
            $city_name      = $db->loadResult();
        }

        if($city_name != "" || $state_name != "")
        {
            $address = "";
            if($city_name != "")
            {
                $address = $city_name;
            }
            if($state_name != "")
            {
                $address .= " ".$state_name;
            }
            if($country_name != "")
            {
                $address .= " ".$country_name;
            }
            $return = self::getLatlongAdd($address);
            $default_lat  = $return[0];
            $default_long = $return[1];
        }
        else
        {
            $default_lat  = $configClass['goole_default_lat'];
            $default_long = $configClass['goole_default_long'];
        }

        $duplicate = OSPHelper::findGoogleDuplication($rows);

        //setup Leaflet map
        $zoomLevel        = 16;
        ?>
        <div id="map_canvas" class="map2x relative"></div>
        <script type="text/javascript">
        jQuery(document).ready(function(){
            var markerArray = [];
            var latArr      = [];
            var longArr     = [];
            var mymap       = L.map('map_canvas', {
                scrollWheelZoom: false // Tắt tính năng zoom bằng cuộn chuột ban đầu
            }).setView([<?php echo $default_lat; ?>, <?php echo $default_long; ?>], <?php echo $zoomLevel;?>);
            
            L.tileLayer('https://{s}.tile.openstreetmap.org/{z}/{x}/{y}.png', {
                attribution: '',
                maxZoom: 18,
                id: 'mapbox.streets',
                zoom: <?php echo $zoomLevel;?>,
            }).addTo(mymap);
            
            // Thêm thông báo nhỏ cho người dùng
            var mapNotice = L.control({position: 'bottomleft'});
            mapNotice.onAdd = function(map) {
                var div = L.DomUtil.create('div', 'map-notice');
                div.innerHTML = '<div style="background-color: rgba(255,255,255,0.8); padding: 5px; border-radius: 4px; font-size: 12px; margin-bottom: 5px;"><?php echo Text::_('Click on map to enable zoom'); ?></div>';
                return div;
            };
            mapNotice.addTo(mymap);
            
            // Xử lý sự kiện click vào bản đồ để bật/tắt tính năng zoom bằng cuộn chuột
            var scrollWheelEnabled = false;
            mymap.on('click', function() {
                if (!scrollWheelEnabled) {
                    mymap.scrollWheelZoom.enable();
                    scrollWheelEnabled = true;
                    jQuery('.map-notice').hide();
                } else {
                    mymap.scrollWheelZoom.disable();
                    scrollWheelEnabled = false;
                    jQuery('.map-notice').show();
                }
            });

            <?php
            if(count($duplicate) > 0)
            {
                $boundItems = array();
                for ($i = 0; $i < count($duplicate); $i++) {
                    $item = $duplicate[$i];
                    $key = OSPHelper::find_key($item->id, $rows);
                    if (($rows[$key]->show_address == 1) && ($rows[$key]->lat_add != "") && ($rows[$key]->long_add != "")) {
                        $boundItems[] = "[" . $rows[$key]->lat_add . "," . $rows[$key]->long_add . "]";
                    }
                }
                if (count($boundItems)) {
                    $boundItems = implode(",", $boundItems);
                }
                ?>
                mymap.fitBounds([<?php echo $boundItems?>]);
                var markers = new L.MarkerClusterGroup();
                <?php

                for($i=0;$i<count($duplicate);$i++)
                {
                    $item = $duplicate[$i];
                    $key  = OSPHelper::find_key($item->id,$rows);
                    if(count((array)$item->value) == 0){ //having no duplication
                        $row = $rows[$key];
                        $row->mapid = $i;
                        $needs = array();
                        $needs[] = "property_details";
                        $needs[] = $row->id;
                        $itemid	 = OSPRoute::getItemid($needs);
                        $title = "";
                        if($row->ref!="" && $configClass['show_ref'] == 1){
                            $title .= $row->ref.",";
                        }
                        $title 		.= $row->pro_name;
                        $title  	 = str_replace("'","",$title);
                        $title 		 = htmlspecialchars($title);
                        $created_on  = $row->created;
                        $modified_on = $row->modified;

                        $addInfo = array();
                        if($row->bed_room > 0){
                            $addInfo[] = $row->bed_room." ".Text::_('OS_BEDROOMS');
                        }
                        if($row->bath_room > 0){
                            $addInfo[] = OSPHelper::showBath($row->bath_room)." ".Text::_('OS_BATHROOMS');
                        }
                        if($row->rooms > 0){
                            $addInfo[] = $row->rooms." ".Text::_('OS_ROOMS');
                        }

						if(OSPHelper::getLanguageFieldValue($row, 'price_text') != "")
						{
							$price = OSPHelper::showPriceText(OSPHelper::getLanguageFieldValue($row, 'price_text'));
						}
						elseif($row->price_call == 0)
						{
							$price = OSPHelper::generatePrice($row->curr, $row->price);
							if($row->rent_time != "")
							{
								$price .= " /".Text::_($row->rent_time);
							}
						}
						else
						{
							$price .= Text::_('OS_CALL_FOR_DETAILS_PRICE');
						}

                        ?>
                        var contentString<?php echo $row->id?> = '<div class="<?php echo $bootstrapHelper->getClassMapping('row-fluid'); ?>">'+
                            '<div class="<?php echo $bootstrapHelper->getClassMapping('span12'); ?> ospitem-maptitle title-blue"><a style="font-size:13px;" href="<?php echo Route::_("index.php?option=com_osproperty&task=property_details&id=".$row->id."&Itemid=".$itemid); ?>" title="<?php echo $title;?>"><?php echo $title;?></a> - <?php echo $price;?></div></div>';
                        <?php
                        if(count($addInfo) > 0)
                        {
                            ?>
                            //contentString<?php echo $row->id?> += '<div class="ospitem-iconbkgr"><span class="ezitem-leftpad"><?php echo implode(" | ",$addInfo); ?></span></div>';
                            <?php
                        }
                        ?>
                                
                        <?php

                        if($row->show_address == 1 && $row->lat_add != "" && $row->long_add != "")
                        {
                            $type_icon  = OSPHelper::getTypeIcon($row->pro_type);
                            ?>
                            var propertyIcon = L.icon({iconUrl: '<?php echo Uri::root()?>media/com_osproperty/assets/images/googlemapicons/<?php echo $type_icon;?>',
                                                                iconSize:     [33, 44] // size of the icon
                            });
                            var popupContent<?php echo $row->mapid?> = contentString<?php echo $row->id?>;
                            var marker<?php echo $row->mapid?> = L.marker([<?php echo $row->lat_add ?>, <?php echo $row->long_add;?>],{icon: propertyIcon});
                            marker<?php echo $row->mapid?>.bindPopup(popupContent<?php echo $row->mapid?>);
                            markerArray.push(marker<?php echo $row->mapid?>);
                            markers.addLayer(marker<?php echo $row->mapid?>);
                            latArr.push(<?php echo $row->lat_add ?>);
                            longArr.push(<?php echo $row->long_add;?>);

                            jQuery("#openmap<?php echo $row->mapid?>").click(function(){
                                var point = [ <?php echo $row->lat_add ?> , <?php echo $row->long_add;?> ];
                                mymap.flyTo(point,16);
                                marker<?php echo $row->mapid?>.openPopup();
                            });
                            <?php
                        }
                    }
                    else
                    {
                        //having duplication
                        $row = $rows[$key];
                        $row->mapid = $i;
                        $itemIdArr = array();
                        $titleArr  = array();
                        $descArr   = array();

                        $needs = array();
                        $needs[] = "property_details";
                        $needs[] = $row->id;
                        $itemid	 = OSPRoute::getItemid($needs);
                        $itemIdArr[] = $itemid;

                        $title = "";
                        if(($row->ref!="") && ($configClass['show_ref'] == 1)){
                            $title .= $row->ref.",";
                        }
                        $title 		.= $row->pro_name;
                        $title  	 = str_replace("'","",$title);
                        $title 		 = htmlspecialchars($title);
                        $titleArr[]  = $title;

                        $addInfo = array();
                        if($row->bed_room > 0){
                            $addInfo[] = $row->bed_room." ".Text::_('OS_BEDROOMS');
                        }
                        if($row->bath_room > 0){
                            $addInfo[] = OSPHelper::showBath($row->bath_room)." ".Text::_('OS_BATHROOMS');
                        }
                        if($row->rooms > 0){
                            $addInfo[] = $row->rooms." ".Text::_('OS_ROOMS');
                        }

						if(OSPHelper::getLanguageFieldValue($row, 'price_text') != "")
						{
							$price = OSPHelper::showPriceText(OSPHelper::getLanguageFieldValue($row, 'price_text'));
						}
						elseif($row->price_call == 0)
						{
							$price = OSPHelper::generatePrice($row->curr, $row->price);
							if($row->rent_time != "")
							{
								$price .= " /".Text::_($row->rent_time);
							}
						}
						else
						{
							$price .= Text::_('OS_CALL_FOR_DETAILS_PRICE');
						}

                        $desc = '<div class="'.$bootstrapHelper->getClassMapping('row-fluid').'"><div class="'.$bootstrapHelper->getClassMapping('span12').' ospitem-maptitle title-blue"><a style="font-size:13px;" href="'.Route::_("index.php?option=com_osproperty&task=property_details&id=".$row->id."&Itemid=".$itemid).'">'.$title.'</a> - '.$price.'</div></div>';
                        if(count($addInfo) > 0){
                           // $desc .= '<div class="ospitem-iconbkgr"><span class="ezitem-leftpad">'.implode(" | ",$addInfo).'</span></div>';
                        }
                        //$desc .= '</p></div></div>';
                        $descArr[] = $desc;

                        foreach($item->value as $value){
                            $key  = OSPHelper::find_key($value,$rows);
                            $dupItem = $rows[$key];
                            $dupItem->mapid = $i;
                            $needs = array();
                            $needs[] = "property_details";
                            $needs[] = $dupItem->id;
                            $itemid	 = OSPRoute::getItemid($needs);
                            $itemIdArr[] = $itemid;

                            $title = "";
                            if(($dupItem->ref!="") && ($configClass['show_ref'] == 1)){
                                $title .= $dupItem->ref.",";
                            }
                            $title 		.= $dupItem->pro_name;
                            $title  	 = str_replace("'","",$title);
                            $title 		 = htmlspecialchars($title);
                            $titleArr[]  = $title;

                            $addInfo = array();
                            if($dupItem->bed_room > 0){
                                $addInfo[] = $dupItem->bed_room." ".Text::_('OS_BEDROOMS');
                            }
                            if($dupItem->bath_room > 0){
                                $addInfo[] = OSPHelper::showBath($dupItem->bath_room)." ".Text::_('OS_BATHROOMS');
                            }
                            if($dupItem->rooms > 0){
                                $addInfo[] = $dupItem->rooms." ".Text::_('OS_ROOMS');
                            }

							if(OSPHelper::getLanguageFieldValue($dupItem, 'price_text') != "")
							{
								$price = OSPHelper::showPriceText(OSPHelper::getLanguageFieldValue($dupItem, 'price_text'));
							}
							elseif($dupItem->price_call == 0)
							{
								$price = OSPHelper::generatePrice($dupItem->curr, $dupItem->price);
								if($dupItem->rent_time != "")
								{
									$price .= " /".Text::_($row->rent_time);
								}
							}
							else
							{
								$price .= Text::_('OS_CALL_FOR_DETAILS_PRICE');
							}
                            $desc = '<div class="'.$bootstrapHelper->getClassMapping('row-fluid').'"><div class="'.$bootstrapHelper->getClassMapping('span12').' ospitem-maptitle title-blue"><a  style="font-size:13px;" href="'.Route::_("index.php?option=com_osproperty&task=property_details&id=".$dupItem->id."&Itemid=".$itemid).'">'.$title.'</a> - '.$price.'</div></div>';
                            if(count($addInfo) > 0){
                                //$desc .= '<div class="ospitem-iconbkgr"><span class="ezitem-leftpad">'.implode(" | ",$addInfo).'</span></div>';
                            }
                            //$desc .= '</p></div></div>';
                            $descArr[] = $desc;
                        }
                        $desc = implode('<div class="clearfix googleinfordiv"></div>',$descArr);

                        if($row->show_address == 1 && $row->lat_add != "" && $row->long_add != "")
                        {
                            $type_icon  = OSPHelper::getTypeIcon($row->pro_type);
                            ?>
                            var contentString<?php echo $row->id?> = '<?php echo $desc;?>';
                            var propertyIcon<?php echo $row->mapid?> = L.icon({iconUrl: '<?php echo Uri::root()?>media/com_osproperty/assets/images/googlemapicons/<?php echo $type_icon;?>',
                                                                iconSize:     [33, 44] // size of the icon
                            });
                            var popupContent<?php echo $row->mapid?> = '<h5><?php echo Text::_('OS_MULTIPLE_PROPERTIES');?></h5>';
                            popupContent<?php echo $row->mapid?> += contentString<?php echo $row->id?>;
                            var marker<?php echo $row->mapid?> = L.marker([<?php echo $row->lat_add ?>, <?php echo $row->long_add;?>],{icon: propertyIcon});
                            marker<?php echo $row->mapid?>.bindPopup(popupContent<?php echo $row->mapid?>);
                            markerArray.push(marker<?php echo $row->mapid?>);
                            markers.addLayer(marker<?php echo $row->mapid?>);
                            latArr.push(<?php echo $row->lat_add ?>);
                            longArr.push(<?php echo $row->long_add;?>);

                            jQuery("#openmap<?php echo $row->mapid?>").click(function(){
                                var point = [ <?php echo $row->lat_add ?> , <?php echo $row->long_add;?> ];
                                mymap.flyTo(point,16);
                                marker<?php echo $row->mapid?>.openPopup();
                            });
                            <?php
                        }
                    }
                }
            ?>
            mymap.addLayer(markers);
            <?php } ?>

			const locationItems = document.querySelectorAll('.property_item');

			// Thêm sự kiện hover vào từng item trong danh sách
			locationItems.forEach(item => {
				item.addEventListener('mouseover', function () {
					// Lấy thông tin latitude, longitude và nội dung popup từ data attributes
					const lat = parseFloat(item.getAttribute('data-lat'));
					const lng = parseFloat(item.getAttribute('data-lng'));
					//const popupContent = item.getAttribute('data-popup');
					
					if(lat != "" && lng != "")
					{
						const marker = findMarkerByLatLng(lat, lng);
						//alert(marker.getLatLng().lat);
						if (marker) {
							marker.openPopup();
							mymap.panTo([lat, lng]);
						}
					}
				});
			});

			function findMarkerByLatLng(lat, lng) {
				let foundMarker = null;
				mymap.eachLayer(layer => {
					if (layer instanceof L.Marker) {
						const position = layer.getLatLng();
						if (position.lat === lat) {
							foundMarker = layer;
						}
					}
				});
				return foundMarker;
			}
        });
        </script>
        <?php
    }


    /**
     * Load Open Street Map in Details page
     * @param $property
     */
    public static function loadOpenStreetMapDetails($property,$configClass,$style="",$toggleposition = 0)
    {
        $db = Factory::getContainer()->get('db');
        $db->setQuery("Select type_icon from #__osrs_types where id = '$property->pro_type'");
        $type_icon = $db->loadResult();
        if($type_icon == ""){
            $type_icon = "1.png";
        }
        $rootUri = Uri::root(true);
        $document = Factory::getApplication()->getDocument()
            ->addScript($rootUri . '/media/com_osproperty/assets/js/leaflet/leaflet.js');
        OSPHelper::addStyleSheet($rootUri . '/media/com_osproperty/assets/js/leaflet/leaflet.css','osproperty.details.leaflet');
        $zoomLevel   = 16;
        $coordinates = $property->lat_add . ',' . $property->long_add;
        $onPopup = false;
        ?>
        <div id="googlemapdiv" style="<?php echo $style; ?>;" class="relative map2x"></div>
        <script type="text/javascript">
            jQuery(document).ready(function(){
                var mymap = L.map('googlemapdiv').setView([<?php echo $property->lat_add; ?>, <?php echo $property->long_add; ?>], <?php echo $zoomLevel;?>);
                L.tileLayer('https://{s}.tile.openstreetmap.org/{z}/{x}/{y}.png', {
                    attribution: 'Map data &copy; <a href="https://www.openstreetmap.org/">OpenStreetMap</a> contributors, <a href="https://creativecommons.org/licenses/by-sa/2.0/">CC-BY-SA</a>, Imagery © <a href="https://www.mapbox.com/">Mapbox</a>',
                    maxZoom: 18,
                    id: 'mapbox.streets',
                    zoom: <?php echo $zoomLevel;?>,
                }).addTo(mymap);

                <?php
                if($property->show_address == 1)
                {
                ?>
                    var propertyIcon = L.icon({iconUrl: '<?php echo Uri::root()?>media/com_osproperty/assets/images/googlemapicons/<?php echo $type_icon;?>',
                                                        iconSize:     [33, 44] // size of the icon
                    });
                    var marker = L.marker([<?php echo $property->lat_add ?>, <?php echo $property->long_add;?>],{icon: propertyIcon}, {draggable: false}).addTo(mymap);
                <?php
                }
                else
                {
                    ?>
                    var circle = L.circle([<?php echo $property->lat_add ?>, <?php echo $property->long_add;?>], {
                        color: '#1D86A0',
                        fillColor: '#1D86A0',
                        fillOpacity: 0.5,
                        radius: 500
                    }).addTo(mymap);
                    <?php
                }
                ?>
                mymap.scrollWheelZoom.disable()
            });
        </script>
        <?php
    }

	/**
     * Load Google Map in Details page
     * @param $property
     */
    public static function loadOpenStreetMapDetailsClone($property,$configClass,$style="",$toggleposition = 0){
        $db = Factory::getContainer()->get('db');
        $db->setQuery("Select type_icon from #__osrs_types where id = '$property->pro_type'");
        $type_icon = $db->loadResult();
        if($type_icon == ""){
            $type_icon = "1.png";
        }
        $rootUri = Uri::root(true);
        $document = Factory::getApplication()->getDocument()
            ->addScript($rootUri . '/media/com_osproperty/assets/js/leaflet/leaflet.js')
            ->addStyleSheet($rootUri . '/media/com_osproperty/assets/js/leaflet/leaflet.css');
        $zoomLevel   = 16;
        $coordinates = $property->lat_add . ',' . $property->long_add;
        $onPopup = false;
        ?>
        <div id="googlemapdiv1" style="<?php echo $style; ?>" class="relative map2x"></div>
        <script type="text/javascript">
            jQuery(document).ready(function(){
                var mymap = L.map('googlemapdiv1').setView([<?php echo $property->lat_add; ?>, <?php echo $property->long_add; ?>], <?php echo $zoomLevel;?>);
                L.tileLayer('https://{s}.tile.openstreetmap.org/{z}/{x}/{y}.png', {
                    attribution: 'Map data &copy; <a href="https://www.openstreetmap.org/">OpenStreetMap</a> contributors, <a href="https://creativecommons.org/licenses/by-sa/2.0/">CC-BY-SA</a>, Imagery © <a href="https://www.mapbox.com/">Mapbox</a>',
                    maxZoom: 18,
                    id: 'mapbox.streets',
                    zoom: <?php echo $zoomLevel;?>,
                }).addTo(mymap);

                <?php
                if($property->show_address == 1)
                {
                ?>
                var propertyIcon = L.icon({iconUrl: '<?php echo Uri::root()?>media/com_osproperty/assets/images/googlemapicons/<?php echo $type_icon;?>',
                    iconSize:     [33, 44] // size of the icon
                });
                var marker = L.marker([<?php echo $property->lat_add ?>, <?php echo $property->long_add;?>],{icon: propertyIcon}, {draggable: false}).addTo(mymap);
                <?php
                }
                else
                {
                ?>
                var circle = L.circle([<?php echo $property->lat_add ?>, <?php echo $property->long_add;?>], {
                    color: '#1D86A0',
                    fillColor: '#1D86A0',
                    fillOpacity: 0.5,
                    radius: 500
                }).addTo(mymap);
                <?php
                }
                ?>
                mymap.scrollWheelZoom.disable()
            });
        </script>
        <?php
    }

	/**
	 * @param $post_code
	 * @param string $country
	 * @return bool
	 */
	public static function getLocationPostCode($post_code, $country_id = 0)
    {
        global $configClass;
        $db = Factory::getContainer()->get('db');
		$url_zip = '';
		if($country_id > 0){
            $db->setQuery("Select `country_name` from #__osrs_countries where id = '$country_id'");
            $country = $db->loadResult();
			$url_zip = '?address='.urlencode($country);
			$url = self::$url.$url_zip."&components=postal_code:".urlencode($post_code);
		}else{
			$url = self::$url."?components=postal_code:".urlencode($post_code);
		}

        $googlekey = $configClass['goole_aip_key'];
		if($googlekey != ""){
			$url .= '&key='.$googlekey;
		}
		$resp_json = self::curl_file_get_contents($url);
		$resp = json_decode($resp_json, true);

		if($resp['status']='OK' && isset($resp['results'][0])){
			return $resp['results'][0]['geometry']['location'];
		}else{
			Factory::getApplication()->enqueueMessage($resp['error_message'],'error');
			return false;
		}
	}

    static private function curl_file_get_contents($URL)
    {
        $c = curl_init();
        curl_setopt($c, CURLOPT_RETURNTRANSFER, 1);
        curl_setopt($c, CURLOPT_URL, $URL);
        curl_setopt($c, CURLOPT_SSL_VERIFYPEER, false);
        curl_setopt($c, CURLOPT_SSL_VERIFYHOST, false);

        $contents = curl_exec($c);
        curl_close($c);

        if ($contents) return $contents;
        else return FALSE;
    }

    /**
     * Calculate distance between two points using Haversine formula
     * @param float $lat1 Latitude of point 1
     * @param float $lon1 Longitude of point 1
     * @param float $lat2 Latitude of point 2
     * @param float $lon2 Longitude of point 2
     * @return float Distance in kilometers
     */
    public static function calculateDistance($lat1, $lon1, $lat2, $lon2)
    {
        if ($lat1 == 0 || $lon1 == 0 || $lat2 == 0 || $lon2 == 0) {
            return 0;
        }
        
        $earthRadius = 6371; // Earth radius in kilometers
        
        $dLat = deg2rad($lat2 - $lat1);
        $dLon = deg2rad($lon2 - $lon1);
        
        $a = sin($dLat/2) * sin($dLat/2) +
             cos(deg2rad($lat1)) * cos(deg2rad($lat2)) *
             sin($dLon/2) * sin($dLon/2);
        
        $c = 2 * atan2(sqrt($a), sqrt(1-$a));
        $distance = $earthRadius * $c;
        
        return round($distance, 2);
    }

    /**
     * Find properties within radius from center point
     * @param float $lat Center latitude
     * @param float $lon Center longitude
     * @param float $radius Search radius in km
     * @param array $filters Additional filters
     * @return array Properties with distance field
     */
    public static function findPropertiesInRadius($lat, $lon, $radius, $filters = array())
    {
        $db = Factory::getContainer()->get('db');
        $query = $db->getQuery(true);
        
        // Base query
        $query->select('p.*, GROUP_CONCAT(pc.category_id) AS category_ids')
                ->from($db->quoteName('#__osrs_properties', 'p'))
                ->leftJoin($db->quoteName('#__osrs_property_categories', 'pc') . ' ON pc.pid = p.id')
                ->where('p.published = 1')
                ->where('p.lat_add IS NOT NULL')
                ->where('p.long_add IS NOT NULL')
                ->where('p.lat_add != ' . $db->quote(''))
                ->where('p.long_add != ' . $db->quote(''))
                ->where('CAST(p.lat_add AS DECIMAL(10,7)) != 0')
                ->where('CAST(p.long_add AS DECIMAL(10,7)) != 0')
                ->group('p.id');
        
        // Apply filters
        self::applyFilters($query, $filters);
        
        // Execute query
        $db->setQuery($query);
        $properties = $db->loadObjectList();
        
        // Calculate distance and filter by radius
        $results = array();
        foreach ($properties as $property) {
            $distance = self::calculateDistance($lat, $lon, $property->lat_add, $property->long_add);
            
            if ($distance <= $radius) {
                $property->distance = $distance;
                $results[] = $property;
            }
        }
        
        // Sort by distance (closest first)
        usort($results, function($a, $b) {
            return $a->distance <=> $b->distance;
        });
        
        return $results;
    }

    /**
     * Find nearest property to given coordinates
     * @param float $lat Latitude
     * @param float $lon Longitude
     * @return object|null Nearest property with distance
     */
    static function findNearestProperty($lat, $lon)
    {
        $db = Factory::getContainer()->get('db');
        $query = $db->getQuery(true);
        
        // Calculate distance and get nearest property
        $query->select('a.id, a.pro_name, a.lat_add, a.long_add')
            ->select('(' . self::getDistanceFormula($lat, $lon) . ') AS distance')
            ->from('#__osrs_properties AS a')
            ->where('a.published = 1')
            ->where('a.lat_add IS NOT NULL')
            ->where('a.long_add IS NOT NULL')
            ->where('a.lat_add != 0')
            ->where('a.long_add != 0')
            ->order('distance ASC')
            ->setLimit(1);
        
        $db->setQuery($query);
        //echo $db->getQuery();die();
        try {
            $result = $db->loadObject();
            return $result;
        } catch (Exception $e) {
            return null;
        }
    }

    /**
     * Get SQL formula for calculating distance using Haversine formula
     * @param float $lat Latitude of center point
     * @param float $lon Longitude of center point
     * @param string $latField Database field name for latitude (default: 'a.lat')
     * @param string $lonField Database field name for longitude (default: 'a.long')
     * @return string SQL formula string
     */
    static function getDistanceFormula($lat, $lon, $latField = 'a.lat_add', $lonField = 'a.long_add')
    {
        // Earth radius in kilometers
        $earthRadius = 6371;
        
        // Haversine formula in SQL
        // distance = R * acos(cos(lat1) * cos(lat2) * cos(lon2 - lon1) + sin(lat1) * sin(lat2))
        
        $formula = $earthRadius . ' * acos(
            cos(radians(' . floatval($lat) . ')) * 
            cos(radians(' . $latField . ')) * 
            cos(radians(' . $lonField . ') - radians(' . floatval($lon) . ')) + 
            sin(radians(' . floatval($lat) . ')) * 
            sin(radians(' . $latField . '))
        )';
        
        return $formula;
    }


    /**
     * Get center point of all properties
     * @return object|null Object with lat, lon
     */
    static function getPropertiesCenter()
    {
         $db = Factory::getContainer()->get('db');
        $query = $db->getQuery(true);
        
        $query->select('AVG(lat_add) AS center_lat, AVG(long_add) AS center_lon')
            ->select('COUNT(*) AS total_properties')
            ->from('#__osrs_properties')
            ->where('published = 1')
            ->where('lat_add IS NOT NULL')
            ->where('long_add IS NOT NULL')
            ->where('lat_add != 0')
            ->where('long_add != 0');
        
        $db->setQuery($query);
        
        try {
            return $db->loadObject();
        } catch (Exception $e) {
            return null;
        }
    }



    /**
     * Find properties within map bounds
     * @param float $north North boundary
     * @param float $south South boundary
     * @param float $east East boundary
     * @param float $west West boundary
     * @param array $filters Additional filters
     * @return array Properties
     */
    public static function findPropertiesInBounds($north, $south, $east, $west, $filters = array())
    {
        $db = Factory::getContainer()->get('db');
        $query = $db->getQuery(true);
        
        // Base query
        $query->select('p.*')
              ->from($db->quoteName('#__osrs_properties', 'p'))
              ->where('p.published = 1')
              ->where('p.lat_add IS NOT NULL')
              ->where('p.long_add IS NOT NULL')
              ->where('p.lat_add != 0')
              ->where('p.long_add != 0');
        
        // Bounds filter
        $query->where('CAST(p.lat_add AS DECIMAL(10,7)) >= ' . (float)$south)
              ->where('CAST(p.lat_add AS DECIMAL(10,7)) <= ' . (float)$north);
        
        // Handle longitude wrap around (crossing date line)
        if ($west <= $east) {
            $query->where('CAST(p.long_add AS DECIMAL(10,7)) >= ' . (float)$west)
                  ->where('CAST(p.long_add AS DECIMAL(10,7)) <= ' . (float)$east);
        } else {
            // Crossing date line
            $query->where('(CAST(p.long_add AS DECIMAL(10,7)) >= ' . (float)$west . ' OR CAST(p.long_add AS DECIMAL(10,7)) <= ' . (float)$east . ')');
        }
        
        // Apply filters
        self::applyFilters($query, $filters);
        
        // Limit for performance (max 500 properties in view)
        $query->setLimit(500);
        
        $db->setQuery($query);
        return $db->loadObjectList();
    }

    /**
     * Check if point is inside polygon using Ray Casting algorithm
     * @param float $lat Point latitude
     * @param float $lon Point longitude
     * @param array $polygon Array of [lat, lon] coordinates
     * @return bool True if inside polygon
     */
    public static function isPointInPolygon($lat, $lon, $polygon)
    {
        $vertices = count($polygon);
        if ($vertices < 3) {
            return false;
        }
        
        $inside = false;
        
        for ($i = 0, $j = $vertices - 1; $i < $vertices; $j = $i++) {
            $xi = $polygon[$i][0]; // lat
            $yi = $polygon[$i][1]; // lon
            $xj = $polygon[$j][0]; // lat
            $yj = $polygon[$j][1]; // lon
            
            $intersect = (($yi > $lon) != ($yj > $lon))
                         && ($lat < ($xj - $xi) * ($lon - $yi) / ($yj - $yi) + $xi);
            
            if ($intersect) {
                $inside = !$inside;
            }
        }
        
        return $inside;
    }

    /**
     * Find properties within polygon
     * @param array $coordinates Array of [lat, lon] coordinates
     * @param array $filters Additional filters
     * @return array Properties
     */
    public static function findPropertiesInPolygon($coordinates, $filters = array())
    {
        // Get bounding box of polygon for initial filter
        $lats = array_column($coordinates, 0);
        $lons = array_column($coordinates, 1);
        
        $north = max($lats);
        $south = min($lats);
        $east = max($lons);
        $west = min($lons);
        
        // Get properties in bounding box first
        $properties = self::findPropertiesInBounds($north, $south, $east, $west, $filters);
        
        // Filter by actual polygon
        $results = array();
        foreach ($properties as $property) {
            if (self::isPointInPolygon($property->lat_add, $property->long_add, $coordinates)) {
                $results[] = $property;
            }
        }
        
        return $results;
    }

    /**
     * Apply common filters to query
     * @param object $query Database query object
     * @param array $filters Filter array
     */
    private static function applyFilters(&$query, $filters)
    {
        $db = Factory::getContainer()->get('db');
        if (!empty($filters['type_id'])) {
            $query->where('p.pro_type = ' . (int)$filters['type_id']);
        }
        
        if (!empty($filters['category_id'])) {
            $subQuery = $db->getQuery(true)
                ->select('1')
                ->from($db->quoteName('#__osrs_property_categories', 'pc'))
                ->where('pc.pid = p.id')
                ->where('pc.category_id = ' . (int)$filters['category_id']);
            
            $query->where('EXISTS (' . $subQuery . ')');
        }
        
        if (isset($filters['price_min']) && $filters['price_min'] > 0) {
            $query->where('p.price >= ' . (float)$filters['price_min']);
        }
        
        if (isset($filters['price_max']) && $filters['price_max'] > 0) {
            $query->where('p.price <= ' . (float)$filters['price_max']);
        }
        
        if (!empty($filters['bedrooms'])) {
            $query->where('p.bed_room >= ' . (int)$filters['bedrooms']);
        }
        
        if (!empty($filters['bathrooms'])) {
            $query->where('p.bath_room >= ' . (int)$filters['bathrooms']);
        }
        
        if (!empty($filters['agent_id'])) {
            $query->where('p.agent_id = ' . (int)$filters['agent_id']);
        }
        
        if (!empty($filters['featured'])) {
            $query->where('p.isFeatured = 1');
        }
        
        // Search by property name or address
        if (!empty($filters['search'])) {
            $search = $query->quote('%' . $filters['search'] . '%');
            $query->where('(p.pro_name LIKE ' . $search . ' OR p.address LIKE ' . $search . ')');
        }
    }

    /**
     * Format property data for map display
     * @param object $property Property object from database
     * @return array Formatted property data
     */
    public static function formatPropertyForMap($property)
    {
        $db = Factory::getContainer()->get('db');
        
        // Get property type name
        $db->setQuery("SELECT type_name FROM `#__osrs_types` WHERE id = " . (int)$property->pro_type);
        $type_name = $db->loadResult();
        
        // Get category name
        $db->setQuery("SELECT `category_name` FROM `#__osrs_categories` WHERE id = " . (int)$property->category_ids);
        $category_name = $db->loadResult();
        
        
        $db->setQuery("SELECT `image` FROM `#__osrs_photos` 
                        WHERE pro_id = " . (int)$property->id . " 
                        ORDER BY ordering LIMIT 1");
        $main_image = $db->loadResult();
        
        
        // Build image URL
        $image_url = '';
        if ($main_image) {
            $image_path = 'images/osproperty/' . $property->id . '/small_' . $main_image;
            if (file_exists(JPATH_ROOT . '/' . $image_path)) {
                $image_url = Uri::root() . $image_path;
            } else {
                $image_url = Uri::root() . 'images/osproperty/properties/' . $property->id . '/' . $main_image;
            }
        } else {
            $image_url = Uri::root() . 'media/com_osproperty/assets/images/no-image.jpg';
        }
        
        // Build property URL
        $property_url = Route::_('index.php?option=com_osproperty&task=property_details&id=' . $property->id . ':' . $property->alias);
        
        // Format price
        //$price_formatted = OSPHelper::showPrice($property->price);

        ob_start();
        if(OSPHelper::getLanguageFieldValue($property,'price_text') != "")
		{
			echo OSPHelper::showPriceText(OSPHelper::getLanguageFieldValue($property,'price_text')) ;
		}
		elseif($property->price_call == 0)
		{
			if($property->price > 0)
			{
				echo OSPHelper::generatePrice($property->curr,$property->price);
				if($property->rent_time != "")
				{
					echo "/".Text::_($property->rent_time);
				}
			}
		}
		elseif($property->price_call == 1)
		{
			echo Text::_('OS_CALL_FOR_DETAILS_PRICE') ;
		}
        $price_formatted = ob_get_contents();
        ob_end_clean();
        
        
        // Get agent name if exists
        $agent_name = '';
        if ($property->agent_id > 0) {
            $db->setQuery("SELECT name FROM `#__osrs_agents` WHERE id = " . (int)$property->agent_id);
            $agent_name = $db->loadResult();
        }
        
        return array(
            'id' => $property->id,
            'lat' => (float)$property->lat_add,
            'lng' => (float)$property->long_add,
            'title' => $property->pro_name,
            'alias' => $property->alias,
            'address' => $property->address,
            'price' => $price_formatted,
            'price_raw' => $property->price,
            'type' => $type_name,
            'type_id' => $property->type_id,
            'category' => $category_name,
            'category_id' => $property->category_id,
            'bedrooms' => $property->bed_room,
            'bathrooms' => $property->bath_room,
            'area' => $property->square_feet,
            'area_unit' => OSPHelper::showSquareSymbol(),
            'image' => $image_url,
            'url' => $property_url,
            'featured' => $property->isFeatured,
            'agent_name' => $agent_name,
            'agent_id' => $property->agent_id,
            'distance' => isset($property->distance) ? $property->distance : null
        );
    }

    /**
     * Get marker icon class based on property type/status
     * @param object $property Property object
     * @return string Icon class name
     */
    public static function getPropertyMarkerIcon($property)
    {
        if ($property->featured == 1) {
            return 'marker-featured';
        }
        
        // Based on type_id from #__osrs_types table
        // Customize based on your property types
        return 'marker-type-' . $property->type_id;
    }

    /**
     * Geocode address to coordinates using Nominatim
     * @param string $address Address to geocode
     * @return array|false Array with lat, lon, display_name or false on failure
     */
    public static function geocodeAddress($address)
    {
        if (empty($address)) {
            return false;
        }
        
        $url = 'https://nominatim.openstreetmap.org/search?format=json&q=' . urlencode($address) . '&limit=1';
        
        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, $url);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, 1);
        curl_setopt($ch, CURLOPT_USERAGENT, 'OS Property Locator');
        curl_setopt($ch, CURLOPT_TIMEOUT, 10);
        
        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);
        
        if ($httpCode == 200 && $response) {
            $data = json_decode($response, true);
            if (!empty($data) && isset($data[0])) {
                return array(
                    'lat' => $data[0]['lat'],
                    'lon' => $data[0]['lon'],
                    'display_name' => $data[0]['display_name']
                );
            }
        }
        
        return false;
    }

    /**
     * Reverse geocode coordinates to address
     * @param float $lat Latitude
     * @param float $lon Longitude
     * @return array|false Array with address data or false
     */
    public static function reverseGeocode($lat, $lon)
    {
        if ($lat == 0 || $lon == 0) {
            return false;
        }
        
        $url = 'https://nominatim.openstreetmap.org/reverse?format=json&lat=' . $lat . '&lon=' . $lon;
        
        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, $url);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, 1);
        curl_setopt($ch, CURLOPT_USERAGENT, 'OS Property Locator');
        curl_setopt($ch, CURLOPT_TIMEOUT, 10);
        
        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);
        
        if ($httpCode == 200 && $response) {
            $data = json_decode($response, true);
            if (!empty($data)) {
                return array(
                    'display_name' => $data['display_name'],
                    'address' => $data['address']
                );
            }
        }
        
        return false;
    }
}
?>