jQuery(function($){
	'use strict';

	const opts = Joomla.getOptions('jmapViewer');
	if (!opts || !opts.xmlUrl) return;

	const format = (opts.format || 'xml').toLowerCase();

	// --------------------------
	// Utils
	// --------------------------
	const ucfirst = str => (str ? str.charAt(0).toUpperCase() + str.slice(1) : '');
	const formatLastMod = value => {
		if (!value) return '';
		return value
			.replace('T', ' ')
			.replace('Z', '')
			.replace(/:\d{2}$/, ''); // remove seconds
	};

	const escapeHtml = (s) => {
		if (s === null || s === undefined) return '';
		return String(s)
			.replace(/&/g, '&amp;')
			.replace(/</g, '&lt;')
			.replace(/>/g, '&gt;')
			.replace(/"/g, '&quot;')
			.replace(/'/g, '&#039;');
	};
	
	function escapeAttr(str) {
		return escapeHtml(str);
	}

	// Namespace-safe getter (for image/news/video namespaces)
	const getByLocalName = (ctx, localName) => {
		if (!ctx) return '';
		const nodes = ctx.getElementsByTagNameNS('*', localName);
		return nodes && nodes[0] ? (nodes[0].textContent || '') : '';
	};
	
	function showLoader() {
		if (!document.getElementById('jmap-loader')) {
			const loader = document.createElement('div');
			loader.id = 'jmap-loader';
			loader.innerHTML = `
				<div class="jmap-spinner"></div>
				<div class="jmap-loader-text">
					Loading sitemap, please wait…
				</div>
			`;
			document.body.appendChild(loader);
		}
	}

	function hideLoader() {
		const loader = document.getElementById('jmap-loader');
		if (loader) {
			loader.remove();
		}
	}

	// --------------------------
	// Renderer registry
	// --------------------------
	const Renderers = {
		xml: {
			parse(xmlDoc) {
				const urls = Array.from(xmlDoc.getElementsByTagName('url'));
				return {
					totalUrls: urls.length,
					items: urls.map(u => ({
						loc: (u.getElementsByTagName('loc')[0]?.textContent || ''),
						priority: u.getElementsByTagName('priority')[0]?.textContent || '',
						changefreq: u.getElementsByTagName('changefreq')[0]?.textContent || '',
						lastmod: u.getElementsByTagName('lastmod')[0]?.textContent || ''
					}))
				};
			},
			render(data) {
				const tbody = $('#sitemap-body');
				tbody.empty();

				data.items.forEach((item) => {
					const pr = Math.round(parseFloat(item.priority || '0') * 100);

					const tr = $('<tr></tr>');
					tr.append(`<td><a target="_blank" rel="noopener" href="${escapeHtml(item.loc)}">${escapeHtml(item.loc)}</a></td>`);
					tr.append(`<td class="right">${isFinite(pr) ? pr : 0}%</td>`);
					tr.append(`<td class="right">${escapeHtml(ucfirst(item.changefreq))}</td>`);
					tr.append(`<td class="right">${escapeHtml(formatLastMod(item.lastmod))}</td>`);

					tbody.append(tr);
				});

				// tablesorter
				$("#sitemap").tablesorter();
			},
			afterRender(data) {
				// Chart data uses hidden counters
				$('#urls_count').text(data.totalUrls);
				$('#urls_count_label').text(data.totalUrls);
			}
		},

		images: {
			parse(xmlDoc) {
				const urls = Array.from(xmlDoc.getElementsByTagName('url'));
				let totalImages = 0;

				const items = urls.map(u => {
					const loc = (u.getElementsByTagName('loc')[0]?.textContent || '');

					// Find all <image:image> regardless of prefix
					const imageNodes = Array.from(u.getElementsByTagNameNS('*', 'image'));
					const images = imageNodes.map(imgNode => {
						// inside <image:image> we have <image:loc>, <image:title>, <image:caption> ...
						const imgLoc = getByLocalName(imgNode, 'loc');
						const imgTitle = getByLocalName(imgNode, 'title');
						const imgCaption = getByLocalName(imgNode, 'caption');

						return {
							loc: imgLoc,
							desc: imgTitle || imgCaption || ''
						};
					});

					totalImages += images.length;

					return {
						pageLoc: loc,
						images
					};
				});

				return {
					totalUrls: items.length,
					totalImages,
					items
				};
			},

			render(data) {
				const tbody = $('#sitemap-body');
				tbody.empty();

				data.items.forEach((item) => {
					// Parent URL row
					const parentTr = $('<tr></tr>');
					parentTr.append(`<td><a target="_blank" rel="noopener" href="${escapeHtml(item.pageLoc)}">${escapeHtml(item.pageLoc)}</a></td>`);
					parentTr.append('<td></td>');
					parentTr.append('<td></td>');
					parentTr.append(`<td class="right">${item.images.length}</td>`);
					tbody.append(parentTr);

					// Child image rows
					item.images.forEach(img => {
						const imgTr = $('<tr class="image-row"></tr>');

						const imgUrl = escapeHtml(img.loc);
						const desc = escapeHtml(img.desc);

						// URL column: show image URL as text (like XSL)
						imgTr.append(`<td class="image-url">${imgUrl ? `<a target="_blank" rel="noopener" href="${imgUrl}">${imgUrl}</a>` : ''}</td>`);

						// Preview column: try to render <img>, if broken just empty
						imgTr.append(
							`<td class="image-preview">${
								imgUrl ? `<img src="${imgUrl}" loading="lazy" alt="">` : ''
							}</td>`
						);

						// Description column
						imgTr.append(`<td class="image-desc">${desc}</td>`);

						// Images count column empty for child rows
						imgTr.append('<td></td>');

						tbody.append(imgTr);
					});
				});

			},

			afterRender(data) {
				$('#urls_count').text(data.totalUrls);
				$('#urls_count_label').text(data.totalUrls);

				$('#images_count').text(data.totalImages);

				// Optional visible label if you add it in template
				const $imgLabel = $('#images_count_label');
				if ($imgLabel.length) $imgLabel.text(data.totalImages);
			}
		},
		gnews: {
			parse(xmlDoc) {
				const urls = Array.from(xmlDoc.getElementsByTagName('url'));
				const NS = 'http://www.google.com/schemas/sitemap-news/0.9';

				return urls.map(url => {
					const get = tag =>
						url.getElementsByTagName(tag)[0]?.textContent || '';

					const news = url.getElementsByTagNameNS(NS, 'news')[0];

					const getNews = tag =>
						news?.getElementsByTagNameNS(NS, tag)[0]?.textContent || '';

					const publication = news?.getElementsByTagNameNS(NS, 'publication')[0];

					const getPub = tag =>
						publication?.getElementsByTagNameNS(NS, tag)[0]?.textContent || '';

					return {
						loc: get('loc'),
						title: getNews('title'),
						genre: getNews('genres'),
						publicationDate: getNews('publication_date'),
						publicationName: getPub('name')
					};
				});
			},

			render(data) {
				const tbody = document.getElementById('sitemap-body');
				tbody.innerHTML = '';

				data.forEach((item, index) => {
					const tr = document.createElement('tr');
					if (index % 2 === 1) tr.classList.add('odd');

					tr.innerHTML = `
						<td><a target="_blank" href="${item.loc}">${item.loc}</a></td>
						<td>${item.title}</td>
						<td></td>
						<td>${item.genre}</td>
						<td class="right">${Renderers.gnews.formatDate(item.publicationDate)}</td>
					`;

					tbody.appendChild(tr);
				});
				
				// tablesorter
				$("#sitemap").tablesorter();
			},

			afterRender(data) {
				const total = data.length;

				document.getElementById('urls_count').innerText = total;
				const lbl = document.getElementById('urls_count_label');
				if (lbl) lbl.innerText = total;

				const pubName = data[0]?.publicationName || '';
				const pubEl = document.getElementById('gnews_publication_name');
				if (pubEl) pubEl.innerText = pubName;
			},

			formatDate(value) {
				if (!value) return '';
				return value
					.replace('T', ' ')
					.replace(/\+\d{2}:\d{2}$/, '')
					.replace(/:\d{2}$/, '');
			}
		},
		mobile: {
			parse(xmlDoc) {
				const urls = Array.from(xmlDoc.getElementsByTagName('url'));

				return urls.map(url => {
					const loc = url.getElementsByTagName('loc')[0]?.textContent || '';
					return { loc };
				});
			},

			render(data) {
				const tbody = document.getElementById('sitemap-body');
				tbody.innerHTML = '';

				data.forEach((item, index) => {
					const tr = document.createElement('tr');
					if (index % 2 === 1) tr.classList.add('odd');

					tr.innerHTML = `
						<td>
							<a target="_blank" href="${item.loc}">
								${item.loc}
							</a>
						</td>
					`;

					tbody.appendChild(tr);
				});
				
				// tablesorter
				$("#sitemap").tablesorter();
			},

			afterRender(data) {
				const total = data.length;

				// Chart counters
				document.getElementById('urls_count').innerText = total;
				const lbl = document.getElementById('urls_count_label');
				if (lbl) lbl.innerText = total;
			}
		},
		videos: {
			parse(xmlDoc) {
				const VIDEO_NS = 'http://www.google.com/schemas/sitemap-video/1.1';

				const urlNodes = Array.from(xmlDoc.getElementsByTagName('url'));
				const groups = [];

				let totalVideos = 0;

				const getText = (node, tagName, ns = null) => {
					const el = ns
						? node.getElementsByTagNameNS(ns, tagName)[0]
						: node.getElementsByTagName(tagName)[0];
					return el?.textContent?.trim() || '';
				};

				const getAttr = (node, tagName, attrName) => {
					const el = node.getElementsByTagNameNS(VIDEO_NS, tagName)[0];
					return el?.getAttribute(attrName) || '';
				};

				urlNodes.forEach((urlNode) => {
					const loc = getText(urlNode, 'loc');

					const videoNodes = Array.from(
						urlNode.getElementsByTagNameNS(VIDEO_NS, 'video')
					);

					const videos = videoNodes.map((v) => ({
						thumbnail: getText(v, 'thumbnail_loc', VIDEO_NS),
						title: getText(v, 'title', VIDEO_NS),
						description: getText(v, 'description', VIDEO_NS),
						player: getText(v, 'player_loc', VIDEO_NS),
						playerAllowEmbed: getAttr(v, 'player_loc', 'allow_embed'),
						playerAutoplay: getAttr(v, 'player_loc', 'autoplay'),
						viewCount: getText(v, 'view_count', VIDEO_NS),
						duration: getText(v, 'duration', VIDEO_NS),
						pubDate: getText(v, 'publication_date', VIDEO_NS),
						uploader: getText(v, 'uploader', VIDEO_NS),
						live: getText(v, 'live', VIDEO_NS),
					}));

					totalVideos += videos.length;

					groups.push({
						loc,
						videosCount: videos.length,
						videos
					});
				});

				return {
					groups,
					urlsCount: urlNodes.length,
					videosCount: totalVideos
				};
			},

			render(data) {
				const tbody = document.getElementById('sitemap-body');
				tbody.innerHTML = '';

				const escapeHtml = (s) =>
					(s || '')
						.replaceAll('&', '&amp;')
						.replaceAll('<', '&lt;')
						.replaceAll('>', '&gt;')
						.replaceAll('"', '&quot;')
						.replaceAll("'", '&#039;');

				const formatDate = (v) => {
					if (!v) return '';
					// Match XSL feel: no seconds, keep date + hh:mm
					return v
						.replace('T', ' ')
						.replace('Z', '')
						.replace(/\+\d{2}:\d{2}$/, '')
						.replace(/:\d{2}$/, '');
				};

				const formatDuration = (secondsStr) => {
					const s = parseInt(secondsStr, 10);
					if (!Number.isFinite(s) || s <= 0) return '';
					const mm = Math.floor(s / 60);
					const ss = (s % 60).toString().padStart(2, '0');
					return `${mm}:${ss}`;
				};

				data.groups.forEach((group, groupIndex) => {
					// ==========================
					// MASTER row (URL principale)
					// ==========================
					const trMaster = document.createElement('tr');
					if (groupIndex % 2 === 1) trMaster.classList.add('odd');

					trMaster.innerHTML = `
						<td>
							<a href="${escapeHtml(group.loc)}" target="_blank" rel="noopener">${escapeHtml(group.loc)}</a>
						</td>
						<td></td>
						<td></td>
						<td></td>
						<td></td>
						<td class="right"></td>
						<td class="right"></td>
						<td class="right"></td>
						<td></td>
						<td class="right">${group.videosCount || 0}</td>
					`;

					tbody.appendChild(trMaster);

					// ==========================
					// SUB rows (un video per riga)
					// ==========================
					group.videos.forEach((v) => {
						const tr = document.createElement('tr');
						tr.classList.add('video-row');
						
						// URL col: nello screenshot/XSL, sotto spesso mettono player URL
						const playerLink = v.player
							? `<a href="${escapeHtml(v.player)}" target="_blank" rel="noopener">${escapeHtml(v.player)}</a>`
							: '';

						const thumb = v.thumbnail
							? `<img src="${escapeHtml(v.thumbnail)}" class="video_thumbnail" loading="lazy">`
							: '';

						// Preview iframe (come XSL): embed YouTube/Vimeo ecc
						// Nota: non forzo autoplay, ma puoi aggiungerlo se vuoi
						const preview = v.player
							? `<iframe width="280" height="156"
								src="${escapeHtml(v.player)}"
								frameborder="0"
								allowfullscreen
								loading="lazy"></iframe>`
							: '';

						tr.innerHTML = `
							<td>${playerLink}</td>
							<td>${thumb}</td>
							<td>${preview}</td>
							<td>${escapeHtml(v.title)}</td>
							<td>${escapeHtml(v.description)}</td>
							<td class="right">${escapeHtml(v.viewCount)}</td>
							<td class="right">${formatDuration(v.duration)}</td>
							<td class="right">${formatDate(v.pubDate)}</td>
							<td>${escapeHtml(v.uploader)}</td>
							<td class="right"></td>
						`;

						tbody.appendChild(tr);
					});
				});
			},

			afterRender(data) {
			    const urlsCount = data.urlsCount || 0;
			    const videosCount = data.videosCount || 0;

			    const elUrls = document.getElementById('urls_count');
			    if (elUrls) elUrls.textContent = urlsCount;

			    const elUrlsLabel = document.getElementById('urls_count_label');
			    if (elUrlsLabel) elUrlsLabel.textContent = urlsCount;

			    const elVideos = document.getElementById('videos_count');
			    if (elVideos) elVideos.textContent = videosCount;
				
				const elVideosLabel = document.getElementById('videos_count_label');
				if (elVideosLabel) elVideosLabel.textContent = videosCount;
			}

		},
		hreflang: {
			parse(xmlDoc) {
				const urls = Array.from(xmlDoc.getElementsByTagName('url'));

				const rows = urls.map(urlNode => {
					const loc = (urlNode.getElementsByTagName('loc')[0]?.textContent || '').trim();

					// NB: in DOMParser il namespace può sparire nel tagName, quindi cerchiamo per localName
					const alternates = Array.from(urlNode.getElementsByTagName('*'))
						.filter(n => (n.localName || '').toLowerCase() === 'link') // xhtml:link
						.filter(n => (n.getAttribute('rel') || '').toLowerCase() === 'alternate')
						.map(n => ({
							lang: (n.getAttribute('hreflang') || '').trim(),
							href: (n.getAttribute('href') || '').trim()
						}))
						.filter(a => a.lang && a.href);

					return {
						loc,
						alternateCount: alternates.length,
						alternates
					};
				});

				// opzionale: totale alternates (se vuoi mostrarlo da qualche parte / hidden)
				const hreflangCount = rows.reduce((sum, r) => sum + (r.alternateCount || 0), 0);

				return {
					rows,
					urlsCount: rows.length,
					hreflangCount
				};
			},

			render(data) {
				const tbody = document.getElementById('sitemap-body');
				if (!tbody) return;

				tbody.innerHTML = '';

				data.rows.forEach((row, idx) => {
					// master row
					const trMaster = document.createElement('tr');
					if (idx % 2 === 1) trMaster.classList.add('odd');

					trMaster.innerHTML = `
						<td>
							<a target="_blank" rel="noopener" href="${escapeAttr(row.loc)}">${escapeHtml(row.loc)}</a>
						</td>
						<td class="right"></td>
						<td class="right">${row.alternateCount}</td>
					`;
					tbody.appendChild(trMaster);

					// children rows (one per language)
					row.alternates.forEach(a => {
						const trChild = document.createElement('tr');
						trChild.classList.add('hreflang-child');

						trChild.innerHTML = `
							<td class="hreflang-suburl">
								<a target="_blank" rel="noopener" href="${escapeAttr(a.href)}">${escapeHtml(a.href)}</a>
							</td>
							<td class="right">${escapeHtml(a.lang)}</td>
							<td class="right"></td>
						`;
						tbody.appendChild(trChild);
					});
				});
			},

			afterRender(data) {
				const urlsCount = data.urlsCount || 0;
				const hreflangCount = data.hreflangCount || 0;

				const elUrls = document.getElementById('urls_count');
				if (elUrls) elUrls.textContent = urlsCount;

				const elUrlsLabel = document.getElementById('urls_count_label');
				if (elUrlsLabel) elUrlsLabel.textContent = urlsCount;

				const elH = document.getElementById('hreflang_count');
				if (elH) elH.textContent = hreflangCount;
			}
		},
		amp: {
			parse(xmlDoc) {
				const urls = Array.from(xmlDoc.getElementsByTagName('url'));
				return {
					totalUrls: urls.length,
					items: urls.map(u => ({
						loc: (u.getElementsByTagName('loc')[0]?.textContent || ''),
						priority: u.getElementsByTagName('priority')[0]?.textContent || '',
						changefreq: u.getElementsByTagName('changefreq')[0]?.textContent || '',
						lastmod: u.getElementsByTagName('lastmod')[0]?.textContent || ''
					}))
				};
			},
			render(data) {
				const tbody = $('#sitemap-body');
				tbody.empty();

				data.items.forEach((item) => {
					const pr = Math.round(parseFloat(item.priority || '0') * 100);

					const tr = $('<tr></tr>');
					tr.append(`<td><a target="_blank" rel="noopener" href="${escapeHtml(item.loc)}">${escapeHtml(item.loc)}</a></td>`);
					tr.append(`<td class="right">${isFinite(pr) ? pr : 0}%</td>`);
					tr.append(`<td class="right">${escapeHtml(ucfirst(item.changefreq))}</td>`);
					tr.append(`<td class="right">${escapeHtml(formatLastMod(item.lastmod))}</td>`);

					tbody.append(tr);
				});

				// tablesorter
				$("#sitemap").tablesorter();
			},
			afterRender(data) {
				// Chart data uses hidden counters
				$('#urls_count').text(data.totalUrls);
				$('#urls_count_label').text(data.totalUrls);
			}
		}
	};

	// Fallback safety
	const renderer = Renderers[format] || Renderers.xml;

	
	// --------------------------
	// Fetch + render pipeline
	// --------------------------
	showLoader();
	if (opts.format === 'images') {
		const txt = document.querySelector('.jmap-loader-text');
		if (txt) {
			txt.innerText = opts.imagesCrawlingMessage;
		}
	}
	if (opts.format === 'videos') {
		const txt = document.querySelector('.jmap-loader-text');
		if (txt) {
			txt.innerText = opts.videosCrawlingMessage;
		}
	}
		
	fetch(opts.xmlUrl, { cache: 'no-store' })
		.then(r => r.text())
		.then(xmlText => {
			const xml = new DOMParser().parseFromString(xmlText, 'text/xml');

			// Basic XML parse error guard
			const parseError = xml.getElementsByTagName('parsererror');
			if (parseError && parseError.length) throw new Error('Invalid XML');

			const data = renderer.parse(xml);

			renderer.render(data);
			renderer.afterRender(data);

			// Chart uses #sitemap_type, #urls_count, (#images_count optionally)
			drawStatsChart();
			
			hideLoader();
		})
		.catch(err => {
			hideLoader();
		});

	// --------------------------
	// Chart.js v1 style
	// --------------------------
	function drawStatsChart() {
		function ChartWrapper(canvasSelector) {
			var chartOptions = {
				animation: true,
				scaleFontSize: 11
			};

			var chartData = {};
			var stats = {};
			var sitemapType = 'xml';
			var ctx;

			function buildChart(animate) {
				var chart = new Chart(ctx);

				chartData.labels = [];
				chartData.datasets = [];

				var values = [];

				if (sitemapType === 'images') {
					delete stats[''];
					stats[opts.labelTotalImages] = parseInt($('#images_count').html(), 10) || 0;
				}
				
				if (sitemapType === 'videos') {
	                delete stats[''];
	                stats[opts.labelTotalVideos] = parseInt($('#videos_count').html(), 10) || 0;
	            }

				$.each(stats, function (label, value) {
					chartData.labels.push(label);
					values.push(value);
				});

				chartData.datasets[0] = {
					fillColor: 'rgba(151,187,205,0.5)',
					strokeColor: 'rgba(151,187,205,1)',
					pointColor: 'rgba(151,187,205,1)',
					pointStrokeColor: '#fff',
					data: values
				};

				chartOptions.animation = animate;
				chart.Line(chartData, chartOptions);
			}

			function resizeAndDraw(animate) {
				var canvas = $(canvasSelector).get(0);
				if (!canvas) return;

				var width = $(canvas).parent().width() || 500;
				canvas.width = width;
				canvas.height = 150;

				buildChart(animate);
			}

			(function init() {
				sitemapType = ($('#sitemap_type').html() || 'xml').toLowerCase();
				
				if(sitemapType === 'gnews') {
					sitemapType = 'Google News';
				}

				stats['      ' + sitemapType.charAt(0).toUpperCase() + sitemapType.slice(1) + ' sitemap links'] = 0;
				stats[opts.labelTotalURLs] = parseInt($('#urls_count').html(), 10) || 0;
				stats[''] = 0;

				if (!!document.createElement('canvas').getContext) {
					ctx = $(canvasSelector).get(0).getContext('2d');
					resizeAndDraw(true);
				}
			})();
		}

		new ChartWrapper('#chart_canvas');
	}
});
