<?php

/**
 * @version             6.0.2
 * @package             TAGZ
 * @copyright           Copyright (C) 2025 roosterz.nl
 * @license             GNU/GPLv3 http://www.gnu.org/licenses/gpl-3.0.html
 */

defined('_JEXEC') or die;

/**
 * Joomla compatibility wrapper
 */
if (!class_exists('TagzJoomlaCompat')) {
    class TagzJoomlaCompat
    {
        public static function app()
        {
            return class_exists('\Joomla\CMS\Factory')
                ? \Joomla\CMS\Factory::getApplication()
                : \JFactory::getApplication();
        }

        public static function db()
        {
            return class_exists('\Joomla\CMS\Factory')
                ? \Joomla\CMS\Factory::getDbo()
                : \JFactory::getDbo();
        }

        public static function cache()
        {
            if (class_exists('\Joomla\CMS\Factory')) {
                return \Joomla\CMS\Factory::getCache();
            }
            return \JFactory::getCache();
        }

        public static function text($key, ...$args)
        {
            if (class_exists('\Joomla\CMS\Language\Text')) {
                return empty($args) ? 
                    \Joomla\CMS\Language\Text::_($key) : 
                    \Joomla\CMS\Language\Text::sprintf($key, ...$args);
            }
            return empty($args) ?
                \JText::_($key) :
                \JText::sprintf($key, ...$args);
        }

        public static function version()
        {
            return defined('JVERSION') ? JVERSION : \Joomla\CMS\Version::MAJOR_VERSION;
        }
    }
}

class pkg_tagzInstallerScript
{
    private $min_joomla_version = '3.7';
    private $min_php_version    = '5.4.0';
    private $app;

    public function __construct()
    {
        $this->app = TagzJoomlaCompat::app();
    }

    /**
     * Requirements checks
     */
    public function preflight($route, $adapter)
    {
        if (!$this->passMinimumRequirementVersion("joomla")) {
            return false;
        }

        if (!$this->passMinimumRequirementVersion("php")) {
            return false;
        }

        return true;
    }

    /**
     * Installation / Update actions
     */
    public function postflight($type, $parent)
    {
        if ($type === 'install') {
            $this->enablePlugin('tagz', 'system');

            $this->app->enqueueMessage(
                TagzJoomlaCompat::text('TAGZ successfully installed!')
            );

            $this->app->redirect('index.php?option=com_tagz');
        }

        if ($type === 'update') {
            $this->correctElementField();
            $this->enablePlugin('tagz', 'system');

            $url = 'index.php?option=com_tagz';

            $msg = sprintf(
                '<strong>TAGZ</strong> updated to version <strong>6.0.2 PRO</strong>. ' .
                'Open dashboard <a href="%s">HERE</a>. ' .
                'Changelog <a href="https://www.roosterz.nl/support/changelog/tagz" target="_blank">HERE</a>.',
                $url
            );

            $this->app->enqueueMessage($msg, 'message');
        }

        TagzJoomlaCompat::cache()->clean('com_plugins');
    }

    /**
     * Uninstall actions
     *
     * @param object $parent Installer adapter
     * @return void
     */
    public function uninstall($parent)
    {
        // try to disable the system plugin we enabled on install
        $this->disablePlugin('tagz', 'system');

        $this->app->enqueueMessage(
            TagzJoomlaCompat::text('TAGZ successfully uninstalled.')
        );

        // Clear plugin cache so admin UI updates
        TagzJoomlaCompat::cache()->clean('com_plugins');
    }

    /**
     * Fix incorrect pkg extension name case (J3 cleanup)
     */
    private function correctElementField()
    {
        $db = TagzJoomlaCompat::db();
        $query = $db->getQuery(true)
            ->update($db->quoteName('#__extensions'))
            ->set($db->quoteName('element') . ' = ' . $db->quote('pkg_tagz'))
            ->where($db->quoteName('element') . ' = ' . $db->quote('pkg_TAGZ'))
            ->where($db->quoteName('type') . ' = ' . $db->quote('package'));

        $db->setQuery($query);
        $db->execute();
    }

    /**
     * Joomla / PHP version checks
     */
    private function passMinimumRequirementVersion($type = "joomla")
    {
        if ($type === 'joomla') {
            if (version_compare(TagzJoomlaCompat::version(), $this->min_joomla_version, '<')) {
                $this->app->enqueueMessage(
                    TagzJoomlaCompat::text(
                        'This extension is no longer supported on Joomla! %1$s. Please update to %2$s or higher.',
                        TagzJoomlaCompat::version(),
                        $this->min_joomla_version
                    ),
                    'error'
                );
                return false;
            }
        }

        if ($type === 'php') {
            if (version_compare(PHP_VERSION, $this->min_php_version, '<')) {
                $this->app->enqueueMessage(
                    TagzJoomlaCompat::text(
                        'This extension requires PHP %2$s or higher. Your version: %1$s',
                        PHP_VERSION,
                        $this->min_php_version
                    ),
                    'error'
                );
                return false;
            }
        }

        return true;
    }

    /**
     * Enable required system plugin
     */
    private function enablePlugin($name, $plugin_folder)
    {
        $db = TagzJoomlaCompat::db();
        $query = $db->getQuery(true)
            ->update($db->quoteName('#__extensions'))
            ->set($db->quoteName('enabled') . ' = 1')
            ->where($db->quoteName('type') . ' = ' . $db->quote('plugin'))
            ->where($db->quoteName('folder') . ' = ' . $db->quote($plugin_folder))
            ->where($db->quoteName('element') . ' = ' . $db->quote($name));

        $db->setQuery($query);
        $db->execute();
    }

    /**
     * Disable a plugin by element and folder
     */
    private function disablePlugin($name, $plugin_folder)
    {
        $db = TagzJoomlaCompat::db();
        $query = $db->getQuery(true)
            ->update($db->quoteName('#__extensions'))
            ->set($db->quoteName('enabled') . ' = 0')
            ->where($db->quoteName('type') . ' = ' . $db->quote('plugin'))
            ->where($db->quoteName('folder') . ' = ' . $db->quote($plugin_folder))
            ->where($db->quoteName('element') . ' = ' . $db->quote($name));

        $db->setQuery($query);
        try {
            $db->execute();
        } catch (\Exception $e) {
            // ignore errors during uninstall cleanup
        }
    }
}
