<?php
/**
 * @package   OSDownloads-Pro
 * @contact   www.joomlashack.com, help@joomlashack.com
 * @copyright 2005-2025 Joomlashack.com. All rights reserved
 * @license   https://www.gnu.org/licenses/gpl.html GNU/GPL
 *
 * This file is part of OSDownloads-Pro.
 *
 * OSDownloads-Pro is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 *
 * OSDownloads-Pro is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with OSDownloads-Pro.  If not, see <https://www.gnu.org/licenses/>.
 */

namespace Alledia\OSDownloads\Pro\MailingList;

use Alledia\OSDownloads\Factory;
use Alledia\OSDownloads\Free;
use Joomla\CMS\Log\Log;
use Joomla\CMS\Table\Table;
use Joomla\Event\Dispatcher;
use Joomla\Event\Event;

// phpcs:disable PSR1.Files.SideEffects
defined('_JEXEC') or die();

// phpcs:enable PSR1.Files.SideEffects

class MailChimp extends Free\MailingList\MailChimp
{
    /**
     * @inheritDoc
     */
    protected function registerObservers(): void
    {
        $dispatcher = Factory::getDispatcher();

        if ($dispatcher instanceof \JEventDispatcher) {
            // Joomla 3 sends a table to the event handler
            $dispatcher->register('onOSDownloadsBeforeSaveEmail', [$this, 'onBeforeStore']);

        } elseif ($dispatcher instanceof Dispatcher) {
            // Joomla 4+ sends an event object to the handler
            $dispatcher->addListener('onOSDownloadsBeforeSaveEmail', [$this, 'onTablebeforeStore']);
        }
    }

    /**
     * Check with Mailchimp and indicate synced status
     * Add to selected Mailchimp Audience/groups
     *
     * @param Table $table
     *
     * @return void
     */
    public function onBeforeStore(Table $table)
    {
        if (static::isEnabled() == false) {
            return;
        }

        $email      = $table->get('email');
        $documentId = $table->get('document_id');
        $listId     = $this->getDocumentParam($documentId, 'mailinglist.mailchimp.list_id');
        $groups     = $this->getDocumentParam($documentId, 'mailinglist.mailchimp.groups');

        if ($email && $listId) {
            try {
                $mc         = static::getMailChimp();
                $member     = static::getMember($email, $listId);
                $status     = $member->status ?? null;
                $subscribed = $status == 'subscribed';

                $interests = [];
                if ($groups) {
                    foreach ($groups as $group) {
                        $interests[$group] = true;
                    }
                }

                // The email is not subscribed. Let's subscribe it.
                try {
                    $doubleOptIn = (bool)static::getParams()->get('mailinglist.mailchimp.double_optin', false);

                    $options = [
                        'email_address' => $email,
                        'status'        => $doubleOptIn && $subscribed == false ? 'pending' : 'subscribed',
                    ];
                    if ($interests) {
                        $options['interests'] = $interests;
                    }

                    if ($subscribed) {
                        $response = $mc->lists->updateListMember($listId, $email, $options, true);
                    } else {
                        $response = $mc->lists->addListMember($listId, $options, true);
                    }
                    $status = $response->status ?? false;

                } catch (\Throwable $error) {
                    $this->logMailChimpError($error, $email);
                }

                $table->set('synced', ($status === 'subscribed') ? 1 : 0);

            } catch (\Throwable $error) {
                $this->logMailChimpError($error, $email, Log::ERROR);
            }
        }
    }

    /**
     * Pass J4+ dispatcher call to legacy event handler
     *
     * @param Event $event
     *
     * @return void
     */
    public function onTableBeforeStore(Event $event)
    {
        if ($event->count() == 1) {
            $table = $event->getArgument(0);

            $this->onBeforeStore($table);
        }
    }
}
