<?php
/**
 * @package   OSDownloads-Pro
 * @contact   www.joomlashack.com, help@joomlashack.com
 * @copyright 2019-2025 Joomlashack.com. All rights reserved
 * @license   https://www.gnu.org/licenses/gpl.html GNU/GPL
 *
 * This file is part of OSDownloads-Pro.
 *
 * OSDownloads-Pro is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 *
 * OSDownloads-Pro is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with OSDownloads-Pro.  If not, see <https://www.gnu.org/licenses/>.
 */

use Alledia\OSDownloads\Free\MailingList\MailChimp;
use GuzzleHttp\Exception\ClientException;
use GuzzleHttp\Exception\ConnectException;
use Joomla\CMS\Component\ComponentHelper;
use Joomla\CMS\HTML\HTMLHelper;
use Joomla\CMS\Language\Text;
use Joomla\CMS\Version;
use MailchimpMarketing\ApiClient;

// phpcs:disable PSR1.Files.SideEffects
defined('_JEXEC') or die();

require_once __DIR__ . '/AbstractFormFieldMailingList.php';

// phpcs:enable PSR1.Files.SideEffects
// phpcs:disable PSR1.Classes.ClassDeclaration.MissingNamespace

abstract class AbstractFormFieldMailChimp extends AbstractFormFieldMailingList
{
    /**
     * @var object[]
     */
    protected $mailchimpOptions = null;

    /**
     * The message to display if wasn't able to render the input
     *
     * @var string
     */
    protected $errorMessageForInput = '';

    /**
     * @var int
     */
    protected $requestCount = 100;

    /**
     * @var array[]object[]
     */
    protected static $mailchimpResponses = [];

    /**
     * @inheritDoc
     */
    public function setup(SimpleXMLElement $element, $value, $group = null)
    {
        if (parent::setup($element, $value, $group)) {
            return $this->isMailingListEnabled('mailchimp');
        }

        return false;
    }

    /**
     * @inheritDoc
     */
    protected function getInput()
    {
        $output = parent::getInput();

        return $this->errorMessageForInput
            ? '<em>' . $this->errorMessageForInput . '</em>'
            : $output;
    }

    /**
     * @inheritDoc
     */
    protected function getOptions()
    {
        return array_merge(parent::getOptions(), $this->getMailchimpOptions());
    }

    /**
     * @return object[]
     */
    abstract protected function getMailchimpOptions(): array;

    /**
     * @return ApiClient
     *
     * @throws Exception
     */
    protected function getMailChimp(): ApiClient
    {
        $params = ComponentHelper::getParams('com_osdownloads');
        $apiKey = $params->get('mailinglist.mailchimp.api');
        if ($apiKey) {
            return MailChimp::getMailChimp();
        }

        throw new Exception(Text::_('COM_OSDOWNLOADS_ERROR_INVALID_API_AUTH'));
    }

    /**
     * @return object[]
     * @throws Exception
     */
    protected function getMailchimpLists(): array
    {
        if (isset(static::$mailchimpResponses['lists']) == false) {
            $mc = $this->getMailChimp();

            $lists = $mc->lists->getAllLists(
                ['lists.id', 'lists.name'],
                null,
                $this->requestCount
            );

            static::$mailchimpResponses['lists'] = $lists->lists ?? [];
        }

        return static::$mailchimpResponses['lists'];
    }

    /**
     * @param string $listId
     *
     * @return object[]
     * @throws Exception
     */
    protected function getMailchimpGroups(string $listId): array
    {
        if (isset(static::$mailchimpResponses['groups']) == false) {
            $mc = $this->getMailChimp();

            $categories = $mc->lists->getListInterestCategories(
                $listId,
                ['categories.id', 'categories.title'],
                null,
                $this->requestCount
            );
            $categories = $categories->categories ?? [];

            $groups = [];
            foreach ($categories as $category) {
                $interests = $mc->lists->listInterestCategoryInterests(
                    $listId,
                    $category->id,
                    ['interests.id', 'interests.name']
                );

                $interests = $interests->interests ?: [];
                foreach ($interests as $interest) {
                    $groups[] = (object)[
                        'category' => $category->title,
                        'interest' => $interest->name,
                        'id'       => $interest->id,
                    ];
                }
            }

            static::$mailchimpResponses['groups'] = $groups;
        }

        return static::$mailchimpResponses['groups'];
    }

    /**
     * @param Throwable|string $error
     * @param ?string          $type
     *
     * @return void
     */
    protected function setError($error, ?string $title = null, ?string $type = null): void
    {
        if (is_string($error)) {
            $message = $error;

        } else {
            if ($error instanceof ConnectException) {
                $authError = true;

            } elseif ($error instanceof ClientException) {
                $response  = $error->getResponse();
                $authError = $response->getStatusCode() == 401;

            } else {
                $authError = false;
            }

            $message = $authError ? Text::_('COM_OSDOWNLOADS_ERROR_INVALID_API_AUTH') : $error->getMessage();
            $title   = $authError ? Text::_('COM_OSDOWNLOADS_ML_MAILCHIMP_INVALID_API_TOOLTIP') : $title;
        }

        if ($this->form->getName() == 'com_config.component') {
            HTMLHelper::_('bootstrap.tooltip', '.hasTooltip');
            if (Version::MAJOR_VERSION < 4) {
                HTMLHelper::_('alledia.fontawesome');
            }

            $message = sprintf(
                '<span class="fa fa-info-circle alert alert-info hasTooltip" title="%s">&nbsp;%s</span>',
                $title,
                $message
            );
        }

        if ($type !== null) {
            $this->errorMessageForInput = Text::_('COM_OSDOWNLOADS_ERROR');
            $this->app->enqueueMessage($message, $type);

        } else {
            $this->errorMessageForInput = $message;
        }
    }
}
