<?php
/**
 * @package        Joomla
 * @subpackage     Helpdesk Pro
 * @author         Tuan Pham Ngoc
 * @copyright      Copyright (C) 2013 - 2026 Ossolution Team
 * @license        GNU/GPL, see LICENSE.php
 */

namespace OSSolution\HelpdeskPro\Admin\View\Tickets;

use Joomla\CMS\Factory;
use Joomla\CMS\HTML\HTMLHelper;
use Joomla\CMS\Language\Text;
use Joomla\CMS\Plugin\PluginHelper;
use OSL\Config\Config;
use OSL\Model\State;
use OSSolution\HelpdeskPro\Admin\Model\Tickets;
use OSSolution\HelpdeskPro\Site\Helper\Database as HelpdeskProHelperDatabase;
use OSSolution\HelpdeskPro\Site\Helper\Helper as HelpdeskProHelper;

defined('_JEXEC') or die;

/***
 * @property Tickets $model
 */
trait Common
{
	/**
	 * List of staffs
	 *
	 * @var array
	 */
	protected $staffs;

	/**
	 * Flag to determine if we show staff column
	 *
	 * @var bool
	 */
	protected $showStaffColumn;

	/**
	 * Custom fields
	 *
	 * @var array
	 */
	protected $fields;

	/**
	 * Custom fields values
	 *
	 * @var array
	 */
	protected $fieldValues;

	/**
	 * List of status
	 *
	 * @var array
	 */
	protected $statusList;

	/**
	 * List of priority
	 *
	 * @var array
	 */
	protected $priorityList;

	/**
	 * Component config
	 *
	 * @var Config
	 */
	protected $config;

	/**
	 * Filterable custom fields
	 *
	 * @var array
	 */
	protected $filterableFields = [];

	/**
	 * Prepare and set data required for display
	 *
	 * @param   State   $state        The model state
	 * @param   string  $fieldSuffix  Suffix for field names
	 *
	 * @return  void
	 */
	protected function prepareDisplayData(State $state, string $fieldSuffix = ''): void
	{
		$config = HelpdeskProHelper::getConfig();

		// Ticket status filter
		$rowStatuses = HelpdeskProHelperDatabase::getAllStatuses('ordering', $fieldSuffix);

		if (count($rowStatuses))
		{
			$options   = [];
			$options[] = HTMLHelper::_('select.option', -1, Text::_('HDP_TICKET_STATUS'), 'id', 'title');
			$options[] = HTMLHelper::_('select.option', 0, Text::_('HDP_ALL_STATUSES'), 'id', 'title');
			$options   = array_merge($options, $rowStatuses);

			$this->lists['filter_status_id'] = HTMLHelper::_(
				'select.genericlist',
				$options,
				'filter_status_id',
				[
					'option.text.toHtml' => false,
					'option.text'        => 'title',
					'option.key'         => 'id',
					'list.attr'          => 'class="input-medium form-select" onchange="submit();"',
					'list.select'        => $state->filter_status_id,
				]
			);
		}

		// Ticket priority filter
		$rowPriorities = HelpdeskProHelperDatabase::getAllPriorities('ordering', $fieldSuffix);

		if (count($rowPriorities))
		{
			$options   = [];
			$options[] = HTMLHelper::_('select.option', 0, Text::_('HDP_ALL_PRIORITIES'), 'id', 'title');
			$options   = array_merge($options, $rowPriorities);

			$this->lists['filter_priority_id'] = HTMLHelper::_(
				'select.genericlist',
				$options,
				'filter_priority_id',
				[
					'option.text.toHtml' => false,
					'option.text'        => 'title',
					'option.key'         => 'id',
					'list.attr'          => 'class="input-medium form-select" onchange="submit();"',
					'list.select'        => $state->filter_priority_id,
				]
			);
		}

		$statusList = [];

		foreach ($rowStatuses as $status)
		{
			$statusList[$status->id] = $status->title;
		}

		$priorityList = [];

		foreach ($rowPriorities as $priority)
		{
			$priorityList[$priority->id] = $priority->title;
		}

		// Label filter
		$rowLabels = HelpdeskProHelperDatabase::getAllLabels();

		if (count($rowLabels))
		{
			$options   = [];
			$options[] = HTMLHelper::_(
				'select.option',
				0,
				Text::_('HDP_SELECT_LABEL'),
				'id',
				'title'
			);
			$options   = array_merge($options, $rowLabels);

			$this->lists['filter_label_id'] = HTMLHelper::_(
				'select.genericlist',
				$options,
				'filter_label_id',
				[
					'option.text.toHtml' => false,
					'option.text'        => 'title',
					'option.key'         => 'id',
					'list.attr'          => 'class="input-medium form-select" onchange="submit();"',
					'list.select'        => $state->filter_label_id,
				]
			);
		}

		/**
		 * Filter ticket by staff
		 */
		if (PluginHelper::isEnabled('helpdeskpro', 'assignticket'))
		{
			$staffDisplayField = $config->get('staff_display_field', 'username') ?: 'username';
			$staffs            = HelpdeskProHelperDatabase::getAllStaffs($config->staff_group_id);

			$options   = [];
			$options[] = HTMLHelper::_(
				'select.option',
				0,
				Text::_('HDP_SELECT_STAFF'),
				'id',
				$staffDisplayField
			);
			$options   = array_merge($options, $staffs);

			$this->lists['filter_staff_id'] = HTMLHelper::_(
				'select.genericlist',
				$options,
				'filter_staff_id',
				'class="input-medium form-select" onchange="submit();"',
				'id',
				$staffDisplayField,
				$state->filter_staff_id
			);

			$rowStaffs = [];

			foreach ($staffs as $staff)
			{
				$rowStaffs[$staff->id] = $staff->{$staffDisplayField};
			}

			$this->staffs          = $rowStaffs;
			$this->showStaffColumn = true;
		}

		// Handle background color of tickets base on status
		$cssRows = [];

		foreach ($rowStatuses as $rowStatus)
		{
			if ($rowStatus->bgcolor)
			{
				$cssRows[] = 'tr.hdp-ticket-status-' . $rowStatus->id . ' td {background-color:#' . $rowStatus->bgcolor . ' !important}';
			}
		}

		if (count($cssRows))
		{
			Factory::getApplication()
				->getDocument()
				->getWebAssetManager()
				->addInlineStyle(implode("\n", $cssRows));
		}

		$this->fields       = HelpdeskProHelperDatabase::getFieldsOnListView($fieldSuffix);
		$this->fieldValues  = $this->model->getFieldsData($this->fields);
		$this->statusList   = $statusList;
		$this->priorityList = $priorityList;
		$this->config       = $config;

		// Get filterable custom fields and build their filter dropdowns
		$filterableFields = HelpdeskProHelperDatabase::getFilterableFields($fieldSuffix);

		foreach ($filterableFields as $field)
		{
			$options   = [];
			$options[] = HTMLHelper::_('select.option', '', Text::_('HDP_SELECT') . ' ' . $field->title);

			$fieldValues = explode("\r\n", $field->values);

			foreach ($fieldValues as $value)
			{
				$value = trim($value);

				if ($value)
				{
					$options[] = HTMLHelper::_('select.option', $value, $value);
				}
			}

			$this->lists['filter_field_' . $field->id] = HTMLHelper::_(
				'select.genericlist',
				$options,
				'filter_field_' . $field->id,
				'class="input-medium form-select" onchange="submit();"',
				'value',
				'text',
				$state->{'filter_field_' . $field->id}
			);
		}

		$this->filterableFields = $filterableFields;
	}
}