<?php
/**
 * @package        Joomla
 * @subpackage     Helpdesk Pro
 * @author         Tuan Pham Ngoc
 * @copyright      Copyright (C) 2013 - 2026 Ossolution Team
 * @license        GNU/GPL, see LICENSE.php
 */

namespace OSSolution\HelpdeskPro\Admin\View\Ticket;

use HDPForm;
use HelpdeskProHelperBootstrap;
use Joomla\CMS\Application\CMSApplication;
use Joomla\CMS\Factory;
use Joomla\CMS\HTML\HTMLHelper;
use Joomla\CMS\Language\Text;
use Joomla\CMS\Plugin\PluginHelper;
use Joomla\CMS\Uri\Uri;
use OSL\Config\Config;
use OSSolution\HelpdeskPro\Site\Helper\Database as HelpdeskProHelperDatabase;
use OSSolution\HelpdeskPro\Site\Helper\Helper as HelpdeskProHelper;
use OSSolution\HelpdeskPro\Site\Helper\Html as HelpdeskProHelperHtml;

defined('_JEXEC') or die;

trait Common
{
	/**
	 * Bootstrap Helper
	 *
	 * @var HelpdeskProHelperBootstrap
	 */
	protected $bootstrapHelper;

	/**
	 * List of ticke statues
	 *
	 * @var array
	 */
	protected $rowStatuses;

	/**
	 * List of ticket priorities
	 *
	 * @var array
	 */
	protected $rowPriorities;

	/**
	 * List of categories
	 *
	 * @var array
	 */
	protected $categories;

	/**
	 * Custom Fields
	 *
	 * @var array
	 */
	protected $fields;

	/**
	 * Comments belong to the current ticket
	 *
	 * @var array
	 */
	protected $messages;

	/**
	 * Custom field values for this ticket
	 *
	 * @var array
	 */
	protected $fieldValues;

	/**
	 * Component config
	 *
	 * @var Config
	 */
	protected $config;

	/**
	 * Plugins output
	 *
	 * @var array
	 */
	protected $results;

	/**
	 * The form object
	 *
	 * @var HDPForm
	 */
	protected $form;

	/**
	 * Store custom field IDs which should be hidden
	 *
	 * @var array
	 */
	protected $hiddenFieldIds = [];

	/**
	 * Method to prepare data for display ticket information
	 *
	 * @param   \stdClass  $item
	 * @param   array      $rows
	 * @param   array      $rowStatuses
	 * @param   array      $rowPriorities
	 *
	 * @return void
	 */
	protected function prepareDisplayTicketData($item, $rows)
	{
		/* @var CMSApplication $app */
		$app    = Factory::getApplication();
		$config = HelpdeskProHelper::getConfig();

		if (Factory::getApplication()->isClient('site'))
		{
			$fieldSuffix = HelpdeskProHelper::getFieldSuffix();
		}
		else
		{
			$fieldSuffix = '';
		}

		$children = [];

		// first pass - collect children
		foreach ($rows as $v)
		{
			$pt            = (int) $v->parent_id;
			$list          = $children[$pt] ?? [];
			$list[]        = $v;
			$children[$pt] = $list;
		}

		$categories = HTMLHelper::_('menu.treerecurse', 0, '', [], $children, 9999, 0, 0);

		PluginHelper::importPlugin('helpdeskpro');

		//Trigger plugins
		$results = $app->triggerEvent('onViewTicket', [$item]);

		if ($config->highlight_code)
		{
			HelpdeskProHelper::loadHighlighter();
		}

		Factory::getApplication()
			->getDocument()
			->addScriptOptions('hdpRootUri', Uri::root(true))
			->addScriptOptions('hdpBaseUri', Uri::base(true))
			->getWebAssetManager()
			->addInlineScript('var currentCategory = 0;');

		$this->fields        = HelpdeskProHelper::getFields($item->category_id, [], $fieldSuffix);
		$this->messages      = $this->model->getMessages();
		$this->fieldValues   = $this->model->getFieldsValue();
		$this->rowStatuses   = HelpdeskProHelperDatabase::getAllStatuses('ordering', $fieldSuffix);
		$this->rowPriorities = HelpdeskProHelperDatabase::getAllPriorities('ordering', $fieldSuffix);
		$this->categories    = $categories;
		$this->config        = $config;
		$this->results       = $results;

		$this->setHiddenFields($item);
	}

	/**
	 * Prepare data for submit ticket form
	 *
	 * @param   array  $rows
	 * @param   int    $categoryId
	 * @param   int    $priorityId
	 *
	 * @return void
	 */
	protected function prepareSubmitTicketFormData($rows = [], $categoryId = 0, $priorityId = 0)
	{
		/* @var CMSApplication $app */
		$app    = Factory::getApplication();
		$config = HelpdeskProHelper::getConfig();

		if ($app->isClient('site'))
		{
			$fieldSuffix = HelpdeskProHelper::getFieldSuffix();
			$selectClass = 'uk-select form-select validate[required]';
		}
		else
		{
			$fieldSuffix = '';
			$selectClass = 'form-select';
		}

		$this->lists['category_id'] = HelpdeskProHelperHtml::buildCategoryDropdown(
			$categoryId,
			'category_id',
			'onchange="HDP.showFields(this.form);" class="' . $selectClass . '"',
			$rows
		);

		$rowPriorities = HelpdeskProHelperDatabase::getAllPriorities('ordering', $fieldSuffix);

		if (count($rowPriorities))
		{
			$options                    = [];
			$options[]                  = HTMLHelper::_('select.option', '', Text::_('HDP_CHOOSE_PRIORITY'), 'id', 'title');
			$options                    = array_merge($options, $rowPriorities);
			$this->lists['priority_id'] = HTMLHelper::_(
				'select.genericlist',
				$options,
				'priority_id',
				[
					'option.text.toHtml' => false,
					'option.text'        => 'title',
					'option.key'         => 'id',
					'list.attr'          => 'class="' . $selectClass . '"',
					'list.select'        => $priorityId > 0 ? $priorityId : $config->default_ticket_priority_id,
				]
			);
		}

		// Custom fields
		$rowFields = HelpdeskProHelper::getAllFields($fieldSuffix);
		$form      = new HDPForm($rowFields);

		$relation = HelpdeskProHelper::getFieldCategoryRelation();
		$form->prepareFormField($categoryId, $relation);

		// Bind data
		if ($app->isClient('site'))
		{
			$data = $this->input->getData();

			if ($this->input->getMethod() == 'POST')
			{
				$useDefault = false;
			}
			else
			{
				$useDefault = true;
			}

			$form->bind($data, $useDefault);
		}

		// Check to see if custom field dependency is enabled
		$hasDependency = $this->useCustomFieldDependency($rowFields);

		if ($hasDependency)
		{
			$form->buildFieldsDependency();
		}

		// Passing necessary JS variables
		$fieldJs = "fields = new Array();\n";

		foreach ($relation as $catId => $fieldList)
		{
			$fieldJs .= ' fields[' . $catId . '] = new Array("' . implode('","', $fieldList) . '");' . "\n";
		}

		Factory::getApplication()
			->getDocument()
			->addScriptOptions('hdpRootUri', Uri::root(true))
			->addScriptOptions('hdpBaseUri', Uri::base(true))
			->addScriptOptions('hasDependency', $hasDependency)
			->getWebAssetManager()
			->addInlineScript($fieldJs)
			->addInlineScript("var currentCategory = $categoryId;");

		$this->config = $config;
		$this->form   = $form;
	}

	/**
	 * Prepare data for ticket edit form
	 *
	 * @param   \stdClass  $item
	 * @param   array      $rows
	 *
	 * @return void
	 * @throws \Exception
	 */
	protected function prepareEditTicketFormData($item, $rows = [])
	{
		$app         = Factory::getApplication();
		$config      = HelpdeskProHelper::getConfig();
		$fieldSuffix = HelpdeskProHelper::getFieldSuffix($item->language);

		if ($app->isClient('site'))
		{
			$selectClass = 'uk-select form-select validate[required]';
		}
		else
		{
			$selectClass = 'form-select';
		}

		$this->lists['category_id'] = HelpdeskProHelperHtml::buildCategoryDropdown(
			$item->category_id,
			'category_id',
			'class="' . $selectClass . '" onchange="HDP.showFields(this.form);"',
			$rows
		);

		$rowPriorities = HelpdeskProHelperDatabase::getAllPriorities('ordering', $fieldSuffix);

		if (count($rowPriorities))
		{
			$options                    = [];
			$options[]                  = HTMLHelper::_('select.option', '', Text::_('HDP_CHOOSE_PRIORITY'), 'id', 'title');
			$options                    = array_merge($options, $rowPriorities);
			$this->lists['priority_id'] = HTMLHelper::_(
				'select.genericlist',
				$options,
				'priority_id',
				[
					'option.text.toHtml' => false,
					'option.text'        => 'title',
					'option.key'         => 'id',
					'list.attr'          => 'class="uk-select validate[required] form-select"',
					'list.select'        => $item->priority_id,
				]
			);
		}

		// Custom fields
		$rowFields = HelpdeskProHelper::getAllFields($fieldSuffix);
		$formData  = $this->getTicketCustomFieldsData($item);
		$relation  = HelpdeskProHelper::getFieldCategoryRelation();

		$form = new HDPForm($rowFields);
		$form->bind($formData);
		$form->prepareFormField($item->category_id, $relation);

		// Check to see if custom field dependency is enabled
		$hasDependency = $this->useCustomFieldDependency($rowFields);

		if ($hasDependency)
		{
			$form->buildFieldsDependency();
		}

		// Passing necessary JS variables
		$fieldJs = "fields = new Array();\n";

		foreach ($relation as $catId => $fieldList)
		{
			$fieldJs .= ' fields[' . $catId . '] = new Array("' . implode('","', $fieldList) . '");' . "\n";
		}

		$categoryId = (int) $item->category_id;

		Factory::getApplication()
			->getDocument()
			->addScriptOptions('hdpRootUri', Uri::root(true))
			->addScriptOptions('hdpBaseUri', Uri::base(true))
			->addScriptOptions('hasDependency', $hasDependency)
			->getWebAssetManager()
			->addInlineScript($fieldJs)
			->addInlineScript("var currentCategory = $categoryId;");

		$this->form   = $form;
		$this->config = $config;
	}

	/**
	 * Calculate and set hidden fields for showing extra information of the ticket
	 *
	 * @param   \stdClass  $item
	 *
	 * @return void
	 */
	protected function setHiddenFields($item)
	{
		$formData  = $this->getTicketCustomFieldsData($item);
		$rowFields = HelpdeskProHelper::getAllFields();
		$form      = new HDPForm($rowFields);
		$form->bind($formData);

		$relation = HelpdeskProHelper::getFieldCategoryRelation();
		$form->prepareFormField($item->category_id, $relation);
		$form->buildFieldsDependency();

		$hiddenFieldIds = [];

		foreach ($form->getFields() as $field)
		{
			if (!$field->visible)
			{
				$hiddenFieldIds[] = $field->id;
			}
		}

		$this->hiddenFieldIds = $hiddenFieldIds;
	}

	/**
	 * Get ticket custom fields data
	 *
	 * @param   \stdClass  $item
	 *
	 * @return array
	 */
	private function getTicketCustomFieldsData($item)
	{
		/* @var \Joomla\Database\DatabaseDriver $db */
		$db    = Factory::getContainer()->get('db');
		$query = $db->getQuery(true)
			->select('a.name, b.field_value')
			->from('#__helpdeskpro_fields AS a')
			->innerJoin('#__helpdeskpro_field_value AS b ON a.id = b.field_id')
			->where('b.ticket_id = ' . (int) $item->id);
		$db->setQuery($query);
		$formData = [];

		foreach ($db->loadObjectList() as $rowFieldValue)
		{
			$formData[$rowFieldValue->name] = $rowFieldValue->field_value;
		}

		return $formData;
	}

	/**
	 * Check to see if custom field dependency is used
	 *
	 * @param   array  $rowFields
	 *
	 * @return bool
	 */
	private function useCustomFieldDependency(array $rowFields)
	{
		foreach ($rowFields as $rowField)
		{
			if ($rowField->depend_on_field_id)
			{
				return true;
			}
		}

		return false;
	}
}