<?php
/**
 * @package        Joomla
 * @subpackage     Helpdesk Pro
 * @author         Tuan Pham Ngoc
 * @copyright      Copyright (C) 2013 - 2026 Ossolution Team
 * @license        GNU/GPL, see LICENSE.php
 */

namespace OSSolution\HelpdeskPro\Admin\Controller;

use Exception;
use HDPForm;
use Joomla\CMS\Application\CMSApplication;
use Joomla\CMS\Captcha\Captcha;
use Joomla\CMS\Component\ComponentHelper;
use Joomla\CMS\Factory;
use Joomla\CMS\HTML\HTMLHelper;
use Joomla\CMS\Language\Text;
use Joomla\CMS\Mail\MailHelper;
use Joomla\CMS\Plugin\PluginHelper;
use Joomla\CMS\Router\Route;
use Joomla\CMS\Session\Session;
use Joomla\CMS\Uri\Uri;
use Joomla\Database\DatabaseDriver;
use Joomla\Filesystem\Exception\FilesystemException;
use Joomla\Filesystem\File;
use Joomla\Filesystem\Folder;
use Joomla\Filesystem\Path;
use OSL\Container\Container;
use OSL\Controller\Download;
use OSSolution\HelpdeskPro\Admin\Model\Tickets;
use OSSolution\HelpdeskPro\Site\Helper\Database as HelpdeskProHelperDatabase;
use OSSolution\HelpdeskPro\Site\Helper\Helper as HelpdeskProHelper;
use OSSolution\HelpdeskPro\Site\Helper\Route as RouteHelper;
use RuntimeException;

defined('_JEXEC') or die;

class Ticket extends Controller
{
	use Download;

	/**
	 * Constructor.
	 *
	 * @param   array  $config  An optional associative array of configuration settings.
	 *
	 * @see OSFControlleAdmin
	 */
	public function __construct(Container $container, array $config = [])
	{
		parent::__construct($container, $config);

		$this->registerTask('comment_and_close', 'add_comment');
		$this->registerTask('comment_and_list', 'add_comment');
	}

	/**
	 * Display form allows adding a new support ticket
	 *
	 * @return void
	 */
	public function add()
	{
		$this->input->set('layout', 'form');

		parent::add();
	}

	/**
	 * Set layout to edit on editing ticket
	 *
	 * @return void
	 */
	public function edit()
	{
		$this->input->set('layout', 'edit');

		parent::edit();
	}

	/**
	 * Override allow add method to control subject support ticket permission in the frontend
	 *
	 * @param   array  $data
	 *
	 * @return bool
	 */
	protected function allowAdd($data = [])
	{
		if ($this->app->isClient('site'))
		{
			return true;
		}

		return parent::allowAdd($data);
	}

	/**
	 * Store a support ticket
	 *
	 * @throws Exception
	 */
	public function save()
	{
		$this->csrfProtection();

		if ($this->app->isClient('site'))
		{
			$this->antiSpam();
		}

		$config = HelpdeskProHelper::getConfig();
		$app    = $this->app;
		$user   = $app->getIdentity();
		$input  = $this->input;
		$isNew  = $input->getInt('id', 0) === 0;

		if (!$config->enable_captcha || ($user->id && $config->enable_captcha == '2'))
		{
			$enableCaptcha = false;
		}
		else
		{
			$enableCaptcha = true;
		}

		if ($enableCaptcha && $app->isClient('site') && !$this->validateCaptcha())
		{
			$app->enqueueMessage(Text::_('HD_INVALID_CAPTCHA_ENTERED'), 'warning');
			$input->set('view', 'ticket');
			$input->set('layout', 'form');

			$this->display();

			return;
		}

		$post = $input->post->getData();

		// Perform some basic validation when ticket is submitted from frontend
		if ($app->isClient('site'))
		{
			$errors = [];
			$user   = $app->getIdentity();

			if (!$user->id && !$config->allow_public_user_submit_ticket)
			{
				throw new Exception(Text::_('HDP_LOGIN_TO_SUBMIT_TICKET'), 403);
			}

			if (!$user->id)
			{
				// Make sure the name and email address is valid
				if (empty($post['name']))
				{
					$errors[] = Text::_('HDP_ENTER_YOUR_NAME');
				}

				$email = $post['email'];

				if (empty($email))
				{
					$errors[] = Text::_('HDP_ENTER_YOUR_EMAIL');
				}

				if ($email && !MailHelper::isEmailAddress($email))
				{
					$errors[] = Text::_('HDP_INVALID_EMAIL_ADDRESS');
				}
			}

			if (empty($post['subject']))
			{
				$errors[] = Text::_('HDP_ENTER_SUBJECT');
			}

			if (empty($post['category_id']))
			{
				$errors[] = Text::_('HDP_SELECT_A_CATEGORY');
			}

			$formErrors = $this->validateFormFields($post);

			if (count($formErrors))
			{
				$errors = array_merge($errors, $formErrors);
			}

			// OK, if error founds, we need to redirect back to submit ticket page
			if (count($errors))
			{
				foreach ($errors as $error)
				{
					$app->enqueueMessage($error, 'error');
				}

				$input->set('view', 'ticket');
				$input->set('layout', 'form');

				$this->display();

				return;
			}
		}

		$input->post->set('message', ComponentHelper::filterText($post['message']));

		/* @var \OSSolution\HelpdeskPro\Admin\Model\Ticket $model */
		$model = $this->getModel('ticket');

		try
		{
			$model->store($input);

			if ($isNew)
			{
				$msg = Text::_('HDP_TICKET_SUBMITTED');
			}
			else
			{
				$msg = Text::_('HDP_TICKET_UPDATED');
			}
		}
		catch (Exception $e)
		{
			$msg = Text::_('HDP_ERROR_SAVING_TICKET');
		}

		if ($app->isClient('administrator'))
		{
			$this->setRedirect('index.php?option=com_helpdeskpro&view=tickets', $msg);
		}
		else
		{
			$user = $app->getIdentity();

			if (!$user->id)
			{
				if ($config->get('submit_ticket_complete_url'))
				{
					$redirectUrl = $config->get('submit_ticket_complete_url');
				}
				else
				{
					$redirectUrl = Route::_(RouteHelper::getTicketRoute($input->get('ticket_code'), false), false);
				}

				$this->setRedirect($redirectUrl, $msg);
			}
			else
			{
				$this->setRedirect(Route::_(RouteHelper::getTicketRoute($input->getInt('id')), false), $msg);
			}
		}
	}

	/**
	 * Add comment to support ticket
	 *
	 * @throws Exception
	 */
	public function add_comment()
	{
		$this->csrfProtection();

		if ($this->app->isClient('site'))
		{
			$this->antiSpam();
		}

		/* @var CMSApplication $app */
		$app    = $this->app;
		$user   = $app->getIdentity();
		$input  = $this->input;
		$config = HelpdeskProHelper::getConfig();

		if (!$config->allow_public_user_submit_ticket && !$user->id)
		{
			if ($app->isClient('administrator'))
			{
				$return = base64_encode('index.php?option=com_helpdeskpro&view=ticket&id=' . $input->getInt('id'));
			}
			else
			{
				$return = base64_encode(RouteHelper::getTicketRoute($input->getInt('id')));
			}

			$app->redirect(
				Route::_('index.php?option=com_users&view=login&return=' . $return),
				Text::_('HDP_SESSION_EXPIRED')
			);
		}

		if (!$config->enable_captcha || ($user->id && $config->enable_captcha == '2'))
		{
			$enableCaptcha = false;
		}
		else
		{
			$enableCaptcha = true;
		}

		if ($enableCaptcha && $app->isClient('site') && !$this->validateCaptcha())
		{
			$app->enqueueMessage(Text::_('HD_INVALID_CAPTCHA_ENTERED'), 'warning');
			$this->input->set('view', 'ticket');
			$this->input->set('layout', 'default');
			$this->input->set('captcha_invalid', 1);
			$this->display();

			return;
		}

		$input->post->set('message', ComponentHelper::filterText($input->post->getString('message')));
		$model = $this->getModel('ticket');

		$task = $this->getTask();

		if ($task == 'comment_and_close')
		{
			$closeTicket = true;
		}
		else
		{
			$closeTicket = false;
		}

		/* @var \OSSolution\HelpdeskPro\Admin\Model\Ticket $model */
		try
		{
			$model->addComment($input, $closeTicket);

			if ($closeTicket)
			{
				$msg = Text::_('HDP_COMMENT_ADDED_TICKET_CLOSED');
			}
			else
			{
				$msg = Text::_('HDP_COMMENT_ADDED');
			}
		}
		catch (Exception $e)
		{
			$msg = Text::_('HDP_ERROR_ADDING_COMMENT');
		}

		$ticketCode = $input->post->getString('ticket_code');
		$ticketId   = $input->post->getInt('id', 0);

		if ($ticketCode)
		{
			$this->setRedirect(Route::_(RouteHelper::getTicketRoute($ticketCode, false), false), $msg);
		}
		else
		{
			if ($app->isClient('administrator'))
			{
				if (in_array($task, ['comment_and_close', 'comment_and_list']))
				{
					$this->setRedirect('index.php?option=com_helpdeskpro&view=tickets', $msg);
				}
				else
				{
					$this->setRedirect('index.php?option=com_helpdeskpro&view=ticket&id=' . $ticketId, $msg);
				}
			}
			else
			{
				if ($closeTicket)
				{
					$this->setRedirect(Route::_(RouteHelper::getTicketsRoute(), false), $msg);
				}
				else
				{
					$this->setRedirect(Route::_(RouteHelper::getTicketRoute($ticketId), false), $msg);
				}
			}
		}
	}

	/**
	 * Change category for the ticket
	 */
	public function update_category()
	{
		$this->csrfProtection();

		$post  = $this->input->post->getData();
		$model = $this->getModel('ticket');

		/* @var \OSSolution\HelpdeskPro\Admin\Model\Ticket $model */
		try
		{
			$model->updateCategory($post);
			$msg = Text::_('HDP_TICKET_CATEGORY_UPDATED');
		}
		catch (Exception $e)
		{
			$msg = Text::_('HDP_ERROR_UPDATE_TICKET_CATEGORY');
		}

		if (isset($post['ticket_code']))
		{
			$this->setRedirect(Route::_(RouteHelper::getTicketRoute($post['ticket_code'], false), false));
		}
		else
		{
			if ($this->app->isClient('administrator'))
			{
				$this->setRedirect('index.php?option=com_helpdeskpro&view=ticket&id=' . $post['id'], $msg);
			}
			else
			{
				$this->setRedirect(Route::_(RouteHelper::getTicketRoute($post['id']), false), $msg);
			}
		}
	}

	/**
	 * Update ticket status
	 *
	 * @throws Exception
	 */
	public function update_status()
	{
		$this->csrfProtection();

		$app  = $this->app;
		$post = $this->input->post->getData();

		/* @var \OSSolution\HelpdeskPro\Admin\Model\Ticket $model */
		$model = $this->getModel('ticket');

		try
		{
			$model->updateStatus($post);
			$msg = Text::_('HDP_TICKET_STATUS_UPDATED');
		}
		catch (Exception $e)
		{
			$msg = Text::_('HDP_ERROR_UPDATING_TICKET_STATUS');
		}

		if (isset($post['ticket_code']))
		{
			$this->setRedirect(Route::_(RouteHelper::getTicketRoute($post['ticket_code'], false), false), $msg);
		}
		else
		{
			if ($app->isClient('administrator'))
			{
				$this->setRedirect('index.php?option=com_helpdeskpro&view=ticket&id=' . $post['id'], $msg);
			}
			else
			{
				$this->setRedirect(Route::_(RouteHelper::getTicketRoute($post['id']), false), $msg);
			}
		}
	}

	/**
	 * Update ticket priority
	 *
	 * @throws Exception
	 */
	public function update_priority()
	{
		$this->csrfProtection();

		$app  = $this->app;
		$post = $this->input->post->getData();

		/* @var \OSSolution\HelpdeskPro\Admin\Model\Ticket $model */
		$model = $this->getModel('ticket');

		try
		{
			$model->updatePriority($post);
			$msg = Text::_('HDP_TICKET_PRIORITY_UPDATED');
		}
		catch (Exception $e)
		{
			$msg = Text::_('HDP_ERROR_UPDATING_PRIORITY');
		}

		if (isset($post['ticket_code']))
		{
			$this->setRedirect(Route::_(RouteHelper::getTicketRoute($post['ticket_code'], false), false), $msg);
		}
		else
		{
			if ($app->isClient('administrator'))
			{
				$this->setRedirect('index.php?option=com_helpdeskpro&view=ticket&id=' . $post['id'], $msg);
			}
			else
			{
				$this->setRedirect(Route::_(RouteHelper::getTicketRoute($post['id']), false), $msg);
			}
		}
	}

	/**
	 * Apply label to a support ticket
	 *
	 * @throws Exception
	 */
	public function apply_label()
	{
		$this->csrfProtection();

		$post = $this->input->post->getData();

		/* @var \OSSolution\HelpdeskPro\Admin\Model\Ticket $model */
		$model = $this->getModel('ticket');

		try
		{
			$model->applyLabel($post);
			$msg = Text::_('HDP_TICKET_LABEL_UPDATED');
		}
		catch (Exception $e)
		{
			$msg = Text::_('HDP_ERROR_UPDATING_TICKET_LABEL');
		}

		$this->setRedirect('index.php?option=com_helpdeskpro&view=ticket&id=' . $post['id'], $msg);
	}

	/**
	 * Save rating and close the ticket
	 */
	public function save_rating()
	{
		$this->csrfProtection();

		$post = $this->input->post->getData();

		/* @var \OSSolution\HelpdeskPro\Admin\Model\Ticket $model */
		$model = $this->getModel('ticket');

		try
		{
			$model->saveRating($post);
			$msg = Text::_('HDP_RATING_SAVED');
		}
		catch (Exception $e)
		{
			$msg = Text::_('HDP_ERROR_SAVING_RATING');
		}

		if (isset($post['ticket_code']))
		{
			$this->setRedirect(Route::_(RouteHelper::getTicketRoute($post['ticket_code'], false), false), $msg);
		}
		else
		{
			if ($this->app->isClient('administrator'))
			{
				$this->setRedirect('index.php?option=com_helpdeskpro&view=ticket&id=' . $post['id'], $msg);
			}
			else
			{
				$this->setRedirect(Route::_(RouteHelper::getTicketRoute($post['id']), false), $msg);
			}
		}
	}

	/**
	 * Assign ticket to a staff
	 */
	public function assign_ticket()
	{
		$user = $this->app->getIdentity();

		if (!$user->authorise('helpdeskpro.assignticket', 'com_helpdeskpro'))
		{
			return;
		}

		$id = $this->input->getInt('id', 0);

		if (!$id)
		{
			return;
		}

		$userId = $this->input->getInt('user_id', 0);
		$db     = $this->container->db;
		$query  = $db->getQuery(true)
			->update('#__helpdeskpro_tickets')
			->set('staff_id=' . (int) $userId)
			->where('id=' . $id);
		$db->setQuery($query)
			->execute();

		// Send notification email to user in case he does not assign ticket to himself
		if ($userId && ($user->id != $userId))
		{
			$query->clear()
				->select('*')
				->from('#__helpdeskpro_tickets')
				->where('id = ' . (int) $id);
			$db->setQuery($query);
			$row    = $db->loadObject();
			$config = HelpdeskProHelper::getConfig();
			HelpdeskProHelper::sendTicketAssignedEmails($row, $config);
		}
	}

	/**
	 * Update a comment
	 *
	 * @throws Exception
	 */
	public function update_comment()
	{
		$user      = $this->app->getIdentity();
		$messageId = $this->input->json->getInt('message_id', 0);

		if (HelpdeskProHelper::canUpdateComment($user, $messageId))
		{
			$db         = $this->container->db;
			$newMessage = $db->quote($this->input->json->getHtml('new_message'));
			$query      = $db->getQuery(true)
				->update('#__helpdeskpro_messages')
				->set('message=' . $newMessage)
				->where('id = ' . (int) $messageId);
			$db->setQuery($query);
			$db->execute();
		}

		$this->app->close();
	}

	/**
	 * Remove comment from a ticket
	 *
	 * @throws Exception
	 */
	public function remove_comment()
	{
		$user      = $this->app->getIdentity();
		$messageId = $this->input->getInt('message_id', 0);

		if (HelpdeskProHelper::canDeleteComment($user, $messageId))
		{
			$db    = $this->container->db;
			$query = $db->getQuery(true)
				->delete('#__helpdeskpro_messages')
				->where('id = ' . (int) $messageId);
			$db->setQuery($query)
				->execute();
		}

		$this->app->close();
	}

	/**
	 * Convert the current open ticket to a knowledge article
	 */
	public function convert_to_kb()
	{
		/* @var \OSSolution\HelpdeskPro\Admin\Model\Ticket $model */
		$model = $this->getModel();

		try
		{
			$model->convertTicketToArticle();
			$this->setMessage(Text::_('HDP_CONVERT_TICKET_SUCCESS'));
		}
		catch (Exception $e)
		{
			$this->setMessage($e->getMessage(), 'error');
		}

		$this->setRedirect('index.php?option=com_helpdeskpro&view=ticket&id=' . $this->input->getInt('id', 0));
	}

	/**
	 * Export support ticket into a csv file
	 */
	public function export()
	{
		if ($this->app->isClient('site'))
		{
			$this->csrfProtection();

			if (!$this->app->getIdentity()->id)
			{
				throw new RuntimeException('You do not have permission to export tickets', 403);
			}
		}

		$config = HelpdeskProHelper::getConfig();

		$rowStatuses = HelpdeskProHelperDatabase::getAllStatuses();
		$statusList  = [];

		foreach ($rowStatuses as $status)
		{
			$statusList[$status->id] = $status->title;
		}

		$rowPriorities = HelpdeskProHelperDatabase::getAllPriorities();
		$priorityList  = [];

		foreach ($rowPriorities as $priority)
		{
			$priorityList[$priority->id] = $priority->title;
		}

		$rowFields = HelpdeskProHelper::getAllFields();

		/* @var Tickets $model */
		$model = $this->getModel('tickets');
		$model->setState('limitstart', 0)
			->setState('limit', 0);
		$rows = $model->getData();

		if (count($rows))
		{
			$csvDelimiter = ',';
			$output       = fopen('php://output', 'w');

			$fieldValues = $model->getFieldsData($rowFields);

			$this->app->setHeader('Content-Type', 'application/csv', true)
				->setHeader('Content-Disposition', 'attachment; filename="tickets.csv"', true)
				->setHeader('Cache-Control', 'must-revalidate', true)
				->sendHeaders();

			$rowStaffs = [];
			$showStaff = false;

			if (PluginHelper::isEnabled('helpdeskpro', 'assignticket'))
			{
				$staffDisplayField = $config->get('staff_display_field', 'username') ?: 'username';
				$staffs            = HelpdeskProHelperDatabase::getAllStaffs($config->staff_group_id);

				$rowStaffs = [];

				foreach ($staffs as $staff)
				{
					$rowStaffs[$staff->id] = $staff->{$staffDisplayField};
				}

				$showStaff = true;
			}

			$csvRowData   = [];
			$csvRowData[] = Text::_('HDP_TITLE');
			$csvRowData[] = Text::_('HDP_MESSAGE');
			$csvRowData[] = Text::_('HDP_CATEGORY');
			$csvRowData[] = Text::_('HDP_USER');
			$csvRowData[] = Text::_('HDP_EMAIL');
			$csvRowData[] = Text::_('HDP_CREATED_DATE');
			$csvRowData[] = Text::_('HDP_MODIFIED_DATE');
			$csvRowData[] = Text::_('HDP_STATUS');
			$csvRowData[] = Text::_('HDP_PRIORITY');

			if ($showStaff)
			{
				$csvRowData[] = Text::_('HDP_STAFF');
			}

			$csvRowData[] = Text::_('HDP_ID');

			foreach ($rowFields as $rowField)
			{
				$csvRowData[] = $rowField->title;
			}

			fputcsv($output, $csvRowData, $csvDelimiter);

			foreach ($rows as $row)
			{
				$csvRowData   = [];
				$csvRowData[] = $row->subject;
				$csvRowData[] = $row->message;
				$csvRowData[] = $row->category_title;

				if ($row->username)
				{
					$csvRowData[] = $row->name . '(' . $row->username . ')';
				}
				else
				{
					$csvRowData[] = $row->name;
				}

				$csvRowData[] = $row->email;
				$csvRowData[] = HTMLHelper::_('date', $row->created_date, $config->date_format);
				$csvRowData[] = HTMLHelper::_('date', $row->modified_date, $config->date_format);
				$csvRowData[] = @$statusList[$row->status_id];
				$csvRowData[] = @$priorityList[$row->priority_id];

				if ($showStaff)
				{
					$csvRowData[] = $rowStaffs[$row->staff_id] ?? '';
				}

				$csvRowData[] = $row->id;

				foreach ($rowFields as $rowField)
				{
					$fieldId = $rowField->id;
					if (!empty($fieldValues[$row->id][$fieldId]))
					{
						$csvRowData[] = $fieldValues[$row->id][$fieldId];
					}
					else
					{
						$csvRowData[] = '';
					}
				}

				fputcsv($output, $csvRowData, $csvDelimiter);
			}

			fclose($output);
			$this->app->close();
		}
	}

	/**
	 * Download a ticket attachment
	 *
	 * @throws Exception
	 */
	public function delete_attachment(): void
	{
		Session::checkToken('GET');

		$user = $this->app->getIdentity();

		// Redirect to original page
		$return = base64_decode($this->input->getString('return'));

		if (!$return)
		{
			$return = Uri::base();
		}

		$this->setRedirect($return);

		if (!$user->id)
		{
			throw new Exception('You do not have permission to delete attachment', 403);
		}

		$ticketId  = $this->input->getInt('ticket_id', 0);
		$messageId = $this->input->getInt('message_id', 0);

		if (!$ticketId && !$messageId)
		{
			throw new Exception('Invalid Delete Attachment Link', 404);
		}

		/* @var DatabaseDriver $db */
		$db    = Factory::getContainer()->get('db');
		$query = $db->getQuery(true);

		if ($ticketId)
		{
			$query->select('*')
				->from('#__helpdeskpro_tickets')
				->where('id = ' . $ticketId);
		}
		else
		{
			$query->select('*')
				->from('#__helpdeskpro_messages')
				->where('id = ' . $messageId);
		}

		$db->setQuery($query);
		$item = $db->loadObject();

		if (!$item)
		{
			throw new Exception('Invalid Delete Attachment Link', 404);
		}

		$role     = HelpdeskProHelper::getUserRole();
		$fileName = $this->input->getString('filename', '');

		if (in_array($role, ['user', 'staff']))
		{
			if ($item->user_id != $user->id)
			{
				throw new Exception('You cannot delete attachment uploaded by others', 403);
			}

			$attachments = explode('|', $item->attachments);

			if (!in_array($fileName, $attachments))
			{
				throw new Exception('You cannot delete attachment uploaded by others', 403);
			}
		}

		$filePath = JPATH_ROOT . '/media/com_helpdeskpro/attachments/' . $fileName;

		if (file_exists($filePath))
		{
			File::delete($filePath);

			$this->setMessage(Text::_('HDP_ATTACHMENT_DELETED'));
		}
		else
		{
			$this->setMessage(Text::_('HDP_FILE_NOT_EXIST'), 'warning');
		}
	}

	/**
	 * Download a ticket attachment
	 *
	 * @throws Exception
	 */
	public function download_attachment()
	{
		$user   = $this->app->getIdentity();
		$config = HelpdeskProHelper::getConfig();

		if (!$config->allow_public_user_submit_ticket && !$user->id)
		{
			throw new Exception('You do not have permission to download attachment');
		}

		$fileName         = $this->input->getString('filename', '');
		$originalFilename = $this->input->getString('original_filename', '');
		$originalFilename = File::makeSafe($originalFilename);
		$filePath         = JPATH_ROOT . '/media/com_helpdeskpro/attachments/' . $fileName;

		if (file_exists($filePath))
		{
			$cleanFilePath = Path::clean($filePath, '/');
			$cleanRealPath = Path::clean(realpath($filePath), '/');

			if ($cleanRealPath != $cleanFilePath)
			{
				throw new RuntimeException('Invalid File Path');
			}

			$fileExt = HelpdeskProHelper::getFileExt($fileName);

			$imageFileTypes = ['gif', 'jpg', 'jpeg', 'png', 'bmp', 'pdf'];

			if (in_array(strtolower($fileExt), $imageFileTypes))
			{
				$inline = true;
			}
			else
			{
				$inline = false;
			}

			$this->processDownloadFile(
				$filePath,
				$originalFilename,
				$inline
			);
		}
		else
		{
			$this->setRedirect('index.php?option=com_helpdeskpro&view=tickets');
			$this->setMessage(Text::_('HDP_FILE_NOT_EXIST'), 'warning');
		}
	}

	/**
	 * Validate captcha
	 *
	 * @return bool
	 */
	protected function validateCaptcha()
	{
		/* @var $app CMSApplication */
		$app = $this->app;

		$captchaPlugin = $app->getParams()->get('captcha', $app->get('captcha'));

		if (!$captchaPlugin)
		{
			// Hardcode to recaptcha, reduce support request
			$captchaPlugin = 'recaptcha';
		}

		$plugin = PluginHelper::getPlugin('captcha', $captchaPlugin);

		$captchaValid = true;

		if ($plugin)
		{
			try
			{
				$captchaValid = Captcha::getInstance($captchaPlugin)->checkAnswer(
					$this->input->post->get(
						'dynamic_recaptcha_1',
						'',
						'string'
					)
				);
			}
			catch (Exception $e)
			{
				// Captcha is not checked, return false
				$captchaValid = false;
			}
		}

		return $captchaValid;
	}

	/**
	 * Method to upload file uploaded by dropzone JS
	 */
	public function upload()
	{
		if (!Session::checkToken('get'))
		{
			return;
		}

		$config = HelpdeskProHelper::getConfig();

		if (!$config->enable_attachment)
		{
			return;
		}

		if (!$this->app->getIdentity()->id && !$config->allow_public_user_submit_ticket)
		{
			return;
		}

		$session = $this->container->session;

		$allowedFileTypes = explode('|', $config->allowed_file_types);

		for ($i = 0, $n = count($allowedFileTypes); $i < $n; $i++)
		{
			$allowedFileTypes[$i] = trim(strtoupper($allowedFileTypes[$i]));
		}

		$attachmentsPath = JPATH_ROOT . '/media/com_helpdeskpro/attachments';

		$attachment = $this->input->files->get('file', [], 'raw');

		$name = File::makeSafe($attachment['name']);

		if (empty($name))
		{
			return;
		}

		$fileExt = strtoupper(HelpdeskProHelper::getFileExt($name));

		if (!in_array($fileExt, $allowedFileTypes))
		{
			return;
		}

		$date = Factory::getDate(Factory::getApplication()->get('offset'));
		$path = $date->format('m', true) . $date->format('Y', true);

		if (!is_dir($attachmentsPath . '/' . $path))
		{
			Folder::create($attachmentsPath . '/' . $path);
		}

		$fileName = File::stripExt($name) . '_' . time() . '_' . uniqid() . '.' . $fileExt;

		$uploadedFiles              = $session->get('hdp_uploaded_files', '');
		$uploadedFilesOriginalNames = $session->get('hdp_uploaded_files_original_names', '');

		if ($uploadedFiles)
		{
			$uploadedFiles              = explode('|', $uploadedFiles);
			$uploadedFilesOriginalNames = explode('|', $uploadedFilesOriginalNames);
		}
		else
		{
			$uploadedFiles              = [];
			$uploadedFilesOriginalNames = [];
		}

		try
		{
			File::upload($attachment['tmp_name'], $attachmentsPath . '/' . $path . '/' . $fileName);

			$uploadedFiles[]              = $path . '/' . $fileName;
			$uploadedFilesOriginalNames[] = $name;
		}
		catch (FilesystemException $e)
		{
		}

		$session->set('hdp_uploaded_files', implode('|', $uploadedFiles));
		$session->set('hdp_uploaded_files_original_names', implode('|', $uploadedFilesOriginalNames));
	}

	/**
	 * Method to add some checks to prevent spams
	 *
	 * @throws Exception
	 */
	protected function antiSpam()
	{
		$config = HelpdeskProHelper::getConfig();

		$honeypotFieldName = $config->get('honeypot_fieldname', 'hdp_my_own_website_name');

		if ($this->input->getString($honeypotFieldName))
		{
			throw new Exception(Text::_('HDP_HONEYPOT_SPAM_DETECTED'), 403);
		}

		if ((int) $config->minimum_form_time > 0)
		{
			$startTime = $this->input->getInt(HelpdeskProHelper::getHashedFieldName(), 0);

			if ((time() - $startTime) < (int) $config->minimum_form_time)
			{
				throw new Exception(Text::_('HDP_FORM_SUBMIT_TOO_FAST'), 403);
			}
		}
	}

	/**
	 * Get show/hide status for custom fields when category changed
	 *
	 * @return void
	 */
	public function get_depend_fields_status_for_category()
	{
		$data       = $this->input->post->getData();
		$showFields = [];
		$hideFields = [];

		$fieldSuffix = HelpdeskProHelper::getFieldSuffix();
		$categoryId  = $this->input->getInt('category_id', 0);

		// Custom fields
		$rowFields = HelpdeskProHelper::getAllFields($fieldSuffix);
		$form      = new HDPForm($rowFields);

		$relation = HelpdeskProHelper::getFieldCategoryRelation();
		$form->prepareFormField($categoryId, $relation);
		$form->bind($data);
		$form->buildFieldsDependency();

		foreach ($form->getFields() as $field)
		{
			if ($field->visible)
			{
				$showFields[] = 'field_' . $field->id;
			}
			else
			{
				$hideFields[] = 'field_' . $field->id;
			}
		}

		echo json_encode(['show_fields' => implode(',', $showFields), 'hide_fields' => implode(',', $hideFields)]);

		$this->app->close();
	}

	/**
	 * Get depend fields status to show/hide custom fields based on selected options
	 *
	 * @return void
	 */
	public function get_depend_fields_status()
	{
		/* @var DatabaseDriver $db */
		$db      = Factory::getContainer()->get('db');
		$fieldId = $this->input->get('field_id', 'int');

		$hiddenFields = [];

		//Get list of dependency fields
		$allFieldIds = HelpdeskProHelper::getAllDependencyFields($fieldId);

		//Get list of dependency fields
		$languageSuffix = HelpdeskProHelper::getFieldSuffix();
		$query          = $db->getQuery(true)
			->select('*')
			->from('#__helpdeskpro_fields')
			->whereIn('id', $allFieldIds)
			->where('published = 1')
			->order('ordering');

		if ($languageSuffix)
		{
			$query->select($db->quoteName('depend_on_options' . $languageSuffix, 'depend_on_options'));
		}

		$db->setQuery($query);
		$rowFields = $db->loadObjectList();

		$masterFields = [];
		$fieldsAssoc  = [];

		foreach ($rowFields as $rowField)
		{
			if ($rowField->depend_on_field_id)
			{
				$masterFields[] = $rowField->depend_on_field_id;
			}

			$fieldsAssoc[$rowField->id] = $rowField;
		}

		$masterFields = array_unique($masterFields);

		if (count($masterFields))
		{
			foreach ($rowFields as $rowField)
			{
				if ($rowField->depend_on_field_id && isset($fieldsAssoc[$rowField->depend_on_field_id]))
				{
					// If master field is hided, then children field should be hided, too
					if (in_array($rowField->depend_on_field_id, $hiddenFields))
					{
						$hiddenFields[] = $rowField->id;
					}
					else
					{
						$fieldName = $fieldsAssoc[$rowField->depend_on_field_id]->name;

						$masterFieldValues = $this->input->get($fieldName, '', 'none');

						if (is_array($masterFieldValues))
						{
							$selectedOptions = $masterFieldValues;
						}
						else
						{
							$selectedOptions = [$masterFieldValues];
						}

						if (is_string($rowField->depend_on_options) && is_array(
								json_decode($rowField->depend_on_options)
							))
						{
							$dependOnOptions = json_decode($rowField->depend_on_options);
						}
						else
						{
							$dependOnOptions = explode(',', $rowField->depend_on_options);
						}

						if (!count(array_intersect($selectedOptions, $dependOnOptions)))
						{
							$hiddenFields[] = $rowField->id;
						}
					}
				}
			}
		}

		$showFields = [];
		$hideFields = [];

		foreach ($rowFields as $rowField)
		{
			if (in_array($rowField->id, $hiddenFields))
			{
				$hideFields[] = 'field_' . $rowField->id;
			}
			else
			{
				$showFields[] = 'field_' . $rowField->id;
			}
		}

		echo json_encode(['show_fields' => implode(',', $showFields), 'hide_fields' => implode(',', $hideFields)]);

		$this->app->close();
	}

	/**
	 * Validate form fields
	 *
	 * @param   array  $data
	 *
	 * @return array
	 */
	protected function validateFormFields($data)
	{
		$errors     = [];
		$categoryId = $data['category_id'] ?? 0;

		// Custom fields
		$rowFields = HelpdeskProHelper::getAllFields();
		$form      = new HDPForm($rowFields);

		$relation = HelpdeskProHelper::getFieldCategoryRelation();
		$form->prepareFormField($categoryId, $relation);
		$form->bind($data);
		$form->buildFieldsDependency();

		foreach ($form->getFields() as $field)
		{
			if (!$field->visible || !$field->required)
			{
				continue;
			}

			$value = $field->value;

			// Validate required field
			if (is_null($value) || (is_string($value) && trim($value) === ''))
			{
				$errors[] = Text::sprintf('HDP_FIELD_NAME_IS_REQUIRED', $field->title);
			}
		}

		return $errors;
	}
}